#include <stdlib.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"

#include "xsmesaP.h"
#include "glx_log.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "vga.h"
#include "vgaPCI.h"
#include "mach64.h"
#include "xaa/xf86xaa.h"

#include "mach64glx.h"

/*
 * mach64BackToFront
 * Blit the visible rectangles from the back buffer to the screen
 * Can't really do this from a direct context - don't have the clip
 * info, and getting it would be a lot slower than just sending a
 * request to the server to do the blit there.
 */
static void mach64BackToFront( DrawablePtr drawable, mach64BufferPtr buf ) {
    RegionPtr	prgnClip;
    BoxPtr	pbox;
    int 	i, nbox;
    int 	xorg, yorg, pitch;
    DMALOCALS;

    if ( ( drawable->width != buf->width ) ||
	 ( drawable->height != buf->height ) ||
	 ( drawable->type != DRAWABLE_WINDOW ) )
    {
	mach64Error( "BackToFront(): bad drawable\n" );
	return;
    }

    prgnClip = &((WindowPtr)drawable)->clipList;
    pbox = REGION_RECTS( prgnClip );
    nbox = REGION_NUM_RECTS( prgnClip );
    if( !nbox ) {
	/* window is completely covered */
	return;
    }

    xorg = drawable->x;
    yorg = drawable->y;
    pitch = buf->pitch;

    /* add the blit commands to the dma buffer */
	DMAGETPTR( 10 + 10 * nbox );

	DMAOUTREG( MACH64REG_Z_CNTL, 0 );			/* no z buffering */ 
	DMAOUTREG( MACH64REG_SCALE_3D_CNTL, 0 );
	DMAOUTREG( MACH64REG_CLR_CMP_CNTL, 0 );			/* disable color compare */

	DMAOUTREG( MACH64REG_DP_SRC, 
		DP_bkgd_src_foreground | DP_frgd_src_foreground | DP_mono_src_1 );
	
	DMAOUTREG( MACH64REG_DP_PIX_WIDTH, 0x30333 );	/* all 15 bit */
	DMAOUTREG( MACH64REG_DP_WRITE_MASK, 0xffffffff );	/* write to all */
	DMAOUTREG( MACH64REG_DP_MIX, 0x00070003 );	/* bkground leave alone */
	DMAOUTREG( MACH64REG_DP_SRC, 0x300 );		/* blit */
		
	DMAOUTREG( MACH64REG_DST_OFF_PITCH, mach64glx.registers[MACH64REG_DST_OFF_PITCH>>2] );

	DMAOUTREG( MACH64REG_SRC_OFF_PITCH, ( (mach64DB->pitch/8) << 22 ) | ( mach64DB->backBufferBlock->ofs >> 3 ) );
	
	DMAOUTREG( MACH64REG_GUI_TRAJ_CNTL, 3 );	/* left to right, top to bottom */ 

	DMAOUTREG( MACH64REG_SC_LEFT_RIGHT, 0 | ( 4095 << 16 ) );
	DMAOUTREG( MACH64REG_SC_TOP_BOTTOM, 0 | ( 4095 << 16 )  );

    /* xy bitblt each visible rectangle */
    for (i=nbox; i > 0; i--) {
    	int	x = pbox->x1 - xorg;
	int	y = pbox->y1 - yorg;
	int	w = pbox->x2 - pbox->x1;
	int	h = pbox->y2 - pbox->y1;
	DMAOUTREG( MACH64REG_SRC_WIDTH1, w );
	DMAOUTREG( MACH64REG_SRC_Y_X, (x<<16)|y );
	DMAOUTREG( MACH64REG_DST_X_Y, (pbox->y1<<16)|pbox->x1 );
	DMAOUTREG( MACH64REG_DST_WIDTH_HEIGHT, (h<<16)|w );

	pbox++;
    }

    DMAADVANCE();
}


/*
 * ClearBox
 * Add hardware commands to draw a filled box for the
 * debugging display.
 */
static void ClearBox( int x, int y, int w, int h, int r, int g, int b )
{
    	DMALOCALS;
	int	color;
	
    	DMAGETPTR( 32 );

	DMAOUTREG( MACH64REG_DST_OFF_PITCH, ( (mach64DB->pitch/8) << 22 ) | (mach64DB->backBufferBlock->ofs>>3) ) ;

	DMAOUTREG( MACH64REG_Z_CNTL, 0 );			/* no z buffering */ 
	DMAOUTREG( MACH64REG_SCALE_3D_CNTL, 0 );
	DMAOUTREG( MACH64REG_CLR_CMP_CNTL, 0 );			/* disable color compare */

	DMAOUTREG( MACH64REG_DP_SRC, 
		DP_bkgd_src_foreground | DP_frgd_src_foreground | DP_mono_src_1 );

	color = MACH64PACKCOLOR565(r,g,b);
	color = (color<<16) | color;
	
	DMAOUTREG( MACH64REG_DP_FRGD_CLR, color );
	DMAOUTREG( MACH64REG_DP_WRITE_MASK, 0xffffffff );	/* write to all */
	DMAOUTREG( MACH64REG_DP_MIX, 0x00070003 );			/* bkground leave alone */
	DMAOUTREG( MACH64REG_DP_SRC, 
		DP_bkgd_src_foreground | DP_frgd_src_foreground | DP_mono_src_1 );
	DMAOUTREG( MACH64REG_GUI_TRAJ_CNTL, 3 );			/* left to right, top to bottom */ 
	DMAOUTREG( MACH64REG_DST_X_Y, (y<<16)|x );
	DMAOUTREG( MACH64REG_DST_WIDTH_HEIGHT, (h<<16)|w );

	/* put it back where X expects it, until we get full dma going */
	DMAOUTREG( MACH64REG_DST_OFF_PITCH, ( (mach64glx.displayWidth/8) << 22 ) | 0 );


    	DMAADVANCE();
}


/*
 * performanceBoxes
 * Draw some small boxesin the corner of the buffer
 * based on some performance information
 */
static void mach64PerformanceBoxes( int is_direct )
{
    int		w, t;

    if ( !mach64glx.boxes || !mach64DB ) {
	return;
    }

    /* draw a box to show we are active, so if it is't seen
       it means that it is completely software based rendering  */
    /* draw a purple box if we are direct rendering */
    if ( is_direct ) {			/* purple = direct (client dma) rendering */
	ClearBox( 4, 4, 8, 8, 255, 0, 255 );
    } else if ( mach64glx.dmaDriver ) {	/* white = server dma rendering */
	ClearBox( 4, 4, 8, 8, 255, 255, 255 );
    } else {				/* grey = servery PDMA */
	ClearBox( 4, 4, 8, 8, 128, 128, 128 );
    }

    /* draw a red box if we had to wait for drawing to complete
       (software render or texture swap) */
    if ( mach64glx.c_drawWaits ) {
	ClearBox( 16, 4, 8, 8, 255, 0, 0 );
	mach64glx.c_drawWaits = 0;
    }

    /* draw a blue box if the register protection signal was hit */
    if ( mach64glx.c_signals ) {
	ClearBox( 28, 4, 8, 8, 0, 0, 255 );
	mach64glx.c_signals = 0;
    }

    /* draw a yellow box if textures were swapped */
    if ( mach64glx.c_textureSwaps ) {
	ClearBox( 40, 4, 8, 8, 255, 255, 0 );
	mach64glx.c_textureSwaps = 0;
    }


    /* draw a green box if we had to wait for dma to complete (full
       utilization) on the previous frame */
    if ( !mach64glx.hardwareWentIdle ) {
	ClearBox( 64, 4, 8, 8, 0, 255, 0 );
    }
    mach64glx.hardwareWentIdle = 0;

    /* show buffer utilization */
    if ( mach64glx.c_dmaFlush > 1 ) {
	/* draw a solid bar if we flushed more than one buffer */
	ClearBox( 4, 16, 252, 4, 255, 32, 32 );
    } else {
	/* draw bars to represent the utilization of primary buffer */
	ClearBox( 4, 16, 252, 4, 32, 32, 32 );
	t = dma_buffer->maxBufferDwords;
	w = 252 * dma_buffer->bufferDwords / t;
	if ( w < 1 ) {
	    w = 1;
	}
	ClearBox( 4, 16, w, 4, 196, 128, 128 );
    }
    mach64glx.c_dmaFlush = 0;
}


/*
 * Copy the back buffer to the front buffer.  If there's no back buffer
 * this is a no-op.  Only called in indirect contexts.
 */
void mach64ServerSwapBuffers( XSMesaBuffer b ) {
	mach64BufferPtr	buf;

	/* make sure mesa gives us everything */
	/* 3.1 leaves unsent verts without this, but there is probably
	a better internal call to get them sent... */
	if ( mach64Ctx && mach64Ctx->gl_ctx ) {
		glFlush();
	}

	mach64glx.swapBuffersCount++;

	ValidateGC( b->frontbuffer, b->cleargc );

	if ( !b->backimage ) {
		return;	/* when is this ever hit? */
	}
	
	buf = (mach64BufferPtr)b->backimage->devPriv;
	if ( !VALID_MACH64_BUFFER( buf ) ) {
		mach64Error( "BackToFront(): invalid back buffer\n" );
		return;
	}
	
	if ( buf->backBufferBlock ) {
		/* diagnostic drawing tools */
		mach64PerformanceBoxes( 0 );

		/* hardware accelerated back to front blit */
		mach64BackToFront( (DrawablePtr)b->frontbuffer,buf );

		/* make sure all dma is going to get executed */
		/* if everything has gone well, this will be the only
		   flush each frame */
		mach64DmaFlush();
	} else {
		/* software blit ... */
	    	/* Use backimage's dimension (not buffer's) */
		(*b->cleargc->ops->PutImage)( (DrawablePtr)b->frontbuffer,
					      b->cleargc,
					      b->frontbuffer->depth,
					      0, 0,
					      b->backimage->width,
					      b->backimage->height,
					      0, ZPixmap, b->backimage->data );
	}

	/* report performance counters */
	mach64Msg( 9, "swapBuffers: c_triangles:%i  c_setup:%i c_textures:%i\n",
	       mach64glx.c_triangles, 
	       mach64glx.c_setupPointers, mach64glx.c_textureSwaps );

	mach64glx.c_triangles = 0;
	mach64glx.c_setupPointers = 0;

	mach64Msg( 9, "---------------------------------------------------------\n" );
}
