
/* this file deals wilth allowing software mesa to interact with our buffers */

#include <stdlib.h>
#include <string.h>
#include "mesaglx/context.h"
#include "mesaglx/types.h"
#include "types.h"

#include "mach64glx.h"


/*
 * Return the address of the Z-buffer value for window coordinate (x,y):
 */
#define Z_SETUP							\
  GLdepth *zbstart = (GLdepth *)mach64DB->depthBuffer;	\
  GLint zbpitch = mach64DB->pitch; \
  GLint zbheight = mach64DB->height;
#define Z_ADDRESS( X, Y )  \
  (zbstart + zbpitch * (zbheight - (Y) - 1) + (X))


/**********************************************************************/
/*****                   Depth Testing Functions                  *****/
/**********************************************************************/


/*
 * Depth test horizontal spans of fragments.  These functions are called
 * via ctx->Driver.depth_test_span only.
 *
 * Input:  n - number of pixels in the span
 *         x, y - location of leftmost pixel in span in window coords
 *         z - array [n] of integer depth values
 * In/Out:  mask - array [n] of flags (1=draw pixel, 0=don't draw) 
 * Return:  number of pixels which passed depth test
 */


/*
 * glDepthFunc( any ) and glDepthMask( GL_TRUE or GL_FALSE ).
 */
static GLuint depth_test_span_generic( GLcontext* ctx,
                                   GLuint n, GLint x, GLint y,
                                   const GLdepth z[],
                                   GLubyte mask[] )
{
   Z_SETUP
   GLdepth *zptr = Z_ADDRESS( x, y );
   GLubyte *m = mask;
   GLuint i;
   GLuint passed = 0;

  mach64DmaFinish();

   /* switch cases ordered from most frequent to less frequent */
   switch (ctx->Depth.Func) {
      case GL_LESS:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] < *zptr) {
		     /* pass */
		     *zptr = z[i];
		     passed++;
		  }
		  else {
		     /* fail */
		     *m = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] < *zptr) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_LEQUAL:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] <= *zptr) {
		     *zptr = z[i];
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] <= *zptr) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_GEQUAL:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] >= *zptr) {
		     *zptr = z[i];
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] >= *zptr) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_GREATER:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] > *zptr) {
		     *zptr = z[i];
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] > *zptr) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_NOTEQUAL:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] != *zptr) {
		     *zptr = z[i];
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] != *zptr) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_EQUAL:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] == *zptr) {
		     *zptr = z[i];
		     passed++;
		  }
		  else {
		     *m =0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] == *zptr) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m =0;
		  }
	       }
	    }
	 }
	 break;
      case GL_ALWAYS:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  *zptr = z[i];
		  passed++;
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer or mask */
	    passed = n;
	 }
	 break;
      case GL_NEVER:
	 for (i=0;i<n;i++) {
	    mask[i] = 0;
	 }
	 break;
      default:
         gl_problem(ctx, "Bad depth func in gl_depth_test_span_generic");
   } /*switch*/

   return passed;
}


/*
 * Depth test an array of randomly positioned fragments.
 */

/*
 * glDepthFunc( any ) and glDepthMask( GL_TRUE or GL_FALSE ).
 */
static void depth_test_pixels_generic( GLcontext* ctx,
                                   GLuint n, const GLint x[], const GLint y[],
                                   const GLdepth z[], GLubyte mask[] )
{
   Z_SETUP
   register GLdepth *zptr;
   register GLuint i;

  mach64DmaFinish();

   /* switch cases ordered from most frequent to less frequent */
   switch (ctx->Depth.Func) {
      case GL_LESS:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] < *zptr) {
		     /* pass */
		     *zptr = z[i];
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] < *zptr) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_LEQUAL:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] <= *zptr) {
		     /* pass */
		     *zptr = z[i];
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] <= *zptr) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_GEQUAL:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] >= *zptr) {
		     /* pass */
		     *zptr = z[i];
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] >= *zptr) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_GREATER:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] > *zptr) {
		     /* pass */
		     *zptr = z[i];
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] > *zptr) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_NOTEQUAL:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] != *zptr) {
		     /* pass */
		     *zptr = z[i];
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] != *zptr) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_EQUAL:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] == *zptr) {
		     /* pass */
		     *zptr = z[i];
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] == *zptr) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_ALWAYS:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  *zptr = z[i];
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer or mask */
	 }
	 break;
      case GL_NEVER:
	 /* depth test never passes */
	 for (i=0;i<n;i++) {
	    mask[i] = 0;
	 }
	 break;
      default:
         gl_problem(ctx, "Bad depth func in gl_depth_test_pixels_generic");
   } /*switch*/
}

/**********************************************************************/
/*****                      Read Depth Buffer                     *****/
/**********************************************************************/


/*
 * Return a span of depth values from the depth buffer as floats in [0,1].
 * This function is only called through Driver.read_depth_span_float()
 * Input:  n - how many pixels
 *         x,y - location of first pixel
 * Output:  depth - the array of depth values
 */
static void read_depth_span_float( GLcontext* ctx,
                               GLuint n, GLint x, GLint y, GLfloat depth[] )
{
   Z_SETUP
   GLdepth *zptr;
   GLfloat scale;
   GLuint i;

  mach64DmaFinish();

   scale = 1.0F / DEPTH_SCALE;

   if ( mach64DB->depthBuffer ) {
      zptr = Z_ADDRESS( x, y );
      for (i=0;i<n;i++) {
	 depth[i] = (GLfloat) zptr[i] * scale;
      }
   }
   else {
      for (i=0;i<n;i++) {
	 depth[i] = 0.0F;
      }
   }
}


/*
 * Return a span of depth values from the depth buffer as integers in
 * [0,MAX_DEPTH].
 * This function is only called through Driver.read_depth_span_int()
 * Input:  n - how many pixels
 *         x,y - location of first pixel
 * Output:  depth - the array of depth values
 */
static void read_depth_span_int( GLcontext* ctx,
                             GLuint n, GLint x, GLint y, GLdepth depth[] )
{
   Z_SETUP
   
   mach64DmaFinish();

   if ( mach64DB->depthBuffer ) {
      GLdepth *zptr = Z_ADDRESS( x, y );
      MEMCPY( depth, zptr, n * sizeof(GLdepth) );
   }
   else {
      GLuint i;
      for (i=0;i<n;i++) {
	 depth[i] = 0;
      }
   }
}

//======================================================

static void (*xsmWriteRGBASpan)( const GLcontext *ctx,
				 GLuint n, GLint x, GLint y,
				 CONST GLubyte rgba[][4], 
				 const GLubyte mask[] );
static void (*xsmWriteRGBSpan)( const GLcontext *ctx,
				GLuint n, GLint x, GLint y,
				CONST GLubyte rgb[][3], const GLubyte mask[] );
static void (*xsmWriteMonoRGBASpan)(const GLcontext *ctx, GLuint n, 
				    GLint x, GLint y,const GLubyte mask[] );
static void (*xsmWriteRGBAPixels)(const GLcontext *ctx,
				  GLuint n, const GLint x[], const GLint y[],
				  CONST GLubyte rgba[][4], 
				  const GLubyte mask[] );
static void (*xsmWriteMonoRGBAPixels)(const GLcontext *ctx,GLuint n, 
				      const GLint x[], const GLint y[],
				      const GLubyte mask[] );
static void (*xsmReadRGBASpan)(const GLcontext *ctx, GLuint n, 
			       GLint x, GLint y,GLubyte rgba[][4] );
static void (*xsmReadRGBAPixels)(const GLcontext *ctx,GLuint n, 
				 const GLint x[], const GLint y[],
				 GLubyte rgba[][4], const GLubyte mask[] );

/* Wrapper functions */

static void WriteRGBASpan( const GLcontext *ctx,
			   GLuint n, GLint x, GLint y,
			   CONST GLubyte rgba[][4], 
			   const GLubyte mask[] )
{
  mach64DmaFinish();
  (*xsmWriteRGBASpan)(ctx,n,x,y,rgba,mask);
}

static void WriteRGBSpan( const GLcontext *ctx,
			  GLuint n, GLint x, GLint y,
			  CONST GLubyte rgb[][3], const GLubyte mask[] )
{
  mach64DmaFinish();
  (*xsmWriteRGBSpan)(ctx,n,x,y,rgb,mask);
}

static void WriteMonoRGBASpan(const GLcontext *ctx, GLuint n, 
			      GLint x, GLint y,const GLubyte mask[] )
{
  mach64DmaFinish();
  (*xsmWriteMonoRGBASpan)(ctx,n,x,y,mask);
}

static void WriteRGBAPixels(const GLcontext *ctx,
			    GLuint n, const GLint x[], const GLint y[],
			    CONST GLubyte rgba[][4], 
			    const GLubyte mask[] )
{
  mach64DmaFinish();
  (*xsmWriteRGBAPixels)(ctx,n,x,y,rgba,mask);
}

static void WriteMonoRGBAPixels(const GLcontext *ctx,GLuint n, 
				const GLint x[], const GLint y[],
				const GLubyte mask[] )
{
  mach64DmaFinish();
  (*xsmWriteMonoRGBAPixels)(ctx,n,x,y,mask);
}

static void ReadRGBASpan(const GLcontext *ctx, GLuint n, 
			 GLint x, GLint y,GLubyte rgba[][4] )
{
  mach64DmaFinish();
  (*xsmReadRGBASpan)(ctx,n,x,y,rgba);
}

static void ReadRGBAPixels(const GLcontext *ctx,GLuint n, 
			   const GLint x[], const GLint y[],
			   GLubyte rgba[][4], const GLubyte mask[] )
{
  mach64DmaFinish();
  (*xsmReadRGBAPixels)(ctx,n,x,y,rgba,mask);
}

#define WRAP(x) 				\
    xsm ## x = ctx->Driver.x;			\
    ctx->Driver.x = x

void mach64DDInitSpans( GLcontext *ctx )
{
   /* Need another hook from mesa to synchronize hardware before
    * write-pixels and read-spans/pixels.
    */ 
   WRAP(WriteRGBASpan);
   WRAP(WriteRGBSpan);
   WRAP(WriteMonoRGBASpan);
   WRAP(WriteRGBAPixels);
   WRAP(WriteMonoRGBAPixels);
   WRAP(ReadRGBASpan);
   WRAP(ReadRGBAPixels);
   
 	ctx->Driver.ReadDepthSpanFloat = read_depth_span_float;
	ctx->Driver.ReadDepthSpanInt = read_depth_span_int;
	ctx->Driver.DepthTestSpan = depth_test_span_generic;
	ctx->Driver.DepthTestPixels = depth_test_pixels_generic;
}
