/* $Id: mach64glx.c,v 1.4 1999/12/12 10:44:17 johnc Exp $ */

/*
 * GLX Hardware Device Driver for ATI Rage Pro
 * Copyright (C) 1999 Gareth Hughes
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on MGA driver: mgaglx.c 1.6
 *
 *    Gareth Hughes <garethh@bell-labs.com>
 */

#include <stdlib.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"

#include "xsmesaP.h"
#include "glx_log.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "vga.h"
#include "vgaPCI.h"
#include "mach64.h"
#include "xaa/xf86xaa.h"

#include "mach64glx.h"

/* our collected globals */
mach64Glx_t		mach64glx;
mach64BufferPtr		mach64DB = NULL;
mach64ContextPtr	mach64Ctx = NULL;

void (*mach64GLXSwapBuffers)(XSMesaBuffer b) = mach64ServerSwapBuffers;

/* are these for the multi-hardware glx support? */
GLboolean (*hwInitGLX)() = mach64InitGLX;
GLboolean (*hwInitVisuals)( VisualPtr *, DepthPtr *, int *, int *,
			    int *, VisualID *, unsigned long, int ) = NULL;

int usec( void )
{
    struct timeval	tv;
    struct timezone	tz;

    gettimeofday( &tv, &tz );

    return (tv.tv_sec & 2047) * 1000000 + tv.tv_usec;
}


void mach64GLXCreateDepthBuffer( GLcontext* ctx )
{
    XSMesaContext	xsmesa = (XSMesaContext) ctx->DriverCtx;
    mach64BufferPtr	buf;

    mach64Msg( 1, "mach64GLXCreateDepthBuffer()\n" );


	buf = (mach64BufferPtr)xsmesa->xsm_buffer->backimage->devPriv;

 	buf->depthBufferBlock = mmAllocMem( cardHeap, buf->pitch * buf->height * 2, 7, 0 );
	if ( buf->depthBufferBlock == 0 ) {
		/* put it in main memory and don't hw accel */
		mach64Msg( 1, "Depth buffer forced to system memory.\n" );
		buf->depthBuffer = malloc( buf->pitch * buf->height * 2 );
		if ( !buf->depthBuffer ) {
			FatalError( "Malloc for depth huffer failed" );
		}
	} else {
		buf->depthBuffer = (unsigned char *) mach64glx.linearBase +
			buf->depthBufferBlock->ofs;
		if ( mach64glx.logLevel >= 1 )  {
		    mmDumpMemInfo( cardHeap );
		}
	}
}

/*
 * mach64GLXDestroyImage
 */
void mach64GLXDestroyImage( GLXImage* image ) {
	mach64BufferPtr	buf;

	mach64Msg( 1, "mach64GLXDestroyImage( %p )\n", image->devPriv );

	buf = (mach64BufferPtr)image->devPriv;

	/* free memory either in card space or VM */
	if ( buf->backBufferBlock ) {
		mmFreeMem( buf->backBufferBlock );
	} else {
		free( buf->backBuffer );
	}

	if ( buf->depthBufferBlock ) {
		mmFreeMem( buf->depthBufferBlock );
	} else {
		free( buf->depthBuffer );
	}

	if ( mach64glx.logLevel >= 1 )  {
	    mmDumpMemInfo( cardHeap );
	}

	xfree( image );
}

/*
 * mach64GLXCreateImage
 *
 * This will always succeed, but if the buffer couldn't be allocated in
 * card memory, HW accel won't be usable.
 */
GLXImage *mach64GLXCreateImage( WindowPtr pwindow, int depth,
				int width, int height, GLXImage *old_image ){
	mach64BufferPtr	buf;
	GLXImage		*image;

	mach64Msg( 1, "mach64GLXCreateImage( %i, %i )\n", width, height );

	buf = (mach64BufferPtr)calloc(1,sizeof(mach64Buffer));
	if ( !buf ) {
		FatalError( "Malloc for buf failed\n" );
	}

	buf->magic = mach64BufferMagic;

	/* we may want to support 32 bit buffers in the future */

	buf->width = width;
	buf->height = height;
	buf->pitch = ( width + 31 ) & ~31;
	buf->backBufferBlock = mmAllocMem( cardHeap, buf->pitch * height * 2, 7, 0 );

	if ( buf->backBufferBlock == 0 ) {
		/* put it in main memory and don't hw accel */
		mach64Msg( 1, "Back buffer forced to system memory.\n" );
		buf->backBuffer = malloc( buf->pitch * height * 2 );
		if ( !buf->backBuffer ) {
			FatalError( "Malloc for back buffer failed" );
		}
	} else {
		buf->backBuffer = (unsigned char *) mach64glx.linearBase +
			buf->backBufferBlock->ofs;
	}

	image = (GLXImage *) xalloc( sizeof(GLXImage) );
	if ( !image ) {
		FatalError( "Malloc for back ximage failed" );
	}
	image->pwin = pwindow;
	image->width = width;
	image->height = height;
	image->bits_per_pixel = depth;
	image->data = buf->backBuffer;
	image->bytes_per_line = buf->pitch * 2;
	image->devPriv = buf;

	/* How can I draw XImage with any line pad? */
	/* e.g. bytes_per_line != PixmapBytePad */
	/* My work around: adjust width instead */
	image->width = buf->pitch;

	if ( mach64glx.logLevel >= 1 )  {
	    mmDumpMemInfo( cardHeap );
	}

	return image;
}


/*
 * DoMakeCurrent
 * Changing either the context or the buffer will force this.
 */
static int DoMakeCurrent( XSMesaContext c, XSMesaBuffer b )
{
    mach64ContextPtr	ctx, oldctx = mach64Ctx;
    mach64BufferPtr	buf, oldbuf = mach64DB;

    mach64Msg( 10, "DoMakeCurrent( %p, %p )\n", c, b );

    mach64Ctx = NULL;
    mach64DB = NULL;

    if ( c )
    {
	ctx = c->hw_ctx;
	if ( !VALID_MACH64_CONTEXT( ctx ) ) {
	    return -1;
	}
    }
    else
    {
	ctx = NULL;
    }

    if ( b && ( b->buffer == XIMAGE ) )
    {
	buf = (mach64BufferPtr) b->backimage->devPriv;
	if ( buf->magic != mach64BufferMagic ) {
		FatalError( "DoMakeCurrent: != mach64BufferMagic" );
	}
    }
    else
    {
	buf = NULL;
    }

    if ( !ctx && buf ) {
	return -1;
    }
    if ( !ctx ) {
	return 0;
    }

    ctx->DB = buf;

    /* valid conditions to render with */
    mach64Ctx = ctx;
    mach64DB = ctx->DB;

    /* Even this is overkill: */
    if ( ( oldctx != ctx ) || ( oldbuf != mach64DB ) )
    {
	mach64Ctx->new_state |= MACH64_NEW_CONTEXT;

	if ( MESA_VERBOSE & VERBOSE_DRIVER ) {
	    fprintf( stderr, "needEnter3D in DoMakeCurrent\n" );
	}
    }

    return 0;
}


/*
 * mach64GLXMakeCurrent
 * Note that GLX calls this every batch of commands,
 * so it will show up several times a frame even if
 * the client application isn't issuing makeCurrent calls.
 */
GLboolean mach64GLXMakeCurrent( XSMesaContext c )
{
    if ( c == XSMesa ) {
	return ( GL_TRUE );
    }

    /* mesa 3.1 will sometimes leave unflushed vertexes */
    if ( XSMesa ) {
	glFlush();
    }

    mach64Msg( 10, "mach64GLXMakeCurrent( %p )\n", c );
    XSMesa = c;
    if ( c )
    {
	DoMakeCurrent( c, c->xsm_buffer );
	gl_make_current( c->gl_ctx,
			 ( c->xsm_buffer ) ? c->xsm_buffer->gl_buffer : NULL );
    }
    else
    {
	/* detach */
	DoMakeCurrent( NULL, NULL );
	gl_make_current( NULL, NULL );
    }

    return GL_TRUE;
}

/*
 * Bind buffer b to context c and make c the current rendering context.
 */
GLboolean mach64GLXBindBuffer( XSMesaContext c, XSMesaBuffer b )
{
    mach64Msg( 10, "mach64GLXBindBuffer( %p, %p )\n", c, b );

    DoMakeCurrent( c, b );
    return XSMesaBindBuffer( c, b );
}


mach64ContextPtr mach64CreateContext( GLcontext *ctx )
{
    mach64ContextPtr	c;

    c = (mach64ContextPtr) calloc( 1, sizeof(mach64Context) );
    if ( !c ) {
	return NULL;
    }

    c->magic = mach64ContextMagic;
    c->gl_ctx = ctx;
#if 0 /* FIXME: GH */
    c->regDWGCTL = ( DC_clipdis_disable | (0xC << DC_bop_SHIFT) |
		     DC_shftzero_enable | DC_zmode_nozcmp |
		     DC_atype_zi );

    c->regALPHACTRL = AC_src_one | AC_dst_zero | AC_amode_FCOL |
	AC_astipple_disable | AC_aten_disable | AC_atmode_noacmp |
	AC_alphasel_fromtex;
#endif

    c->renderindex = -1;		/* impossible value */
    c->new_state = ~0;
    c->reg_dirty = ~0;

    /* XXX Fix me: callback not registered when main VB is created.
     */
    if ( ctx->VB ) {
#if 0 /* FIXME: GH */
	mach64DDRegisterVB( ctx->VB );
#endif
    } else {
	fprintf(stderr, "**** Didn't create vb driver data\n");
    }

    if ( ctx->NrPipelineStages ) {
#if 0 /* FIXME: GH */
	ctx->NrPipelineStages =
	    mach64DDRegisterPipelineStages( ctx->PipelineStage,
					    ctx->PipelineStage,
					    ctx->NrPipelineStages );
#endif
    }

    if ( !getenv("MGA_NO_FAST_PATH") )
    {
#if 0 /* FIXME: GH */
	ctx->Driver.BuildPrecalcPipeline = mach64DDBuildPrecalcPipeline;

	/* This isn't quite finished.  It's also not very fast yet.
	 */
	if ( getenv("MGA_FAST_IMMEDIATE") ) {
	    ctx->Driver.BuildEltPipeline = mach64DDBuildImmediatePipeline;
	}
#endif
    } else {
  	mach64Msg( 1, "enabling MGA_NO_FAST_PATH" );
    }

#if 0 /* FIXME: GH */
    mach64DDRenderInit();
    mach64DDTrifuncInit();
    mach64DDFastPathInit();
    mach64DDSetupInit();
    mach64DDExtensionsInit( ctx );

    mach64InitFogAttrib( &c->Fog );
#endif
    mach64Msg( 2, "mach64CreateContext(): successful.\n" );
    return c;
}



/*
 * Create a new XSMesaContext.
 * Input:  v - XSMesaVisual
 *         share_list - another XSMesaContext with which to share display
 *                      lists or NULL if no sharing is wanted.
 * Return:  an XSMesaContext or NULL if error.
 */
XSMesaContext mach64GLXCreateContext( XSMesaVisual v,
				      XSMesaContext share_list )
{
    XSMesaContext	c;

    mach64Msg( 0, "### Creating new xsmesa context for Rage Pro...\n" );

    c = (XSMesaContext) calloc( 1, sizeof(struct xsmesa_context) );
    if ( ! c ) {
        return NULL;
    }

    c->gl_ctx = gl_create_context( v->gl_visual,
				   ( share_list ) ? share_list->gl_ctx : NULL,
				   (void *) c,
				   GL_TRUE /* direct rendering */ );
    if ( !c->gl_ctx ) {
        free( c );
        return NULL;
    }

    c->xsm_visual = v;
    c->xsm_buffer = NULL;       /* set later by XSMesaMakeCurrent */
    c->pixelformat = v->dithered_pf;    /* Dithering is enabled by default */
    c->hw_ctx = (void *) mach64CreateContext( c->gl_ctx );
    if ( !c->hw_ctx ) {
	mach64Error( "Cannot create hardware specific context.\n" );
	return NULL;
    }
    ((mach64ContextPtr)(c->hw_ctx))->refcount++;
    c->gl_ctx->Driver.UpdateState = mach64_setup_DD_pointers;

    mach64Msg( 1, "mach64GLXCreateContext succeeded: %p\n", c );

    return c;
}


int mach64DestroyContext( mach64ContextPtr ctx )
{
    if ( !ctx ) {
	return 0;
    }
    if ( ctx->magic != mach64ContextMagic ) {
	return -1;
    }
    if ( --(ctx->refcount) > 0 ) {
	return 0;
    }

#if 0 /* FIXME: GH */
    mach64DestroyContextTextures( ctx );
#endif

    if ( ctx == mach64Ctx ) {
    	mach64Ctx = NULL;
    }
    ctx->magic = 0;
    free( ctx );
    mach64Msg( 2, "mach64DestroyContext(): successfully destroyed.\n" );
    return 0;
}


void mach64GLXDestroyContext( XSMesaContext c )
{
    mach64Msg( 1, "mach64GLXDestroyContext( %p )\n", c );

    /* make sure all drawing is completed */
    mach64DmaFinish();

    if ( mach64DestroyContext( (mach64ContextPtr) c->hw_ctx ) != 0 ) {
	mach64Error( "mach64GLXDestroyContext(): mach64DestroyContext() failed!\n" );
    }

    XSMesaDestroyContext( c );
    if ( XSMesa == c ) {
	XSMesa = 0;
    }
}


/*
 * Local Variables:
 * mode: c
 * c-basic-offset: 4
 * End:
 */
