/*
 * GLX Hardware Device Driver for Intel i810
 * Copyright (C) 1999 Keith Whitwell
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * KEITH WHITWELL, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *
 */

/* $Id: i810lib.h,v 1.1 1999/12/13 12:47:45 keithw Exp $ */

#ifndef I810LIB_INC
#define I810LIB_INC

#include <stdio.h>

#include "i810common.h"
#include "i810buf.h"
#include "i810context.h"
#include "mm.h"
#include "i810log.h"
#include "i810dma.h"

#ifndef MESA31
#error "The I810 driver now requires Mesa 3.1 or higher"
#endif

extern struct i810_dest_buffer *i810DB;
extern i810ContextPtr i810Ctx;
extern int i810In3DContext;

typedef struct {
   /* logging stuff */
   GLuint logLevel;
   FILE *logFile;

   /* dma stuff */
   GLuint dmaDriver;	
   GLuint dmaSize;
   GLuint dmaAdr;
   GLuint cmdSize;

   /* bookkeeping for texture swaps */
   GLuint dma_buffer_age;
   GLuint current_texture_age;
        
   /* options */
   GLuint nullprims;		/* skip all primitive generation */
   GLuint boxes;		/* draw performance boxes */
   GLuint noFallback;		/* don't fall back to software */
   GLuint skipDma;		/* don't send anything to hardware */

   /* performance counters */	
   GLuint c_setupPointers;
   GLuint c_triangles;
   GLuint c_points;
   GLuint c_lines;
   GLuint c_drawWaits;
   GLuint c_textureSwaps;
   GLuint c_signals;
   GLuint c_dmaFlush;
   GLuint c_overflows;
	
   GLuint hardwareWentIdle;	/* cleared each swapbuffers, set if a
				   waitfordmacompletion ever exited
				   without having to wait */

   /* Primitive managment (like warpSerie) */
   GLuint prim_start;
   GLuint prim_dwords;

   /* card == dcache
    * sysmem == normal gart memory
    */
   memHeap_t *cardHeap;
   memHeap_t *sysmemHeap;
   unsigned char *sysmemVirtual;

   i810BatchBuffer	*dma_buffer;

} i810Glx_t;

extern i810Glx_t	i810glx;

int	i810_usec( void );
void i810LibInit();
void i810SoftReset();

int i810MakeCurrent(i810ContextPtr ctx, struct i810_dest_buffer *buf);
void i810LoadTexturePalette(unsigned short *pal, int start, int len);

#define I810PACKCOLOR1555(r,g,b,a) \
  ((((r) & 0xf8) << 7) | (((g) & 0xf8) << 2) | (((b) & 0xf8) >> 3) | \
    ((a) ? 0x8000 : 0))

#define I810PACKCOLOR565(r,g,b) \
  ((((r) & 0xf8) << 8) | (((g) & 0xfc) << 3) | (((b) & 0xf8) >> 3))

#define I810PACKCOLOR4444(r,g,b,a) \
  ((((a) & 0xf0) << 8) | (((r) & 0xf0) << 4) | ((g) & 0xf0) | ((b) >> 4))


#include "i810_3d_reg.h"

static __inline__ GLuint i810PackColor(GLubyte r, GLubyte g, 
				       GLubyte b, GLubyte a)
{
  switch (i810DB->Format) {
  case DV_PF_555:
    return I810PACKCOLOR1555(r,g,b,a);
  case DV_PF_565:
    return I810PACKCOLOR565(r,g,b);
  default:
    return 0;
  }
}




#endif
