
/*
 * GLX Server Extension
 * Copyright (C) 1996  Steven G. Parker  (sparker@cs.utah.edu)
 * Copyright (C) 1998, 1999  Terence Ripperda (ripperda@sgi.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * STEPHEN PARKER, TERENCE RIPPERDA, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR 
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, 
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "GL/gl.h"
#include <stdio.h>
#include <assert.h>
#include "glxcommon.h"
#include "glx_varray.h"

#ifdef DYNAMIC_MODULE
#include "glx_log.h"
#endif


#ifdef XFree86LOADER
#include "xf86_libc.h"
#endif

int GLX_data_size(GLenum data_type)
{
    switch (data_type)
    {
        case GL_BYTE:
        case GL_UNSIGNED_BYTE:
            return sizeof(char);
        case GL_SHORT:
        case GL_UNSIGNED_SHORT:
            return sizeof(short);
        case GL_INT:
        case GL_UNSIGNED_INT:
            return sizeof(int);
        case GL_FLOAT:
            return sizeof(float);
        case GL_DOUBLE:
            return sizeof(double);
        default:
            fprintf(stderr, "invalid data type to GLX_data_size: 0x%x!!\n",
                    data_type);
            return 0;
    }
}

/* XXX still missing many formats! */
int GLX_num_elements(GLenum format) {
    switch (format)
    {
        case GL_COLOR_INDEX:
        case GL_STENCIL_INDEX:
        case GL_DEPTH_COMPONENT:
        case GL_RED:
        case GL_GREEN:
        case GL_BLUE:
        case GL_ALPHA:
        case GL_LUMINANCE:
            return 1;
        case GL_LUMINANCE_ALPHA:
            return 2;
        case GL_RGB:
        case GL_BGR:
            return 3;
        case GL_RGBA:
        case GL_BGRA:
        case GL_ABGR_EXT:
            return 4;
        default:
            fprintf(stderr, "Unsupported format type to GLX_num_elements:");
            fprintf(stderr, " 0x%x!!\n", format);
            return 0;
    }
}


int GLX_light_size(GLenum pname)
{
    int n;
    switch (pname)
    {
        case GL_QUADRATIC_ATTENUATION:
        case GL_LINEAR_ATTENUATION:
        case GL_CONSTANT_ATTENUATION:
        case GL_SPOT_CUTOFF:
        case GL_SPOT_EXPONENT:
            n=1;
            break;
        case GL_SPOT_DIRECTION:
            n=3;
            break;
        case GL_AMBIENT:
        case GL_DIFFUSE:
        case GL_SPECULAR:
        case GL_POSITION:
            n=4;
            break;
        default:
            fprintf(stderr, "Invalid value to GLX_light_size\n");
            n=0;
    }
    return n;
}

int GLX_map1_size(GLenum target)
{
    int k;
    switch (target)
    {
        case GL_MAP1_TEXTURE_COORD_4:
        case GL_MAP1_COLOR_4:
        case GL_MAP1_VERTEX_4:
            k=4;
            break;
        case GL_MAP1_TEXTURE_COORD_1:
        case GL_MAP1_INDEX:
            k=1;
            break;
        case GL_MAP1_TEXTURE_COORD_2:
            k=2;
            break;
        case GL_MAP1_TEXTURE_COORD_3:
        case GL_MAP1_NORMAL:
        case GL_MAP1_VERTEX_3:
            k=3;
            break;
        default:
            fprintf(stderr, "Invalid value to GLX_map1_size\n");
            k=0;
    }
    return k;
}

int GLX_map2_size(GLenum target)
{
    int k;
    switch (target)
    {
        case GL_MAP2_TEXTURE_COORD_4:
        case GL_MAP2_COLOR_4:
        case GL_MAP2_VERTEX_4:
            k=4;
            break;
        case GL_MAP2_TEXTURE_COORD_1:
        case GL_MAP2_INDEX:
            k=1;
            break;
        case GL_MAP2_TEXTURE_COORD_2:
            k=2;
            break;
        case GL_MAP2_TEXTURE_COORD_3:
        case GL_MAP2_NORMAL:
        case GL_MAP2_VERTEX_3:
            k=3;
            break;
        default:
            fprintf(stderr, "Invalid value to GLX_map2_size: %x\n", target);
            k=0;
    }
    return k;
}

int GLX_material_size(GLenum pname)
{
    int n;
    switch (pname)
    {
        case GL_SHININESS:
            n=1;
            break;
        case GL_COLOR_INDEXES:
            n=3;
            break;
        case GL_AMBIENT:
        case GL_DIFFUSE:
        case GL_SPECULAR:
        case GL_EMISSION:
        case GL_AMBIENT_AND_DIFFUSE:
            n=4;
            break;
        default:
            fprintf(stderr, "Invalid value to GLX_material_size: %x\n", pname);
            n=0;
    }
    return n;
}

int GLX_get_size(GLenum pname)
{
    int n;
    switch (pname)
    {
        case GL_ACCUM_ALPHA_BITS:
        case GL_ACCUM_BLUE_BITS:
        case GL_ACCUM_GREEN_BITS:
        case GL_ACCUM_RED_BITS:
            n=1;
            break;
        case GL_ACCUM_CLEAR_VALUE:
            n=4;
            break;
        case GL_ALPHA_BIAS:
        case GL_ALPHA_BITS:
        case GL_ALPHA_SCALE:
        case GL_ALPHA_TEST:
        case GL_ALPHA_TEST_FUNC:
        case GL_ALPHA_TEST_REF:
        case GL_ATTRIB_STACK_DEPTH:
        case GL_AUTO_NORMAL:
        case GL_AUX_BUFFERS:
        case GL_BLEND:
        case GL_BLEND_DST:
        case GL_BLEND_SRC:
        case GL_BLEND_EQUATION_EXT:
            n=1;
            break;
        case GL_BLEND_COLOR_EXT:
            n=4;
            break;
        case GL_BLUE_BIAS:
        case GL_BLUE_BITS:
        case GL_BLUE_SCALE:
            n=1;
            break;
        case GL_CLIP_PLANE0:
        case GL_CLIP_PLANE1:
        case GL_CLIP_PLANE2:
        case GL_CLIP_PLANE3:
        case GL_CLIP_PLANE4:
        case GL_CLIP_PLANE5:
            n=1;
            break;
        case GL_COLOR_CLEAR_VALUE:
            n=4;
        case GL_COLOR_MATERIAL:
        case GL_COLOR_MATERIAL_FACE:
        case GL_COLOR_MATERIAL_PARAMETER:
            n=1;
            break;
        case GL_COLOR_WRITEMASK:
            n=4;
            break;
        case GL_CULL_FACE:
        case GL_CULL_FACE_MODE:
            n=1;
            break;
        case GL_CURRENT_COLOR:
            n=3;
            break;
        case GL_CURRENT_INDEX:
            n=1;
            break;
        case GL_CURRENT_NORMAL:
            n=3;
            break;
        case GL_CURRENT_RASTER_COLOR:
            n=4;
            break;
        case GL_CURRENT_RASTER_DISTANCE:
        case GL_CURRENT_RASTER_INDEX:
            n=1;
            break;
        case GL_CURRENT_RASTER_POSITION:
            n=4;
            break;
        case GL_CURRENT_RASTER_POSITION_VALID:
            n=1;
            break;
        case GL_CURRENT_RASTER_TEXTURE_COORDS:
        case GL_CURRENT_TEXTURE_COORDS:
            n=4;
            break;
        case GL_DEPTH_BIAS:
        case GL_DEPTH_BITS:
        case GL_DEPTH_CLEAR_VALUE:
        case GL_DEPTH_FUNC:
            n=1;
            break;
        case GL_DEPTH_RANGE:
            n=2;
            break;
        case GL_DEPTH_SCALE:
        case GL_DEPTH_TEST:
        case GL_DEPTH_WRITEMASK:
        case GL_DITHER:
        case GL_DOUBLEBUFFER:
        case GL_DRAW_BUFFER:
        case GL_EDGE_FLAG:
        case GL_FOG:
            n=1;
            break;
        case GL_FOG_COLOR:
            n=4;
            break;
        case GL_FOG_DENSITY:
        case GL_FOG_END:
        case GL_FOG_HINT:
        case GL_FOG_INDEX:
        case GL_FOG_MODE:
        case GL_FOG_START:
        case GL_FRONT_FACE:
        case GL_GREEN_BIAS:
        case GL_GREEN_BITS:
        case GL_GREEN_SCALE:
        case GL_INDEX_BITS:
        case GL_INDEX_CLEAR_VALUE:
        case GL_INDEX_MODE:
        case GL_INDEX_OFFSET:
        case GL_INDEX_SHIFT:
        case GL_INDEX_WRITEMASK:
        case GL_LIGHT0:
        case GL_LIGHT1:
        case GL_LIGHT2:
        case GL_LIGHT3:
        case GL_LIGHT4:
        case GL_LIGHT5:
        case GL_LIGHT6:
        case GL_LIGHT7:
        case GL_LIGHTING:
            n=1;
            break;
        case GL_LIGHT_MODEL_AMBIENT:
            n=4;
            break;
        case GL_LIGHT_MODEL_LOCAL_VIEWER:
        case GL_LIGHT_MODEL_TWO_SIDE:
        case GL_LINE_SMOOTH:
        case GL_LINE_SMOOTH_HINT:
        case GL_LINE_STIPPLE:
        case GL_LINE_STIPPLE_PATTERN:
        case GL_LINE_STIPPLE_REPEAT:
        case GL_LINE_WIDTH:
        case GL_LINE_WIDTH_GRANULARITY:
            n=1;
            break;
        case GL_LINE_WIDTH_RANGE:
            n=2;
            break;
        case GL_LIST_BASE:
        case GL_LIST_INDEX:
        case GL_LIST_MODE:
        case GL_LOGIC_OP:
        case GL_LOGIC_OP_MODE:
        case GL_MAP1_COLOR_4:
            n=1;
            break;
        case GL_MAP1_GRID_DOMAIN:
            n=2;
            break;
        case GL_MAP1_GRID_SEGMENTS:
        case GL_MAP1_INDEX:
        case GL_MAP1_NORMAL:
        case GL_MAP1_TEXTURE_COORD_1:
        case GL_MAP1_TEXTURE_COORD_2:
        case GL_MAP1_TEXTURE_COORD_3:
        case GL_MAP1_TEXTURE_COORD_4:
        case GL_MAP1_VERTEX_3:
        case GL_MAP1_VERTEX_4:
        case GL_MAP2_COLOR_4:
            n=1;
            break;
        case GL_MAP2_GRID_DOMAIN:
            n=4;
            break;
        case GL_MAP2_GRID_SEGMENTS:
            n=2;
            break;
        case GL_MAP2_INDEX:
        case GL_MAP2_NORMAL:
        case GL_MAP2_TEXTURE_COORD_1:
        case GL_MAP2_TEXTURE_COORD_2:
        case GL_MAP2_TEXTURE_COORD_3:
        case GL_MAP2_TEXTURE_COORD_4:
        case GL_MAP2_VERTEX_3:
        case GL_MAP2_VERTEX_4:
        case GL_MAP_COLOR:
        case GL_MAP_STENCIL:
        case GL_MATRIX_MODE:
        case GL_MAX_ATTRIB_STACK_DEPTH:
        case GL_MAX_CLIP_PLANES:
        case GL_MAX_EVAL_ORDER:
        case GL_MAX_LIGHTS:
        case GL_MAX_LIST_NESTING:
        case GL_MAX_MODELVIEW_STACK_DEPTH:
        case GL_MAX_NAME_STACK_DEPTH:
        case GL_MAX_PIXEL_MAP_TABLE:
        case GL_MAX_PROJECTION_STACK_DEPTH:
        case GL_MAX_TEXTURE_SIZE:
        case GL_MAX_TEXTURE_STACK_DEPTH:
            n=1;
            break;
        case GL_MAX_VIEWPORT_DIMS:
            n=2;
            break;
        case GL_MODELVIEW_MATRIX:
            n=16;
            break;
        case GL_MODELVIEW_STACK_DEPTH:
        case GL_NAME_STACK_DEPTH:
        case GL_NORMALIZE:
        case GL_PACK_ALIGNMENT:
        case GL_PACK_LSB_FIRST:
        case GL_PACK_ROW_LENGTH:
        case GL_PACK_SKIP_PIXELS:
        case GL_PACK_SKIP_ROWS:
        case GL_PACK_SWAP_BYTES:
        case GL_PERSPECTIVE_CORRECTION_HINT:
        case GL_PIXEL_MAP_A_TO_A_SIZE:
        case GL_PIXEL_MAP_B_TO_B_SIZE:
        case GL_PIXEL_MAP_G_TO_G_SIZE:
        case GL_PIXEL_MAP_I_TO_A_SIZE:
        case GL_PIXEL_MAP_I_TO_B_SIZE:
        case GL_PIXEL_MAP_I_TO_G_SIZE:
        case GL_PIXEL_MAP_I_TO_I_SIZE:
        case GL_PIXEL_MAP_I_TO_R_SIZE:
        case GL_PIXEL_MAP_R_TO_R_SIZE:
        case GL_PIXEL_MAP_S_TO_S_SIZE:
        case GL_POINT_SIZE:
        case GL_POINT_SIZE_GRANULARITY:
            n=1;
            break;
        case GL_POINT_SIZE_RANGE:
            n=2;
            break;
        case GL_POINT_SMOOTH:
        case GL_POINT_SMOOTH_HINT:
            n=1;
            break;
        case GL_POLYGON_MODE:
            n=2;
            break;
        case GL_POLYGON_SMOOTH:
        case GL_POLYGON_SMOOTH_HINT:
        case GL_POLYGON_STIPPLE:
            n=1;
            break;
        case GL_PROJECTION_MATRIX:
        case GL_PROJECTION_STACK_DEPTH:
            n=16;
            break;
        case GL_READ_BUFFER:
        case GL_RED_BIAS:
        case GL_RED_BITS:
        case GL_RED_SCALE:
        case GL_RENDER_MODE:
        case GL_RGBA_MODE:
            n=1;
            break;
        case GL_SCISSOR_BOX:
            n=4;
            break;
        case GL_SCISSOR_TEST:
        case GL_SHADE_MODEL:
        case GL_STENCIL_BITS:
        case GL_STENCIL_CLEAR_VALUE:
        case GL_STENCIL_FAIL:
        case GL_STENCIL_FUNC:
        case GL_STENCIL_PASS_DEPTH_FAIL:
        case GL_STENCIL_PASS_DEPTH_PASS:
        case GL_STENCIL_REF:
        case GL_STENCIL_TEST:
        case GL_STENCIL_VALUE_MASK:
        case GL_STENCIL_WRITEMASK:
        case GL_STEREO:
        case GL_SUBPIXEL_BITS:
        case GL_TEXTURE_1D:
        case GL_TEXTURE_2D:
            n=1;
            break;
        case GL_TEXTURE_ENV_COLOR:
            n=4;
            break;
        case GL_TEXTURE_ENV_MODE:
        case GL_TEXTURE_GEN_Q:
        case GL_TEXTURE_GEN_R:
        case GL_TEXTURE_GEN_S:
        case GL_TEXTURE_GEN_T:
            n=1;
            break;
        case GL_TEXTURE_MATRIX:
            n=16;
            break;
        case GL_TEXTURE_STACK_DEPTH:
        case GL_UNPACK_ALIGNMENT:
        case GL_UNPACK_LSB_FIRST:
        case GL_UNPACK_ROW_LENGTH:
        case GL_UNPACK_SKIP_PIXELS:
        case GL_UNPACK_SKIP_ROWS:
        case GL_UNPACK_SWAP_BYTES:
            n=1;
            break;
        case GL_VIEWPORT:
            n=4;
            break;
        case GL_ZOOM_X:
        case GL_ZOOM_Y:
            n=1;
            break;
        default:
            fprintf(stderr, "Invalid value to GLX_get_size\n");
            n=0;
            break;
    }
    return n;
}       

int GLX_lightmodel_size(GLenum pname)
{
    int n;
    switch (pname)
    {
        case GL_LIGHT_MODEL_LOCAL_VIEWER:
        case GL_LIGHT_MODEL_TWO_SIDE:
            n=1;
            break;
        case GL_LIGHT_MODEL_AMBIENT:
            n=4;
            break;
        default:
            fprintf(stderr, "Invalid value to GLX_lightmodel_size\n");
            n=0;
            break;
    }
    return n;
}

int GLX_tex_size(GLenum pname)
{
    int n;
    switch (pname)
    {
        case GL_TEXTURE_ENV_MODE:
            n=1;
            break;
        case GL_TEXTURE_ENV_COLOR:
            n=4;
            break;
        default:
            fprintf(stderr, "Invalid value to GLX_tex_size\n");
            n=0;
            break;
    }
    return n;
}

int GLX_texparameter_size(GLenum pname)
{
    int n;
    switch (pname)
    {
        case GL_TEXTURE_MIN_FILTER:
        case GL_TEXTURE_MAG_FILTER:
        case GL_TEXTURE_WRAP_S:
        case GL_TEXTURE_WRAP_T:
            n=1;
            break;
        case GL_TEXTURE_BORDER_COLOR:
            n=4;
            break;
        default:    
            fprintf(stderr, "Invalid value to GLX_texparameter_size\n");
            n=0;
            break;
    }
    return n;
}

int GLX_texlevelparameter_size(GLenum pname)
{
    int n;
    switch (pname)
    {
        case GL_TEXTURE_WIDTH:
        case GL_TEXTURE_HEIGHT:
        case GL_TEXTURE_COMPONENTS:
        case GL_TEXTURE_BORDER:
            n=1;
            break;
        case GL_TEXTURE_BORDER_COLOR:
            n=4;
            break;
        default:    
            fprintf(stderr, "Invalid value to GLX_texparameter_size\n");
            n=0;
            break;
    }
    return n;
}

int GLX_texgen_size(GLenum pname)
{
    int n;
    switch (pname)
    {
        case GL_TEXTURE_GEN_MODE:
            n=1;
            break;
        case GL_OBJECT_PLANE:
        case GL_EYE_PLANE:
            n=4;
            break;
        default:
            fprintf(stderr, "Invalid value to GLX_texgen_size\n");
            n=0;
            break;
    }
    return n;
}

int GLX_fog_size(GLenum pname)
{
    int n;
    switch (pname)
    {
        case GL_FOG_INDEX:
        case GL_FOG_DENSITY:
        case GL_FOG_START:
        case GL_FOG_END:
        case GL_FOG_MODE:
            n=1;
            break;
        case GL_FOG_COLOR:
            n=4;
            break;
        default:
            fprintf(stderr, "Invalid value to GLX_fog_size\n");
            n=0;
            break;
    }
    return n;
}

int GLX_list_size(GLint n, GLenum type)
{
    return n * GLX_data_size(type);
}

int GLX_list_pad(GLint n, GLenum type)
{
    int s=GLX_list_size(n, type);
    return s%4==0?0:4-s%4;
}

int GLX_texenv_size(GLenum pname)
{
    int k;
    switch (pname)
    {
        case GL_TEXTURE_ENV_MODE:
            k=1;
            break;
        case GL_TEXTURE_ENV_COLOR:
            k=4;
            break;
        default:
            fprintf(stderr, "Invalid value to GLX_texenv_size\n");
            k=0;
            break;
    }
    return k;
}

int GLX_texture_size(GLsizei width, GLsizei height,
                   GLenum format, GLenum type, GLenum target)
{
    /* proxy textures have no size */
    if ( (target == GL_PROXY_TEXTURE_1D) ||
         (target == GL_PROXY_TEXTURE_2D) ||
         (target == GL_PROXY_TEXTURE_3D) ||
         (target == GL_PROXY_TEXTURE_3D_EXT) )
             return 0;
    return GLX_image_size(width, height, format, type);
}

int GLX_image_size(GLsizei width, GLsizei height,
                   GLenum format, GLenum type)
{
    int s = GLX_num_elements(format);
    if (width < 0) return -1;
    if (height < 0) return -1;
    if (s < 0) return -1;
    switch (type)
    {
        case GL_BITMAP:
            width=(width+7)/8;
            break;
        case GL_BYTE:
        case GL_UNSIGNED_BYTE:
            s*=1;
            break;
        case GL_SHORT:
        case GL_UNSIGNED_SHORT:
            s*=2;
            break;
        case GL_INT:
        case GL_UNSIGNED_INT:
        case GL_FLOAT:
            s*=4;
            break;
        default:
            fprintf(stderr, "Invalid value to GLX_image_size - type: %x\n", type);
            s=0;
            break;
    }
    return s*width*height;
}

int GLX_texture_pad(GLsizei width, GLsizei height,
                  GLenum format, GLenum type, GLenum target)
{
    int s=GLX_texture_size(width, height, format, type, target);
    return s%4==0?0:4-s%4;
}

int GLX_image_pad(GLsizei width, GLsizei height,
                  GLenum format, GLenum type)
{
    int s=GLX_image_size(width, height, format, type);
    return s%4==0?0:4-s%4;
}


int GLX_vertex_data_size(struct array_info *arrays, int num_arrays) {
    int s = 0, i = 0;

    assert(arrays != NULL);
    for (i = 0; i < num_arrays; i++)
    {
        s += arrays[i].j * GLX_pad(GLX_data_size(arrays[i].data_type));
    }
    return s;
}
