/*
   rdesktop: A Remote Desktop Protocol client.
   Support for the Matrox "mtx" channel
   Copyright (C) 2006 Matrox Graphics Inc. 

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "rdesktop.h"
#include <sys/types.h>
#include <unistd.h>
#include <X11/Xlib.h>

#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "mtxrdpmsg.h"

#define MTX_MSG_INCOMPLETE -1
#define MTX_CHANNEL_NAME "MtxVC"

typedef BOOL(*handle_cmds_t) (const char *buf, size_t avail);

extern Display *g_display;

static VCHANNEL *mtx_channel;

static void mtx_send(const char *server_name);

static int 
mtx_get_server_ip( char server_name[128] )
{
    int fd = 0;
    struct ifreq* ifr;
    struct ifconf ifc;
    int numreqs = 30;
    int n;
    char name[128];
    int ret = -1;

    ifc.ifc_buf = NULL;

    fd = socket(PF_INET, SOCK_DGRAM, 0);
    if (fd < 0) {
        return ret;
    }

    for (;;) {
        ifc.ifc_len = sizeof(struct ifreq) * numreqs;
        ifc.ifc_buf = realloc(ifc.ifc_buf, ifc.ifc_len);

        if (ifc.ifc_buf == NULL){
            goto out;
        }

        if (ioctl(fd, SIOCGIFCONF, &ifc) < 0) {
            goto out;
        }

       if (ifc.ifc_len == sizeof(struct ifreq) * numreqs) {
           /* assume it overflowed and try again */
           numreqs += 10;
           continue;
        }
        break;
    }

    ifr = ifc.ifc_req;
    for (n = 0; n < ifc.ifc_len; n += sizeof(struct ifreq)) {
        name[sizeof(name)-1] = '\0';
        strncpy(name, (const char*)inet_ntoa(((struct sockaddr_in *) &ifr->ifr_addr)->sin_addr), sizeof(name)-1);
        if (strcmp("127.0.0.1",name)){ /* skip default loopback address */
            strcpy(server_name,name);
            ret = 0; /* we use the first address */
            goto out;
        }
        ifr++;
    }
out:
    if (ifc.ifc_buf){
        free(ifc.ifc_buf);
    }

    if (fd) {
        close(fd);
    }

    return ret;
}

/* Process one command of input from virtual channel */
static BOOL
mtx_process_cmd(const char *buf, size_t avail)
{
    if ( buf[0] == MTX_MSG_INIT )
    {
        mtx_send("ok");
        return 1;
    }
    else if ( buf[0] == MTX_MSG_GETCLIENTADDR )
    {
        char name[128];
        if ( mtx_get_server_ip(name) != 0 )
        {
            mtx_send("err");
        }
        else
        {
            char* display = XDisplayString(g_display);
            char* p = strchr(display,':');
            if (p)
            {
                strcat(name,p);
            }
            else
            {
                strcat(name,":0"); /* default to :0 */
            }
            mtx_send(name);
        }
        return 1;
    }
    else if ( buf[0] ==  MTX_MSG_GETALLOWEDHOST )
    {
        int nhosts,i;
        Bool state;
        XHostAddress * hosts = XListHosts(g_display, &nhosts, &state);
        char str[30];
        sprintf(str,"%d",nhosts);
        mtx_send(str);
        if (hosts)
        {
            for (i=0;i<nhosts;i++)
            {
                if (hosts[i].family==FamilyInternet && hosts[i].length==4 )
                {
                    mtx_send(inet_ntoa(*(struct in_addr *)hosts[i].address));
                }
                else
                {
                    mtx_send("err");
                }
            }
            XFree(hosts);
        }
        return 1;
    }
    else if( buf[0] == MTX_MSG_SETALLOWEDHOST || buf[0] == MTX_MSG_REMOVEALLOWEDHOST )
    {
        XHostAddress ha;
        struct in_addr addr;
        char client[256];
        unsigned char clen;

        if ( avail < 2 )
            return MTX_MSG_INCOMPLETE;

        clen = buf[1];

        if ( avail < (clen + 2) )
            return MTX_MSG_INCOMPLETE;

        memcpy(client, buf+2, clen);
        client[clen] = 0;
        if (inet_aton(client, &addr) == 0)
        {
            error("mtx protocol error: cant convert client to numeric '%s'\n", client);
        }
        ha.family = FamilyInternet;
        ha.length = 4;
        ha.address = (char *)&addr;
        if ( buf[0] == MTX_MSG_SETALLOWEDHOST ){
            XAddHost( g_display, &ha);
        } else {
            XRemoveHost( g_display, &ha);
        }

        return clen + 2;
    }
    else
    {
        error("mtx protocol error: Invalid command '%d'\n", buf[0]);
    }
    return 0;
}

void
handle_buffer(const char *input, unsigned int inputlen, handle_cmds_t cmdhandler)
{
    char *buf, *p;
    size_t buflen;
    static char* rest = NULL;
    static size_t restlen = 0;

    /* Copy data to buffer */
    buflen = restlen + inputlen;
    buf = (char *) xmalloc(buflen);
    if (restlen)
        memcpy(buf, rest, restlen);
    memcpy(buf + restlen , input, inputlen);
    p = buf;

    while (buflen > (p - buf) )
    {
        int consummed = cmdhandler(p, buflen - (p - buf) );
        if (consummed == MTX_MSG_INCOMPLETE)
        {
            break;
        }
        p += consummed;
    }

    /* Save in rest */
    if (rest)
        xfree(rest);
    restlen = buf + buflen - p;
    if (restlen)
    {
        rest = (char *) xmalloc(restlen);
        memcpy(rest, p, restlen);
    }
    else
    {
        rest = NULL;
    }
    xfree(buf);
}

/* Process new data from the virtual channel */
static void
mtx_process(STREAM s)
{
    unsigned int pkglen;

    pkglen = s->end - s->p;
    handle_buffer(s->p, pkglen, mtx_process_cmd);
}

/* Initialize this module: Register the mtx channel */
BOOL
mtx_init(void)
{
    mtx_channel =
            channel_register(MTX_CHANNEL_NAME, CHANNEL_OPTION_INITIALIZED | CHANNEL_OPTION_ENCRYPT_RDP,
                             mtx_process);
    return (mtx_channel != NULL);
}

/* Send data to channel */
static void
mtx_send(const char *output)
{
    STREAM s;
    unsigned long len;
    unsigned char c;

    if (!output) return;

    len = strlen(output);

    if ( len > 254 )
    {
        error("mtx protocol error: string too long to send '%s'\n", output);
    }

    c = len;

    s = channel_init(mtx_channel, len + 1 );
    out_uint8p(s, &c, 1);
    out_uint8p(s, output, len);
    s_mark_end(s);

    channel_send(s, mtx_channel);
}
