/***************************************************************************************\

Module Name:    MtxPci.c

Description:    Interface for PCI devices.

References:     MtxPci.htm

    Copyright (c) 2000, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/


// --------------------------------------------------------------------------------------
//                      H E A D E R   F I L E S   R E Q U I R E D
// --------------------------------------------------------------------------------------

#include "precomp.h"

// --------------------------------------------------------------------------------------
//                        C O N S T A N T S   A N D   T Y P E S
// --------------------------------------------------------------------------------------

// --------------------------------------------------------------------------------------
//                           G L O B A L   V A R I A B L E S
// --------------------------------------------------------------------------------------

// --------------------------------------------------------------------------------------
//                    L O C A L   F U N C T I O N   P R O T O T Y P E S
// --------------------------------------------------------------------------------------


// --------------------------------------------------------------------------------------
//                           I N L I N E S   A N D   M A C R O S
// --------------------------------------------------------------------------------------


/***************************************************************************************\

Function:       pciScanBus

Description:    Scan all devices on the PCI bus, calling the PCISEARCH functions at
                each node.

Parameters:     fnSearch        Compare function execute for each node
                pvRefData       Reference data passed to the compare function

Return Value:   DWORD 
                TRUE if the compare function returned TRUE, FALSE if the compare
                function returned FALSE for all devices on the PCI bus.

Algorithm:      |
                | For all bus
                | | For all devices
                | | | For all functions
                | | | | If pciIsDevicePresent
                | | | | | Detect multi-function
                | | | | | Detect pci-pci bridge
                | | | | | Call fnSearch function
                | | | | | If fnSearch return TRUE
                | | | | | | Exit all loops
                |

Comments:       None.

\***************************************************************************************/
STACK_LINKAGE BOOL32 pciScanBus(
    PCISEARCH   fnSearch, 
    void*       pvRefData)
{
    PCIHANDLE   hPci;
    UINT32      ulMaxBus;
    UINT32      ulMaxDevice;
    UINT32      ulBus;
    UINT32      ulDevice;
    UINT32      ulFunction;
    UINT32      ulMaxFunction;
    UINT32      ulSecBus;
    UINT32      dwHEADER;
    UINT32      dwBUSCFG;
    UINT16      wVendorID;

    hPci = 0;

    ulMaxBus = 0 + 1;
    for (ulBus=0; (ulBus<=ulMaxBus); ulBus++)
    {
        ulMaxDevice = 31;
        for (ulDevice=0; (ulDevice<=ulMaxDevice); ulDevice++)
        {
            ulMaxFunction = 0;
            for (ulFunction=0; (ulFunction<=ulMaxFunction); ulFunction++)
            {
                hPci = ((ulBus << 8) | (ulDevice << 3) | ulFunction);

                csReadWord(hPci, PCI_DEVID, &wVendorID);
                if ((wVendorID != 0x0000) && (wVendorID != 0xFFFF))
                {
                    csReadDword(hPci, PCI_HEADER, &dwHEADER);

                    if (ulFunction == 0)
                    {
                        if ((dwHEADER & PCI_HEADER_MULTIFUNC_MASK) == PCI_HEADER_MULTIFUNC_1)
                        {
                            ulMaxFunction = 7;
                        }
                    }

                    if ((dwHEADER & PCI_HEADER_TYPE_MASK) == PCI_HEADER_TYPE_BRIDGE)
                    {
                        csReadDword(hPci, PCI_BRIDGE_BUS_CFG, &dwBUSCFG);

                        ulSecBus = ((dwBUSCFG & PCI_BRIDGE_BUS_CFG_SECBUSNB_MASK) >> PCI_BRIDGE_BUS_CFG_SECBUSNB_SHIFT);

                        if ((ulSecBus != 0xFFL) && (ulSecBus != 0x00L))
                        {
                            if (ulSecBus > ulMaxBus)
                            {
                                ulMaxBus = ulSecBus + 1;
                                if (ulMaxBus > 255)
                                {
                                    ulMaxBus = 255;
                                }
                            }
                        }
                    }

                    if (fnSearch(hPci, pvRefData))
                    {
                        return (TRUE);
                    }
                }
            }
        }
    }

    return (FALSE);
}


/***************************************************************************************\

Function:       pciScan

Description:    See pciScanBus for more details.

Parameters:     

Return Value:   DWORD 

Comments:       None.

\***************************************************************************************/
STACK_LINKAGE BOOL32 pciScan(
    PCISEARCH   fnSearch,
    void*       pvRefData)
{
    return pciScanBus(fnSearch, pvRefData);
}


/***************************************************************************************\

Function:       pciFindCapability

Description:    Search for a capability section in the capability list.

Parameters:     hPci            PCI address
                pdwOffset       Address of DWORD to store offset of found section
                dwCapId         ID of capability section to find

Return Value:   DWORD 
                TRUE if the section is found, FALSE otherwise.

Algorithm:      |
                | If caplist is availlable
                | | Read CAP_PTR register
                | |
                | | While cap_ptr is valid
                | | | Read CAP_HEADER register
                | | |
                | | | If cap_id == dwCapId
                | | | | Section is found
                | | |
                | | | Goto cap_next_ptr
                |

Comments:       None.

\***************************************************************************************/
STACK_LINKAGE BOOL32 pciFindCapability(
    IN      PCIHANDLE       hPci, 
    OUT     UINT32*         pdwOffset, 
    IN      UINT32          dwCapId)
{
    UINT32      bFoundCap;
    UINT32      dwDEVCTRL;
    UINT32      dwCAPPTR;
    UINT32      dwCAPHEADER;
    UINT32      dwOffset;

    bFoundCap = FALSE;

    csReadDword(hPci, PCI_DEVCTRL, &dwDEVCTRL);
    if (dwDEVCTRL & PCI_DEVCTRL_CAPLIST_MASK)
    {
        csReadDword(hPci, PCI_CAP_PTR, &dwCAPPTR);

        dwOffset = ((dwCAPPTR & PCI_CAP_PTR_MASK) >> PCI_CAP_PTR_SHIFT);

        while ((dwOffset >= PCI_CAP_PTR_MIN) && (dwOffset <= PCI_CAP_PTR_MAX))
        {
            csReadDword(hPci, dwOffset, &dwCAPHEADER);

            if (((dwCAPHEADER & PCI_CAP_ID_MASK) >> PCI_CAP_ID_SHIFT) == dwCapId)
            {
                *pdwOffset = dwOffset;
                bFoundCap = TRUE;
                break;
            }

            dwOffset = ((dwCAPHEADER & PCI_CAP_NEXT_MASK) >> PCI_CAP_NEXT_SHIFT);
        }
    }

    return (bFoundCap);
}


/***************************************************************************************\

Function:       pciFindPMCapability

Description:    None.

Parameters:     hPci
                pdwOffset

Return Value:   DWORD 
                Not Specified.

Comments:       None.

\***************************************************************************************/
STACK_LINKAGE BOOL32 pciFindPMCapability(
    IN      PCIHANDLE       hPci, 
    OUT     UINT32*         pdwOffset)
{
    return pciFindCapability(hPci, pdwOffset, CAP_ID_PM);
}


/***************************************************************************************\

Function:       pciFindAGPCapability

Description:    None.

Parameters:     hPci
                pdwOffset

Return Value:   DWORD 
                Not Specified.

Comments:       None.

\***************************************************************************************/
STACK_LINKAGE BOOL32 pciFindAGPCapability(
    IN      PCIHANDLE       hPci, 
    OUT     UINT32*         pdwOffset)
{
    return pciFindCapability(hPci, pdwOffset, CAP_ID_AGP);
}


/***************************************************************************************\

Function:       pciGetTermbase

Description:    None.

Parameters:     hPci
                dwOffset
                pdwBase
                pdwSize

Return Value:   DWORD 
                Not Specified.

Comments:       None.

\***************************************************************************************/
STACK_LINKAGE BOOL32 pciGetTermbase(
    IN      PCIHANDLE       hPci, 
    IN      UINT32          dwOffset, 
    OUT     UINT32*         pdwBase, 
    OUT     UINT32*         pdwSize
)
{
    UINT32 dwTmp;
    UINT32 dwMask;
    
    csReadDword (hPci, dwOffset, &dwTmp);
    csWriteDword(hPci, dwOffset, 0xFFFFFFFF);
    csReadDword (hPci, dwOffset, &dwMask);
    csWriteDword(hPci, dwOffset, dwTmp);

    *pdwBase = (dwTmp & 0xFFFFFFF0);
    *pdwSize = (dwMask & 0xFFFFFFF0);

    return (TRUE);
}


/***************************************************************************************\

Function:       pciGetLocation

Description:    Get the location information for the PCI device.

Parameters:     hPci
                poInfo

Return Value:   DWORD 
                Not Specified.

Comments:       None.

\***************************************************************************************/
STACK_LINKAGE BOOL32 pciGetLocation(
    PCIHANDLE       hPci,
    PCILOCATION*    poInfo)
{
    poInfo->bus = ((hPci & PCI_LOCATION_BUS_MASK) >> PCI_LOCATION_BUS_SHIFT);
    poInfo->device = ((hPci & PCI_LOCATION_DEVICE_MASK) >> PCI_LOCATION_DEVICE_SHIFT);
    poInfo->function = ((hPci & PCI_LOCATION_FUNCTION_MASK) >> PCI_LOCATION_FUNCTION_SHIFT);
    poInfo->reserved = 0;

    return TRUE;
}
