/***************************************************************************************\

Module Name:    MtxIo.c

Description:    Access to IO registers.

References:     

    Copyright (c) 2000, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/


// --------------------------------------------------------------------------------------
//                               H E A D E R   F I L E S
// --------------------------------------------------------------------------------------

#include "precomp.h"

// --------------------------------------------------------------------------------------
//                        C O N S T A N T S   A N D   T Y P E S
// --------------------------------------------------------------------------------------

// --------------------------------------------------------------------------------------
//                           G L O B A L   V A R I A B L E S
// --------------------------------------------------------------------------------------

// --------------------------------------------------------------------------------------
//                  L O C A L   F U N C T I O N   P R O T O T Y P E S
// --------------------------------------------------------------------------------------

// --------------------------------------------------------------------------------------
//                         I N L I N E S   A N D   M A C R O S
// --------------------------------------------------------------------------------------


/***************************************************************************************\

Function:       ioReadWrite

Description:    Read/write of an IO register

Parameters:     hPort           IO port number
                pvBuffer        Output data
                lSize           Number of bytes to read
                lAccess         Access type

Return Value:   void 
                None.

Comments:       None.

\***************************************************************************************/
STACK_LINKAGE BOOL32 ioReadWrite(
    IN      IOHANDLE    hPort,
    IN OUT  void*       pvBuffer,
    IN      UINT32      lSize,
    IN      UINT32      lAccess)
{
    DWORD lOffset = 0;

    while (lOffset < lSize)
    {
        if ((((hPort + lOffset) & 3) == 0) && ((lSize - lOffset) >= 4))
        {
            cpuReadWrite(CPU_BANK_IO_DWORD, hPort + lOffset, (BYTE*)pvBuffer + lOffset, lAccess);

            lOffset += 4;
        }
        else if ((((hPort + lOffset) & 1) == 0) && ((lSize - lOffset) >= 2))
        {
            cpuReadWrite(CPU_BANK_IO_WORD, hPort + lOffset, (BYTE*)pvBuffer + lOffset, lAccess);

            lOffset += 2;
        }
        else
        {
            cpuReadWrite(CPU_BANK_IO_BYTE, hPort + lOffset, (BYTE*)pvBuffer + lOffset, lAccess);

            lOffset ++;
        }
    }

    return TRUE;
}


/***************************************************************************************\

Function:       ioRead

Description:    Read from an IO register

Parameters:     hPort           IO port number
                pvBuffer        Output data
                lSize           Number of bytes to read

Return Value:   void 
                None.

Comments:       None.

\***************************************************************************************/
STACK_LINKAGE BOOL32 ioRead(
    IN      IOHANDLE    hPort, 
    OUT     void*       pvBuffer, 
    IN      UINT32      lSize)
{
    return ioReadWrite(hPort, pvBuffer, lSize, 0);
}


/****************************************************************************************\

Function:       ioReadDword

Description:    Read from an IO register

Parameters:     hPort           IO port number
                pvBuffer        Output data

Return Value:   

Comments:       None.

\****************************************************************************************/
STACK_LINKAGE BOOL32 ioReadDword(
    IN      IOHANDLE    hPort,
    OUT     UINT32*     pvBuffer)
{
    return ioRead(hPort, pvBuffer, sizeof(*pvBuffer));
}


/****************************************************************************************\

Function:       ioReadWord

Description:    Read from an IO register

Parameters:     hPort           IO port number
                pvBuffer        Output data

Return Value:   

Comments:       None.

\****************************************************************************************/
STACK_LINKAGE BOOL32 ioReadWord(
    IN      IOHANDLE    hPort,
    OUT     UINT16*     pvBuffer)
{
    return ioRead(hPort, pvBuffer, sizeof(*pvBuffer));
}


/****************************************************************************************\

Function:       ioReadByte

Description:    Read from an IO register

Parameters:     hPort           IO port number
                pvBuffer        Output data

Return Value:   

Comments:       None.

\****************************************************************************************/
STACK_LINKAGE BOOL32 ioReadByte(
    IN      IOHANDLE    hPort,
    OUT     UINT8*      pvBuffer)
{
    return ioRead(hPort, pvBuffer, sizeof(*pvBuffer));
}


/****************************************************************************************\

Function:       ioWrite

Description:    Write to an IO register

Parameters:     hPort           IO port number
                pvBuffer        Input data
                lSize           Number of bytes to write

Return Value:   

Comments:       None.

\****************************************************************************************/
STACK_LINKAGE BOOL32 ioWrite(
    IN      IOHANDLE    hPort, 
    IN      void*       pvBuffer, 
    IN      UINT32      lSize)
{
    return ioReadWrite(hPort, pvBuffer, lSize, 1);
}


/****************************************************************************************\

Function:       ioWriteDword

Description:    Write to an IO register

Parameters:     hPort           IO port number
                pvBuffer        Input data

Return Value:   

Comments:       None.

\****************************************************************************************/
STACK_LINKAGE BOOL32 ioWriteDword(
    IN      IOHANDLE    hPort,
    IN      UINT32      dData)
{
    return ioWrite(hPort, &dData, sizeof(dData));
}


/****************************************************************************************\

Function:       ioWriteWord

Description:    Write to an IO register

Parameters:     hPort           IO port number
                pvBuffer        Input data

Return Value:   

Comments:       None.

\****************************************************************************************/
STACK_LINKAGE BOOL32 ioWriteWord(
    IN      IOHANDLE    hPort,
    IN      UINT16      wData)
{
    return ioWrite(hPort, &wData, sizeof(wData));
}


/****************************************************************************************\

Function:       ioWriteByte

Description:    Write to an IO register

Parameters:     hPort           IO port number
                pvBuffer        Input data

Return Value:   

Comments:       None.

\****************************************************************************************/
STACK_LINKAGE BOOL32 ioWriteByte(
    IN      IOHANDLE    hPort,
    IN      UINT8       bData)
{
    return ioWrite(hPort, &bData, sizeof(bData));
}
