/***************************************************************************************\

Module Name:    mtx_mem.h

Description:    This module defines an abstraction layer of the kernel services used by 
                the kernel module.

Comments:       This is simply used to allow the binary to be compatible with kernels 
                compiled with switches that modifies the ABI.  It allows the binary to
                be kernel-independant.

    Copyright (C) 2004, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#ifndef INC_MEM_H
#define INC_MEM_H

/*-------------------------------------------------------------------------------------*\
                                 H E A D E R   F I L E S
\*-------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------*\
                          C O N S T A N T S   A N D   T Y P E S
\*-------------------------------------------------------------------------------------*/


/*-------------------------------------------------------------------------------------*\
                   G L O B A L   V A R I A B L E   R E F E R E N C E S
\*-------------------------------------------------------------------------------------*/

extern  ULONG last_alloc_size;

/*-------------------------------------------------------------------------------------*\
                 I N T E R F A C E   F U N C T I O N   P R O T O T Y P E S
\*-------------------------------------------------------------------------------------*/

/* Memory management functions called from binary modules */
#if MEMORY_STATS
INLINE void* mtx_mem_stats_alloc(ULONG size, const char* filename, int line);
INLINE void  mtx_mem_stats_free(void* ptr);
#endif
void* mtx_mem_alloc(ULONG size, HREGION region);
void  mtx_mem_free(void* ptr, ULONG size);

INLINE void* mtx_mem_io_remap(ULONG_PTR offset, ULONG size);
INLINE void  mtx_mem_io_unmap(void* ptr);

/* Memory management functions called from within open source code only */
void    mtx_mem_stats_dump(void);
void    mtx_mem_stats_dump_list(INT);
void*   mtx_mem_alloc_pci(ULONG size);
void    mtx_mem_free_pci(void* ptr, ULONG size);
LONG_PTR  mtx_mem_alloc_agp(ULONG size);
void    mtx_mem_free_agp(LONG_PTR ptr, ULONG size);
void*   mtx_vmalloc(ULONG size);
void    mtx_vfree(void* pAddr);
void    mtx_enable_dev_space(void *dev_id);

/*-------------------------------------------------------------------------------------*\
                             I N L I N E S   A N D   M A C R O S
\*-------------------------------------------------------------------------------------*/

#if MEMORY_STATS
  extern  INT next_allocation_mark;
  #define MTXALLOC(size)      mtx_mem_stats_alloc(size, __FILE__, __LINE__)
  #define MTXFREE(ptr, size)  mtx_mem_stats_free(ptr)
#else
  #define MTXALLOC(size)      mtx_mem_alloc(size, NULL)
  #define MTXFREE(ptr, size)  mtx_mem_free(ptr, size)
#endif

#if MEMORY_STATS

typedef struct mtx_mem_info
{
    ULONG                size;      /* Size of memory chunk */

    const char*          filename;  /* Name of file where allocation was called */
    int                  line;      /* Line number where allocation was called */
    struct mtx_mem_info* prev;      /* Previous memory info record */
    struct mtx_mem_info* next;      /* Next memory info record */
    INT                  allocation_mark; /* ID gived to allocation */

} mtx_mem_info_t;

extern ULONG alloc_count;
extern ULONG alloc_mem;
extern ULONG alloc_pci_count;
extern ULONG alloc_pci_mem;
extern ULONG alloc_agp_count;
extern ULONG alloc_agp_mem;
extern ULONG ioremap_count;

extern mtx_mem_info_t mem_info_head;
extern INT next_allocation_mark;

/***************************************************************************************
 * Function:       mtx_mem_stats_alloc
 *
 * Description:    Allocate memory in kernel address space and attach a mem_info
 *                 record to it so we can keep track of memory statistics.
 *
 * Parameters:     size             Size in bytes of memory space to allocate
 *                 filename         Name of file calling this allocation
 *                 line             Line number in file where allocation was called
 *
 * Return Value:   Ptr to allocated memory, NULL if failed
 *
 * Comments:        
 */
void* mtx_mem_stats_alloc(ULONG size, const char* filename, int line)
{
    void *ptr = mtx_mem_alloc(size + sizeof(mtx_mem_info_t));
    
    if (!ptr) {
        
        MTX_ERROR("Failed to alloc chunk at %s:%d of size %u\n", filename, line, size);
        mtx_mem_stats_dump();
        
    } else {

        mtx_mem_info_t *mem_info = (mtx_mem_info_t*)ptr;

        if (mem_info_head.size == 0) {

            /* size of mem_info head is used as a flag for first allocation */
            mem_info_head.next = &mem_info_head;
            mem_info_head.prev = &mem_info_head;
            mem_info_head.size = ~0;
        }

        /* Fill mem_info record at the beginning of chunk */
        mem_info->size = size;
        mem_info->filename = filename;
        mem_info->line = line;
        mem_info->allocation_mark = next_allocation_mark++;

        /* Add record to list */
        mem_info->next = mem_info_head.next;
        mem_info->prev = &mem_info_head;
        mem_info_head.next->prev = mem_info;
        mem_info_head.next = mem_info;
    }

    return (void*)((char*)ptr + sizeof(mtx_mem_info_t));
}

/***************************************************************************************
 * Function:       mtx_mem_stats_free
 *
 * Description:    Free a memory chunk and the mem_info record attached to it.
 *
 * Parameters:     ptr          Pointer to memory chunk
 *
 * Return Value:   Ptr to allocated memory, NULL if failed
 *
 * Comments:        
 */
void mtx_mem_stats_free(void* ptr)
{
    if (ptr != NULL) {
        
        mtx_mem_info_t *mem_info = (mtx_mem_info_t*)((char*)ptr - sizeof(mtx_mem_info_t));

        /* Remove record from list */
        mem_info->next->prev = mem_info->prev;
        mem_info->prev->next = mem_info->next;
        
        /* Free chunk of memory */
        mtx_mem_free((void*)mem_info, mem_info->size + sizeof(mtx_mem_info_t));
    }
}
#endif

/***************************************************************************************
 * Function:       mtx_mem_io_remap
 *
 * Description:    Map any memory to kernel memory space.
 *
 * Parameters:     offset           Start address of the mapping
 *                 size             Size of the mapping
 *
 * Return Value:   Ptr to mapped memory, NULL if failed
 *
 * Comments:
 *
 */
void* mtx_mem_io_remap(ULONG_PTR offset, ULONG size)
{
    void *ptr = ioremap(offset, size);

    if (!ptr) return NULL;
    
  #if MEMORY_STATS
    ++ioremap_count;
  #endif

    return ptr;
}

/***************************************************************************************
 * Function:       mtx_mem_io_unmap
 *
 * Description:    Free kernel memory map.
 *
 * Parameters:     ptr              Pointer to mapped memory
 *
 * Return Value:   None
 *
 * Comments:
 *
 */
void mtx_mem_io_unmap(void* ptr)
{
    if (!ptr) return;

    iounmap(ptr);
    
  #if MEMORY_STATS
    --ioremap_count;
  #endif
}

#endif  /* #ifndef INC_MEM_H */
