/***************************************************************************************\

Module Name:    mtx_client.c

Description:    This module defines an abstraction layer of the kernel services used by 
                the kernel module.

Comments:       This is simply used to allow the binary to be compatible with kernels 
                compiled with switches that modifies the ABI.  It allows the binary to
                be kernel-independant.

    Copyright (c) 2004, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/


/* -------------------------------------------------------------------------------------- *\
                          I N C L U D E S   A N D   U S I N G S
\* -------------------------------------------------------------------------------------- */
#include "linux/string.h"
#include "asm/delay.h"

#include "mtx_drv.h"

/* -------------------------------------------------------------------------------------- *\
                          C O N S T A N T S   A N D   T Y P E S
\* -------------------------------------------------------------------------------------- */

/* -------------------------------------------------------------------------------------- *\
                 G L O B A L   V A R I A B L E S   D E C L A R A T I O N S
\* -------------------------------------------------------------------------------------- */

/* -------------------------------------------------------------------------------------- *\
                   L O C A L   F U N C T I O N   D E C L A R A T I O N S
\* -------------------------------------------------------------------------------------- */

/* -------------------------------------------------------------------------------------- *\
                                         M A C R O S
\* -------------------------------------------------------------------------------------- */

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,10))

#define MEMR8   ioread8
#define MEMR16  ioread16
#define MEMR32  ioread32
#define MEMW8   iowrite8
#define MEMW16  iowrite16
#define MEMW32  iowrite32

#define IOR8    ioread8
#define IOR16   ioread16
#define IOR32   ioread32
#define IOW8    iowrite8
#define IOW16   iowrite16
#define IOW32   iowrite32

#else

#define MEMR8   readb
#define MEMR16  readw
#define MEMR32  readl
#define MEMW8   writeb
#define MEMW16  writew
#define MEMW32  writel

#define IOR8    inb
#define IOR16   inw
#define IOR32   inl
#define IOW8    outb
#define IOW16   outw
#define IOW32   outl

#endif

/* -------------------------------------------------------------------------------------- *\
                                           C O D E
\* -------------------------------------------------------------------------------------- */
/* Use inline function to resolve implicit calls to memset during initialization of variables. */
STACK_LINKAGE void*   LocMemSet(void* pvDst, int iValue, int szByteCount)
{
    return memset(pvDst, iValue, szByteCount);
}

STACK_LINKAGE void*   LocMemCpy(void* pvDst, const void* pvSrc, int szByteCount)
{
    return memcpy(pvDst, pvSrc, szByteCount);
}

#if MEMORY_STATS

STACK_LINKAGE void* ClientMemStatsAlloc(size_t size, const char* filename, int line)
{
    return mtx_mem_stats_alloc( size, filename, line );
}

STACK_LINKAGE void ClientMemStatsFree(void* ptr)
{
    mtx_mem_stats_free( ptr );
}
#endif

STACK_LINKAGE void* ClientMemAlloc(size_t size, HREGION region)
{
    return mtx_mem_alloc( size, (mtx_region_t*)region );
}

STACK_LINKAGE void ClientMemFree(void* ptr, size_t size)
{
    mtx_mem_free(ptr, size);
}

STACK_LINKAGE void* ClientIoRemap(ULONG_PTR offset, ULONG size)
{
    return mtx_mem_io_remap(offset, size);
}

STACK_LINKAGE void ClientIoUnmap(void* ptr)
{
    mtx_mem_io_unmap(ptr);
}

STACK_LINKAGE ULONG ClientRegisterReadDword(HDEVICE hDevice, ULONG ulOffset, UINT32 *pdwData)
{
    *pdwData = MEMR32( ((BYTE*) GetRegisterHandle( hDevice ) ) + ulOffset );

    return *pdwData;
}

STACK_LINKAGE ULONG ClientRegisterReadWord(HDEVICE hDevice, ULONG ulOffset, UINT16 *pwData)
{
    *pwData = MEMR16( ((BYTE*) GetRegisterHandle( hDevice ) ) + ulOffset );
    return *pwData;
}

STACK_LINKAGE ULONG ClientRegisterReadByte(HDEVICE hDevice, ULONG ulOffset, UINT8 *pbData)
{
    *pbData = MEMR8( ((BYTE*) GetRegisterHandle( hDevice ) ) + ulOffset );
    return *pbData;
}

STACK_LINKAGE ULONG ClientRegisterWriteDword(HDEVICE hDevice, ULONG ulOffset, UINT32 dwData)
{
    MEMW32( dwData, ((BYTE*) GetRegisterHandle( hDevice ) + ulOffset) );
    return dwData;
}

STACK_LINKAGE ULONG ClientRegisterWriteWord(HDEVICE hDevice, ULONG ulOffset, UINT16 wData)
{
    MEMW16( wData, ((BYTE*) GetRegisterHandle( hDevice ) + ulOffset ) );
    return wData;
}

STACK_LINKAGE ULONG ClientRegisterWriteByte(HDEVICE hDevice, ULONG ulOffset, UINT8 bData)
{
    MEMW8( bData, ((BYTE*) GetRegisterHandle( hDevice ) + ulOffset ) );
    return bData;
}

STACK_LINKAGE BYTE ClientIoReadByte(UINT_PTR iPort)
{
    return IOR8((BYTE*)iPort);
}

STACK_LINKAGE UINT16 ClientIoReadWord(UINT_PTR iPort)
{
    return IOR16((BYTE*)iPort);
}

STACK_LINKAGE UINT32 ClientIoReadDword(UINT_PTR iPort)
{
    return IOR32((BYTE*)iPort);
}

STACK_LINKAGE void ClientIoWriteByte(UINT_PTR iPort, BYTE iValue)
{
    return IOW8(iValue, (BYTE*)iPort);
}

STACK_LINKAGE void ClientIoWriteWord(UINT_PTR iPort, UINT16 iValue)
{
    return IOW16(iValue, (BYTE*)iPort);
}

STACK_LINKAGE void ClientIoWriteDword(UINT_PTR iPort, UINT32 iValue)
{
    return IOW32(iValue, (BYTE*)iPort);
}

STACK_LINKAGE ULONG ClientPciConfigReadByte(PCIHANDLE hDevice, ULONG iOffset, UINT8* piValue)
{
    return pci_read_config_byte((struct pci_dev*) hDevice, iOffset, piValue);
}

STACK_LINKAGE ULONG ClientPciConfigReadWord(PCIHANDLE hDevice, ULONG iOffset, UINT16* piValue)
{
    return pci_read_config_word((struct pci_dev*) hDevice, iOffset, piValue);
}

STACK_LINKAGE ULONG ClientPciConfigReadDword(PCIHANDLE hDevice, ULONG iOffset, UINT32* piValue)
{
    return pci_read_config_dword((struct pci_dev*) hDevice, iOffset, piValue);
}

STACK_LINKAGE ULONG ClientPciConfigWriteByte(PCIHANDLE hDevice, ULONG iOffset, UINT8 iValue)
{
    return pci_write_config_byte((struct pci_dev*) hDevice, iOffset, iValue);
}

STACK_LINKAGE ULONG ClientPciConfigWriteWord(PCIHANDLE hDevice, ULONG iOffset, UINT16 iValue)
{
    return pci_write_config_word((struct pci_dev*) hDevice, iOffset, iValue);
}

STACK_LINKAGE ULONG ClientPciConfigWriteDword(PCIHANDLE hDevice, ULONG iOffset, UINT32 iValue)
{
    return pci_write_config_dword((struct pci_dev*) hDevice, iOffset, iValue);
}


#define MAX_STRING_LENGTH 255

STACK_LINKAGE void ClientPrintLevel(ULONG iVerboseLevel, const char* pszFormat, ...)
{
    char    acBuffer[MAX_STRING_LENGTH];
    va_list va;
    size_t  iLogLevelLength;

    switch (iVerboseLevel)
    {
        case MTX_MSG_CRIT:
            sprintf(acBuffer, KERN_CRIT);
            break;
        case MTX_MSG_ERROR:
            sprintf(acBuffer, KERN_ERR);
            break;
        case MTX_MSG_WARNING:
            sprintf(acBuffer, KERN_WARNING);
            break;        
        default:
            sprintf(acBuffer, KERN_INFO);
    }
    
    iLogLevelLength = strlen( acBuffer );
    
    va_start(va, pszFormat);
    vsnprintf(acBuffer + iLogLevelLength, MAX_STRING_LENGTH - iLogLevelLength, pszFormat, va);
    va_end(va);

    printk(acBuffer);
}

STACK_LINKAGE INT ClientVSPrintf(char *buf, const char *fmt, va_list args)
{
    return vsprintf(buf, fmt, args);
}

STACK_LINKAGE INT ClientVSNPrintf(char *buf, size_t size, const char *fmt, va_list args)
{
    return vsnprintf(buf, size, fmt, args);
}

STACK_LINKAGE void ClientMicroDelay(ULONG iMicroSecs)
{
    udelay( iMicroSecs );
}

STACK_LINKAGE void ClientSchedule(void)
{
    schedule();
}

STACK_LINKAGE INT ClientGetNbCPUs( void )
{
#ifdef KERNEL_2_6
    return num_booting_cpus();
#else
    return smp_num_cpus;
#endif
}

STACK_LINKAGE INT ClientGetPageSize( void )
{
    return PAGE_SIZE; 
}

STACK_LINKAGE INT ClientCopyFromUserSpace(void* pvKnlData, void* pvUserData, ULONG ulSize)
{    
    return copy_from_user(pvKnlData, pvUserData, ulSize);
}

STACK_LINKAGE int ClientCopyToUserSpace(void* pvUserData, void* pvKnlData, ULONG ulSize)
{
    return copy_to_user(pvUserData, pvKnlData, ulSize);   
}

STACK_LINKAGE void ClientAtomicInc(void* pvValue)
{
    atomic_inc( (atomic_t*)pvValue );
}

STACK_LINKAGE void ClientAtomicDec(void* pvValue)
{
    atomic_dec( (atomic_t*)pvValue );
}

STACK_LINKAGE int ClientAtomicDecAndTest(void* pvValue)
{
    return atomic_dec_and_test( (atomic_t*)pvValue );
}
