#!/bin/sh
#############################################################
# Copyright 2004, All Rights Reserved, Matrox Graphics Inc. #
#############################################################


#############################################################
#
# Vars Setup Starts Here
#
#############################################################

MACHINEARCH=`uname -m`
TOP=`pwd`
ROOT=$TOP
BACKUP_SUFFIX="mtxbak"
BACKUP_DIR=/usr/share/matrox
RESTARTX=0
UNDO=0
OVERWRITE=0
XPRESENT=0
DISPLAYHELP=0
FAILEDCOMPILATION=0
INSTALLXDRIVER=0
INSTALLKERNELMODULE=0
INSTALLGLLIBRARY=0
INSTALLGLXLIBRARY=0
INSTALLGLHEADER=0
INSTALLGLEXTHEADER=0
MD5SUMBIN=`which md5sum 2> /dev/null`

BLACK='\E[30m'
RED='\E[31m'
GREEN='\E[32m'
YELLOW='\E[33m'
BLUE='\E[34m'
MAGENTA='\E[35m'
CYAN='\E[36m'
WHITE='\E[37m'

if [ "$MACHINEARCH" == "x86_64" ]; then
    if test -e /lib64; then
        XLIBRARYPATH=lib64
    else
        XLIBRARYPATH=lib
    fi
    if test -e /lib32; then
        XLIBRARY32PATH=lib32
    else
        XLIBRARY32PATH=lib
    fi
else
    XLIBRARYPATH=lib
fi

if test -e /usr/X11R6/$XLIBRARYPATH/modules/extensions/libextmod.so; then
    LIBSO="yes"
else
    LIBSO="no"
fi


#############################################################
#
# Script Functions Start Here
#
#############################################################

function IsRoot
{
    # check for superuser mode
    if [ $UID -ne 0 ]; then
        echo "---------------------------------------------------------"
        echo -e "$RED\033[1mERROR\033[0m: \033[1mYou must be logged in as Root to run this program.\033[0m"
        echo "---------------------------------------------------------"
        echo ""
        exit 1
    fi
}

function ParseOptions
{
    n=$#
    i=0
    for arg in $* ; do
        param=`echo $arg | cut -d'=' -f1`
        value=`echo $arg | cut -d'=' -f2`
        case $param in
            "-h") DISPLAYHELP=1 ;;
            "-root") ROOT=$value ;;
            "-uninstall") UNDO=1 ;;
            "-overwrite") OVERWRITE=1 ;;
        esac;
    done
}

function GetXPath
{
    XPRESENT=0
    XPATH=`which X 2> /dev/null`

    if [ $? -ne 0 ] ; then
        XPATH="/usr/X11R6/bin/X"
        if test -h $XPATH ; then
            WHEREIS=`whereis X 2> /dev/null`
            field=`echo $WHEREIS | grep -s '/usr/X11R6/bin/X'`
            if [ $? -eq 0 ] ; then
                XPATH="/usr/X11R6"
            else
                echo ""
                echo "Please enter the full path to your current X11R6 directory: "
                echo "Example: /usr/X11R6"
                read XPATH
            fi
        else
            echo -e "$RED\033[1mERROR\033[0m: \033[1mNo X server found.\033[0m"
            echo ""
            exit 1
        fi
    else
        WHEREIS=`whereis X 2> /dev/null`
        field=`echo $WHEREIS | grep -s '/usr/X11R6/bin/X'`
        if [ $? -eq 0 ] ; then
            XPATH="/usr/X11R6"
        else
            echo ""
            echo "Please enter the full path to your current X11R6 directory: "
            echo "Example: /usr/X11R6"
            read XPATH
        fi
    fi
    
    export XPATH
    
    if  test -z $XPATH ; then
        echo -e "$RED\033[1mERROR\033[0m: \033[1mThe X11R6 path could not be found. Please make"
        echo -e "       sure that an X server is installed.\033[0m"
        echo ""
        exit 1
    fi
}

function GetXVersion
{
    XVERSION=`X -version 2>&1 | grep -s "XFree86 Version" | cut -d" " -f3 | sed -e "s/\([^\.]*\.[^\.]*\.[^\.]*\)\.[^\.]*/\1/"`

    if test -z $XVERSION ; then 
        XVERSION=`X -version 2>&1 | grep -s "X Protocol Version" | sed -e "s/Revision[^0-9]*\([0-9]*\), Release[^0-9]*\([0-9.]*\)/\2\.\1/" | cut -d" " -f 5-`
    fi
        
    export XVERSION

    MAJOR=`echo $XVERSION | awk 'BEGIN {FS="."; } $1 != "" { print $1; }'`
    MINOR=`echo $XVERSION | awk 'BEGIN {FS="."; } $2 != "" { print $2; }'`
    PATCH=`echo $XVERSION | awk 'BEGIN {FS="."; } $3 != "" { print $3; }'`
    export XVERSION_PATH=$MAJOR"_"$MINOR"_"$PATCH

    if ! test -e ./xserver/$XVERSION_PATH ; then
    	# HACK fix for SuSE 9.1
    	if [ "$XVERSION" == "4.3.99" ]; then
                XPRESENT=1
                XVERSION="6.7.0"
    	elif [ "$XVERSION" == "4.3.99.902" ]; then
                XPRESENT=1
                XVERSION="6.7.0"
    	# HACK fix for Distros that use XFree86 4.4.0
    	elif [ "$XVERSION" == "4.4.0" ]; then
    	    XPRESENT=1
            XVERSION="6.7.0"
    	elif [ "$XVERSION" == "6.8.1.0" ]; then
    	    XPRESENT=1
            XVERSION="6.8.1"
    	elif [ "$XVERSION" == "6.8.2.0" ]; then
    	    XPRESENT=1
            XVERSION="6.8.2"
    	# HACK fix for Mandriva 2006
    	elif [ "$XVERSION" == "6.8.99" ]; then
    	    XPRESENT=1
            XVERSION="6.9.0"
    	elif [ "$XVERSION" == "6.8.99.900" ]; then
    	    XPRESENT=1
            XVERSION="6.9.0"
    	elif [ "$XVERSION" == "6.8.99.900.0" ]; then
    	    XPRESENT=1
            XVERSION="6.9.0"
    	else
                echo -e "$RED\033[1mERROR\033[0m: \033[1mThe X server drivers included in this installation package"
                echo -e "       do not support the current version of your X server.\033[0m"
        	    echo ""
                exit 1
    	fi
    else
        XPRESENT=1
    fi
}

function SetXInstallPath
{
    if [ "$XPRESENT"=="1" ]; then
        export XINSTALLDIR=$XPATH/$XLIBRARYPATH/modules/drivers
        if ! test -e $XINSTALLDIR ; then
            echo -e "$RED\033[1mERROR\033[0m: \033[1mCould find X server driver install path.\033[0m"
            echo ""
            exit 1
        fi
    else
        echo -e "$RED\033[1mERROR\033[0m: \033[1mX server not present.\033[0m"
        echo ""
        exit 1
    fi
}

function GetKernelInfo
{
    export VERSION=`uname -r` 
    export KERNEL_FAMILY=`uname -r | cut -b 1,2,3`
}

function GetKernelPath
{
    KERNELMODULESDIR=/lib/modules/$VERSION

    if ! test -e $KERNELMODULESDIR ; then
        echo -e "$RED\033[1mERROR\033[0m: \033[1mCould not find the kernel modules source code path."
        echo -e "       Make sure you have installed the kernel source code files.\033[0m"
        echo ""
        exit 1
    fi
    
    export KERNELMODULESDIR
}

function SetKernelInstallPath
{
    if test -d $KERNELMODULESDIR/kernel; then
        if test -d $KERNELMODULESDIR/kernel/drivers; then
            if test -d $KERNELMODULESDIR/kernel/drivers/video; then
                KNLINSTALLDIR=$KERNELMODULESDIR/kernel/drivers/video
            elif test -d $KERNELMODULESDIR/kernel/drivers/char; then
                KNLINSTALLDIR=$KERNELMODULESDIR/kernel/drivers/char
            else
                KNLINSTALLDIR=$KERNELMODULESDIR/kernel/drivers
            fi
        else
            KNLINSTALLDIR=$KERNELMODULESDIR/kernel
        fi
    elif test -d $KERNELMODULESDIR/video; then
        KNLINSTALLDIR=$KERNELMODULESDIR/video
    elif test -d $KERNELMODULESDIR/char; then
        KNLINSTALLDIR=$KERNELMODULESDIR/char
    else
        KNLINSTALLDIR=$KERNELMODULESDIR
    fi
    
    if ! test -e $KNLINSTALLDIR ; then
        echo -e "$RED\033[1mERROR\033[0m: \033[1mCould not find an installation location for."
        echo -e "       the kernel module. Make sure a kernel is installed.\033[0m"
        echo ""
        exit 1
    fi

    export KNLINSTALLDIR
}

function SetFilenames
{
    GetKernelInfo
    KERNELDRIVER=mtx.o
    if [ "$KERNEL_FAMILY" = "2.6" ]; then
        KERNELDRIVER=mtx.ko
    fi

    export KERNELDRIVER
    if [ "$LIBSO" == "yes" ]; then
        export XDRIVER=mtx_drv.so
        export GLXLIBRARY=libglx.so
    else
        export XDRIVER=mtx_drv.o
        export GLXLIBRARY=libglx.a
    fi
    export GLLIBRARY=libGL.so.1.3.0
    export GLHEADER=gl.h
    export GLEXTHEADER=glext.h
    export GLXHEADER=glx.h
}

function InstallXDriver
{
    MAJOR=`echo $XVERSION | awk 'BEGIN {FS="."; } $1 != "" { print $1; }'`
    MINOR=`echo $XVERSION | awk 'BEGIN {FS="."; } $2 != "" { print $2; }'`
    PATCH=`echo $XVERSION | awk 'BEGIN {FS="."; } $3 != "" { print $3; }'`
    export XVERSION_PATH=$MAJOR"_"$MINOR"_"$PATCH
    
    if [ "$OVERWRITE" == "1" ]; then
        ####################################
        # Install the driver
        ####################################

        echo -e "\033[1mYou are about to install the X server driver,"
        echo -en "do you wish to continue\033[0m $YELLOW\033[1m(y/n)\033[0m\033[1m?\033[0m "
        read STRCONT 
        echo ""
        if [ $STRCONT == 'y' -o $STRCONT == 'Y' ]; then
    
            # copying driver file for X
            if [ "$XVERSION" == "4.3.0" ]; then
                XSERVERTYPE="XFree86"
            else
                XSERVERTYPE="X.org"
            fi
            echo "Installing X server driver for $XSERVERTYPE $XVERSION"
            echo ""
            cp -f xserver/$XVERSION_PATH/$XDRIVER $XINSTALLDIR/$XDRIVER
        else
            echo -e "\033[1mAborting X server driver installation.\033[0m"
            echo ""
        fi
    
    else

        #######################################
        # driver has been installed before
        #######################################
        if test -e $XINSTALLDIR/$XDRIVER ; then
    
            ####################################
            # Is there a backup dir
            ####################################
            if ! test -e $BACKUP_DIR ; then
            
                ####################################
                # No create it
                ####################################
                mkdir $BACKUP_DIR
    
                TEST1="`$MD5SUMBIN -b ./xserver/$XVERSION_PATH/$XDRIVER | cut -b 1-32`"
                TEST2="`$MD5SUMBIN -b $XINSTALLDIR/$XDRIVER | cut -b 1-32`"
    
                ####################################
                # no backups create a new one
                ####################################
                if test -e $XINSTALLDIR/$XDRIVER ; then 
                    if ! [ "$TEST1" == "$TEST2" ]; then
                        echo "Creating a new X server driver backup file"
                        echo "$XDRIVER.$BACKUP_SUFFIX in $BACKUP_DIR."
                        echo ""
                        mv $XINSTALLDIR/$XDRIVER $BACKUP_DIR/$XDRIVER.$BACKUP_SUFFIX
                    fi
                fi
                
                ####################################
                # Install the driver
                ####################################
    
                # copying driver file for X
                if [ "$XVERSION" == "4.3.0" ]; then
                    XSERVERTYPE="XFree86"
                else
                    XSERVERTYPE="X.org"
                fi

                if ! [ "$TEST1" == "$TEST2" ]; then
                    echo "Installing X server driver for $XSERVERTYPE $XVERSION"
                    echo ""
                    cp xserver/$XVERSION_PATH/$XDRIVER $XINSTALLDIR/$XDRIVER
                else
                    echo -e "\033[1mInstalled driver is the same file as the installer"
                    echo -e "version, not installing driver file.\033[0m"
                    echo ""
                fi
            ####################################
            # Yes there is a backup dir
            ####################################
            else
    
                ####################################
                # Yes backup dir is there!
                # Is there a backup?
                ####################################
                if test -e $BACKUP_DIR/$XDRIVER.$BACKUP_SUFFIX ; then
    
                    ####################################
                    # Yes there is a backup!
                    # Is the same as the current file?
                    ####################################
                    TEST1="`$MD5SUMBIN -b $BACKUP_DIR/$XDRIVER.$BACKUP_SUFFIX | cut -b 1-32`"
                    TEST2="`$MD5SUMBIN -b $XINSTALLDIR/$XDRIVER | cut -b 1-32`"
    
                    if ! [ "$TEST1" == "$TEST2" ] ; then
    
                        ####################################
                        # No but is the current installed
                        # driver the same as the installer?
                        ####################################
                        TEST1="`$MD5SUMBIN -b ./xserver/$XVERSION_PATH/$XDRIVER | cut -b 1-32`"
                        TEST2="`$MD5SUMBIN -b $XINSTALLDIR/$XDRIVER | cut -b 1-32`"
    
                        if [ "$TEST1" == "$TEST2" ]; then
    
                            ####################################
                            # Yes they are so do nothing
                            ####################################
                            echo -e "\033[1mCurrently installed driver is the same as the installer file."
                            echo -e "X server driver not installed.\033[0m"
                            echo ""
                            INSTALLXDRIVER=1
                        else
    
                            ####################################
                            # No they aren't so create a new 
                            # backup file
                            ####################################
                            CURRENTDATE=`date +%y%m%d%H%M%S`
                            echo "Renaming current X server driver backup file"
                            echo "to $XDRIVER.$CURRENTDATE in $BACKUP_DIR."
                            echo ""
                            mv $BACKUP_DIR/$XDRIVER.$BACKUP_SUFFIX $BACKUP_DIR/$XDRIVER.$CURRENTDATE
                            
                            echo "Creating a new X server driver backup file"
                            echo "$XDRIVER.$BACKUP_SUFFIX in $BACKUP_DIR."
                            echo ""
                            mv $XINSTALLDIR/$XDRIVER $BACKUP_DIR/$XDRIVER.$BACKUP_SUFFIX
                        fi
                    else
    
                        ####################################
                        # Yes so do nothing
                        ####################################
                        echo -e "\033[1mCurrent X server driver backup file is the same"
                        echo -e "as the currently installed file. No backups"
                        echo -e "created.\033[0m"
                        echo ""
                    fi
                else

                    ####################################
                    # no backups create a new on
                    # only if driver is not the same 
                    # file.
                    ####################################
                    TEST3="`$MD5SUMBIN -b ./xserver/$XVERSION_PATH/$XDRIVER | cut -b 1-32`"
                    TEST4="`$MD5SUMBIN -b $XINSTALLDIR/$XDRIVER | cut -b 1-32`"
                    if [ "$TEST3" == "$TEST4" ]; then
                        echo -e "\033[1mCurrently installed driver is the same as the installer file."
                        echo -e "X server driver not installed.\033[0m"
                        echo ""
                        INSTALLXDRIVER=1
                    else
                        echo "Creating an X server driver backup file in $BACKUP_DIR."
                        echo ""
                        mv $XINSTALLDIR/$XDRIVER $BACKUP_DIR/$XDRIVER.$BACKUP_SUFFIX
                    fi
                fi
    
                ####################################
                # Install the driver
                ####################################
                if [ "$INSTALLXDRIVER" == "0" ]; then
        
                    if test -e $XINSTALLDIR/$XDRIVER ; then
                        TEST5="`$MD5SUMBIN -b ./xserver/$XVERSION_PATH/$XDRIVER | cut -b 1-32`"
                        TEST6="`$MD5SUMBIN -b $XINSTALLDIR/$XDRIVER | cut -b 1-32`"
                    else
                        TEST5="5"
                        TEST6="6"
                    fi

                    if [ "$XVERSION" == "4.3.0" ]; then
                        XSERVERTYPE="XFree86"
                    else
                        XSERVERTYPE="X.org"
                    fi

                    if ! [ "$TEST5" == "$TEST6" ]; then
                        echo "Installing X server driver for $XSERVERTYPE $XVERSION"
                        echo ""
                        cp xserver/$XVERSION_PATH/$XDRIVER $XINSTALLDIR/$XDRIVER
                    else
                        echo -e "\033[1mInstalled driver is the same file as the installer"
                        echo -e "version, not installing driver file.\033[0m"
                        echo ""
                    fi
                fi
            fi
    
        #######################################
        # driver hasn't been installed before
        #######################################
        else
    
            ####################################
            # Is there a backup dir
            ####################################
            if ! test -e $BACKUP_DIR ; then
    
                ####################################
                # No create it
                ####################################
                mkdir $BACKUP_DIR
            fi
            
            ####################################
            # Install the driver
            ####################################
    
            # copying driver file for X
            if [ "$XVERSION" == "4.3.0" ]; then
                XSERVERTYPE="XFree86"
            else
                XSERVERTYPE="X.org"
            fi

            echo "Installing X server driver for $XSERVERTYPE $XVERSION"
            echo ""
            cp xserver/$XVERSION_PATH/$XDRIVER $XINSTALLDIR/$XDRIVER
        fi
    fi
}

function CompileKernelModule
{
    cd $TOP
    
    if ! test -e $TOP/kernel/src/$KERNELDRIVER ; then
        GCCBIN=`which gcc`
        if [ $? -ne 0 ] ; then
            echo -e "$RED\033[1mERROR\033[0m: \033[1mPlease make sure that you have installed"
            echo -e "       the GNU GCC compiler (version 3.3.2 recommended).\033[0m"
            echo ""
            exit 1
        fi
	
        if test -d /usr/src/linux-obj; then
            cd /usr/src/linux
            make cloneconfig
            make modules_prepare
            cd $TOP
        fi
	
        echo "Messages are being logged in file /tmp/make.log,"
        echo "this might take some time."
        echo ""
        echo -n "Compiling $KERNELDRIVER ..."
        export PARHELIUX=$TOP/kernel/src
        export INSTALLERBUILD=1
        cd kernel/src
        make clean > /dev/null 2>&1
        cd parhelia
        make clean > /dev/null 2>&1
        ln -sf ../../mtx_parhelia.o . > /dev/null 2>&1
        cd ..
        make > /tmp/make.log 2>&1
        if [ $? -ne 0 ]; then
            echo ""
            echo -e "$RED\033[1mERROR\033[0m: \033[1mThere has been an error compiling the kernel module. "
            echo -e "       A log file has been created in the file /tmp/make.log\033[0m"
            echo ""
            exit 1
        else
            echo " done."
            echo ""
            cd ../..
        fi 
    fi
}

function InstallKernelModule
{
    cd $TOP
    
    if test -e $TOP/kernel/src/$KERNELDRIVER ; then
        if [ "$OVERWRITE" == "1" ]; then
            echo -e "\033[1mYou are about to install the mtx kernel module,"
            echo -en "do you wish to continue\033[0m $YELLOW\033[1m(y/n)\033[0m\033[1m?\033[0m "
            read STRCONT 
            echo ""
            if [ $STRCONT = 'y' -o $STRCONT = 'Y' ]; then
                echo "Installing the $KERNELDRIVER kernel module to $KNLINSTALLDIR"
                echo ""
                cp -f $TOP/kernel/src/$KERNELDRIVER $KNLINSTALLDIR
                /sbin/depmod > /dev/null 2>&1 
            else
                echo -e "\033[1mAborting kernel module installation.\033[0m"
                echo ""
            fi
        else
            ########################################
            # Is there already an installed module?
            ########################################
            if test -e $KNLINSTALLDIR/$KERNELDRIVER ; then

                ########################################
                # YES we have a module
                # Is there a backup dir
                ####################################
                if ! test -e $BACKUP_DIR ; then
            
                    ####################################
                    # No create it
                    ####################################
                    mkdir $BACKUP_DIR
    
                    TEST1="`$MD5SUMBIN -b ./kernel/src/$KERNELDRIVER | cut -b 1-32`"
                    TEST2="`$MD5SUMBIN -b $KNLINSTALLDIR/$KERNELDRIVER | cut -b 1-32`"
    
                    ####################################
                    # no backups create a new one
                    ####################################
                    if test -e $KNLINSTALLDIR/$KERNELDRIVER ; then 
                        if ! [ "$TEST1" == "$TEST2" ]; then
                            echo "Creating a new kernel module backup file"
                            echo "$KERNELDRIVER.$BACKUP_SUFFIX in $BACKUP_DIR."
                            echo ""
                            mv $KNLINSTALLDIR/$KERNELDRIVER $BACKUP_DIR/$KERNELDRIVER.$BACKUP_SUFFIX
                        fi
                    fi
                
                    ####################################
                    # Install the driver
                    ####################################
    
                    if ! [ "$TEST1" == "$TEST2" ]; then
                        echo "Installing X server driver for XFree86 4.2.0"
                        echo ""
                        cp kernel/src/$KERNELDRIVER $KNLINSTALLDIR/$KERNELDRIVER
                        /sbin/depmod > /dev/null 2>&1 
                    else
                        echo -e "\033[1mInstalled kernel module is the same file as the installer"
                        echo -e "version, not installing kernel module file.\033[0m"
                        echo ""
                    fi
    
                ####################################
                # Yes there is a backup dir
                ####################################
                else
    
                    ####################################
                    # Yes backup dir is there!
                    # Is there a backup?
                    ####################################
                    if test -e $BACKUP_DIR/$KERNELDRIVER.$BACKUP_SUFFIX ; then
    
                        ####################################
                        # Yes there is a backup!
                        # Is the same as the current file?
                        ####################################
                        TEST1="`$MD5SUMBIN -b $BACKUP_DIR/$KERNELDRIVER.$BACKUP_SUFFIX | cut -b 1-32`"
                        TEST2="`$MD5SUMBIN -b $KNLINSTALLDIR/$KERNELDRIVER | cut -b 1-32`"
    
                        if ! [ "$TEST1" == "$TEST2" ] ; then
        
                            ####################################
                            # No but is the current installed
                            # driver the same as the installer?
                            ####################################
                            TEST1="`$MD5SUMBIN -b kernel/src/$KERNELDRIVER | cut -b 1-32`"
                            TEST2="`$MD5SUMBIN -b $KNLINSTALLDIR/$KERNELDRIVER | cut -b 1-32`"
        
                            if [ "$TEST1" == "$TEST2" ]; then
        
                                ####################################
                                # Yes they are so do nothing
                                ####################################
                                echo -e "\033[1mInstalled kernel module is the same file as the installer"
                                echo -e "version, not installing kernel module file.\033[0m"
                                echo ""
                                INSTALLKERNELMODULE=1
                            else
        
                                ####################################
                                # No they aren't so create a new 
                                # backup file
                                ####################################
                                CURRENTDATE=`date +%y%m%d%H%M%S`
                                echo "Renaming current kernel module backup file"
                                echo "to $XDRIVER.$CURRENTDATE in $BACKUP_DIR."
                                echo ""
                                mv $BACKUP_DIR/$KERNELDRIVER.$BACKUP_SUFFIX $BACKUP_DIR/$KERNELDRIVER.$CURRENTDATE
                                
                                echo "Creating a new kernel module backup file"
                                echo "$KERNELDRIVER.$BACKUP_SUFFIX in $BACKUP_DIR."
                                echo ""
                                mv $KNLINSTALLDIR/$KERNELDRIVER $BACKUP_DIR/$KERNELDRIVER.$BACKUP_SUFFIX
                            fi
                        else
        
                            ####################################
                            # Yes so do nothing
                            ####################################
                            echo -e "\033[1mCurrent kernel module backup file is the same"
                            echo -e "as the currently installed file. No backups"
                            echo -e "created.\033[0m"
                            echo ""
                        fi
                    else
    
                        ####################################
                        # no backups create a new on
                        # only if driver is not the same 
                        # file.
                        ####################################
                        TEST3="`$MD5SUMBIN -b kernel/src/$KERNELDRIVER | cut -b 1-32`"
                        TEST4="`$MD5SUMBIN -b $KNLINSTALLDIR/$KERNELDRIVER | cut -b 1-32`"
                        if [ "$TEST3" == "$TEST4" ]; then
                            echo -e "\033[1mCurrently installed kernel module is the same as the installer file."
                            echo -e "Kernel module not installed.\033[0m"
                            echo ""
                            INSTALLKERNELMODULE=1
                        else
                            echo "Creating a kernel module backup file in $BACKUP_DIR."
                            echo ""
                            mv $KNLINSTALLDIR/$KERNELDRIVER $BACKUP_DIR/$KERNELDRIVER.$BACKUP_SUFFIX
                        fi
                    fi
        
                    ####################################
                    # Install the driver
                    ####################################
                    if [ "$INSTALLKERNELMODULE" == "0" ]; then
            
                        if test -e $KNLINSTALLDIR/$KERNELDRIVER ; then
                            TEST5="`$MD5SUMBIN -b kernel/src/$KERNELDRIVER | cut -b 1-32`"
                            TEST6="`$MD5SUMBIN -b $KNLINSTALLDIR/$KERNELDRIVER | cut -b 1-32`"
                        else
                            TEST5="5"
                            TEST6="6"
                        fi
                        
                        if ! [ "$TEST5" == "$TEST6" ]; then
                            echo "Installing the $KERNELDRIVER kernel module to $KNLINSTALLDIR"
                            echo ""
                            cp -f $TOP/kernel/src/$KERNELDRIVER $KNLINSTALLDIR
                            /sbin/depmod > /dev/null 2>&1 
                        else
                            echo -e "\033[1mInstalled kernel module is the same file as the installer"
                            echo -e "version, not installing kernel module file.\033[0m"
                            echo ""
                        fi
                    fi
                fi
                
            else

                ########################################
                # NO we do not have a module
                # Is there a backup dir
                ####################################
                if ! test -e $BACKUP_DIR ; then
    
                    ####################################
                    # No create it
                    ####################################
                    mkdir $BACKUP_DIR
                fi

                echo "Installing the $KERNELDRIVER kernel module to $KNLINSTALLDIR"
                echo ""
                cp -f $TOP/kernel/src/$KERNELDRIVER $KNLINSTALLDIR
                /sbin/depmod > /dev/null 2>&1 
            fi
        fi    
    else
        echo -e "$RED\033[1mERROR\033[0m: \033[1mFile $KERNELDRIVER missing or not compiled.\033[0m"
        echo ""
        exit 1
    fi
}

function SetOpenGLInstallPath
{
    export GLXINSTALLDIR=$XPATH/$XLIBRARYPATH/modules/extensions
    if ! test -e $GLXINSTALLDIR ; then
        echo -e "$RED\033[1mERROR\033[0m: \033[1mCould not find an installation location for."
        echo -e "       the GLX library file.\033[0m"
        echo ""
        exit 1
    fi

    export GLINSTALLDIR=/usr/$XLIBRARYPATH
    if ! test -e $GLINSTALLDIR ; then
        echo -e "$RED\033[1mERROR\033[0m: \033[1mCould not find an installation location for."
        echo -e "       the GL library file.\033[0m"
        echo ""
        exit 1
    fi

    if [ "$MACHINEARCH" == "x86_64" ]; then
        export GLINSTALL32DIR=/usr/$XLIBRARY32PATH
        if ! test -e $GLINSTALL32DIR ; then
            export GLINSTALL32DIR=/usr/lib
            if ! test -e $GLINSTALL32DIR ; then
                echo -e "$RED\033[1mERROR\033[0m: \033[1mCould not find an installation location for."
                echo -e "       the GL library file.\033[0m"
                echo ""
                exit 1
            fi
        fi
    fi

    if test -e /usr/X11R6/include/GL ; then 
        export GLHEADERSINSTALLDIR=/usr/X11R6/include/GL
    else
        export GLHEADERSINSTALLDIR=/usr/include
    fi
    if ! test -e $GLHEADERSINSTALLDIR ; then
        echo -e "$RED\033[1mERROR\033[0m: \033[1mCould not find an installation location for."
        echo -e "       the GL header files.\033[0m"
        echo ""
        exit 1
    fi

    export GLOPTDIR=$XPATH/$XLIBRARYPATH
    if ! test -e $GLOPTDIR ; then
        echo -e "$RED\033[1mERROR\033[0m: \033[1mCould not find an location for older."
        echo -e "       GL library files.\033[0m"
        echo ""
        exit 1
    fi

    if [ "$MACHINEARCH" == "x86_64" ]; then
        export GLINSTALL32DIR=/usr/$XLIBRARY32PATH
        if ! test -e $GLINSTALL32DIR ; then
            export GLINSTALL32DIR=/usr/lib
            if ! test -e $GLINSTALL32DIR ; then
                echo -e "$RED\033[1mERROR\033[0m: \033[1mCould not find an installation location for."
                echo -e "       the GL library (32bit) file.\033[0m"
                echo ""
                exit 1
            fi
        fi

        export GLOPT32DIR=$XPATH/$XLIBRARY32PATH
        if ! test -e $GLOPT32DIR ; then
            export GLOPT32DIR=$XPATH/lib
            if ! test -e $GLOPT32DIR ; then
                echo -e "$RED\033[1mERROR\033[0m: \033[1mCould not find an location for older."
                echo -e "       GL library (32bit) files.\033[0m"
                echo ""
                exit 1
            fi
        fi
    fi
}

function BackupOpenGL
{
    LIBGLFILES=`/sbin/ldconfig -p | grep libGL.so | awk 'BEGIN {FS="=> "}; $2 != "" {print $2;}'`
    LIBGLWRAPPERFILES=`/sbin/ldconfig -p | grep libGLwrapper.so | awk 'BEGIN {FS="=> "}; $2 != "" {print $2;}'`
    CURRENTDATE=`date +%y%m%d`
    CURRENTTIME=`date +%H%M%S`
    GLBACKUPDIR="$BACKUP_DIR/GLBackup-$CURRENTDATE-$CURRENTTIME/"

    if ! test -d $BACKUP_DIR; then
        mkdir $BACKUP_DIR
    fi

    mkdir $GLBACKUPDIR

    if [ "$MACHINEARCH" == "x86_64" ]; then
        if ! test -e $BACKUP_DIR/lib32; then
            mkdir $BACKUP_DIR/lib32
        fi
        if ! test -e $GLBACKUPDIR/lib32; then
            mkdir $GLBACKUPDIR/lib32
        fi
    fi

    if test -e $GLBACKUPDIR/restoregl.sh ; then
        rm -f $GLBACKUPDIR/restoregl.sh
    fi

    echo "Updating library cache"
    echo ""
    /sbin/ldconfig

    echo "Creating OpenGL backups in $GLBACKUPDIR"
    echo ""

    echo "#! /bin/sh" > $GLBACKUPDIR/restoregl.sh
    echo "" >> $GLBACKUPDIR/restoregl.sh

    # Backup libGL
    let "index=0"
    if [ "$LIBGLFILES" != "" ]; then
        for i in $LIBGLFILES
            do
                if test -e $i ; then
                    (( index++ ))
                    FILES1=`echo "$i" | awk 'BEGIN {FS="libGL.so"}; $2 != "" {print $2;}'`
                    FILES2="libGL.so$FILES1"
                    echo "Create backup of $FILES2"
                    echo ""
                    mv "$i" "$GLBACKUPDIR$index.$FILES2"
                    echo "rm -f $i" >> $GLBACKUPDIR/restoregl.sh
                    echo "mv $index.$FILES2 $i" >> $GLBACKUPDIR/restoregl.sh
                fi
            done;
    fi

    ALLGLFILES="libGL.so libGL.so.1 libGL.so.1.2 libGL.so.1.2.0 libGL.so.1.3.0"
    for i in ALLGLFILES
    do
        FILES3="/usr/$XLIBRARYPATH/$i"
        if test -e $FILES3 ; then
            (( index++ ))
            FILES1=`echo "$FILES3" | awk 'BEGIN {FS="libGL.so"}; $2 != "" {print $2;}'`
            FILES2="libGL.so$FILES1"
            echo "Create backup of $FILES2"
            echo ""
            mv "$FILES3" "$GLBACKUPDIR$index.$FILES2"
            echo "rm -f $FILES" >> $GLBACKUPDIR/restoregl.sh
            echo "mv $index.$FILES2 $FILES3" >> $GLBACKUPDIR/restoregl.sh
        fi

        FILES3="/usr/X11R6/$XLIBRARYPATH/$i"
        if test -e $FILES3 ; then
            (( index++ ))
            FILES1=`echo "$FILES3" | awk 'BEGIN {FS="libGL.so"}; $2 != "" {print $2;}'`
            FILES2="libGL.so$FILES1"
            echo "Create backup of $FILES2"
            echo ""
            mv "$FILES3" "$GLBACKUPDIR$index.$FILES2"
            echo "rm -f $FILES" >> $GLBACKUPDIR/restoregl.sh
            echo "mv $index.$FILES2 $FILES3" >> $GLBACKUPDIR/restoregl.sh
        fi

        FILES3="/usr/local/$XLIBRARYPATH/$i"
        if test -e $FILES3 ; then
            (( index++ ))
            FILES1=`echo "$FILES3" | awk 'BEGIN {FS="libGL.so"}; $2 != "" {print $2;}'`
            FILES2="libGL.so$FILES1"
            echo "Create backup of $FILES2"
            echo ""
            mv "$FILES3" "$GLBACKUPDIR$index.$FILES2"
            echo "rm -f $FILES" >> $GLBACKUPDIR/restoregl.sh
            echo "mv $index.$FILES2 $FILES3" >> $GLBACKUPDIR/restoregl.sh
        fi

        FILES3="/usr/$XLIBRARY32PATH/$i"
        if test -e $FILES3 ; then
            (( index++ ))
            FILES1=`echo "$FILES3" | awk 'BEGIN {FS="libGL.so"}; $2 != "" {print $2;}'`
            FILES2="libGL.so$FILES1"
            echo "Create backup of $FILES2"
            echo ""
            mv "$FILES3" "$GLBACKUPDIR/lib32/$index.$FILES2"
            echo "rm -f $FILES" >> $GLBACKUPDIR/restoregl.sh
            echo "mv lib32/$index.$FILES2 $FILES3" >> $GLBACKUPDIR/restoregl.sh
        else
            FILES3="/usr/lib/$i"
            if test -e $FILES3 ; then
                (( index++ ))
                FILES1=`echo "$FILES3" | awk 'BEGIN {FS="libGL.so"}; $2 != "" {print $2;}'`
                FILES2="libGL.so$FILES1"
                echo "Create backup of $FILES2"
                echo ""
                mv "$FILES3" "$GLBACKUPDIR/lib/$index.$FILES2"
                echo "rm -f $FILES" >> $GLBACKUPDIR/restoregl.sh
                echo "mv lib/$index.$FILES2 $FILES3" >> $GLBACKUPDIR/restoregl.sh
            fi
        fi

        FILES3="/usr/X11R6/$XLIBRARY32PATH/$i"
        if test -e $FILES3 ; then
            (( index++ ))
            FILES1=`echo "$FILES3" | awk 'BEGIN {FS="libGL.so"}; $2 != "" {print $2;}'`
            FILES2="libGL.so$FILES1"
            echo "Create backup of $FILES2"
            echo ""
            mv "$FILES3" "$GLBACKUPDIR/lib32/$index.$FILES2"
            echo "rm -f $FILES" >> $GLBACKUPDIR/restoregl.sh
            echo "mv lib32/$index.$FILES2 $FILES3" >> $GLBACKUPDIR/restoregl.sh
        else
            FILES3="/usr/X11R6/lib/$i"
            if test -e $FILES3 ; then
                (( index++ ))
                FILES1=`echo "$FILES3" | awk 'BEGIN {FS="libGL.so"}; $2 != "" {print $2;}'`
                FILES2="libGL.so$FILES1"
                echo "Create backup of $FILES2"
                echo ""
                mv "$FILES3" "$GLBACKUPDIR/lib/$index.$FILES2"
                echo "rm -f $FILES" >> $GLBACKUPDIR/restoregl.sh
                echo "mv lib/$index.$FILES2 $FILES3" >> $GLBACKUPDIR/restoregl.sh
            fi
        fi

        FILES3="/usr/local/$XLIBRARY32PATH/$i"
        if test -e $FILES3 ; then
            (( index++ ))
            FILES1=`echo "$FILES3" | awk 'BEGIN {FS="libGL.so"}; $2 != "" {print $2;}'`
            FILES2="libGL.so$FILES1"
            echo "Create backup of $FILES2"
            echo ""
            mv "$FILES3" "$GLBACKUPDIR/lib32/$index.$FILES2"
            echo "rm -f $FILES" >> $GLBACKUPDIR/restoregl.sh
            echo "mv lib32/$index.$FILES2 $FILES3" >> $GLBACKUPDIR/restoregl.sh
        else
            FILES3="/usr/local/lib/$i"
            if test -e $FILES3 ; then
                (( index++ ))
                FILES1=`echo "$FILES3" | awk 'BEGIN {FS="libGL.so"}; $2 != "" {print $2;}'`
                FILES2="libGL.so$FILES1"
                echo "Create backup of $FILES2"
                echo ""
                mv "$FILES3" "$GLBACKUPDIR/lib/$index.$FILES2"
                echo "rm -f $FILES" >> $GLBACKUPDIR/restoregl.sh
                echo "mv lib/$index.$FILES2 $FILES3" >> $GLBACKUPDIR/restoregl.sh
            fi
        fi
    done

    echo "" >> $GLBACKUPDIR/restoregl.sh

    # One last pass to make sure we backup it all up
    FILEPATTERN="libGL.so*"
    if [ "$MACHINEARCH" == "x86_64" ]; then
        if test -e /lib64; then
            LISTPATH64="/lib64 /usr/lib64 /usr/X11R6/lib64 /usr/X11/lib64"
        else
            LISTPATH64="/lib /usr/lib /usr/X11R6/lib /usr/X11/lib"
        fi

        if test -e /lib32; then
            LISTPATH32="/lib32 /usr/lib32 /usr/X11R6/lib32 /usr/X11/lib32"
        else
            LISTPATH32="/lib /usr/lib /usr/X11R6/lib /usr/X11/lib"
        fi

        LISTPATH=$LISTPATH64" "$LISTPATH32
    else
        LISTPATH32="/lib /usr/lib /usr/X11R6/lib /usr/X11/lib"
        LISTPATH64=""

        LISTPATH=$LISTPATH32
    fi

    LISTLDSOCONF=`cat /etc/ld.so.conf`

    for FILEPATH in $LISTPATH
    do
        if test -e $FILEPATH ; then
            LISTIT=`find $FILEPATH -name "$FILEPATTERN"`
            for FILES in $LISTIT
            do
                StrLength=${#FILES}
                let "X=-1"
                let "StrLength-=1"
                FileStr=""
                PathStr=""
                while [ $StrLength -ge 0 ]
                do
                    if [ "${FILES:StrLength:1}" == "/" ] ; then
                        if [ $X -eq -1 ]; then 
                            let "X=StrLength"
                        fi
                    fi
                    if [ $X -eq -1 ]; then
                        FileStr="${FILES:StrLength:1}"$FileStr
                    fi
                    let "StrLength-=1"
                done
                PathStr="${FILES:0:$X}"
                FilePath=$PathStr/$FileStr
                (( index++ ))
                mv $FilePath $PathStr/$index.$FileStr
                echo "rm -f $FilePath" >> $GLBACKUPDIR/restoregl.sh
                echo "mv $index.$FileStr $FilePath" >> $GLBACKUPDIR/restoregl.sh
            done 
        fi
    done

    for FILEPATH in $LISTLDSOCONF
    do
        if [[ $FILEPATH == /* ]]; then
            LISTIT=`find $FILEPATH -name "$FILEPATTERN"`
            for FILES in $LISTIT
            do
                StrLength=${#FILES}
                let "X=-1"
                let "StrLength-=1"
                FileStr=""
                PathStr=""
                while [ $StrLength -ge 0 ]
                do
                    if [ "${FILES:StrLength:1}" == "/" ] ; then
                        if [ $X -eq -1 ]; then 
                            let "X=StrLength"
                        fi
                    fi
                    if [ $X -eq -1 ]; then
                        FileStr="${FILES:StrLength:1}"$FileStr
                    fi
                    let "StrLength-=1"
                done
                PathStr="${FILES:0:$X}"
                FilePath=$PathStr/$FileStr
                (( index++ ))
                mv $FilePath $PathStr/$index.$FileStr
                echo "rm -f $FilePath" >> $GLBACKUPDIR/restoregl.sh
                echo "mv $index.$FileStr $FilePath" >> $GLBACKUPDIR/restoregl.sh
            done 
        fi
    done
    
    echo "" >> $GLBACKUPDIR/restoregl.sh
    
    # Backup libGLwrapper
    index=0
    if [ "$LIBGLWRAPPERFILES" != "" ]; then
        for i in $LIBGLWRAPPERFILES
            do
                if test -e $i ; then
                    (( index++ ))
                    FILES1=`echo "$i" | awk 'BEGIN {FS="libGLwrapper.so"}; $2 != "" {print $2;}'`
                    FILES2=`libGLwrapper.so$FILES1`
                    echo "Create backup of $FILES2"
                    echo ""
                    mv $i $GLBACKUPDIR/$index.$FILES2
                    echo "rm -f $i" >> $GLBACKUPDIR/restoregl.sh
                    echo "mv $index.$FILES2 $i" >> $GLBACKUPDIR/restoregl.sh
                fi
            done;
    fi

    echo "" >> $GLBACKUPDIR/restoregl.sh

    echo "Updating library cache"
    echo ""
    /sbin/ldconfig

    if test -e $GLBACKUPDIR/restoregl.sh ; then
        chmod 755 $GLBACKUPDIR/restoregl.sh
    fi

    echo "These files can be restored by launching the following"
    echo "command as root:"
    echo ""
    echo "    sh $GLBACKUPDIR/restoregl.sh"
    echo ""
}

function InstallOpenGL
{
    cd $TOP
    
    if [ "$OVERWRITE" == "1" ]; then
        echo -e "\033[1mYou are about to install the OpenGL files,"
        echo -en "do you wish to continue\033[0m $YELLOW\033[1m(y/n)\033[0m\033[1m?\033[0m "
        read STRCONT 
        echo ""
        if [ $STRCONT = 'y' -o $STRCONT = 'Y' ]; then

            #########################################
            # Backup current OpenGL Files
            #########################################
            BackupOpenGL

            echo "Installing the OpenGL library file to $GLINSTALLDIR"
            echo ""
            rm -f $GLINSTALLDIR/libGL.so*
            cp -f $TOP/opengl/$GLLIBRARY $GLINSTALLDIR
            ln -s $GLINSTALLDIR/$GLLIBRARY $GLINSTALLDIR/libGL.so.1.2.0
            ln -s $GLINSTALLDIR/libGL.so.1.2.0 $GLINSTALLDIR/libGL.so.1
            ln -s $GLINSTALLDIR/libGL.so.1 $GLINSTALLDIR/libGL.so

            if [ "$MACHINEARCH" == "x86_64" ]; then
                rm -f $GLINSTALL32DIR/libGL.so*
                cp -f $TOP/opengl/lib32/$GLLIBRARY $GLINSTALL32DIR
                ln -s $GLINSTALL32DIR/$GLLIBRARY $GLINSTALL32DIR/libGL.so.1.2.0
                ln -s $GLINSTALL32DIR/libGL.so.1.2.0 $GLINSTALL32DIR/libGL.so.1
                ln -s $GLINSTALL32DIR/libGL.so.1 $GLINSTALL32DIR/libGL.so
            fi

            echo "Installing the GLX library file to $GLXINSTALLDIR"
            echo ""
            cp -f $TOP/opengl/$GLXLIBRARY $GLXINSTALLDIR

            echo "Installing the $GLHEADER header file to $GLHEADERSINSTALLDIR"
            echo ""
            cp -f $TOP/opengl/include/$GLHEADER $GLHEADERSINSTALLDIR

            echo "Installing the $GLEXTHEADER header file to $GLHEADERSINSTALLDIR"
            echo ""
            cp -f $TOP/opengl/include/$GLEXTHEADER $GLHEADERSINSTALLDIR

            echo "Installing the $GLXHEADER header file to $GLHEADERSINSTALLDIR"
            echo ""
            cp -f $TOP/opengl/include/$GLXHEADER $GLHEADERSINSTALLDIR
            
            echo "Updating library cache"
            echo ""
            /sbin/ldconfig
        else
            echo -e "\033[1mAborting OpenGL installation.\033[0m"
            echo ""
        fi
    else
        #########################################
        # Backup current OpenGL Files
        #########################################
        BackupOpenGL
    
        #########################################
        # Is there already an installed library?
        #########################################
        if test -e $GLINSTALLDIR/$GLLIBRARY ; then

            ####################################
            # YES we have a module
            # Is there a backup dir
            ####################################
            if ! test -e $BACKUP_DIR ; then
            
                ####################################
                # No create it
                ####################################
                mkdir $BACKUP_DIR
    
                ####################################
                # no backups create a new one
                ####################################
                TEST1="`$MD5SUMBIN -b opengl/$GLLIBRARY | cut -b 1-32`"
                TEST2="`$MD5SUMBIN -b $GLINSTALLDIR/$GLLIBRARY | cut -b 1-32`"
    
                if test -e $GLINSTALLDIR/$GLLIBRARY ; then 
                    if ! [ "$TEST1" == "$TEST2" ]; then
                        echo "Creating a new OpenGL library backup file"
                        echo "$GLLIBRARY.$BACKUP_SUFFIX in $BACKUP_DIR."
                        echo ""
                        mv $GLINSTALLDIR/$GLLIBRARY $BACKUP_DIR/$GLLIBRARY.$BACKUP_SUFFIX
                    fi
                fi

                if [ "$MACHINEARCH" == "x86_64" ]; then
                    TEST1="`$MD5SUMBIN -b opengl/lib32/$GLLIBRARY | cut -b 1-32`"
                    TEST2="`$MD5SUMBIN -b $GLINSTALL32DIR/$GLLIBRARY | cut -b 1-32`"
    
                    if test -e $GLINSTALL32DIR/$GLLIBRARY ; then 
                        if ! [ "$TEST1" == "$TEST2" ]; then
                            echo "Creating a new OpenGL library backup file"
                            echo "$GLLIBRARY.$BACKUP_SUFFIX in $BACKUP_DIR/lib32."
                            echo ""
                            mv $GLINSTALL32DIR/$GLLIBRARY $BACKUP_DIR/lib32/$GLLIBRARY.$BACKUP_SUFFIX
                        fi
                    fi
                fi

                TEST1="`$MD5SUMBIN -b opengl/$GLXLIBRARY | cut -b 1-32`"
                TEST2="`$MD5SUMBIN -b $GLXINSTALLDIR/$GLXLIBRARY | cut -b 1-32`"
    
                if test -e $GLXINSTALLDIR/$GLXLIBRARY ; then 
                    if ! [ "$TEST1" == "$TEST2" ]; then
                        echo "Creating a new GLX library backup file"
                        echo "$GLXLIBRARY.$BACKUP_SUFFIX in $BACKUP_DIR."
                        echo ""
                        mv $GLXINSTALLDIR/$GLXLIBRARY $BACKUP_DIR/$GLXLIBRARY.$BACKUP_SUFFIX
                    fi
                fi

                TEST1="`$MD5SUMBIN -b opengl/include/$GLHEADER | cut -b 1-32`"
                TEST2="`$MD5SUMBIN -b $GLHEADERSINSTALLDIR/$GLHEADER | cut -b 1-32`"
    
                if test -e $GLHEADERSINSTALLDIR/$GLHEADER ; then 
                    if ! [ "$TEST1" == "$TEST2" ]; then
                        echo "Creating a new $GLHEADER backup file"
                        echo "$GLHEADER.$BACKUP_SUFFIX in $BACKUP_DIR."
                        echo ""
                        mv $GLHEADERSINSTALLDIR/$GLHEADER $BACKUP_DIR/$GLHEADER.$BACKUP_SUFFIX
                    fi
                fi

                TEST1="`$MD5SUMBIN -b opengl/include/$GLEXTHEADER | cut -b 1-32`"
                TEST2="`$MD5SUMBIN -b $GLHEADERSINSTALLDIR/$GLEXTHEADER | cut -b 1-32`"
    
                if test -e $GLHEADERSINSTALLDIR/$GLEXTHEADER ; then 
                    if ! [ "$TEST1" == "$TEST2" ]; then
                        echo "Creating a new $GLEXTHEADER backup file"
                        echo "$GLEXTHEADER.$BACKUP_SUFFIX in $BACKUP_DIR."
                        echo ""
                        mv $GLHEADERSINSTALLDIR/$GLEXTHEADER $BACKUP_DIR/$GLEXTHEADER.$BACKUP_SUFFIX
                    fi
                fi
                
                ####################################
                # Install the library
                ####################################
                TEST1="`$MD5SUMBIN -b opengl/$GLLIBRARY | cut -b 1-32`"
                TEST2="`$MD5SUMBIN -b $GLINSTALLDIR/$GLLIBRARY | cut -b 1-32`"
    
                if ! [ "$TEST1" == "$TEST2" ]; then
                    echo "Installing the OpenGL library file to $GLINSTALLDIR"
                    echo ""
                    cp -f $TOP/opengl/$GLLIBRARY $GLINSTALLDIR
                    rm -f $GLINSTALLDIR/libGL.so*
                    ln -s $GLINSTALLDIR/$GLLIBRARY $GLINSTALLDIR/libGL.so.1.2.0
                    ln -s $GLINSTALLDIR/libGL.so.1.2.0 $GLINSTALLDIR/libGL.so.1
                    ln -s $GLINSTALLDIR/libGL.so.1 $GLINSTALLDIR/libGL.so
                else
                    echo -e "\033[1mInstalled OpenGL library is the same file as the installer"
                    echo -e "version, not installing OpenGL library file.\033[0m"
                    echo ""
                fi

                if [ "$MACHINEARCH" == "x86_64" ]; then
                    TEST1="`$MD5SUMBIN -b opengl/lib32/$GLLIBRARY | cut -b 1-32`"
                    TEST2="`$MD5SUMBIN -b $GLINSTALL32DIR/$GLLIBRARY | cut -b 1-32`"
    
                    if ! [ "$TEST1" == "$TEST2" ]; then
                        echo "Installing the OpenGL library file to $GLINSTALL32DIR"
                        echo ""
                        cp -f $TOP/opengl/lib32/$GLLIBRARY $GLINSTALL32DIR
                        rm -f $GLINSTALL32DIR/libGL.so*
                        ln -s $GLINSTALL32DIR/$GLLIBRARY $GLINSTALL32DIR/libGL.so.1.2.0
                        ln -s $GLINSTALL32DIR/libGL.so.1.2.0 $GLINSTALL32DIR/libGL.so.1
                        ln -s $GLINSTALL32DIR/libGL.so.1 $GLINSTALL32DIR/libGL.so
                    else
                        echo -e "\033[1mInstalled OpenGL library is the same file as the installer"
                        echo -e "version, not installing OpenGL library (32bit) file.\033[0m"
                        echo ""
                    fi
                fi

                TEST1="`$MD5SUMBIN -b opengl/$GLXLIBRARY | cut -b 1-32`"
                TEST2="`$MD5SUMBIN -b $GLXINSTALLDIR/$GLXLIBRARY | cut -b 1-32`"
    
                if ! [ "$TEST1" == "$TEST2" ]; then
                    echo "Installing the GLX library file to $GLXINSTALLDIR"
                    echo ""
                    cp -f $TOP/opengl/$GLXLIBRARY $GLXINSTALLDIR
                else
                    echo -e "\033[1mInstalled GLX library is the same file as the installer"
                    echo -e "version, not installing GLX library file.\033[0m"
                    echo ""
                fi

                TEST1="`$MD5SUMBIN -b opengl/include/$GLEXTHEADER | cut -b 1-32`"
                TEST2="`$MD5SUMBIN -b $GLHEADERSINSTALLDIR/$GLEXTHEADER | cut -b 1-32`"
    
                if ! [ "$TEST1" == "$TEST2" ]; then
                    echo "Installing the glext.h header file to $GLHEADERSINSTALLDIR"
                    echo ""
                    cp -f $TOP/opengl/$GLEXTHEADER $GLHEADERSINSTALLDIR
                else
                    echo -e "\033[1mInstalled glext.h header is the same file as the installer"
                    echo -e "version, not installing glext.h file.\033[0m"
                    echo ""
                fi

                echo "Updating library cache"
                echo ""
                /sbin/ldconfig
    
            ####################################
            # Yes there is a backup dir
            ####################################
            else
    
                ####################################
                # Yes backup dir is there!
                # Is there a backup?
                ####################################
                if test -e $BACKUP_DIR/$GLLIBRARY.$BACKUP_SUFFIX ; then
    
                    ####################################
                    # Yes there is a backup!
                    # Is the same as the current file?
                    ####################################
                    TEST1="`$MD5SUMBIN -b $BACKUP_DIR/$GLLIBRARY.$BACKUP_SUFFIX | cut -b 1-32`"
                    TEST2="`$MD5SUMBIN -b $GLINSTALLDIR/$GLLIBRARY | cut -b 1-32`"
    
                    if ! [ "$TEST1" == "$TEST2" ] ; then
        
                        ####################################
                        # No but is the current installed
                        # driver the same as the installer?
                        ####################################
                        TEST1="`$MD5SUMBIN -b opengl/$GLLIBRARY | cut -b 1-32`"
                        TEST2="`$MD5SUMBIN -b $GLINSTALLDIR/$GLLIBRARY | cut -b 1-32`"
        
                        if [ "$TEST1" == "$TEST2" ]; then
        
                            ####################################
                            # Yes they are so do nothing
                            ####################################
                            echo -e "\033[1mInstalled OpenGL library file is the same file as the installer"
                            echo -e "version, not installing OpenGL library file.\033[0m"
                            echo ""
                            INSTALLGLLIBRARY=1
                        else
        
                            ####################################
                            # No they aren't so create a new 
                            # backup file
                            ####################################
                            CURRENTDATE=`date +%y%m%d%H%M%S`
                            echo "Renaming current OpenGL library backup file"
                            echo "to $GLLIBRARY.$CURRENTDATE in $BACKUP_DIR."
                            echo ""
                            mv $BACKUP_DIR/$GLLIBRARY.$BACKUP_SUFFIX $BACKUP_DIR/$GLLIBRARY.$CURRENTDATE
                                
                            echo "Creating an OpenGL library backup file"
                            echo "$GLLIBRARY.$BACKUP_SUFFIX in $BACKUP_DIR."
                            echo ""
                            mv $GLINSTALLDIR/$GLLIBRARY $BACKUP_DIR/$GLLIBRARY.$BACKUP_SUFFIX
                        fi
                    else
        
                        ####################################
                        # Yes so do nothing
                        ####################################
                        echo -e "\033[1mCurrent OpenGL library backup file is the same"
                        echo -e "as the currently installed file. No backups"
                        echo -e "created.\033[0m"
                        echo ""
                    fi
                else
    
                    ####################################
                    # no backups create a new on
                    # only if library is not the same 
                    # file.
                    ####################################
                    TEST3="`$MD5SUMBIN -b opengl/$GLLIBRARY | cut -b 1-32`"
                    TEST4="`$MD5SUMBIN -b $GLINSTALLDIR/$GLLIBRARY | cut -b 1-32`"
                    if [ "$TEST3" == "$TEST4" ]; then
                        echo -e "\033[1mCurrently installed OpenGL library file is the same as the installer file."
                        echo -e "OpenGL library file not installed.\033[0m"
                        echo ""
                        INSTALLGLLIBRARY=1
                    else
                        echo "Creating an OpenGL library backup file in $BACKUP_DIR."
                        echo ""
                        mv $GLINSTALLDIR/$GLLIBRARY $BACKUP_DIR/$GLLIBRARY.$BACKUP_SUFFIX
                    fi
                fi

                if [ "$MACHINEARCH" == "x86_64" ]; then
                    ####################################
                    # Yes backup dir is there!
                    # Is there a backup?
                    ####################################
                    if test -e $BACKUP_DIR/lib32/$GLLIBRARY.$BACKUP_SUFFIX ; then
        
                        ####################################
                        # Yes there is a backup!
                        # Is the same as the current file?
                        ####################################
                        TEST1="`$MD5SUMBIN -b $BACKUP_DIR/lib32/$GLLIBRARY.$BACKUP_SUFFIX | cut -b 1-32`"
                        TEST2="`$MD5SUMBIN -b $GLINSTALL32DIR/$GLLIBRARY | cut -b 1-32`"
        
                        if ! [ "$TEST1" == "$TEST2" ] ; then
            
                            ####################################
                            # No but is the current installed
                            # driver the same as the installer?
                            ####################################
                            TEST1="`$MD5SUMBIN -b opengl/lib32/$GLLIBRARY | cut -b 1-32`"
                            TEST2="`$MD5SUMBIN -b $GLINSTALL32DIR/$GLLIBRARY | cut -b 1-32`"
        
                            if [ "$TEST1" == "$TEST2" ]; then
            
                                ####################################
                                # Yes they are so do nothing
                                ####################################
                                echo -e "\033[1mInstalled OpenGL library file is the same file as the installer"
                                echo -e "version, not installing OpenGL library (32bit) file.\033[0m"
                                echo ""
                                INSTALLGLLIBRARY32=1
                            else
        
                                ####################################
                                # No they aren't so create a new 
                                # backup file
                                ####################################
                                CURRENTDATE=`date +%y%m%d%H%M%S`
                                echo "Renaming current OpenGL library backup file"
                                echo "to $GLLIBRARY.$CURRENTDATE in $BACKUP_DIR."
                                echo ""
                                mv $BACKUP_DIR/lib32/$GLLIBRARY.$BACKUP_SUFFIX $BACKUP_DIR/lib32/$GLLIBRARY.$CURRENTDATE
                                    
                                echo "Creating an OpenGL library backup file"
                                echo "$GLLIBRARY.$BACKUP_SUFFIX in $BACKUP_DIR/lib32."
                                echo ""
                                mv $GLINSTALL32DIR/$GLLIBRARY $BACKUP_DIR/lib32/$GLLIBRARY.$BACKUP_SUFFIX
                            fi
                        else
        
                            ####################################
                            # Yes so do nothing
                            ####################################
                            echo -e "\033[1mCurrent OpenGL library (32bit) backup file is the same"
                            echo -e "as the currently installed file. No backups"
                            echo -e "created.\033[0m"
                            echo ""
                        fi
                    fi
                else
    
                    ####################################
                    # no backups create a new on
                    # only if library is not the same 
                    # file.
                    ####################################
                    TEST3="`$MD5SUMBIN -b opengl/$GLLIBRARY | cut -b 1-32`"
                    TEST4="`$MD5SUMBIN -b $GLINSTALLDIR/$GLLIBRARY | cut -b 1-32`"
                    if [ "$TEST3" == "$TEST4" ]; then
                        echo -e "\033[1mCurrently installed OpenGL library file is the same as the installer file."
                        echo -e "OpenGL library file not installed.\033[0m"
                        echo ""
                        INSTALLGLLIBRARY=1
                    else
                        echo "Creating an OpenGL library backup file in $BACKUP_DIR."
                        echo ""
                        mv $GLINSTALLDIR/$GLLIBRARY $BACKUP_DIR/$GLLIBRARY.$BACKUP_SUFFIX
                    fi

                    if [ "$MACHINEARCH" == "x86_64" ]; then
                        TEST3="`$MD5SUMBIN -b opengl/lib32/$GLLIBRARY | cut -b 1-32`"
                        TEST4="`$MD5SUMBIN -b $GLINSTALL32DIR/$GLLIBRARY | cut -b 1-32`"
                        if [ "$TEST3" == "$TEST4" ]; then
                            echo -e "\033[1mCurrently installed OpenGL library (32bit) file is the same as the installer file."
                            echo -e "OpenGL library file not installed.\033[0m"
                            echo ""
                            INSTALLGLLIBRARY32=1
                        else
                            echo "Creating an OpenGL library (32bit) backup file in $BACKUP_DIR."
                            echo ""
                            mv $GLINSTALL32DIR/$GLLIBRARY $BACKUP_DIR/lib32/$GLLIBRARY.$BACKUP_SUFFIX
                        fi
                    fi
                fi
        
                ####################################
                # Install the library file
                ####################################
                if [ "$INSTALLGLLIBRARY" == "0" ]; then
            
                    if test -e $GLINSTALLDIR/$GLLIBRARY ; then
                        TEST5="`$MD5SUMBIN -b opengl/$GLLIBRARY | cut -b 1-32`"
                        TEST6="`$MD5SUMBIN -b $GLINSTALLDIR/$GLLIBRARY | cut -b 1-32`"
                    else
                        TEST5="5"
                        TEST6="6"
                    fi
                        
                    if ! [ "$TEST5" == "$TEST6" ]; then
                        echo "Installing the $GLLIBRARY OpenGL library file to $GLINSTALLDIR"
                        echo ""
                        cp -f $TOP/opengl/$GLLIBRARY $GLINSTALLDIR
                    else
                        echo -e "\033[1mInstalled OpenGL library file is the same file as the installer"
                        echo -e "version, not installing the OpenGL library file.\033[0m"
                        echo ""
                    fi
                fi

                if [ "$MACHINEARCH" == "x86_64" ]; then
                    if [ "$INSTALLGLLIBRARY32" == "0" ]; then
            
                        if test -e $GLINSTALL32DIR/$GLLIBRARY ; then
                            TEST5="`$MD5SUMBIN -b opengl/lib32/$KERNELDRIVER | cut -b 1-32`"
                            TEST6="`$MD5SUMBIN -b $GLINSTALL32DIR/$GLLIBRARY | cut -b 1-32`"
                        else
                            TEST5="5"
                            TEST6="6"
                        fi
                        
                        if ! [ "$TEST5" == "$TEST6" ]; then
                            echo "Installing the $GLLIBRARY OpenGL library (32bit) file to $GLINSTALL32DIR"
                            echo ""
                            cp -f $TOP/opengl/lib32/$GLLIBRARY $GLINSTALL32DIR
                        else
                            echo -e "\033[1mInstalled OpenGL library (32bit) file is the same file as the installer"
                            echo -e "version, not installing the OpenGL library file.\033[0m"
                            echo ""
                        fi
                    fi
                fi
            #libGL

                ####################################
                # Yes backup dir is there!
                # Is there a backup?
                ####################################
                if test -e $BACKUP_DIR/$GLXLIBRARY.$BACKUP_SUFFIX ; then
    
                    ####################################
                    # Yes there is a backup!
                    # Is the same as the current file?
                    ####################################
                    TEST1="`$MD5SUMBIN -b $BACKUP_DIR/$GLXLIBRARY.$BACKUP_SUFFIX | cut -b 1-32`"
                    TEST2="`$MD5SUMBIN -b $GLXINSTALLDIR/$GLXLIBRARY | cut -b 1-32`"
    
                    if ! [ "$TEST1" == "$TEST2" ] ; then
        
                        ####################################
                        # No but is the current installed
                        # driver the same as the installer?
                        ####################################
                        TEST1="`$MD5SUMBIN -b opengl/$GLXLIBRARY | cut -b 1-32`"
                        TEST2="`$MD5SUMBIN -b $GLXINSTALLDIR/$GLXLIBRARY | cut -b 1-32`"
        
                        if [ "$TEST1" == "$TEST2" ]; then
        
                            ####################################
                            # Yes they are so do nothing
                            ####################################
                            echo -e "\033[1mInstalled GLX library file is the same file as the installer"
                            echo -e "version, not installing GLX library file.\033[0m"
                            echo ""
                            INSTALLGLXLIBRARY=1
                        else
        
                            ####################################
                            # No they aren't so create a new 
                            # backup file
                            ####################################
                            CURRENTDATE=`date +%y%m%d%H%M%S`
                            echo "Renaming current GLX library backup file"
                            echo "to $GLXLIBRARY.$CURRENTDATE in $BACKUP_DIR."
                            echo ""
                            mv $BACKUP_DIR/$GLXLIBRARY.$BACKUP_SUFFIX $BACKUP_DIR/$GLXLIBRARY.$CURRENTDATE
                                
                            echo "Creating an GLX library backup file"
                            echo "$GLXLIBRARY.$BACKUP_SUFFIX in $BACKUP_DIR."
                            echo ""
                            mv $GLXINSTALLDIR/$GLXLIBRARY $BACKUP_DIR/$GLXLIBRARY.$BACKUP_SUFFIX
                        fi
                    else
        
                        ####################################
                        # Yes so do nothing
                        ####################################
                        echo -e "\033[1mCurrent GLX library backup file is the same"
                        echo -e "as the currently installed file. No backups"
                        echo -e "created.\033[0m"
                        echo ""
                    fi
                else
    
                    ####################################
                    # no backups create a new on
                    # only if library is not the same 
                    # file.
                    ####################################
                    TEST3="`$MD5SUMBIN -b opengl/$GLXLIBRARY | cut -b 1-32`"
                    TEST4="`$MD5SUMBIN -b $GLXINSTALLDIR/$GLXLIBRARY | cut -b 1-32`"
                    if [ "$TEST3" == "$TEST4" ]; then
                        echo -e "\033[1mCurrently installed GLX library file is the same as the installer file."
                        echo -e "GLX library file not installed.\033[0m"
                        echo ""
                        INSTALLGLXLIBRARY=1
                    else
                        echo "Creating an GLX library backup file in $BACKUP_DIR."
                        echo ""
                        mv $GLXINSTALLDIR/$GLXLIBRARY $BACKUP_DIR/$GLXLIBRARY.$BACKUP_SUFFIX
                    fi
                fi
        
                ####################################
                # Install the library file
                ####################################
                if [ "$INSTALLGLXLIBRARY" == "0" ]; then
            
                    if test -e $GLXINSTALLDIR/$GLXLIBRARY ; then
                        TEST5="`$MD5SUMBIN -b opengl/$GLXLIBRARY | cut -b 1-32`"
                        TEST6="`$MD5SUMBIN -b $GLXINSTALLDIR/$GLXLIBRARY | cut -b 1-32`"
                    else
                        TEST5="5"
                        TEST6="6"
                    fi
                        
                    if ! [ "$TEST5" == "$TEST6" ]; then
                        echo "Installing the $GLXLIBRARY GLX library file to $GLXINSTALLDIR"
                        echo ""
                        cp -f $TOP/opengl/$GLXLIBRARY $GLXINSTALLDIR
                    else
                        echo -e "\033[1mInstalled GLX library file is the same file as the installer"
                        echo -e "version, not installing the GLX library file.\033[0m"
                        echo ""
                    fi
                fi
            #libglx

                ####################################
                # Yes backup dir is there!
                # Is there a backup?
                ####################################
                if test -e $BACKUP_DIR/$GLHEADER.$BACKUP_SUFFIX ; then
    
                    ####################################
                    # Yes there is a backup!
                    # Is the same as the current file?
                    ####################################
                    TEST1="`$MD5SUMBIN -b $BACKUP_DIR/$GLHEADER.$BACKUP_SUFFIX | cut -b 1-32`"
                    TEST2="`$MD5SUMBIN -b $GLHEADERSINSTALLDIR/$GLHEADER | cut -b 1-32`"
    
                    if ! [ "$TEST1" == "$TEST2" ] ; then
        
                        ####################################
                        # No but is the current installed
                        # driver the same as the installer?
                        ####################################
                        TEST1="`$MD5SUMBIN -b opengl/include/$GLHEADER | cut -b 1-32`"
                        TEST2="`$MD5SUMBIN -b $GLHEADERSINSTALLDIR/$GLHEADER | cut -b 1-32`"
        
                        if [ "$TEST1" == "$TEST2" ]; then
        
                            ####################################
                            # Yes they are so do nothing
                            ####################################
                            echo -e "\033[1mInstalled $GLHEADER file is the same file as the installer"
                            echo -e "version, not installing $GLHEADER file.\033[0m"
                            echo ""
                            INSTALLGLHEADER=1
                        else
        
                            ####################################
                            # No they aren't so create a new 
                            # backup file
                            ####################################
                            CURRENTDATE=`date +%y%m%d%H%M%S`
                            echo "Renaming current $GLHEADER backup file"
                            echo "to $GLHEADER.$CURRENTDATE in $BACKUP_DIR."
                            echo ""
                            mv $BACKUP_DIR/$GLHEADER.$BACKUP_SUFFIX $BACKUP_DIR/$GLHEADER.$CURRENTDATE
                                
                            echo "Creating an $GLHEADER library backup file"
                            echo "$GLHEADER.$BACKUP_SUFFIX in $BACKUP_DIR."
                            echo ""
                            mv $GLHEADERSINSTALLDIR/$GLHEADER $BACKUP_DIR/$GLHEADER.$BACKUP_SUFFIX
                        fi
                    else
        
                        ####################################
                        # Yes so do nothing
                        ####################################
                        echo -e "\033[1mCurrent $GLHEADER backup file is the same"
                        echo -e "as the currently installed file."
                        echo -e "No backups created.\033[0m"
                        echo ""
                    fi
                else
    
                    ####################################
                    # no backups create a new on
                    # only if library is not the same 
                    # file.
                    ####################################
                    TEST3="`$MD5SUMBIN -b opengl/include/$GLHEADER | cut -b 1-32`"
                    TEST4="`$MD5SUMBIN -b $GLHEADERSINSTALLDIR/$GLHEADER | cut -b 1-32`"
                    if [ "$TEST3" == "$TEST4" ]; then
                        echo -e "\033[1mCurrently installed $GLHEADER file is the same as the installer file."
                        echo -e "$GLHEADER file not installed.\033[0m"
                        echo ""
                        INSTALLGLHEADER=1
                    else
                        echo "Creating a $GLHEADER backup file in $BACKUP_DIR."
                        echo ""
                        mv $GLHEADERSINSTALLDIR/$GLHEADER $BACKUP_DIR/$GLHEADER.$BACKUP_SUFFIX
                    fi
                fi
        
                ####################################
                # Install the library file
                ####################################
                if [ "$INSTALLGLHEADER" == "0" ]; then
            
                    if test -e $GLHEADERSINSTALLDIR/$GLHEADER ; then
                        TEST5="`$MD5SUMBIN -b opengl/include/$GLHEADER | cut -b 1-32`"
                        TEST6="`$MD5SUMBIN -b $GLHEADERSINSTALLDIR/$GLHEADER | cut -b 1-32`"
                    else
                        TEST5="5"
                        TEST6="6"
                    fi
                        
                    if ! [ "$TEST5" == "$TEST6" ]; then
                        echo "Installing the $GLHEADER file to $GLHEADERSINSTALLDIR"
                        echo ""
                        cp -f $TOP/opengl/include/$GLHEADER $GLHEADERSINSTALLDIR
                    else
                        echo -e "\033[1mInstalled $GLHEADER file is the same file as the installer"
                        echo -e "version, not installing the $GLHEADER file.\033[0m"
                        echo ""
                    fi
                fi
            #gl.h

                ####################################
                # Yes backup dir is there!
                # Is there a backup?
                ####################################
                if test -e $BACKUP_DIR/$GLEXTHEADER.$BACKUP_SUFFIX ; then
    
                    ####################################
                    # Yes there is a backup!
                    # Is the same as the current file?
                    ####################################
                    TEST1="`$MD5SUMBIN -b $BACKUP_DIR/$GLEXTHEADER.$BACKUP_SUFFIX | cut -b 1-32`"
                    TEST2="`$MD5SUMBIN -b $GLHEADERSINSTALLDIR/$GLEXTHEADER | cut -b 1-32`"
    
                    if ! [ "$TEST1" == "$TEST2" ] ; then
        
                        ####################################
                        # No but is the current installed
                        # driver the same as the installer?
                        ####################################
                        TEST1="`$MD5SUMBIN -b opengl/include/$GLEXTHEADER | cut -b 1-32`"
                        TEST2="`$MD5SUMBIN -b $GLHEADERSINSTALLDIR/$GLEXTHEADER | cut -b 1-32`"
        
                        if [ "$TEST1" == "$TEST2" ]; then
        
                            ####################################
                            # Yes they are so do nothing
                            ####################################
                            echo -e "\033[1mInstalled $GLEXTHEADER file is the same file as the installer"
                            echo -e "version, not installing $GLEXTHEADER file.\033[0m"
                            echo ""
                            INSTALLGLEXTHEADER=1
                        else
        
                            ####################################
                            # No they aren't so create a new 
                            # backup file
                            ####################################
                            CURRENTDATE=`date +%y%m%d%H%M%S`
                            echo "Renaming current $GLEXTHEADER backup file"
                            echo "to $GLEXTHEADER.$CURRENTDATE in $BACKUP_DIR."
                            echo ""
                            mv $BACKUP_DIR/$GLEXTHEADER.$BACKUP_SUFFIX $BACKUP_DIR/$GLEXTHEADER.$CURRENTDATE
                                
                            echo "Creating an $GLEXTHEADER library backup file"
                            echo "$GLEXTHEADER.$BACKUP_SUFFIX in $BACKUP_DIR."
                            echo ""
                            mv $GLHEADERSINSTALLDIR/$GLEXTHEADER $BACKUP_DIR/$GLEXTHEADER.$BACKUP_SUFFIX
                        fi
                    else
        
                        ####################################
                        # Yes so do nothing
                        ####################################
                        echo -e "\033[1mCurrent $GLEXTHEADER backup file is the same"
                        echo -e "as the currently installed file."
                        echo -e "No backups created.\033[0m"
                        echo ""
                    fi
                else
    
                    ####################################
                    # no backups create a new on
                    # only if library is not the same 
                    # file.
                    ####################################
                    TEST3="`$MD5SUMBIN -b opengl/include/$GLEXTHEADER | cut -b 1-32`"
                    TEST4="`$MD5SUMBIN -b $GLHEADERSINSTALLDIR/$GLEXTHEADER | cut -b 1-32`"
                    if [ "$TEST3" == "$TEST4" ]; then
                        echo -e "\033[1mCurrently installed $GLEXTHEADER file is the same as the installer file."
                        echo -e "$GLEXTHEADER file not installed.\033[0m"
                        echo ""
                        INSTALLGLEXTHEADER=1
                    else
                        echo "Creating a $GLEXTHEADER backup file in $BACKUP_DIR."
                        echo ""
                        mv $GLHEADERSINSTALLDIR/$GLEXTHEADER $BACKUP_DIR/$GLEXTHEADER.$BACKUP_SUFFIX
                    fi
                fi
        
                ####################################
                # Install the library file
                ####################################
                if [ "$INSTALLGLEXTHEADER" == "0" ]; then
            
                    if test -e $GLHEADERSINSTALLDIR/$GLEXTHEADER ; then
                        TEST5="`$MD5SUMBIN -b opengl/include/$GLEXTHEADER | cut -b 1-32`"
                        TEST6="`$MD5SUMBIN -b $GLHEADERSINSTALLDIR/$GLEXTHEADER | cut -b 1-32`"
                    else
                        TEST5="5"
                        TEST6="6"
                    fi
                        
                    if ! [ "$TEST5" == "$TEST6" ]; then
                        echo "Installing the $GLEXTHEADER file to $GLHEADERSINSTALLDIR"
                        echo ""
                        cp -f $TOP/opengl/include/$GLEXTHEADER $GLHEADERSINSTALLDIR
                    else
                        echo -e "\033[1mInstalled $GLEXTHEADER file is the same file as the installer"
                        echo -e "version, not installing the $GLEXTHEADER file.\033[0m"
                        echo ""
                    fi
                fi

                ####################################
                # Yes backup dir is there!
                # Is there a backup?
                ####################################
                if test -e $BACKUP_DIR/$GLXHEADER.$BACKUP_SUFFIX ; then
    
                    ####################################
                    # Yes there is a backup!
                    # Is the same as the current file?
                    ####################################
                    TEST1="`$MD5SUMBIN -b $BACKUP_DIR/$GLXHEADER.$BACKUP_SUFFIX | cut -b 1-32`"
                    TEST2="`$MD5SUMBIN -b $GLHEADERSINSTALLDIR/$GLXHEADER | cut -b 1-32`"
    
                    if ! [ "$TEST1" == "$TEST2" ] ; then
        
                        ####################################
                        # No but is the current installed
                        # driver the same as the installer?
                        ####################################
                        TEST1="`$MD5SUMBIN -b opengl/include/$GLXHEADER | cut -b 1-32`"
                        TEST2="`$MD5SUMBIN -b $GLHEADERSINSTALLDIR/$GLXHEADER | cut -b 1-32`"
        
                        if [ "$TEST1" == "$TEST2" ]; then
        
                            ####################################
                            # Yes they are so do nothing
                            ####################################
                            echo -e "\033[1mInstalled $GLXHEADER file is the same file as the installer"
                            echo -e "version, not installing $GLXHEADER file.\033[0m"
                            echo ""
                            INSTALLGLXHEADER=1
                        else
        
                            ####################################
                            # No they aren't so create a new 
                            # backup file
                            ####################################
                            CURRENTDATE=`date +%y%m%d%H%M%S`
                            echo "Renaming current $GLXHEADER backup file"
                            echo "to $GLXHEADER.$CURRENTDATE in $BACKUP_DIR."
                            echo ""
                            mv $BACKUP_DIR/$GLXHEADER.$BACKUP_SUFFIX $BACKUP_DIR/$GLXHEADER.$CURRENTDATE
                                
                            echo "Creating an $GLXHEADER library backup file"
                            echo "$GLXHEADER.$BACKUP_SUFFIX in $BACKUP_DIR."
                            echo ""
                            mv $GLHEADERSINSTALLDIR/$GLXHEADER $BACKUP_DIR/$GLXHEADER.$BACKUP_SUFFIX
                        fi
                    else
        
                        ####################################
                        # Yes so do nothing
                        ####################################
                        echo -e "\033[1mCurrent $GLXHEADER backup file is the same"
                        echo -e "as the currently installed file."
                        echo -e "No backups created.\033[0m"
                        echo ""
                    fi
                else
    
                    ####################################
                    # no backups create a new on
                    # only if library is not the same 
                    # file.
                    ####################################
                    TEST3="`$MD5SUMBIN -b opengl/include/$GLXHEADER | cut -b 1-32`"
                    TEST4="`$MD5SUMBIN -b $GLHEADERSINSTALLDIR/$GLXHEADER | cut -b 1-32`"
                    if [ "$TEST3" == "$TEST4" ]; then
                        echo -e "\033[1mCurrently installed $GLXHEADER file is the same as the installer file."
                        echo -e "$GLXHEADER file not installed.\033[0m"
                        echo ""
                        INSTALLGLXHEADER=1
                    else
                        echo "Creating a $GLXHEADER backup file in $BACKUP_DIR."
                        echo ""
                        mv $GLHEADERSINSTALLDIR/$GLXHEADER $BACKUP_DIR/$GLXHEADER.$BACKUP_SUFFIX
                    fi
                fi
        
                ####################################
                # Install the library file
                ####################################
                if [ "$INSTALLGLXHEADER" == "0" ]; then
            
                    if test -e $GLHEADERSINSTALLDIR/$GLXHEADER ; then
                        TEST5="`$MD5SUMBIN -b opengl/include/$GLXHEADER | cut -b 1-32`"
                        TEST6="`$MD5SUMBIN -b $GLHEADERSINSTALLDIR/$GLXHEADER | cut -b 1-32`"
                    else
                        TEST5="5"
                        TEST6="6"
                    fi
                        
                    if ! [ "$TEST5" == "$TEST6" ]; then
                        echo "Installing the $GLXHEADER file to $GLHEADERSINSTALLDIR"
                        echo ""
                        cp -f $TOP/opengl/include/$GLXHEADER $GLHEADERSINSTALLDIR
                    else
                        echo -e "\033[1mInstalled $GLXHEADER file is the same file as the installer"
                        echo -e "version, not installing the $GLXHEADER file.\033[0m"
                        echo ""
                    fi
                fi
            #glx.h

                echo "Updating library cache"
                echo ""
                /sbin/ldconfig

            #END
            fi
                
        else

            ########################################
            # NO we do not have a module
            # Is there a backup dir
            ####################################
            if ! test -e $BACKUP_DIR ; then
    
                ####################################
                # No create it
                ####################################
                mkdir $BACKUP_DIR
            fi

            echo "Installing the OpenGL library file to $GLINSTALLDIR"
            echo ""
            rm -f $GLINSTALLDIR/libGL.so*
            cp -f $TOP/opengl/$GLLIBRARY $GLINSTALLDIR
            ln -s $GLINSTALLDIR/$GLLIBRARY $GLINSTALLDIR/libGL.so.1.2.0
            ln -s $GLINSTALLDIR/libGL.so.1.2.0 $GLINSTALLDIR/libGL.so.1
            ln -s $GLINSTALLDIR/libGL.so.1 $GLINSTALLDIR/libGL.so

            if [ "$MACHINEARCH" == "x86_64" ]; then
                rm -f $GLINSTALL32DIR/libGL.so*
                cp -f $TOP/opengl/lib32/$GLLIBRARY $GLINSTALL32DIR
                ln -s $GLINSTALL32DIR/$GLLIBRARY $GLINSTALL32DIR/libGL.so.1.2.0
                ln -s $GLINSTALL32DIR/libGL.so.1.2.0 $GLINSTALL32DIR/libGL.so.1
                ln -s $GLINSTALL32DIR/libGL.so.1 $GLINSTALL32DIR/libGL.so
            fi

            echo "Installing the GLX library file to $GLXINSTALLDIR"
            echo ""
            cp -f $TOP/opengl/$GLXLIBRARY $GLXINSTALLDIR

            echo "Installing the $GLHEADER header file to $GLHEADERSINSTALLDIR"
            echo ""
            cp -f $TOP/opengl/include/$GLHEADER $GLHEADERSINSTALLDIR

            echo "Installing the $GLEXTHEADER header file to $GLHEADERSINSTALLDIR"
            echo ""
            cp -f $TOP/opengl/include/$GLEXTHEADER $GLHEADERSINSTALLDIR

            echo "Installing the $GLXHEADER header file to $GLHEADERSINSTALLDIR"
            echo ""
            cp -f $TOP/opengl/include/$GLXHEADER $GLHEADERSINSTALLDIR

            echo "Updating library cache"
            echo ""
            /sbin/ldconfig
        fi
    fi
}

function Uninstall
{
    BACKUPS=0
    if test -e $BACKUP_DIR ; then
        if test -e $BACKUP_DIR/$XDRIVER.$BACKUP_SUFFIX ; then
             BACKUPS=1
        fi
        if test -e $BACKUP_DIR/$KERNELDRIVER.$BACKUP_SUFFIX ; then
             BACKUPS=1
        fi
        if test -e $BACKUP_DIR/$GLLIBRARY.$BACKUP_SUFFIX ; then
             BACKUPS=1
        fi
        if test -e $BACKUP_DIR/$GLXLIBRARY.$BACKUP_SUFFIX ; then
             BACKUPS=1
        fi
        if test -e $BACKUP_DIR/$GLHEADER.$BACKUP_SUFFIX ; then
             BACKUPS=1
        fi
        if test -e $BACKUP_DIR/$GLEXTHEADER.$BACKUP_SUFFIX ; then
             BACKUPS=1
        fi
        
        if [ "$BACKUPS" == "1" ] ; then
            echo -e "\033[1mDo you wish to uninstall or recover"
            echo -en "a backup or cancel \033[0m$YELLOW\033[1m(u/r/c)\033[0m? "
            read YESNOCANCEL
            echo ""
            if [ $YESNOCANCEL = 'u' -o $YESNOCANCEL = 'U' ]; then
                if test -e $XINSTALLDIR/$XDRIVER ; then 
                    echo "Uninstalling X driver"
                    rm -f $XINSTALLDIR/$XDRIVER
                fi
                if test -e $KNLINSTALLDIR/$KERNELDRIVER ; then 
                    echo "Uninstalling kernel module"
                    rm -f $KNLINSTALLDIR/$KERNELDRIVER
                fi
                if test -e $GLINSTALLDIR/$GLLIBRARY ; then 
                    echo "Uninstalling OpenGL library"
                    rm -f $GLINSTALLDIR/$GLLIBRARY
                    if test -e $GLINSTALLDIR/libGL.so ; then
                        rm -f $GLINSTALLDIR/libGL.so
                    fi
                    if test -e $GLINSTALLDIR/libGL.so.1 ; then
                        rm -f $GLINSTALLDIR/libGL.so.1
                    fi
                    if test -e $GLINSTALLDIR/libGL.so.1.2.0 ; then
                        rm -f $GLINSTALLDIR/libGL.so.1.2.0
                    fi
                    if [ "$MACHINEARCH" == "x86_64" ]; then
                        rm -f $GLINSTALL32DIR/$GLLIBRARY
                        if test -e $GLINSTALL32DIR/libGL.so ; then
                            rm -f $GLINSTALL32DIR/libGL.so
                        fi
                        if test -e $GLINSTALL32DIR/libGL.so.1 ; then
                            rm -f $GLINSTALL32DIR/libGL.so.1
                        fi  
                        if test -e $GLINSTALL32DIR/libGL.so.1.2.0 ; then
                            rm -f $GLINSTALL32DIR/libGL.so.1.2.0
                        fi
                    fi
                fi
                if test -e $GLXINSTALLDIR/$GLXLIBRARY ; then 
                    echo "Uninstalling GLX library"
                    rm -f $GLXINSTALLDIR/$GLXLIBRARY
                fi
                if test -e $GLHEADERSINSTALLDIR/$GLHEADER ; then 
                    echo "Uninstalling $GLHEADER"
                    rm -f $GLHEADERSINSTALLDIR/$GLHEADER
                fi
                if test -e $GLHEADERSINSTALLDIR/$GLEXTHEADER ; then 
                    echo "Uninstalling $GLEXTHEADER"
                    rm -f $GLHEADERSINSTALLDIR/$GLEXTHEADER
                fi

                echo "Uninstallation complete"
                exit 0
            fi

            if [ $YESNOCANCEL = 'r' -o $YESNOCANCEL = 'R' ]; then
                if test -e $BACKUP_DIR/$XDRIVER.$BACKUP_SUFFIX ; then 
                    echo "Restoring X driver"
                    cp -f $BACKUP_DIR/$XDRIVER.$BACKUP_SUFFIX $XINSTALLDIR/$XDRIVER
                    rm -f $BACKUP_DIR/$XDRIVER.$BACKUP_SUFFIX
                fi
                if test -e $BACKUP_DIR/$KERNELDRIVER.$BACKUP_SUFFIX ; then 
                    echo "Restoring kernel module"
                    cp -f $BACKUP_DIR/$KERNELDRIVER.$BACKUP_SUFFIX $KNLINSTALLDIR/$KERNELDRIVER
                    rm -f $BACKUP_DIR/$KERNELDRIVER.$BACKUP_SUFFIX
                fi
                if test -e $BACKUP_DIR/$GLLIBRARY.$BACKUP_SUFFIX ; then 
                    echo "Restoring OpenGL library"
                    cp -f $BACKUP_DIR/$GLLIBRARY.$BACKUP_SUFFIX $GLINSTALLDIR/$GLLIBRARY
                    rm -f $BACKUP_DIR/$GLLIBRARY.$BACKUP_SUFFIX
                    ln -s $GLINSTALLDIR/$GLLIBRARY $GLINSTALLDIR/libGL.so.1.2.0
                    ln -s $GLINSTALLDIR/libGL.so.1.2.0 $GLINSTALLDIR/libGL.so.1
                    ln -s $GLINSTALLDIR/libGL.so.1 $GLINSTALLDIR/libGL.so
                fi
                if test -e $BACKUP_DIR/$GLXLIBRARY.$BACKUP_SUFFIX ; then 
                    echo "Restoring GLX library"
                    cp -f $BACKUP_DIR/$GLXLIBRARY.$BACKUP_SUFFIX $GLXINSTALLDIR/$GLXLIBRARY
                    rm -f $BACKUP_DIR/$GLXLIBRARY.$BACKUP_SUFFIX
                fi
                if test -e $BACKUP_DIR/$GLHEADER.$BACKUP_SUFFIX ; then 
                    echo "Restoring $GLHEADER"
                    cp -f $BACKUP_DIR/$GLHEADER.$BACKUP_SUFFIX $GLHEADERSINSTALLDIR/$GLHEADER
                    rm -f $BACKUP_DIR/$GLHEADER.$BACKUP_SUFFIX
                fi
                if test -e $BACKUP_DIR/$GLEXTHEADER.$BACKUP_SUFFIX ; then 
                    echo "Restoring $GLEXTHEADER"
                    cp -f $BACKUP_DIR/$GLEXTHEADER.$BACKUP_SUFFIX $GLHEADERSINSTALLDIR/$GLEXTHEADER
                    rm -f $BACKUP_DIR/$GLEXTHEADER.$BACKUP_SUFFIX
                fi
                if [ "$MACHINEARCH" == "x86_64" ]; then
                    if test -e $BACKUP_DIR/lib32/$GLLIBRARY.$BACKUP_SUFFIX ; then 
                        echo "Restoring OpenGL library"
                        cp -f $BACKUP_DIR/lib32/$GLLIBRARY.$BACKUP_SUFFIX $GLINSTALL32DIR/$GLLIBRARY
                        rm -f $BACKUP_DIR/lib32/$GLLIBRARY.$BACKUP_SUFFIX
                        ln -s $GLINSTALL32DIR/$GLLIBRARY $GLINSTALL32DIR/libGL.so.1.2.0
                        ln -s $GLINSTALL32DIR/libGL.so.1.2.0 $GLINSTALL32DIR/libGL.so.1
                        ln -s $GLINSTALL32DIR/libGL.so.1 $GLINSTALL32DIR/libGL.so
                    fi
                fi

                echo "Updating library cache"
                echo ""
                /sbin/ldconfig

                echo "Restoration complete"
                exit 0
            fi

            echo "Canceling operation."
            echo ""
            exit 0
        else
            echo -e "\033[1mNo files were previous backed up,"
            echo -en "do you wish to uninstall \033[0m$YELLOW\033[1m(y/n)\033[0m? "
            read YESNOCANCEL
            echo ""
            if [ $YESNOCANCEL = 'y' -o $YESNOCANCEL = 'Y' ]; then
            if test -e $XINSTALLDIR/$XDRIVER ; then 
                    echo "Uninstalling X driver"
                    rm -f $XINSTALLDIR/$XDRIVER
                fi
                if test -e $KNLINSTALLDIR/$KERNELDRIVER ; then 
                    echo "Uninstalling kernel module"
                    rm -f $KNLINSTALLDIR/$KERNELDRIVER
                fi
                if test -e $GLINSTALLDIR/$GLLIBRARY ; then 
                    echo "Uninstalling OpenGL library"
                    rm -f $GLINSTALLDIR/$GLLIBRARY
                    if test -e $GLINSTALLDIR/libGL.so ; then
                        rm -f $GLINSTALLDIR/libGL.so
                    fi
                    if test -e $GLINSTALLDIR/libGL.so.1 ; then
                        rm -f $GLINSTALLDIR/libGL.so.1
                    fi
                    if test -e $GLINSTALLDIR/libGL.so.1.2.0 ; then
                        rm -f $GLINSTALLDIR/libGL.so.1.2.0
                    fi
                fi
                if test -e $GLXINSTALLDIR/$GLXLIBRARY ; then 
                    echo "Uninstalling GLX library"
                    rm -f $GLXINSTALLDIR/$GLXLIBRARY
                fi
                if test -e $GLHEADERSINSTALLDIR/$GLHEADER ; then 
                    echo "Uninstalling $GLHEADER"
                    rm -f $GLHEADERSINSTALLDIR/$GLHEADER
                fi
                if test -e $GLHEADERSINSTALLDIR/$GLEXTHEADER ; then 
                    echo "Uninstalling $GLEXTHEADER"
                    rm -f $GLHEADERSINSTALLDIR/$GLEXTHEADER
                fi
                exit 0
            fi

            echo "Canceling operation."
            echo ""
            exit 0
        fi

    else
        echo -e "\033[1mNo files were previous backed up,"
        echo -en "do you wish to uninstall \033[0m$YELLOW\033[1m(y/n)\033[0m? "
        read YESNOCANCEL
        echo ""
        if [ $YESNOCANCEL = 'y' -o $YESNOCANCEL = 'Y' ]; then
            if test -e $XINSTALLDIR/$XDRIVER ; then 
                echo "Uninstalling X driver"
                rm -f $XINSTALLDIR/$XDRIVER
            fi
            if test -e $KNLINSTALLDIR/$KERNELDRIVER ; then 
                echo "Uninstalling kernel module"
                rm -f $KNLINSTALLDIR/$KERNELDRIVER
            fi
            if test -e $GLINSTALLDIR/$GLLIBRARY ; then 
                echo "Uninstalling OpenGL library"
                rm -f $GLINSTALLDIR/$GLLIBRARY
                if test -e $GLINSTALLDIR/libGL.so ; then
                    rm -f $GLINSTALLDIR/libGL.so
                fi
                if test -e $GLINSTALLDIR/libGL.so.1 ; then
                    rm -f $GLINSTALLDIR/libGL.so.1
                fi
                if test -e $GLINSTALLDIR/libGL.so.1.2.0 ; then
                    rm -f $GLINSTALLDIR/libGL.so.1.2.0
                fi
            fi
            if test -e $GLXINSTALLDIR/$GLXLIBRARY ; then 
                echo "Uninstalling GLX library"
                rm -f $GLXINSTALLDIR/$GLXLIBRARY
            fi
            if test -e $GLHEADERSINSTALLDIR/$GLHEADER ; then 
                echo "Uninstalling $GLHEADER"
                rm -f $GLHEADERSINSTALLDIR/$GLHEADER
            fi
            if test -e $GLHEADERSINSTALLDIR/$GLEXTHEADER ; then 
                echo "Uninstalling $GLEXTHEADER"
                rm -f $GLHEADERSINSTALLDIR/$GLEXTHEADER
            fi
        fi

        echo "Canceling operation."
        echo ""
        exit 0
    fi
}


#############################################################
#
# Script Starts Here
#
#############################################################


clear
echo -e $GREEN"\033[1m"
echo "========================================"
echo "   Matrox Linux Driver Install Script   "
echo "========================================"
echo -e $GREEN"\033[0m"

#################################
# verify if user has root access
#################################
IsRoot

#################################
# Parse command line options
#################################
ParseOptions $*

if [ "$DISPLAYHELP" == "1" ] ; then
    echo "Options :"
    echo ""
    echo "    -h            Displays this message."
    echo "    -overwrite    Used to force driver installation."
    echo "    -uninstall    Reinstalls back if available or removes"
    echo "                  the driver if none are available."
    echo ""
    exit 0
fi

if [ "$UNDO" == "1" ]; then

    #################################
    # Get installed X path
    #################################
    GetXPath
    
    #################################
    # Get installed X version
    #################################
    GetXVersion
    
    #################################
    # Setup File Vars
    #################################
    SetFilenames
   
    #################################
    # Set X install path
    #################################
    SetXInstallPath
    
    #if [ "$MACHINEARCH" != "x86_64" ]; then
        #################################
        # Get Kernel Information
        #################################
        GetKernelInfo
        
        #################################
        # Get Kernel Path
        #################################
        GetKernelPath
    
        #################################
        # Set Kernel Path
        #################################
        SetKernelInstallPath

        #################################
        # Set OpenGL Install Path
        #################################
        SetOpenGLInstallPath
    #fi    

    #################################
    # Uninstall
    #################################
    Uninstall
    exit 0

else

    if [ "$OVERWRITE" == "1" ]; then
        echo -e "$RED\033[1mOverwrite option enabled."
        echo -e "No backups will be created.\033[0m"
        echo -en "\033[1mDo you wish to continue\033[0m $YELLOW\033[1m(y/n)\033[0m\033[1m?\033[0m "
        read STRCONT 
        echo ""
        if [ $STRCONT = 'n' -o $STRCONT = 'N' ]; then
            exit 1
        fi
        if [ $STRCONT = 'NO' -o $STRCONT = 'no' ]; then
            exit 1
        fi
    fi    

    #################################
    # Get installed X path
    #################################
    GetXPath
    
    #################################
    # Get installed X version
    #################################
    GetXVersion
    
    #################################
    # Set X install path
    #################################
    SetXInstallPath
    
    #################################
    # Setup File Vars
    #################################
    SetFilenames
    
    #################################
    # Install X server driver   
    #################################
    InstallXDriver

    #if [ "$MACHINEARCH" != "x86_64" ]; then
        #################################
        # Get Kernel Information
        #################################
        GetKernelInfo
    
        #################################
        # Get Kernel Path
        #################################
        GetKernelPath
    
        #################################
        # Set Kernel Path
        #################################
        SetKernelInstallPath
    
        #################################
        # Compile kernel module
        #################################
        CompileKernelModule

        #################################
        # Install kernel module
        #################################
        InstallKernelModule
    
        #################################
        # Set OpenGL Install Path
        #################################
        SetOpenGLInstallPath
    
        #################################
        # Install OpenGL 
        #################################
        InstallOpenGL
    #fi

    exit 0
fi
