/*
 *  Intelligent Platform Management Interface driver for Linux 2.x
 *
 *  (c) Copyright 1999  San Mehat & VA Linux Systems
 *        1382 Bordeaux Dr.
 *        Sunnyvale, California
 *        94089
 *
 *  http://www.valinux.com
 *
 *  This driver is provided under the GNU public license, incorporated
 *  herein by reference. The driver is provided without warranty or
 *  support.
 *
 *  IOCTL definitions for IPMI drivers
 */

/*
 * Note: The following macros should be used on the IPMI_XFER structure.
 *       DO NOT try to muck with this structure directly.. use the macros
 *       to ensure future compatibility:
 *
 *  INIT_XFER(IPMI_XFER *);  
 *    -- Zero out a IPMI_XFER structure and initialize it for use
 *
 *  SET_REQUEST_LUN(IPMI_XFER *, unsigned char lun);
 *    -- Set the request packet logical unit
 *
 *  SET_REQUEST_NETFN(IPMI_XFER *, unsigned char netfn);
 *    -- Set the request packet network function code
 *
 *  SET_REQUEST_CMD(IPMI_XFER *, unsigned char cmd);
 *    -- Set the request packet IPMI command code
 *
 *  SET_REQUEST_DATA(IPMI_XFER *, unsigned char *data, int length);
 *    -- Set the request packet optional argument data field
 * 
 *  GET_RESPONSE_LUN(IPMI_XFER *, unsigned char lun);
 *    -- Get the response packet logical unit
 *
 *  GET_RESPONSE_NETFN(IPMI_XFER *, unsigned char netfn);
 *    -- Get the response packet network function code
 * 
 *  GET_RESPONSE_CMD(IPMI_XFER *, unsigned char cmd);
 *    -- Get the response packet command
 *
 *  GET_RESPONSE_CC(IPMI_XFER *, unsigned char cc);
 *    -- Get the response packet completion code
 * 
 *  GET_RESPONSE_DATA_LENGTH(IPMI_XFER *, int len);
 *    -- Get the response packet data length
 * 
 *  GET_RESPONSE_DATA(IPMI_XFER *, unsigned char *buffer);
 *    -- Copy the response packet data into local buffer
 */

#ifndef _IPMI_IOCTLS_H
#define _IPMI_IOCTLS_H

#define	IOCTL_IPMI_XFER		0x01
#define	IOCTL_DRIVER_INFO	0x02

typedef struct ipmi_xfer
	{
	unsigned char	request[64];
	unsigned char	response[64];
	int						request_len;
	int						response_len;
	} IPMI_XFER;

struct ipmi_driver_info
	{
	char						driver_name[64];			/* Name of the driver */
	int							major_ver;					
	int							minor_ver;
	unsigned short	mb_chipset_vendor;		/* PCI host bridge vendor tag */
	unsigned short	mb_chipset_device;		/* PCI host bridge vendor device id */
	unsigned int		flags;								/* driver specific flags */
	unsigned int		reserved;
	};

/* flags definitions for the 'ipmi_kcs' driver */
#define KCS_FLAG_BLINKY	0x01	/* Set if blinky works (only on Intel L440GX) */
#define KCS_FLAG_LEGACY	0x02  /* Set if using legacy KCS interface ( < IPMI 1.0) */

#define INIT_XFER(_xferp) \
	memset(_xferp, 0, sizeof(IPMI_XFER)); \
	_xferp->request_len = 2; \
	_xferp->response_len = sizeof(_xferp->response);

#define SET_REQUEST_LUN(_xferp, _lun) \
	{ \
	unsigned char _netfn_copy; \
	\
	_netfn_copy = (_xferp->request[0] & 0xFC); \
	_xferp->request[0] = _lun; \
	_xferp->request[0]|= _netfn_copy; \
	}

#define SET_REQUEST_NETFN(_xferp, netfn) \
	{ \
	unsigned char __lun_copy; \
	\
	__lun_copy = (_xferp->request[0] & 0x3); \
	_xferp->request[0] = (netfn << 2); \
	_xferp->request[0]|= __lun_copy; \
	}

#define SET_REQUEST_CMD(_xferp, _cmd) \
	_xferp->request[1] = _cmd;

#define SET_REQUEST_DATA(_xferp, datap, _len) \
	{ \
	memcpy(&_xferp->request[2], datap, _len); \
	_xferp->request_len = (_len + 2); \
	}

#define GET_RESPONSE_LUN(_xferp, _lun) \
	_lun = (_xferp->response[0] & 0x3);

#define GET_RESPONSE_NETFN(_xferp, netfn) \
	netfn = ((_xferp->response[0] & 0xFC) >> 2);

#define GET_RESPONSE_CMD(_xferp, _cmd) \
	_cmd = _xferp->response[1];

#define GET_RESPONSE_CC(_xferp, cc) \
	cc = _xferp->response[2];

#define GET_RESPONSE_DATA_LENGTH(_xferp, _len) \
	_len = (_xferp->response_len - 3);

#define GET_RESPONSE_DATA(_xferp, datap) \
	memcpy(datap, &_xferp->response[3], (_xferp->response_len -3));

/*
 * The Netfn codes
 */
#define CHASSIS_REQUEST   0x00
#define CHASSIS_RESPONSE  0x01
#define BRIDGE_REQUEST    0x02
#define BRIDGE_RESPONSE   0x03
#define SENSOR_REQUEST    0x04
#define SENSOR_RESPONSE   0x05
#define APP_REQUEST       0x06
#define APP_RESPONSE      0x07
#define FIRMWARE_REQUEST  0x08
#define FIRMWARE_RESPONSE 0x09
#define STORAGE_REQUEST   0x0A
#define STORAGE_RESPONSE  0x0B

#endif
