/*******************************************************************************

  
  Copyright(c) 1999 - 2004 Intel Corporation. All rights reserved.
  
  This program is free software; you can redistribute it and/or modify it 
  under the terms of the GNU General Public License as published by the Free 
  Software Foundation; either version 2 of the License, or (at your option) 
  any later version.
  
  This program is distributed in the hope that it will be useful, but WITHOUT 
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
  more details.
  
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 
  Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
  The full GNU General Public License is included in this distribution in the
  file called LICENSE.
  
  Contact Information:
  Linux NICS <linux.nics@intel.com>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

*******************************************************************************/

#ifndef _KCOMPAT_H_
#define _KCOMPAT_H_

#include <linux/version.h>

#ifdef E100_NAPI
#undef CONFIG_E100_NAPI
#define CONFIG_E100_NAPI
#endif

#ifdef E100_NO_NAPI
#undef CONFIG_E100_NAPI
#endif

/*****************************************************************************/
/* 2.4.0 => 2.2.0 */
#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0) )

/**************************************/
/* MODULE API */

#ifndef __init
	#define __init
#endif

#ifndef __exit
	#define __exit
#endif

#ifndef __devinit
	#define __devinit
#endif

#ifndef __devexit
	#define __devexit
#endif

#ifndef __devinitdata
	#define __devinitdata
#endif

#ifndef module_init
	#define module_init(f) int init_module() { return f(); }
#endif

#ifndef module_exit
	#define module_exit(f) void cleanup_module() { return f(); }
#endif

#ifndef MODULE_DEVICE_TABLE
	#define MODULE_DEVICE_TABLE(bus, table)
#endif

#ifndef SET_MODULE_OWNER
	#define SET_MODULE_OWNER(X)
#else
	#undef MOD_INC_USE_COUNT
	#undef MOD_DEC_USE_COUNT
	#define MOD_INC_USE_COUNT
	#define MOD_DEC_USE_COUNT
#endif

/**************************************/
/* PCI DRIVER API */

#ifndef pci_device_id
#define pci_device_id _kc_pci_device_id
struct _kc_pci_device_id {
	unsigned int vendor, device;
	unsigned int subvendor, subdevice;
	unsigned int class, classmask;
	unsigned long driver_data;
};
#endif

#ifndef pci_driver
#define pci_driver _kc_pci_driver
struct _kc_pci_driver {
	char *name;
	struct pci_device_id *id_table;
	int (*probe)(struct pci_dev *dev, const struct pci_device_id *id);
	void (*remove)(struct pci_dev *dev);
	int (*save_state)(struct pci_dev *dev, u32 state);
	int (*suspend)(struct pci_dev *dev, u32 state);
	int (*resume)(struct pci_dev *dev);
	int (*enable_wake)(struct pci_dev *dev, u32 state, int enable);
};
#endif

#ifndef PCI_ANY_ID
	#define PCI_ANY_ID (~0U)
#endif

#ifndef pci_module_init
#define pci_module_init _kc_pci_module_init
extern int _kc_pci_module_init(struct pci_driver *drv);
#endif

#ifndef pci_unregister_driver
#define pci_unregister_driver _kc_pci_unregister_driver
extern void _kc_pci_unregister_driver(struct pci_driver *drv);
#endif

#ifndef pci_set_drvdata
#define pci_set_drvdata _kc_pci_set_drvdata
extern void _kc_pci_set_drvdata(struct pci_dev *dev, void *data);
#endif

#ifndef pci_get_drvdata
#define pci_get_drvdata _kc_pci_get_drvdata
extern void * _kc_pci_get_drvdata(struct pci_dev *dev);
#endif

#ifndef pci_enable_device
#define pci_enable_device _kc_pci_enable_device
extern int _kc_pci_enable_device(struct pci_dev *dev);
#endif

#ifndef pci_resource_start
#define pci_resource_start _kc_pci_resource_start
extern int _kc_pci_resource_start(struct pci_dev *dev, int bar);
#endif

#ifndef pci_resource_len
#define pci_resource_len _kc_pci_resource_len
extern unsigned long _kc_pci_resource_len(struct pci_dev *dev, int bar);
#endif

#undef IORESOURCE_IO
#define IORESOURCE_IO PCI_BASE_ADDRESS_SPACE_IO

#undef pci_resource_flags
#define pci_resource_flags(dev, i) (dev->base_address[i] & IORESOURCE_IO)

/**************************************/
/* PCI DMA MAPPING */

#ifndef PCI_DMA_TODEVICE
	#define PCI_DMA_TODEVICE 1
#endif

#ifndef PCI_DMA_FROMDEVICE
	#define PCI_DMA_FROMDEVICE 2
#endif

#ifndef dma_addr_t
#define dma_addr_t _kc_dma_addr_t
typedef u64 _kc_dma_addr_t;
#endif

#ifndef pci_alloc_consistent
#define pci_alloc_consistent _kc_pci_alloc_consistent
extern void * _kc_pci_alloc_consistent(struct pci_dev *dev, size_t size, u64 *dma_handle);
#endif

#ifndef pci_free_consistent
#define pci_free_consistent _kc_pci_free_consistent
extern void _kc_pci_free_consistent(struct pci_dev *dev, size_t size, void *vaddr, u64 dma_handle);
#endif

#ifndef pci_map_single
#define pci_map_single _kc_pci_map_single
extern u64 _kc_pci_map_single(struct pci_dev *dev, void *addr, size_t size, int direction);
#endif

#ifndef pci_unmap_single
#define pci_unmap_single _kc_pci_unmap_single
extern void _kc_pci_unmap_single(struct pci_dev *dev, u64 dma_addr, size_t size, int direction);
#endif

#ifndef pci_dma_sync_single
#define pci_dma_sync_single _kc_pci_dma_sync_single
extern void _kc_pci_dma_sync_single(struct pci_dev *dev, u64 dma_addr, size_t size, int direction);
#endif

/**************************************/
/* NETWORK DRIVER API */

#ifndef net_device
	#define net_device device
#endif

#ifndef dev_kfree_skb_irq
	#define dev_kfree_skb_irq dev_kfree_skb
#endif

#ifndef dev_kfree_skb_any
	#define dev_kfree_skb_any dev_kfree_skb
#endif

#ifndef netif_running
	#define netif_running(dev) (!!(int)(dev->flags & IFF_RUNNING))
#endif

#ifndef netif_start_queue
	#define netif_start_queue(dev) clear_bit(0, &dev->tbusy)
#endif

#ifndef netif_stop_queue
	#define netif_stop_queue(dev) set_bit(0, &dev->tbusy)
#endif

#ifndef netif_wake_queue
	#define netif_wake_queue(dev) do { clear_bit(0, &dev->tbusy); \
		                           mark_bh(NET_BH); } while(0)
#endif

#ifndef netif_queue_stopped
	#define netif_queue_stopped(dev) (dev->tbusy)
#endif

#ifndef netif_device_attach
#define netif_device_attach _kc_netif_device_attach
extern void _kc_netif_device_attach(struct net_device *dev);
#endif

#ifndef netif_device_detach
#define netif_device_detach _kc_netif_device_detach
extern void _kc_netif_device_detach(struct net_device *dev);
#endif

#ifndef netif_carrier_on
#define netif_carrier_on _kc_netif_carrier_on
extern void _kc_netif_carrier_on(struct net_device *dev);
#endif

#ifndef netif_carrier_off
#define netif_carrier_off _kc_netif_carrier_off
extern void _kc_netif_carrier_off(struct net_device *dev);
#endif

#ifndef netif_carrier_ok
#define netif_carrier_ok _kc_netif_carrier_ok
extern int _kc_netif_carrier_ok(struct net_device *dev);
#endif

/**************************************/
/* OTHER */

#ifndef del_timer_sync
	#define del_timer_sync del_timer
#endif

#ifndef BUG
	#define BUG() printk(KERN_CRIT "BUG in %s at line %d\n", __FILE__, __LINE__)
#endif

#ifndef set_current_state
	#define set_current_state(S) current->state = (S)
#endif

#ifndef list_for_each
	#define list_for_each(pos, head) \
		for(pos = (head)->next; pos != (head); pos = pos->next)
#endif

#ifndef list_add_tail
	#define list_add_tail(new, head) __list_add(new, (head)->prev, (head))
#endif

#ifndef ARRARY_SIZE
	#define ARRAY_SIZE(x) (sizeof(x) / sizeof((x)[0]))
#endif

#define msec_delay(x) mdelay(x)

#else /* remove 2.2.x only compatibility stuff */

	#undef MOD_INC_USE_COUNT
	#undef MOD_DEC_USE_COUNT
	#define MOD_INC_USE_COUNT
	#define MOD_DEC_USE_COUNT

#endif /* 2.4.0 => 2.2.0 */


/*****************************************************************************/
/* 2.4.3 => 2.4.0 */
#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,4,3) )

/**************************************/
/* PCI DRIVER API */

#ifndef pci_set_dma_mask
#define pci_set_dma_mask _kc_pci_set_dma_mask
extern int _kc_pci_set_dma_mask(struct pci_dev *dev, dma_addr_t mask);
#endif

#ifndef pci_request_regions
#define pci_request_regions _kc_pci_request_regions
extern int _kc_pci_request_regions(struct pci_dev *pdev, char *res_name);
#endif

#ifndef pci_release_regions
#define pci_release_regions _kc_pci_release_regions
extern void _kc_pci_release_regions(struct pci_dev *pdev);
#endif

/**************************************/
/* NETWORK DRIVER API */

#ifndef alloc_etherdev
#define alloc_etherdev _kc_alloc_etherdev
extern struct net_device * _kc_alloc_etherdev(int sizeof_priv);
#endif

#ifndef is_valid_ether_addr
#define is_valid_ether_addr _kc_is_valid_ether_addr
extern int _kc_is_valid_ether_addr(u8 *addr);
#endif

#endif /* 2.4.3 => 2.4.0 */

/*****************************************************************************/
/* 2.4.6 => 2.4.3 */
#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,4,6) )

#ifndef pci_set_power_state
#define pci_set_power_state _kc_pci_set_power_state
extern int _kc_pci_set_power_state(struct pci_dev *dev, int state);
#endif

#ifndef pci_save_state
#define pci_save_state _kc_pci_save_state
extern int _kc_pci_save_state(struct pci_dev *dev, u32 *buffer);
#endif

#ifndef pci_restore_state
#define pci_restore_state _kc_pci_restore_state
extern int _kc_pci_restore_state(struct pci_dev *pdev, u32 *buffer);
#endif

#ifndef pci_enable_wake
#define pci_enable_wake _kc_pci_enable_wake
extern int _kc_pci_enable_wake(struct pci_dev *pdev, u32 state, int enable);
#endif

#ifndef pci_disable_device
#define pci_disable_device _kc_pci_disable_device
extern void _kc_pci_disable_device(struct pci_dev *pdev);
#endif

/* PCI PM entry point syntax changed, so don't support suspend/resume */
#undef CONFIG_PM

#endif /* 2.4.6 => 2.4.3 */

/*****************************************************************************/
/* 2.4.10 => 2.4.6 */
#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,4,10) )

/**************************************/
/* MODULE API */

#ifndef MODULE_LICENSE
	#define MODULE_LICENSE(X)
#endif

/**************************************/
/* OTHER */

#undef min
#define min(x,y) ({ \
	const typeof(x) _x = (x);	\
	const typeof(y) _y = (y);	\
	(void) (&_x == &_y);		\
	_x < _y ? _x : _y; })

#undef max
#define max(x,y) ({ \
	const typeof(x) _x = (x);	\
	const typeof(y) _y = (y);	\
	(void) (&_x == &_y);		\
	_x > _y ? _x : _y; })

#ifndef list_for_each_safe
#define list_for_each_safe(pos, n, head) \
	for (pos = (head)->next, n = pos->next; pos != (head); \
		pos = n, n = pos->next)
#endif

#endif /* 2.4.10 -> 2.4.6 */

/*****************************************************************************/
/* 2.4.13 => 2.4.10 */
#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,4,13) )

/**************************************/
/* PCI DMA MAPPING */

#ifndef virt_to_page
	#define virt_to_page(v) (mem_map + (virt_to_phys(v) >> PAGE_SHIFT))
#endif

#ifndef pci_map_page
#define pci_map_page _kc_pci_map_page
extern u64 _kc_pci_map_page(struct pci_dev *dev, struct page *page, unsigned long offset, size_t size, int direction);
#endif

#ifndef pci_unmap_page
#define pci_unmap_page _kc_pci_unmap_page
extern void _kc_pci_unmap_page(struct pci_dev *dev, u64 dma_addr, size_t size, int direction);
#endif

/* pci_set_dma_mask takes dma_addr_t, which is only 32-bits prior to 2.4.13 */

#undef PCI_DMA_32BIT
#define PCI_DMA_32BIT	0xffffffff
#undef PCI_DMA_64BIT
#define PCI_DMA_64BIT	0xffffffff

/**************************************/
/* OTHER */

#ifndef cpu_relax
#define cpu_relax()	rep_nop()
#endif

#endif /* 2.4.13 => 2.4.10 */

/*****************************************************************************/
/* 2.4.19 => 2.4.12 */
#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,4,19) )

#ifndef __devexit_p
	#define __devexit_p(x) &(x)
#endif

#ifndef VLAN_HLEN
#define VLAN_HLEN 4
#endif

#ifndef VLAN_ETH_HLEN
#define VLAN_ETH_HLEN 18
#endif

#ifndef VLAN_ETH_FRAME_LEN
#define VLAN_ETH_FRAME_LEN 1518
#endif

#endif /* 2.4.19 => 2.4.13 */

/*****************************************************************************/
/* 2.4.20 => 2.4.19 */
#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,4,20) )

#ifndef HAVE_NETIF_MSG
/*
 * Network interface message level settings
 */
#define HAVE_NETIF_MSG 1

enum {
	NETIF_MSG_DRV		= 0x0001,
	NETIF_MSG_PROBE		= 0x0002,
	NETIF_MSG_LINK		= 0x0004,
	NETIF_MSG_TIMER		= 0x0008,
	NETIF_MSG_IFDOWN	= 0x0010,
	NETIF_MSG_IFUP		= 0x0020,
	NETIF_MSG_RX_ERR	= 0x0040,
	NETIF_MSG_TX_ERR	= 0x0080,
	NETIF_MSG_TX_QUEUED	= 0x0100,
	NETIF_MSG_INTR		= 0x0200,
	NETIF_MSG_TX_DONE	= 0x0400,
	NETIF_MSG_RX_STATUS	= 0x0800,
	NETIF_MSG_PKTDATA	= 0x1000,
	NETIF_MSG_HW		= 0x2000,
	NETIF_MSG_WOL		= 0x4000,
};
#else
#define NETIF_MSG_HW	0x2000
#define NETIF_MSG_WOL	0x4000
#endif

#ifndef MII_RESV1
#define MII_RESV1		0x17		/* Reserved...		*/
#endif

#endif /* 2.4.20 => 2.4.18 */

#ifndef ETHTOOL_BUSINFO_LEN
#define ETHTOOL_BUSINFO_LEN	32
#endif

#undef ethtool_drvinfo
#define ethtool_drvinfo _kc_ethtool_drvinfo
struct _kc_ethtool_drvinfo {
	u32	cmd;
	char	driver[32];	/* driver short name, "tulip", "eepro100" */
	char	version[32];	/* driver version string */
	char	fw_version[32];	/* firmware version string, if applicable */
	char	bus_info[ETHTOOL_BUSINFO_LEN];	/* Bus info for this IF. */
				/* For PCI devices, use pci_dev->slot_name. */
	char	reserved1[32];
	char	reserved2[16];
	u32	n_stats;	/* number of u64's from ETHTOOL_GSTATS */
	u32	testinfo_len;
	u32	eedump_len;	/* Size of data from ETHTOOL_GEEPROM (bytes) */
	u32	regdump_len;	/* Size of data from ETHTOOL_GREGS (bytes) */
};

#undef ethtool_value
#define ethtool_value _kc_ethtool_value
/* for passing single values */
struct _kc_ethtool_value {
	u32	cmd;
	u32	data;
};

#undef ethtool_regs
#define ethtool_regs _kc_ethtool_regs
struct _kc_ethtool_regs {
	u32	cmd;
	u32	version; /* driver-specific, indicates different chips/revs */
	u32	len; /* bytes */
	u8	data[0];
};

#ifndef ETH_GSTRING_LEN
#define ETH_GSTRING_LEN		32
#endif

#undef ethtool_stringset
#define ethtool_stringset _kc_ethtool_stringset
#undef ETH_SS_TEST
#define ETH_SS_TEST _kc_ETH_SS_TEST
#undef ETH_SS_STATS
#define ETH_SS_STATS _kc_ETH_SS_STATS
enum _kc_ethtool_stringset {
	_kc_ETH_SS_TEST		= 0,
	_kc_ETH_SS_STATS,
};

#undef ethtool_eeprom
#define ethtool_eeprom _kc_ethtool_eeprom
struct ethtool_eeprom {
	u32	cmd;
	u32	magic;
	u32	offset; /* in bytes */
	u32	len; /* in bytes */
	u8	data[0];
};

#undef ethtool_coalesce
#define ethtool_coalesce _kc_ethtool_coalesce
/* for configuring coalescing parameters of chip */
struct _kc_ethtool_coalesce {
	u32	cmd;	/* ETHTOOL_{G,S}COALESCE */

	/* How many usecs to delay an RX interrupt after
	 * a packet arrives.  If 0, only rx_max_coalesced_frames
	 * is used.
	 */
	u32	rx_coalesce_usecs;

	/* How many packets to delay an RX interrupt after
	 * a packet arrives.  If 0, only rx_coalesce_usecs is
	 * used.  It is illegal to set both usecs and max frames
	 * to zero as this would cause RX interrupts to never be
	 * generated.
	 */
	u32	rx_max_coalesced_frames;

	/* Same as above two parameters, except that these values
	 * apply while an IRQ is being serviced by the host.  Not
	 * all cards support this feature and the values are ignored
	 * in that case.
	 */
	u32	rx_coalesce_usecs_irq;
	u32	rx_max_coalesced_frames_irq;

	/* How many usecs to delay a TX interrupt after
	 * a packet is sent.  If 0, only tx_max_coalesced_frames
	 * is used.
	 */
	u32	tx_coalesce_usecs;

	/* How many packets to delay a TX interrupt after
	 * a packet is sent.  If 0, only tx_coalesce_usecs is
	 * used.  It is illegal to set both usecs and max frames
	 * to zero as this would cause TX interrupts to never be
	 * generated.
	 */
	u32	tx_max_coalesced_frames;

	/* Same as above two parameters, except that these values
	 * apply while an IRQ is being serviced by the host.  Not
	 * all cards support this feature and the values are ignored
	 * in that case.
	 */
	u32	tx_coalesce_usecs_irq;
	u32	tx_max_coalesced_frames_irq;

	/* How many usecs to delay in-memory statistics
	 * block updates.  Some drivers do not have an in-memory
	 * statistic block, and in such cases this value is ignored.
	 * This value must not be zero.
	 */
	u32	stats_block_coalesce_usecs;

	/* Adaptive RX/TX coalescing is an algorithm implemented by
	 * some drivers to improve latency under low packet rates and
	 * improve throughput under high packet rates.  Some drivers
	 * only implement one of RX or TX adaptive coalescing.  Anything
	 * not implemented by the driver causes these values to be
	 * silently ignored.
	 */
	u32	use_adaptive_rx_coalesce;
	u32	use_adaptive_tx_coalesce;

	/* When the packet rate (measured in packets per second)
	 * is below pkt_rate_low, the {rx,tx}_*_low parameters are
	 * used.
	 */
	u32	pkt_rate_low;
	u32	rx_coalesce_usecs_low;
	u32	rx_max_coalesced_frames_low;
	u32	tx_coalesce_usecs_low;
	u32	tx_max_coalesced_frames_low;

	/* When the packet rate is below pkt_rate_high but above
	 * pkt_rate_low (both measured in packets per second) the
	 * normal {rx,tx}_* coalescing parameters are used.
	 */

	/* When the packet rate is (measured in packets per second)
	 * is above pkt_rate_high, the {rx,tx}_*_high parameters are
	 * used.
	 */
	u32	pkt_rate_high;
	u32	rx_coalesce_usecs_high;
	u32	rx_max_coalesced_frames_high;
	u32	tx_coalesce_usecs_high;
	u32	tx_max_coalesced_frames_high;

	/* How often to do adaptive coalescing packet rate sampling,
	 * measured in seconds.  Must not be zero.
	 */
	u32	rate_sample_interval;
};

#undef ethtool_ringparam
#define ethtool_ringparam _kc_ethtool_ringparam
/* for configuring RX/TX ring parameters */
struct _kc_ethtool_ringparam {
	u32	cmd;	/* ETHTOOL_{G,S}RINGPARAM */

	/* Read only attributes.  These indicate the maximum number
	 * of pending RX/TX ring entries the driver will allow the
	 * user to set.
	 */
	u32	rx_max_pending;
	u32	rx_mini_max_pending;
	u32	rx_jumbo_max_pending;
	u32	tx_max_pending;

	/* Values changeable by the user.  The valid values are
	 * in the range 1 to the "*_max_pending" counterpart above.
	 */
	u32	rx_pending;
	u32	rx_mini_pending;
	u32	rx_jumbo_pending;
	u32	tx_pending;
};

#undef ethtool_pauseparam
#define ethtool_pauseparam _kc_ethtool_pauseparam
/* for configuring link flow control parameters */
struct _kc_ethtool_pauseparam {
	u32	cmd;	/* ETHTOOL_{G,S}PAUSEPARAM */

	/* If the link is being auto-negotiated (via ethtool_cmd.autoneg
	 * being true) the user may set 'autonet' here non-zero to have the
	 * pause parameters be auto-negotiated too.  In such a case, the
	 * {rx,tx}_pause values below determine what capabilities are
	 * advertised.
	 *
	 * If 'autoneg' is zero or the link is not being auto-negotiated,
	 * then {rx,tx}_pause force the driver to use/not-use pause
	 * flow control.
	 */
	u32	autoneg;
	u32	rx_pause;
	u32	tx_pause;
};

#undef ethtool_gstrings
#define ethtool_gstrings _kc_ethtool_gstrings
/* for passing string sets for data tagging */
struct _kc_ethtool_gstrings {
	u32	cmd;		/* ETHTOOL_GSTRINGS */
	u32	string_set;	/* string set id e.c. ETH_SS_TEST, etc*/
	u32	len;		/* number of strings in the string set */
	u8	data[0];
};

#undef ethtool_test_flags
#define ethtool_test_flags _kc_ethtool_test_flags
#undef ETH_TEST_FL_OFFLINE
#define ETH_TEST_FL_OFFLINE _kc_ETH_TEST_FL_OFFLINE
#undef ETH_TEST_FL_FAILED
#define ETH_TEST_FL_FAILED _kc_ETH_TEST_FL_FAILED
enum _kc_ethtool_test_flags {
	_kc_ETH_TEST_FL_OFFLINE	= (1 << 0),	/* online / offline */
	_kc_ETH_TEST_FL_FAILED	= (1 << 1),	/* test passed / failed */
};

#undef ethtool_test
#define ethtool_test _kc_ethtool_test
/* for requesting NIC test and getting results*/
struct _kc_ethtool_test {
	u32	cmd;		/* ETHTOOL_TEST */
	u32	flags;		/* ETH_TEST_FL_xxx */
	u32	reserved;
	u32	len;		/* result length, in number of u64 elements */
	u64	data[0];
};

#undef ethtool_stats
#define ethtool_stats _kc_ethtool_stats
/* for dumping NIC-specific statistics */
struct _kc_ethtool_stats {
	u32	cmd;		/* ETHTOOL_GSTATS */
	u32	n_stats;	/* number of u64's being returned */
	u64	data[0];
};

#ifndef ETHTOOL_GMSGLVL
#define ETHTOOL_GMSGLVL		0x00000007 /* Get driver message level */
#endif
#ifndef ETHTOOL_SMSGLVL
#define ETHTOOL_SMSGLVL		0x00000008 /* Set driver msg level, priv. */
#endif
#ifndef ETHTOOL_NWAY_RST
#define ETHTOOL_NWAY_RST	0x00000009 /* Restart autonegotiation, priv */
#endif
#ifndef ETHTOOL_GLINK
#define ETHTOOL_GLINK		0x0000000a /* Get link status */
#endif
#ifndef ETHTOOL_GEEPROM
#define ETHTOOL_GEEPROM		0x0000000b /* Get EEPROM data */
#endif
#ifndef ETHTOOL_SEEPROM
#define ETHTOOL_SEEPROM		0x0000000c /* Set EEPROM data */
#endif
#ifndef ETHTOOL_GCOALESCE
#define ETHTOOL_GCOALESCE	0x0000000e /* Get coalesce config */
#endif
#ifndef ETHTOOL_SCOALESCE
#define ETHTOOL_SCOALESCE	0x0000000f /* Set coalesce config. */
#endif
#ifndef ETHTOOL_GRINGPARAM
#define ETHTOOL_GRINGPARAM	0x00000010 /* Get ring parameters */
#endif
#ifndef ETHTOOL_SRINGPARAM
#define ETHTOOL_SRINGPARAM	0x00000011 /* Set ring parameters, priv. */
#endif
#ifndef ETHTOOL_GPAUSEPARAM
#define ETHTOOL_GPAUSEPARAM	0x00000012 /* Get pause parameters */
#endif
#ifndef ETHTOOL_SPAUSEPARAM
#define ETHTOOL_SPAUSEPARAM	0x00000013 /* Set pause parameters. */
#endif
#ifndef ETHTOOL_GRXCSUM
#define ETHTOOL_GRXCSUM		0x00000014 /* Get RX hw csum enable (ethtool_value) */
#endif
#ifndef ETHTOOL_SRXCSUM
#define ETHTOOL_SRXCSUM		0x00000015 /* Set RX hw csum enable (ethtool_value) */
#endif
#ifndef ETHTOOL_GTXCSUM
#define ETHTOOL_GTXCSUM		0x00000016 /* Get TX hw csum enable (ethtool_value) */
#endif
#ifndef ETHTOOL_STXCSUM
#define ETHTOOL_STXCSUM		0x00000017 /* Set TX hw csum enable (ethtool_value) */
#endif
#ifndef ETHTOOL_GSG
#define ETHTOOL_GSG		0x00000018 /* Get scatter-gather enable
					    * (ethtool_value) */
#endif
#ifndef ETHTOOL_SSG
#define ETHTOOL_SSG		0x00000019 /* Set scatter-gather enable
					    * (ethtool_value). */
#endif
#ifndef ETHTOOL_TEST
#define ETHTOOL_TEST		0x0000001a /* execute NIC self-test, priv. */
#endif
#ifndef ETHTOOL_GSTRINGS
#define ETHTOOL_GSTRINGS	0x0000001b /* get specified string set */
#endif
#ifndef ETHTOOL_PHYS_ID
#define ETHTOOL_PHYS_ID		0x0000001c /* identify the NIC */
#endif
#ifndef ETHTOOL_GSTATS
#define ETHTOOL_GSTATS		0x0000001d /* get NIC-specific statistics */
#endif
#ifndef ETHTOOL_GTSO
#define ETHTOOL_GTSO		0x0000001e /* Get TSO enable (ethtool_value) */
#endif
#ifndef ETHTOOL_STSO
#define ETHTOOL_STSO		0x0000001f /* Set TSO enable (ethtool_value) */
#endif

#define mii_if_info _kc_mii_if_info
struct _kc_mii_if_info {
	int phy_id;
	int advertising;
	int phy_id_mask;
	int reg_num_mask;

	unsigned int full_duplex : 1;	/* is full duplex? */
	unsigned int force_media : 1;	/* is autoneg. disabled? */

	struct net_device *dev;
	int (*mdio_read) (struct net_device *dev, int phy_id, int location);
	void (*mdio_write) (struct net_device *dev, int phy_id, int location, int val);
};

struct ethtool_cmd;
struct mii_ioctl_data;

#undef mii_link_ok
#define mii_link_ok _kc_mii_link_ok
#undef mii_nway_restart
#define mii_nway_restart _kc_mii_nway_restart
#undef mii_ethtool_gset
#define mii_ethtool_gset _kc_mii_ethtool_gset
#undef mii_ethtool_sset
#define mii_ethtool_sset _kc_mii_ethtool_sset
#undef mii_check_link
#define mii_check_link _kc_mii_check_link
#undef generic_mii_ioctl
#define generic_mii_ioctl _kc_generic_mii_ioctl
extern int _kc_mii_link_ok (struct mii_if_info *mii);
extern int _kc_mii_nway_restart (struct mii_if_info *mii);
extern int _kc_mii_ethtool_gset(struct mii_if_info *mii,
		struct ethtool_cmd *ecmd);
extern int _kc_mii_ethtool_sset(struct mii_if_info *mii,
		struct ethtool_cmd *ecmd);
extern void _kc_mii_check_link (struct mii_if_info *mii);
extern int _kc_generic_mii_ioctl(struct mii_if_info *mii_if,
                struct mii_ioctl_data *mii_data, int cmd,
		unsigned int *duplex_changed);


struct _kc_pci_dev_ext {
	struct pci_dev *dev;
	void *pci_drvdata;
	struct pci_driver *driver;
};

struct _kc_net_dev_ext {
	struct net_device *dev;
	unsigned int carrier;
};


/**************************************/
/* mii support */

int _kc_mii_ethtool_gset(struct mii_if_info *mii, struct ethtool_cmd *ecmd)
{
	struct net_device *dev = mii->dev;
	u32 advert, bmcr, lpa, nego;

	ecmd->supported =
	    (SUPPORTED_10baseT_Half | SUPPORTED_10baseT_Full |
	     SUPPORTED_100baseT_Half | SUPPORTED_100baseT_Full |
	     SUPPORTED_Autoneg | SUPPORTED_TP | SUPPORTED_MII);

	/* only supports twisted-pair */
	ecmd->port = PORT_MII;

	/* only supports internal transceiver */
	ecmd->transceiver = XCVR_INTERNAL;

	/* this isn't fully supported at higher layers */
	ecmd->phy_address = mii->phy_id;

	ecmd->advertising = ADVERTISED_TP | ADVERTISED_MII;
	advert = mii->mdio_read(dev, mii->phy_id, MII_ADVERTISE);
	if (advert & ADVERTISE_10HALF)
		ecmd->advertising |= ADVERTISED_10baseT_Half;
	if (advert & ADVERTISE_10FULL)
		ecmd->advertising |= ADVERTISED_10baseT_Full;
	if (advert & ADVERTISE_100HALF)
		ecmd->advertising |= ADVERTISED_100baseT_Half;
	if (advert & ADVERTISE_100FULL)
		ecmd->advertising |= ADVERTISED_100baseT_Full;

	bmcr = mii->mdio_read(dev, mii->phy_id, MII_BMCR);
	lpa = mii->mdio_read(dev, mii->phy_id, MII_LPA);
	if (bmcr & BMCR_ANENABLE) {
		ecmd->advertising |= ADVERTISED_Autoneg;
		ecmd->autoneg = AUTONEG_ENABLE;
		
		nego = mii_nway_result(advert & lpa);
		if (nego == LPA_100FULL || nego == LPA_100HALF)
			ecmd->speed = SPEED_100;
		else
			ecmd->speed = SPEED_10;
		if (nego == LPA_100FULL || nego == LPA_10FULL) {
			ecmd->duplex = DUPLEX_FULL;
			mii->full_duplex = 1;
		} else {
			ecmd->duplex = DUPLEX_HALF;
			mii->full_duplex = 0;
		}
	} else {
		ecmd->autoneg = AUTONEG_DISABLE;

		ecmd->speed = (bmcr & BMCR_SPEED100) ? SPEED_100 : SPEED_10;
		ecmd->duplex = (bmcr & BMCR_FULLDPLX) ? DUPLEX_FULL : DUPLEX_HALF;
	}

	/* ignore maxtxpkt, maxrxpkt for now */

	return 0;
}

int _kc_mii_ethtool_sset(struct mii_if_info *mii, struct ethtool_cmd *ecmd)
{
	struct net_device *dev = mii->dev;

	if (ecmd->speed != SPEED_10 && ecmd->speed != SPEED_100)
		return -EINVAL;
	if (ecmd->duplex != DUPLEX_HALF && ecmd->duplex != DUPLEX_FULL)
		return -EINVAL;
	if (ecmd->port != PORT_MII)
		return -EINVAL;
	if (ecmd->transceiver != XCVR_INTERNAL)
		return -EINVAL;
	if (ecmd->phy_address != mii->phy_id)
		return -EINVAL;
	if (ecmd->autoneg != AUTONEG_DISABLE && ecmd->autoneg != AUTONEG_ENABLE)
		return -EINVAL;
				  
	/* ignore supported, maxtxpkt, maxrxpkt */
	
	if (ecmd->autoneg == AUTONEG_ENABLE) {
		u32 bmcr, advert, tmp;

		if ((ecmd->advertising & (ADVERTISED_10baseT_Half |
					  ADVERTISED_10baseT_Full |
					  ADVERTISED_100baseT_Half |
					  ADVERTISED_100baseT_Full)) == 0)
			return -EINVAL;

		/* advertise only what has been requested */
		advert = mii->mdio_read(dev, mii->phy_id, MII_ADVERTISE);
		tmp = advert & ~(ADVERTISE_ALL | ADVERTISE_100BASE4);
		if (ADVERTISED_10baseT_Half)
			tmp |= ADVERTISE_10HALF;
		if (ADVERTISED_10baseT_Full)
			tmp |= ADVERTISE_10FULL;
		if (ADVERTISED_100baseT_Half)
			tmp |= ADVERTISE_100HALF;
		if (ADVERTISED_100baseT_Full)
			tmp |= ADVERTISE_100FULL;
		if (advert != tmp) {
			mii->mdio_write(dev, mii->phy_id, MII_ADVERTISE, tmp);
			mii->advertising = tmp;
		}
		
		/* turn on autonegotiation, and force a renegotiate */
		bmcr = mii->mdio_read(dev, mii->phy_id, MII_BMCR);
		bmcr |= (BMCR_ANENABLE | BMCR_ANRESTART);
		mii->mdio_write(dev, mii->phy_id, MII_BMCR, bmcr);

		mii->force_media = 0;
	} else {
		u32 bmcr, tmp;

		/* turn off auto negotiation, set speed and duplexity */
		bmcr = mii->mdio_read(dev, mii->phy_id, MII_BMCR);
		tmp = bmcr & ~(BMCR_ANENABLE | BMCR_SPEED100 | BMCR_FULLDPLX);
		if (ecmd->speed == SPEED_100)
			tmp |= BMCR_SPEED100;
		if (ecmd->duplex == DUPLEX_FULL) {
			tmp |= BMCR_FULLDPLX;
			mii->full_duplex = 1;
		} else
			mii->full_duplex = 0;
		if (bmcr != tmp)
			mii->mdio_write(dev, mii->phy_id, MII_BMCR, tmp);

		mii->force_media = 1;
	}
	return 0;
}

int _kc_mii_link_ok (struct mii_if_info *mii)
{
	/* first, a dummy read, needed to latch some MII phys */
	mii->mdio_read(mii->dev, mii->phy_id, MII_BMSR);
	if (mii->mdio_read(mii->dev, mii->phy_id, MII_BMSR) & BMSR_LSTATUS)
		return 1;
	return 0;
}

int _kc_mii_nway_restart (struct mii_if_info *mii)
{
	int bmcr;
	int r = -EINVAL;

	/* if autoneg is off, it's an error */
	bmcr = mii->mdio_read(mii->dev, mii->phy_id, MII_BMCR);

	if (bmcr & BMCR_ANENABLE) {
		bmcr |= BMCR_ANRESTART;
		mii->mdio_write(mii->dev, mii->phy_id, MII_BMCR, bmcr);
		r = 0;
	}

	return r;
}

void _kc_mii_check_link (struct mii_if_info *mii)
{
	int cur_link = mii_link_ok(mii);
	int prev_link = netif_carrier_ok(mii->dev);

	if (cur_link && !prev_link)
		netif_carrier_on(mii->dev);
	else if (prev_link && !cur_link)
		netif_carrier_off(mii->dev);
}

int _kc_generic_mii_ioctl(struct mii_if_info *mii_if,
		      struct mii_ioctl_data *mii_data, int cmd,
		      unsigned int *duplex_chg_out)
{
	int rc = 0;
	unsigned int duplex_changed = 0;

	if (duplex_chg_out)
		*duplex_chg_out = 0;

	mii_data->phy_id &= mii_if->phy_id_mask;
	mii_data->reg_num &= mii_if->reg_num_mask;

	switch(cmd) {
	case SIOCDEVPRIVATE:	/* binary compat, remove in 2.5 */
	case SIOCGMIIPHY:
		mii_data->phy_id = mii_if->phy_id;
		/* fall through */

	case SIOCDEVPRIVATE + 1:/* binary compat, remove in 2.5 */
	case SIOCGMIIREG:
		mii_data->val_out =
			mii_if->mdio_read(mii_if->dev, mii_data->phy_id,
					  mii_data->reg_num);
		break;

	case SIOCDEVPRIVATE + 2:/* binary compat, remove in 2.5 */
	case SIOCSMIIREG: {
		u16 val = mii_data->val_in;

		if (!capable(CAP_NET_ADMIN))
			return -EPERM;

		if (mii_data->phy_id == mii_if->phy_id) {
			switch(mii_data->reg_num) {
			case MII_BMCR: {
				unsigned int new_duplex = 0;
				if (val & (BMCR_RESET|BMCR_ANENABLE))
					mii_if->force_media = 0;
				else
					mii_if->force_media = 1;
				if (mii_if->force_media &&
				    (val & BMCR_FULLDPLX))
					new_duplex = 1;
				if (mii_if->full_duplex != new_duplex) {
					duplex_changed = 1;
					mii_if->full_duplex = new_duplex;
				}
				break;
			}
			case MII_ADVERTISE:
				mii_if->advertising = val;
				break;
			default:
				/* do nothing */
				break;
			}
		}

		mii_if->mdio_write(mii_if->dev, mii_data->phy_id,
				   mii_data->reg_num, val);
		break;
	}

	default:
		rc = -EOPNOTSUPP;
		break;
	}

	if ((rc == 0) && (duplex_chg_out) && (duplex_changed))
		*duplex_chg_out = 1;

	return rc;
}

/*****************************************************************************/
/* 2.4.13 => 2.4.3 */
#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,4,13) )

/**************************************/
/* PCI DMA MAPPING */

#if defined(CONFIG_HIGHMEM)

#ifndef PCI_DRAM_OFFSET
#define PCI_DRAM_OFFSET 0
#endif

u64 _kc_pci_map_page(struct pci_dev *dev, struct page *page, unsigned long offset, size_t size, int direction)
{
	return (((u64)(page - mem_map) << PAGE_SHIFT) + offset + PCI_DRAM_OFFSET);
}

#else /* CONFIG_HIGHMEM */

u64 _kc_pci_map_page(struct pci_dev *dev, struct page *page, unsigned long offset, size_t size, int direction)
{
	return pci_map_single(dev, (void *)page_address(page) + offset, size, direction);
}

#endif /* CONFIG_HIGHMEM */

void _kc_pci_unmap_page(struct pci_dev *dev, u64 dma_addr, size_t size, int direction)
{
	return pci_unmap_single(dev, dma_addr, size, direction);
}

#endif /* 2.4.13 => 2.4.3 */


/*****************************************************************************/
/* 2.4.3 => 2.4.0 */
#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,4,3) )

/**************************************/
/* PCI DRIVER API */

#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0) )
int _kc_pci_set_dma_mask(struct pci_dev *dev, dma_addr_t mask) { return 0; }
#else
int _kc_pci_set_dma_mask(struct pci_dev *dev, dma_addr_t mask)
{
	if(!pci_dma_supported(dev, mask))
		return -EIO;
	dev->dma_mask = mask;
	return 0;
}
#endif

int _kc_pci_request_regions(struct pci_dev *dev, char *res_name)
{
	int i;

	for (i = 0; i < 6; i++) {
		if (pci_resource_len(dev, i) == 0)
			continue;

#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0) )
		if ((dev->base_address[i] & PCI_BASE_ADDRESS_SPACE_IO))
			request_region(pci_resource_start(dev, i), pci_resource_len(dev, i), res_name);
#else
		if (pci_resource_flags(dev, i) & IORESOURCE_IO) {
			if (!request_region(pci_resource_start(dev, i), pci_resource_len(dev, i), res_name)) {
				pci_release_regions(dev);
				return -EBUSY;
			}
		} else if (pci_resource_flags(dev, i) & IORESOURCE_MEM) {
			if (!request_mem_region(pci_resource_start(dev, i), pci_resource_len(dev, i), res_name)) {
				pci_release_regions(dev);
				return -EBUSY;
			}
		}
#endif
	}
	return 0;
}

void _kc_pci_release_regions(struct pci_dev *dev)
{
	int i;

	for (i = 0; i < 6; i++) {
		if (pci_resource_len(dev, i) == 0)
			continue;

#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0) )
		if ((dev->base_address[i] & PCI_BASE_ADDRESS_SPACE))
			release_region(pci_resource_start(dev, i), pci_resource_len(dev, i));
#else
		if (pci_resource_flags(dev, i) & IORESOURCE_IO)
			release_region(pci_resource_start(dev, i), pci_resource_len(dev, i));

		else if (pci_resource_flags(dev, i) & IORESOURCE_MEM)
			release_mem_region(pci_resource_start(dev, i), pci_resource_len(dev, i));
#endif
	}
}

/**************************************/
/* NETWORK DRIVER API */

#define _KC_MAX_NET_DEV 32
static int my_net_count = 0;
static struct _kc_net_dev_ext my_net_devices[_KC_MAX_NET_DEV];

struct net_device * _kc_alloc_etherdev(int sizeof_priv)
{
	struct net_device *dev;
	int alloc_size;

	if(my_net_count >= _KC_MAX_NET_DEV)
		return NULL;

	alloc_size = sizeof (*dev) + sizeof_priv + IFNAMSIZ + 31;

	dev = kmalloc(alloc_size, GFP_KERNEL);

	if (!dev) return NULL;

	memset(dev, 0, alloc_size);

	if (sizeof_priv)
		dev->priv = (void *) (((unsigned long)(dev + 1) + 31) & ~31);

#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0) )
	dev->name = (char *) dev->priv + sizeof_priv;
#endif
	dev->name[0] = '\0';

	ether_setup(dev);

	my_net_devices[my_net_count].dev = dev;
	my_net_count++;

	return dev;
}

int _kc_is_valid_ether_addr(u8 *addr)
{
	const char zaddr[6] = {0,};

	return !(addr[0]&1) && memcmp( addr, zaddr, 6);
}

#endif /* 2.4.3 => 2.4.0 */


/*****************************************************************************/
/* 2.4.0 => 2.2.0 */
#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0) )

/**************************************/
/* PCI DRIVER API */

#define _KC_MAX_PCI_DEV 32
static int my_pci_count = 0;
static struct _kc_pci_dev_ext my_pci_devices[_KC_MAX_PCI_DEV];

int _kc_pci_module_init(struct pci_driver *drv)
{
	struct pci_dev *dev;
	struct pci_device_id *pciid;
	uint16_t subvendor, subdevice;

	my_pci_count = 0;

	for(dev = pci_devices; dev; dev = dev->next) {

		if(my_pci_count >= _KC_MAX_PCI_DEV)
			break;

		pciid = &drv->id_table[0];
		pci_read_config_word(dev, PCI_SUBSYSTEM_VENDOR_ID, &subvendor);
		pci_read_config_word(dev, PCI_SUBSYSTEM_ID, &subdevice);

		while(pciid->vendor != 0) {
			if(((pciid->vendor == dev->vendor) ||
			   (pciid->vendor == PCI_ANY_ID)) &&

			  ((pciid->device == dev->device) ||
			   (pciid->device == PCI_ANY_ID)) &&

			  ((pciid->subvendor == subvendor) ||
			   (pciid->subvendor == PCI_ANY_ID)) &&

			  ((pciid->subdevice == subdevice) ||
			   (pciid->subdevice == PCI_ANY_ID))) {

				my_pci_devices[my_pci_count].dev = dev;
				my_pci_devices[my_pci_count].driver = drv;
				my_pci_count++;
				if(drv->probe(dev, pciid)) {
					my_pci_count--;
					my_pci_devices[my_pci_count].dev = NULL;
				}
				break;
			}
			pciid++;
		}
	}
	return (my_pci_count > 0) ? 0 : -ENODEV;
}

void _kc_pci_unregister_driver(struct pci_driver *drv)
{
	int i;
	for(i = 0; i < my_pci_count; i++) {
		if(my_pci_devices[i].dev) {
			drv->remove(my_pci_devices[i].dev);
			my_pci_devices[i].dev = NULL;
		}
	}
	my_pci_count = 0;
}

void _kc_pci_set_drvdata(struct pci_dev *dev, void *data)
{
	int i;
	for(i = 0; i < my_pci_count; i++) {
		if(my_pci_devices[i].dev == dev) {
			my_pci_devices[i].pci_drvdata = data;
		}
	}
}

void * _kc_pci_get_drvdata(struct pci_dev *dev)
{
	int i;
	for(i = 0; i < my_pci_count; i++) {
		if(my_pci_devices[i].dev == dev) {
			return my_pci_devices[i].pci_drvdata;
		}
	}
	return NULL;
}

int _kc_pci_enable_device(struct pci_dev *dev) { return 0; }

int _kc_pci_resource_start(struct pci_dev *dev, int bar)
{
	return ((dev->base_address[bar] & PCI_BASE_ADDRESS_SPACE) ?
		(dev->base_address[bar] & PCI_BASE_ADDRESS_IO_MASK) :
		(dev->base_address[bar] & PCI_BASE_ADDRESS_MEM_MASK));
}

unsigned long _kc_pci_resource_len(struct pci_dev *dev, int bar)
{
	u32 old, len;

	int bar_reg = PCI_BASE_ADDRESS_0 + (bar << 2);

	pci_read_config_dword(dev, bar_reg, &old);
	pci_write_config_dword(dev, bar_reg, ~0);
	pci_read_config_dword(dev, bar_reg, &len);
	pci_write_config_dword(dev, bar_reg, old);

	if((len & PCI_BASE_ADDRESS_SPACE) == PCI_BASE_ADDRESS_SPACE_MEMORY)
		len = ~(len & PCI_BASE_ADDRESS_MEM_MASK);
	else
		len = ~(len & PCI_BASE_ADDRESS_IO_MASK) & 0xffff;
	return (len + 1);
}

struct pci_driver *_kc_pci_dev_driver(struct pci_dev *dev)
{
	int i;
	for(i = 0; i < my_pci_count; i++) {
		if(my_pci_devices[i].dev == dev) {
			return my_pci_devices[i].driver;
		}
	}
	return NULL;
}

/**************************************/
/* PCI DMA MAPPING */

void * _kc_pci_alloc_consistent(struct pci_dev *dev, size_t size, u64 *dma_handle)
{
	void *vaddr = kmalloc(size, GFP_KERNEL);

	if(vaddr)
		*dma_handle = virt_to_bus(vaddr);
	return vaddr;
}

void _kc_pci_free_consistent(struct pci_dev *dev, size_t size, void *addr, u64 dma_handle)
{
	kfree(addr);
}

u64 _kc_pci_map_single(struct pci_dev *dev, void *addr, size_t size, int direction)
{
	return virt_to_bus(addr);
}

void _kc_pci_unmap_single(struct pci_dev *dev, u64 dma_addr, size_t size, int direction) { return; }

void _kc_pci_dma_sync_single(struct pci_dev *dev, u64 dma_addr, size_t size, int direction) { return; }

/**************************************/
/* NETWORK DRIVER API */

void _kc_netif_device_attach(struct net_device *dev)
{
	if(netif_running(dev) && netif_queue_stopped(dev)) {
		netif_wake_queue(dev);
	}
}

void _kc_netif_device_detach(struct net_device *dev)
{
	if(netif_running(dev)) {
		netif_stop_queue(dev);
	}
}

void _kc_netif_carrier_on(struct net_device *dev)
{
	int i;
	for(i = 0; i < my_net_count; i++) {
		if(my_net_devices[i].dev == dev) {
			set_bit(0, &my_net_devices[i].carrier);
		}
	}
}

void _kc_netif_carrier_off(struct net_device *dev)
{
	int i;
	for(i = 0; i < my_net_count; i++) {
		if(my_net_devices[i].dev == dev) {
			clear_bit(0, &my_net_devices[i].carrier);
		}
	}
}

int _kc_netif_carrier_ok(struct net_device *dev)
{
	int i;
	for(i = 0; i < my_net_count; i++) {
		if(my_net_devices[i].dev == dev) {
			return test_bit(0, &my_net_devices[i].carrier);
		}
	}
	return 0;
}

#endif /* 2.4.0 => 2.2.0 */

/*****************************************************************************/
/* 2.4.6 => 2.4.3 */
#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,4,6) )

int _kc_pci_set_power_state(struct pci_dev *dev, int state)
{ return 0; }
int _kc_pci_save_state(struct pci_dev *dev, u32 *buffer)
{ return 0; }
int _kc_pci_restore_state(struct pci_dev *pdev, u32 *buffer)
{ return 0; }
int _kc_pci_enable_wake(struct pci_dev *pdev, u32 state, int enable)
{ return 0; }
void _kc_pci_disable_device(struct pci_dev *pdev)
{}

#endif /* 2.4.6 => 2.4.3 */

#ifndef IRQ_HANDLED
#define irqreturn_t void
#define IRQ_HANDLED
#define IRQ_NONE
#endif

#ifndef SET_NETDEV_DEV
#define SET_NETDEV_DEV(net, pdev)
#endif

#ifndef likely
#define likely(x)	(x)
#define unlikely(x)	(x)
#endif

#ifndef HAVE_FREE_NETDEV
#define free_netdev(x)	kfree(x)
#endif

#ifdef HAVE_POLL_CONTROLLER
#define CONFIG_NET_POLL_CONTROLLER
#endif

#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,4,22) )
#define pci_name(x)	((x)->slot_name)
#endif

#ifndef module_param
#define module_param(v,t,p) MODULE_PARM(v, "i");
#endif

#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,6,4) )
#undef netdev_priv
#define netdev_priv _kc_netdev_priv
static inline void *_kc_netdev_priv(struct net_device *dev)
{
	return dev->priv;
}
#endif

#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,6,5) )
#define pci_dma_sync_single_for_cpu	pci_dma_sync_single
#define pci_dma_sync_single_for_device	pci_dma_sync_single_for_cpu
#endif

#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,6,7) )
#undef if_mii
#define if_mii _kc_if_mii
static inline struct mii_ioctl_data *_kc_if_mii(struct ifreq *rq)
{
	return (struct mii_ioctl_data *) &rq->ifr_ifru;
}
#endif /* < 2.6.7 */

#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,6,8) )
#define msleep(x)	do { if(in_interrupt()) { \
				/* Don't mdelay in interrupt context! */ \
	                	BUG(); \
			} else { \
				set_current_state(TASK_UNINTERRUPTIBLE); \
				schedule_timeout((x * HZ)/1000 + 2); \
			} } while(0)
#endif

#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,6,9) )
#define msleep_interruptible(x)	do {set_current_state(TASK_INTERRUPTIBLE); \
					schedule_timeout((x * HZ)/1000); \
				} while(0)
#endif

#if ( LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,6) && LINUX_VERSION_CODE < KERNEL_VERSION(2,6,10) )
#ifdef pci_save_state
#undef pci_save_state
#endif
#define pci_save_state(X) { \
	int i; \
	if (nic->pm_state) { \
		for (i = 0; i < 16; i++) { \
			pci_read_config_dword((X), \
					      i * 4, \
					      &nic->pm_state[i]); \
		} \
	} \
}

#ifdef pci_restore_state
#undef pci_restore_state
#endif
#define pci_restore_state(X) { \
	int i; \
	if (nic->pm_state) { \
		for (i = 0; i < 16; i++) { \
			pci_write_config_dword((X), \
					       i * 4, \
					       nic->pm_state[i]); \
		} \
	} else { \
		for (i = 0; i < 6; i++) { \
			pci_write_config_dword((X), \
					       PCI_BASE_ADDRESS_0 + (i * 4), \
					       (X)->resource[i].start); \
		} \
		pci_write_config_byte((X), PCI_INTERRUPT_LINE, (X)->irq); \
	} \
}
#endif /* 2.4.6 <= x < 2.6.10 */

#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,6,10) )
#ifdef module_param_array_named
#undef module_param_array_named
#define module_param_array_named(name, array, type, nump, perm)          \
	static struct kparam_array __param_arr_##name                    \
	= { ARRAY_SIZE(array), nump, param_set_##type, param_get_##type, \
	    sizeof(array[0]), array };                                   \
	module_param_call(name, param_array_set, param_array_get,        \
			  &__param_arr_##name, perm)
#endif /* module_param_array_named */
#endif /* < 2.6.10 */

#ifndef NET_IP_ALIGN
#define NET_IP_ALIGN 2
#endif

#include "ethtool.c"

#endif /* _KCOMPAT_H_ */

