#!/bin/bash

#**************************************************************************
#    begin                : Wed Feb 28, 2001
#    copyright            : (C) 2003 by Adaptec
#
#    This is the installer script for Adaptec Linux drivers
#**************************************************************************
#                                                                         *
#   This program is free software; you can redistribute it and/or modify  *
#   it under the terms of the GNU General Public License as published by  *
#   the Free Software Foundation; either version 2 of the License, or     *
#   (at your option) any later version.                                   *
#                                                                         *
#**************************************************************************

#
# Product Specific Settings
#
product=aic79xx
version=2.0.4-1
module_install_path=kernel/drivers/scsi/aic7xxx

installer_path=/usr/adaptec/$product
PATH=$installer_path:$PATH 
export PATH 

verbose=0
root_dir=""
do_mkinitrd=1
do_lilo=1
force=0
module_count=0
module_count_file=/tmp/$product.$$
savail=0
sneeded=0

lilo_conf=$root_dir"/etc/lilo.conf"
grub_conf=$root_dir"/boot/grub/grub.conf"
elilo_conf=`find /boot -type f -name elilo.conf -print`
boot_dir=`echo $elilo_conf | sed -e 's,elilo.conf,,'`
if [ -z "$boot_dir" ] ; then
	boot_dir="/boot"
fi
if [ ! -e "$grub_conf" ] ; then
	grub_conf=$root_dir"/boot/grub/menu.lst"
fi
prebuilt_binaries=/usr/adaptec/$product/$product"_prebuilt.tgz"
tmp_magic=/tmp/magic$$

file=file
if [ -x /usr/bin/file ] ; then
	file=/usr/bin/file
elif [ -x $root_dir/usr/bin/file ] ; then
	file=$root_dir/usr/bin/file
fi

#
# Embedded Awk Scripts
#
parse_grub_conf='
	BEGIN {
		image=""
		initrd="?"
		label="?"
		title="?"
	}
	/^[[:space:]]*title/ {
		if ( label != "?" ) {
			dumpentry()
		}

		initrd="?"
		image="?"

		label=$2
		for (i = 3; i <= NF; i++) {
			label = label " " $i
		}
	}
	/^[[:space:]]*kernel/ {
		image=$2
		sub(/^\(.*\)/, "", image)
	}
	/^[[:space:]]*initrd/ {
		initrd=$2
		sub(/^\(.*\)/, "", initrd)
	}
	function dumpentry() {
		if (image != "?" && image !~ /^\/boot/) {
			if (image !~ /^\//) {
				image = "/" image
			}
			image = "/boot" image
		}
		if (initrd != "?" && initrd !~ /^\/boot/) {
			if (initrd !~ /^\//) {
				initrd = "/" initrd
			}
			initrd = "/boot" initrd
		}
		print image " " initrd " " label
		return
	}
	END {
		dumpentry()
		exit 0
	}'

parse_lilo_conf='
	BEGIN {
		FS="[ \11=]*"
		global=1
		global_initrd="?"
		global_label="?"
		image=""
		initrd="?"
		label="?"
		isImage=0
	}
	/image[ \11]*=/ {
		if ( global == 0 && isImage == 1) {
			print image " " initrd " " label
			label="?"
			initrd="?"
		}
		isImage = 1
		if ( $1 == "" ) {
			image=$3
		} else {
			image=$2
		}
		label="?"
		initrd="?"
		if (( global_initrd != "?" ) && ( global != 0 )) {
			print "? " global_initrd " " global_label
		}
		global=0
	}
	/label[ \11]*=/ {
		if ( $1 == "" ) {
			label=$3
		} else {
			label=$2
		}
		if ( global != 0 ) {
			global_label=$2
		}
	}
	/initrd[ \11]*=/ {
		if ( $1 == "" ) {
			initrd=$3
		} else {
			initrd=$2
		}
		if ( global != 0 ) {
			global_initrd=$2
		}
	}
	/other[ \11]*=/ {
		if ( global == 0 && isImage == 1 ) {
			print image " " initrd " " label
			label="?"
			initrd="?"
		}
		isImage = 0
		label="?"
		initrd="?"
		image="?"
		if (( global_initrd != "?" ) && ( global != 0 )) {
			print "? " global_initrd " " global_label
		}
		global=0
	}
	END {
		if ( global == 0 && isImage == 1) {
			print image " " initrd " " label
		}
	}'

parse_modules_conf='
	BEGIN {
		c=1
	}
	{
		if (($1 == "alias") \
		 && (substr($2, 0, length("scsi_hostadapter")) \
		  == "scsi_hostadapter")) {
			$2="scsi_hostadapter"c
			print $0
			c=c+1
		} else {
			print $0
		}
	}'

#
# Parse df -k output returning the amount of
# free space in bytes for each filesystem.
#
parse_df='
	/^Filesystem/ {
		next
	}
	{
		print $4
	}'

#
# Return the largest filesize in tar t output.
#
parse_tar='
	BEGIN {
		biggest=0
	}
	{
		if ($3 > biggest)
			biggest = $3
	}
	END {
		if (biggest != 0) {
			printf "%d", biggest/1024
			exit 0
		}
		exit 1
	}'

# List of recognized United Linux kernels.  The
# kernel name is the name used for our module that
# supports the listed kernel build.
united_linux_kernel_table()
{
cat << EOF

# -------------------------------------------------------------------------
# Kernel Name       build  md5sum of vmlinuz
# -------------------------------------------------------------------------
2.4.19-82-64GB-SMP   113    81172db51144dc51d8fe1d4b5f3370f0

2.4.19-82-4GB        120    13aaf51654edb0e4939fa594803f090b
					
2.4.19-290-4GB       290    5cb186a3694c85a353c564caea87bc41
					
2.4.19-290-64GB      290    87aa76d6aa9947acdae0ca378f439864

2.4.19-304-4GB       304    f72b37b4038008816359a59d8214e044 

2.4.19-304-64GB-SMP  304    8e1c11ec813a5d88e0f96c72acc32227

2.4.19-333-4GB       333    b1b15f6d6606f1a7462ff77d77228f03 
					
2.4.19-333-64GB-SMP  333    d9586773378cbb42b16d86fa448855f4
					
2.4.19-340-4GB       340    20ad92fec50aca1e9f50367d8ce67c08
					
2.4.19-340-64GB-SMP  340    f0aec500ab3fa3483a6cd79126dc2d9b
					
EOF
}

# output usage statement
usage () {
	echo "usage: `basename $0` product version module_dir" >&2
	echo "            [--version] || [--help] ||" >&2
	echo "            [--verbose] [--root=<root dir>] [--noinitrd]" >&2
	echo "            [--nolilo] [--force]" >&2
	echo
	echo "usage: `basename $0` [--list_modules] " >&2
	echo
	echo "For the proceedure to patch your kernel sources see /usr/adaptec/$product/readme.txt"
	echo
	exit 1
}

set_module_count()
{
	module_count=$1
	echo $module_count > $module_count_file
}

update_module_count()
{
	module_count=`cat $module_count_file 2> /dev/null || echo 0`
}

cleanup_module_count()
{
	rm -f $module_count_file 2> /dev/null
}

# write an updated magic file
write_magic () {
cat <<-EOF >$tmp_magic
	0x202	string		HdrS		Linux/i386 Kernel
	>0x206	short		0x0105		setupV1 zImage
	>0x206	short		0x0200		setupV2
	>>0x211	byte		0		zImage
	>>0x211	byte		1		bzImage
	>(0x20e.s+0x205)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%5.5s"
	>(0x20e.s+0x206)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%6.6s"
	>(0x20e.s+0x207)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%7.7s"
	>(0x20e.s+0x208)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%8.8s"
	>(0x20e.s+0x209)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%9.9s"
	>(0x20e.s+0x20a)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%10.10s"
	>(0x20e.s+0x20b)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%11.11s"
	>(0x20e.s+0x20c)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%12.12s"
	>(0x20e.s+0x20d)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%13.13s"
	>(0x20e.s+0x20e)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%14.14s"
	>(0x20e.s+0x20f)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%15.15s"
	>(0x20e.s+0x210)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%16.16s"
	>(0x20e.s+0x211)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%17.17s"
	>(0x20e.s+0x212)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%18.18s"
	>(0x20e.s+0x213)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%19.19s"
	>(0x20e.s+0x214)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%20.20s"
	>(0x20e.s+0x215)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%21.21s"
	>(0x20e.s+0x216)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%22.22s"
	>(0x20e.s+0x217)	byte	32
	>>(0x20e.s+0x200)	string	>\0	"%23.23s"
EOF
}

cleanup_magic () {
	rm -f $tmp_magic 2> /dev/null
}

list_prebuilt_modules () {

	echo Prebuilt modules:
	list=`tar ztf $prebuilt_binaries 2> /dev/null | sort` 
	display_list=`echo $list | sed "s/$product-//g"`
	for i in $display_list ; do
		echo "	** $i"
	done
	
	echo
}


install_module() {
	if [ ! -e "$1/$module_install_path" ] ; then
		mkdir -p $1/$module_install_path 
	fi

	# Backup Modules
	find $1 -name $product.o -print | \
	while read module; do
		backup_file $module 0
		rm -f $module 2> /dev/null
	done

	saved_pwd=`pwd`
	cd $1/$module_install_path

	versions="ia64 athlon i686 i586 i386 ia32"

	if [ $verbose -eq 1 ] ; then
		echo Installing Module...
	fi

	firstloop=1
	for m in $versions ; do
		if [ $firstloop -eq 1 ] ; then
			if [ $m = $3 ] ; then
				firstloop=0
			else
				continue
			fi
		fi
		if [ ! -n "$m" ] ; then
			m="ia32"
		fi
		if [ $verbose -eq 1 ] ; then
			echo "Checking for prebuilt $2-$m."
		fi
		module_file=$product-$2-$m
		tar zxf $prebuilt_binaries $module_file > /dev/null 2> /dev/null
		if [ -e "$module_file" ] ; then 
			if [ $verbose -eq 1 ] ; then
				echo "Found prebuilt for $2-$m."
			fi
			break;
		fi
	done
	if ! [ -e "$module_file" ] ; then
		echo
		echo "****** PROBLEM *******"
		echo "No prebuilt module for "$2 $3
		echo "**********************"
		cd $saved_pwd
		return 1
	fi	   
	mv -f $module_file $product.o
	if [ $verbose -eq 1 ] ; then
		echo "Module installed for "$2" kernel."
	fi
	cd $saved_pwd
	return 0
}


modify_sysconfig_kernel() {
	sysconfig_kernel=$root_dir"/etc/sysconfig/kernel"
	if [ $verbose -eq 1 ] ; then
		echo "Modifying "$sysconfig_kernel
	fi

	if ! [ -e "$sysconfig_kernel" ] ; then
		if [ $verbose -eq 1 ] ; then
			echo "No rc.config file"
		fi
		return 1
	fi

	backup_file $sysconfig_kernel 0
	if [ $verbose -eq 1 ] ; then
		grep "INITRD_MODULES" $sysconfig_kernel
	else
		grep "INITRD_MODULES" $sysconfig_kernel > /dev/null
	fi
        if [ $? -ne 0 ] ; then
		# Configuration not set up to use ramdisks add it
		echo "INITRD_MODULES= \"\"" >>$sysconfig_kernel 
        fi
	sed "s/INITRD_MODULES[ \t]*=[ \t]*/INITRD_MODULES=/" $sysconfig_kernel >/tmp/sysconfig_kernel.tmp && mv -f /tmp/sysconfig_kernel.tmp $sysconfig_kernel

        current_modules=`grep "INITRD_MODULES[ \t]*=" $sysconfig_kernel | head -1`
        echo $current_modules | grep $product > /dev/null
	if [ $? -ne 0 ] ; then
		# If we cannot find in the config file, add it.
		new_modules=`echo $current_modules | sed "s/\"[ \t]*$/ $product\"/"`
		sed "s/INITRD_MODULES.*/$new_modules/" $sysconfig_kernel >/tmp/sysconfig_kernel.tmp && mv -f /tmp/sysconfig_kernel.tmp $sysconfig_kernel
        fi
	if [ $verbose -eq 1 ] ; then
		grep "INITRD_MODULES" $sysconfig_kernel
	fi
	return 0
}


modify_rc_config() {
	rc_config=$root_dir"/etc/rc.config"
	if [ $verbose -eq 1 ] ; then
		echo "Modifying "$rc_config
	fi

	if ! [ -e "$rc_config" ] ; then
		if [ $verbose -eq 1 ] ; then
			echo "No rc.config file"
		fi
		return 1
	fi

	backup_file $rc_config 0
	if [ $verbose -eq 1 ] ; then
		grep "INITRD_MODULES" $rc_config
	else
		grep "INITRD_MODULES" $rc_config > /dev/null
	fi
        if [ $? -ne 0 ] ; then
		# Configuration not set up to use ramdisks add it
		echo "INITRD_MODULES= \"\"" >>$rc_config 
        fi
	sed "s/INITRD_MODULES[ \t]*=[ \t]*/INITRD_MODULES=/" $rc_config >/tmp/rc.config.tmp && mv -f /tmp/rc.config.tmp $rc_config

        current_modules=`grep "INITRD_MODULES[ \t]*=" $rc_config | head -1`
        echo $current_modules | grep $product > /dev/null
	if [ $? -ne 0 ] ; then
		# If we cannot find in the config file, add it.
		new_modules=`echo $current_modules | sed "s/\"[ \t]*$/ $product\"/"`
		sed "s/INITRD_MODULES.*/$new_modules/" $rc_config >/tmp/rc.config.tmp && mv -f /tmp/rc.config.tmp $rc_config
        fi
	if [ $verbose -eq 1 ] ; then
		grep "INITRD_MODULES" $rc_config
	fi
	return 0
}


modify_modules_conf() {

	modules_conf=""
	if [ -e "$root_dir/etc/modules.conf" ] ; then
		modules_conf="$root_dir/etc/modules.conf"
	elif [ -e "$root_dir/etc/conf.modules" ] ; then
		modules_conf="$root_dir/etc/conf.modules"
	else
		if [ $verbose -eq 1 ] ; then
			echo "No modules.conf file"
		fi
		return 1
	fi
	if [ $verbose -eq 1 ] ; then
		echo "Modifying "$modules_conf
	fi

	backup_file $modules_conf 0

	# Remove references to old names for our drivers like aic7xxx_mod.o
	sed -e "s/${product}_mod/$product/g" $modules_conf \
	    > /tmp/modules.conf$$
	mv /tmp/modules.conf$$ $modules_conf
	grep 'alias scsi_hostadapter' $modules_conf \
	     | grep "${product}[[:space:]]*$" > /dev/null
	if [ $? -ne 0 ] ;  then
		# Put in entry if not already in config file
		echo "alias scsi_hostadapter $product" >> $modules_conf
		if [ $verbose -eq 1 ] ; then
			echo "No $product entry - adding it..."
		fi
	else
		grep 'alias scsi_hostadapter' $modules_conf \
		     | grep "^[^#]*$product" > /dev/null
		if [ $? -ne 0 ] ;  then
			# Module has been explicitly disabled from the
			# initrd.  Skip the mkinitrd step.
			if [ $verbose -eq 1 ] ; then
				echo "Module explicitly disabled in $modules_conf."
				echo "Skipping mkintird step."
			fi
			do_mkinitrd=0
		fi
	fi

	tmp_modules=/tmp/modules.conf.$$

	#
	# Enumerate/Renumber scsi_hostadapter entries
	#
	awk "$parse_modules_conf" $modules_conf >$tmp_modules \
		&& mv -f $tmp_modules  $modules_conf
	if [ $verbose -eq 1 ] ; then
		grep $product $modules_conf 
	fi
}

# Making the initrd is one of the things that are very different 
# between the different distributions.  This script has been tested
# with RedHat, SuSE, Mandrake, and Caldera

run_mkinitrd() {

        echo "Creating initrd..."
# $1 = image
# $2 = kernel version
# $3 = initrd

#	if [ -e $3 ] ; then
		backup_initrd $3
		rm -f $3
#	fi
	mkinitrd=`which mk_initrd 2> /dev/null`

	if [ -z "$mkinitrd" ] ; then	
		# if not the SuSE mk_initrd then try Red Hat/Mandrake style
		if [ "$root_dir" != "" ] ; then
			# if we specified root then use our modified one
			mkinitrd="/usr/adaptec/$product/adpt_mkinitrd --root="$root_dir
		else
			mkinitrd=`which mkinitrd 2> /dev/null`
		fi

		if [ -z "$mkinitrd" ] ; then
			# not Red Hat - check if Caldera
			tmp_initrd=`echo $3 | sed s/.gz$//`
			if [ "$tmp_initrd" != "$initrd" ] ; then
				if [ $verbose -eq 1 ] ; then
					echo "Making Caldera initrd..."
				fi
				if ! [ -e "/initrd" ] ; then
					mkdir -f /initrd
				fi

				mv -f $old_initrd $3

				rootfs="/etc/modules/rootfs"
				grep scsi_mod $rootfs > /dev/null 2> /dev/null
				if [ $? -ne 0 ] ; then
					echo scsi_mod >>$rootfs
				fi
				grep $product $rootfs > /dev/null 2> /dev/null
				if [ $? -ne 0 ] ; then
					echo $product >>$rootfs
				fi
				grep sd $rootfs > /dev/null 2> /dev/null
				if [ $? -ne 0 ] ; then
					echo sd >>$rootfs
				fi
				gunzip $3 > /dev/null 
				mount -o loop $tmp_initrd /initrd > /dev/null 2> /dev/null
				cp -f /lib/modules/$2/$module_install_path/$product.o /initrd/lib/modules/
				cp -f $rootfs /initrd/etc/modules/rootfs
				sync
				umount $tmp_initrd > /dev/null 2> /dev/null
				sync
				gzip $tmp_initrd > /dev/null
				depmod -qa $2
				return 0
			else
				echo "Can not find mkinitrd. Exitting..."
				exit 1
			fi
		fi

		if [ $verbose -eq 1 ] ; then
			echo "$mkinitrd $3 $2"
		fi

		sh -c "$mkinitrd $3 $2"
	else
		grep root_device $mkinitrd > /dev/null 2> /dev/null
		if [ $? -ne 0 ] ; then
			mkinitrd="/usr/adaptec/$product/adpt_mk_initrd"
		fi

		if [ "$root_dir" != "" ] ; then
			mkinitrd=$mkinitrd" "$root_dir
		fi
		if [ $verbose -eq 1 ] ; then
			echo "$mkinitrd -k `basename $1` -i `basename $3`"
		fi

		sh -c "$mkinitrd  -k `basename $1` -i `basename $3`"
	fi

	if [ $? -ne 0 ] ; then
		echo Problem making initrd.  Quitting.
		if [ -e $old_initrd ] ; then
			mv -f $old_initrd $3
		fi
		return 1
	fi
	return 0

}

run_lilo() {

	echo
        echo "Running lilo..."
	if [ "$root_dir" != "" ] ; then
		lilo -r $root_dir -w
	else
		lilo
	fi
	if [ $? -ne 0 ] ; then
		echo Problem running lilo.  Quitting.
		exit 1
	fi
	echo
        echo "Reboot to load Ram Disk."
	echo
}

begin_install_module() {

	echo "$boot_configs" | while read image initrd label; do

		if [ "X$boot_dir" != "X/boot" ] ; then
			cd $boot_dir
		fi
		if [ $image = "?" ] ; then
			global_initrd=$initrd
			continue
		fi
		echo

		echo $image $initrd $label
		#
		# Try to match the kernel by MD5 signature.  This
		# method works for stock kernels distributed for
		# SuSE and UnitedLinux where errata or service pack
		# kernels do not have their version numbers bumped
		# to differentiate them from the in-box kernel.
		#
		kernel=""
		sum=`md5sum $image | awk '{ print $1 }'`
		if [ $? -eq 0 ] ; then
			our_kernel_version=`united_linux_kernel_table | grep "$sum" | awk '{ print $1 }'`
			errata=`united_linux_kernel_table | grep "$sum" | awk '{ print $2 }'`
			kernel=`echo $our_kernel_version | sed "s,\([^-]*\)-[^-]*\(-.*\),\1-$errata\2,"`
			module_dir=`echo $kernel | sed 's,\([^-]*\)-[^-]*\(-.*\),\1\2,'`
		fi

		if [ -z "$kernel" ] ; then
			file_magic=`$file -L -m $tmp_magic $image 2> /dev/null`" \""
			kernel=`echo $file_magic | cut -d \" -f 2 `
			our_kernel_version=$kernel
			module_dir=$kernel
		fi

		if [ -z "$kernel" ] ; then
			# Try grepping.  Our magic file doesn't
			# understand IA64 kernels and making it
			# understand them looks difficult...
			kernel=`grep -a -e 'Linux version .* (' $image |
				tr -c -d [:print:] |
				sed -e 's,.*Linux version \([^ ]*\) (.*,\1,'`
			our_kernel_version=$kernel
			module_dir=$kernel
		fi

		if [ -z "$kernel" ] ; then
			# Same thing, but try gunziping first.
			kernel=`zcat $image |
				grep -a -e 'Linux version .* (' |
				tr -c -d [:print:] |
				sed -e 's,.*Linux version \([^ ]*\) (.*,\1,'`
			our_kernel_version=$kernel
			module_dir=$kernel
		fi

		if [ -z "$kernel" ] ; then
			# Same thing, but try bunziping first.
			kernel=`bzcat $image |
				grep -a -e 'Linux version .* (' |
				tr -c -d [:print:] |
				sed -e 's,.*Linux version \([^ ]*\) (.*,\1,'`
			our_kernel_version=$kernel
			module_dir=$kernel
		fi

		echo "Configuration ["$label"]"
		if [ -z "$kernel" ] ; then
			echo
			echo ***NOTICE: $image " is not a kernel. Skipping."
			echo
			continue
		fi

		if [ $verbose -eq 1 ] ; then
			echo "label="${label}" image="${image}" version="\[$kernel\]" initrd="${initrd}
		fi

		if [ $do_mkinitrd -eq 1 ] ; then
			if [ x$initrd = "x?" ] ; then
				if [ -z "$global_initrd" ] ; then
					echo
					echo "****** PROBLEM *******"
					echo Kernel $image is not configured to use initrd.  Skipping.
					echo
					echo "TO FIX THIS PROBLEM:"
					echo "Modify $loader_file for this config to use an initrd"
					echo "(e.g. initrd=/boot/initrd-$kernel.img)"
					echo "**********************"
					continue
				fi
				if [ -z "$last_initrd_kernel" ] ; then
					initrd=$global_initrd
				elif [ $last_initrd_kernel = $kernel ] ; then
					initrd=$global_initrd
				else
					echo "****** PROBLEM *******"
					echo A globally declared initrd is used for two different 
					echo kernel versions.  Please fix this in $loader_file
					echo and rerun the install.
					echo "**********************"
					exit 1
				fi
				if [ $verbose -eq 1 ] ; then
					echo "Setting initrd to global initrd->"$initrd
				fi
				last_initrd_kernel=$kernel

			fi

			if [ x$initrd = x$last_initrd ] ; then
				if [ x$kernel = x$last_kernel ] ; then
					echo "Same as last initrd. Skipping."
					continue;
				elif [ $force -ne 1 ] ; then
					echo
					echo "****** PROBLEM *******"
					echo In $loader_file there are two different kernel versions 
					echo assigned to the same initrd - fix this and rerun installer.
					echo Or use the --force option
					echo "**********************"
					exit 1
				fi
			fi
		fi

		echo $image" -> "$kernel
		last_kernel=$kernel
		last_initrd=$initrd
		module_root=$root_dir/lib/modules/$module_dir

		machine=`uname -m`
		if [ $machine = "i686" ] ; then
			# athlon check
			cat /proc/cpuinfo | grep "AMD Athlon"
			if [ $? -eq 0 ] ; then
				machine="athlon"
			fi
			cat /proc/cpuinfo | grep "AMD-K7"
			if [ $? -eq 0 ] ; then
				machine="athlon"
			fi
		fi

		install_module $module_root $our_kernel_version $machine
		if [ $? -ne 0 ] ; then
			continue;
		fi
		echo "Installing in "$module_root/$module_install_path

		module_count=`expr $module_count + 1`
		#
		# Allow parent shells to see updated value.
		#
		set_module_count $module_count

		if [ $do_mkinitrd -eq 1 ] ; then
			run_mkinitrd $image $kernel $initrd
			if [ $? -ne 0 ] ; then
				exit 1
			fi
		fi

	done
}

#
# Arg1: Filename to backup
# Arg2: Extra space required for replacement file
#
backup_file()
{
	sneeded=`ls -l $1 | awk '{printf "%d", $5/1024}'`
	sneeded=`expr $sneeded + $2`
	# Roundup to at least 1K.
	if [ $sneeded -eq 0 ] ; then
		sneeded=1;
	fi
	make_space_available
	if [ $? -ne 0 ] ; then
		echo
		echo "************ PROBLEM ***************"
		echo "Insufficient Space to Backup $module"
		echo "************************************"
		return 1
	fi
	if [ $? -eq 0 ] ; then
		if [ ! -d $backupdir ] ; then
			echo
			echo "Backing up all modified files to $backupdir"
			echo
			mkdir -p $backupdir
			if [ $? -ne 0 ] ; then
				echo
				echo "***************** PROBLEM ******************"
				echo "Unable to create backup directory $backupdir"
				echo "********************************************"
    			return 1
			fi
		fi
		targetdir=$backupdir`dirname $1`
		mkdir -p $targetdir
		if [ $? -ne 0 ] ; then
			echo
			echo "***************** PROBLEM ******************"
			echo "Unable to create backup directory $targetdir"
			echo "********************************************"
   			return 1
		fi
		cp $1 $targetdir
		if [ $? -ne 0 ] ; then
			echo
			echo "********** PROBLEM ************"
			echo "Unable to copy $1 to $targetdir"
			echo "*******************************"
			return 1
		fi
	fi
}

backup_initrd()
{
	mod_size=`tar ztvf $prebuilt_binaries \
		| grep $product | awk "$parse_tar"`
	backup_file $1 $mod_size
}

space_available()
{
	savail=`df -k $boot_dir | awk "$parse_df" `
	[ $savail -ge $sneeded ]
}

#
# Ensure there is enough space for our install by deleting
# backup directories - oldest first.
#
make_space_available()
{
	for backup in $backupdirs ; do
		if space_available ; then
			break
		fi

		if [ -d $backup ] ; then
			echo "Removing Backup Directory $backup"
			rm -rf $backup
		fi
	done

	# Return whether space is available
	space_available
}

cleanup() {

	cleanup_module_count
	cleanup_magic
}

#
# Main
#
trap cleanup EXIT

echo
echo Adaptec $product driver installer - V$version
echo 

#
# parse the arguments
#
while [ $# -gt 0 ]; do
	case $1 in
		--version)
			echo `basename $0`": version $version"
			exit 0
			;;

		--verbose)
			verbose=1
			;;

		--root*)
			if echo $1 | grep '=' > /dev/null ; then
				root_dir=`echo $1 | sed 's/^--root=//'`
			else
				root_dir=$2
				shift
			fi
			;;

		--list_modules)
			list_prebuilt_modules
			exit 0
			;;

		--noinitrd)
			do_mkinitrd=0
			do_lilo=0
			;;

		--nolilo)
			do_lilo=0
			;;

		--force)
			force=1
			;;

		--help)
			usage	
			;;
		*)
			usage
			;;
	esac

	shift
done

if [ $verbose -eq 1 ] ; then
	echo "Options: do_mkinitrd="$do_mkinitrd" do_lilo="$do_lilo" root dir="$root_dir
fi

#
# Specify a backup directory for all files we modify.
#
backupdirbase=$boot_dir/$product"-backup"
backupdirs=`ls -d1 $backupdirbase* | sort`
nextbackup=0
if [ -n "$backupdirs" ] ; then
	lastbackup=`echo "$backupdirs" | tail -n 1 | awk -F - '{print $3 }'`
	nextbackup=`expr $lastbackup + 1`
fi
backupdir=$backupdirbase"-"$nextbackup

#
# Determine boot loader.  We give priority to grub configuration files
# over any other configuration files found in the system.
#
if [ ! -e "$lilo_conf" -a ! -e "$grub_conf" -a ! -e "$elilo_conf" ] ; then
	echo "System is not configured to use a boot loader."
	echo "Checked lilo, elilo, and grub."
	echo "Stopping..."
	exit 1
fi

#
# Update config files so that our module will
# be included in any regenerated initrd.
#
if [ -e "${root_dir}/etc/sysconfig/kernel" ]; then
	modify_sysconfig_kernel
fi
if [ -e "${root_dir}/etc/rc.config" ]; then
	modify_rc_config
fi
if [ -e "${root_dir}/etc/modules.conf" ]; then
	modify_modules_conf
fi

#
# Write out a temporary magic file so we can
# use the "file" utility to help identify the
# kernels known to the boot loader.
#
write_magic 

#
# Parse boot loader configuration.
#
do_lilo=0
if [ -e "$grub_conf" ] ; then
	echo "Using GRUB configuration"
	echo "If this is not what you want, rename your $grub_conf file"
	loader_file=$grub_conf
	boot_configs=`awk "$parse_grub_conf" $loader_file | sort +2`
elif [ -e "$lilo_conf" -o -e "$elilo_conf" ] ; then

	if [ -e "$lilo_conf" ] ; then
		echo "Using LILO configuration"
		do_lilo=1
		loader_file=$lilo_conf
	else
		echo "Using ELILO configuration"
		loader_file=$elilo_conf
	fi

	boot_configs=`awk "$parse_lilo_conf" $loader_file | sort +2`
else 
	echo
	echo "**********************************************************"
	echo "******** Unable to find bootloader configuration. ********"
	echo "********          NO MODULES INSTALLED            ********"
	echo "**********************************************************"
	echo
	echo "Done."
	exit 1
fi

begin_install_module

exit_status=$?
if [ $exit_status -ne 0 ] ; then
	exit $exit_status
fi

update_module_count
if [ $module_count -eq 0 ] ; then
	echo
	echo "***************************************"
	echo "******** NO MODULES INSTALLED *********"
	echo "***************************************"
	echo
	echo "Done."
	exit 1
fi

echo

if [ $do_lilo -eq 1 ] ; then
run_lilo	
fi

echo Done.
exit
