#!/bin/bash

#######################################################################
# This is the DRI installation script. It compiles the kernel modules 
# and installs the kernel, DRI and libGL modules.
#
# Authors:
#
#   Frank Worsley   <fworsley@shaw.ca>
#   Alan Hourihane  <alanh@fairlite.demon.co.uk>
#   David Dawes     <dawes@tungstengraphics.com>
#
# Copyright (c) 2002 Frank Worsley
# Copyright (c) 2002, 2003 Alan Hourihane
# Copyright (c) 2003 David Dawes
#
#######################################################################
#
# Permission to use, copy, modify, distribute, and sell this software and its
# documentation for any purpose is hereby granted without fee, provided that
# the above copyright notice appear in all copies and that both that
# copyright notice and this permission notice appear in supporting
# documentation.
# 
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
# OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
# IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
# OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
# ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.
# 
#
#######################################################################

###########################################################
# VARIABLE DEFINITIONS
###########################################################

LOGFILE="dri.log"
LOGFILE_TMP="tmp.log"

SAVE_PREFIX=""
SAVE_SUFFIX=".dri-old"

VARDB="/var/lib"
DB_DIR="$VARDB/dripkg"
INSTALLATIONS="$DB_DIR/installations"
INSTALLED_FILES="$DB_DIR/files"
PREVIOUS_FILES="$DB_DIR/prevfiles"
UNINSTALL_SCRIPT="$DB_DIR/uninstall"
ORIGINAL_FILES="$INSTALLED_FILES-orig"
CURRENT_FILES="$INSTALLED_FILES-current"

REQUIRED_UTILS="awk ed expr md5sum sort wc"

# Defaults
XF86_DIR="/usr/X11R6"
KERNEL_VER=`uname -r`
KERNEL_LIB_DIR="/lib/modules/$KERNEL_VER"
DIST_NAME="unknown"
OS_NAME=`uname -s`

LOGO_HEADLINE="DIRECT RENDERING OPEN SOURCE PROJECT"
LOGO_URL="http://dri.sourceforge.net"

# Check if the required utilities are available.
MISSING_UTILS=""
for UTIL in $REQUIRED_UTILS; do
	if which $UTIL &> /dev/null; then
		: OK
	else
		MISSING_UTILS="$MISSING_UTILS $UTIL"
	fi
	if [ X"$MISSING_UTILS" != X ]; then
		echo "Could not locate some required utilities:$MISSING_UTILS"
		echo "Aborting"
		exit 127
	fi
done

# TODO: Add options to allow the kernel and XFree86 directories to be specified
# from the command line?

# Parse the command line

batch=0
force=0
verbose=0

while getopts bfv c; do
	case $c in
	b)
		batch=1
		shift
		;;
	f)
		force=1
		shift
		;;
	v)
		verbose=1
		shift
		;;
	'?')
		echo "Unrecognised option: -$c"
		exit 127
		;;
	esac
done

OPTION=$1

# Default to install in batch mode
if [ "$batch" = 1 -a X"$OPTION" = X ]; then
	OPTION=install
fi

###########################################################
# FUNCTION DECLARATIONS
###########################################################

msg() {
	if [ $batch = 0 -o $verbose = 1 ]; then
		echo "$@"
	fi
}

errmsg() {
	echo "$@"
}

print_logo() {

	################################################################
	# This function prints a DRI logo and a general text header.
	################################################################

	if [ "$batch" = 0 ]; then
		clear
		msg ""
		msg "$LOGO_HEADLINE  -  DRIVER INSTALLATION SCRIPT"
		msg ""
		msg "[ $LOGO_URL ]"
		msg ""
		msg "=========================================================================="
	fi
	msg ""
}

set_values() {

	#####################################################################
	# Sets up default values.
	#####################################################################

	# Determine driver to be installed
	DRV_NAME=`echo "1 p" | ed -s pkginfo`
	DRV_DESC=`echo "2 p" | ed -s pkginfo`
	DRV_ARCH=`echo "3 p" | ed -s pkginfo`
	DRV_DATE=`echo "4 p" | ed -s pkginfo`
	DRV_MODULE=`echo "5 p" | ed -s pkginfo`
	DRV_VERSION=`echo "6 p" | ed -s pkginfo`
	DRV_BUILD_DESC=`echo "7 p" | ed -s pkginfo`
	if [ -f installinfo ]; then
		INSTALL_STAMP=`echo "1 p" | ed -s installinfo`
	fi
	
	# Determine directories from default or user values
	XF86_MOD_DIR="$XF86_DIR/lib/modules"
	XF86_DRI_DIR="$XF86_MOD_DIR/dri"
	XF86_DRV_DIR="$XF86_MOD_DIR/drivers"
	XF86_GL_DIR="$XF86_DIR/lib"
	XF86_EXT_DIR="$XF86_MOD_DIR/extensions"
	XF86_OS_DIR="$XF86_MOD_DIR/linux"
	KERNEL_DRM_DIR="$KERNEL_LIB_DIR/kernel/drivers/char/drm"
	KERNEL_AGPGART_DIR="$KERNEL_LIB_DIR/kernel/drivers/char/agp"
	INSTALL_AGPGART="0"
	if [ -d agpgart ]; then
		INSTALL_AGPGART="1"
	fi
	USE_EXISTING_KERNEL_MODULE="0"
}

check_values() {

	#####################################################################
	# Checks if default/user given values are correct.
	#####################################################################
	
	# Check if installation directories exist
	MSG=$'The following problems have occured:\n'
	FATAL=0
	ERR=0
	
	if [ ! -d "$XF86_DIR" ]; then
		MSG="$MSG"$'\n	- XFree86 directory does not exist'
		ERR="1"
		FATAL="1"
	fi
	
	if [ ! -d "$KERNEL_LIB_DIR" ]; then
		MSG="$MSG"$'\n	- Kernel module directory does not exist'
		ERR="1"
		FATAL="1"
	fi

	if [ ! -d "$KERNEL_DRM_DIR" ]; then
		MSG="$MSG"$'\n	- Kernel DRM directory does not exist'
		ERR="1"
	fi
	
	if [ ! -d "$KERNEL_AGPGART_DIR" ]; then
		MSG="$MSG"$'\n	- Kernel AGPGART directory does not exist'
		ERR="1"
	fi
	
	if [ ! -d "$XF86_GL_DIR" ]; then
		MSG="$MSG"$'\n	- XFree86 GL library directory does not exist'
		ERR="1"
	fi	
	
	if [ ! -d "$XF86_MOD_DIR" ]; then
		MSG="$MSG"$'\n	- XFree86 modules directory does not exist'
		ERR="1"
	fi		

	if [ ! -d "$XF86_DRI_DIR" ]; then
		MSG="$MSG"$'\n	- XFree86 DRI directory does not exist'
		ERR="1"
	fi		

	if [ ! -d "$XF86_DRV_DIR" ]; then
		MSG="$MSG"$'\n	- XFree86 driver directory does not exist'
		ERR="1"
	fi
	
	if [ ! -d "$XF86_EXT_DIR" ]; then
		MSG="$MSG"$'\n	- XFree86 extensions directory does not exist'
		ERR="1"
	fi	
	
	if [ ! -d "$XF86_OS_DIR" ]; then
		MSG="$MSG"$'\n	- XFree86 Linux OS directory does not exist'
		ERR="1"
	fi	
	
	# No error so return
	if [ "$ERR" == "0" ]; then
		return 0
	fi
	
	# Get out if we have a fatal error that requires user to re-enter values
	if [ "$FATAL" == "1" -a "$batch" = 0 ]; then
		MSG="$MSG"$'\n\nPlease make sure you have entered all information correctly.\n\nPress ENTER to review the information.'
		print_logo
		errmsg "$MSG"
		read KEY
		return -1
	fi
	
	if [ "$batch" = 0 ]; then
		#Fix what we can
		MSG="$MSG"$'\n\nThe script can create these directories for you.\n\nPress ENTER to continue or CTRL-C to abort.'
		print_logo
		msg "$MSG"
		read KEY
	fi
	
	# Now we just create all directories no matter what ...
	mkdir -p $XF86_GL_DIR
	mkdir -p $XF86_MOD_DIR
	mkdir -p $XF86_DRI_DIR
	mkdir -p $XF86_DRV_DIR
	mkdir -p $XF86_EXT_DIR
	mkdir -p $XF86_OS_DIR
	mkdir -p $KERNEL_DRM_DIR
	mkdir -p $KERNEL_AGPGART_DIR
	
	return 0
}

get_osname() {
	if [ -f /etc/redhat-release ]; then
		DIST_NAME="redhat"
		VER=`awk '{ print $5 }' < /etc/redhat-release`
		DIST_NAME="$DIST_NAME-$VER"
		return 0
	fi

	if [ -f /etc/SuSE-release ]; then
		DIST_NAME="suse"
		VER=`grep VERSION /etc/SuSE-release | awk '{ print $3 }'`
		DIST_NAME="$DIST_NAME-$VER"
		return 0
	fi
}

add_stamp() {
	if [ ! -f $INSTALLATIONS ]; then
		touch $INSTALLATIONS
	fi
	if grep -q $INSTALL_STAMP $INSTALLATIONS; then
		:
	else
		echo "$INSTALL_STAMP" >> $INSTALLATIONS
	fi
}

install_file() {
	srcfile=`basename $1`
	target="$2/$srcfile"
	save="$2/$SAVE_PREFIX$srcfile$SUFFIX"
	orig="$2/$SAVE_PREFIX$srcfile$SAVE_SUFFIX"
	remove="$2/$SAVE_PREFIX$srcfile$SAVE_SUFFIX:remove"
	add_stamp
	if [ -e $target ]; then
		if [ ! -e $orig -a ! -e $remove ]; then
			if [ -f $ORIGINAL_FILES ]; then
				rm -f $ORIGINAL_FILES.tmp
				grep -v $target $ORIGINAL_FILES > \
					$ORIGINAL_FILES.tmp
				mv -f $ORIGINAL_FILES.tmp $ORIGINAL_FILES
			fi
			md5sum $target >> $ORIGINAL_FILES
		fi
		md5sum $target >> $PREVIOUS_FILES
		rm -f $save
		mv $target $save >& $LOGFILE_TMP
		echo "test -e $save && mv -f $save $target" >> \
			$UNINSTALL_SCRIPT
		if [ ! -e $orig -a ! -e $remove ]; then
			ln $save $orig
			echo "rm -f $orig" >> $UNINSTALL_SCRIPT
		fi
	else
		rm -f $save
		echo "rm -f $target" >> $UNINSTALL_SCRIPT
		if [ ! -e $orig -a ! -e $remove ]; then
			touch $remove
			echo "rm -f $remove" >> $UNINSTALL_SCRIPT
			if [ -f $ORIGINAL_FILES ]; then
				rm -f $ORIGINAL_FILES.tmp
				grep -v $target $ORIGINAL_FILES > \
					$ORIGINAL_FILES.tmp
				mv -f $ORIGINAL_FILES.tmp $ORIGINAL_FILES
			fi
			echo "remove $target" >> $ORIGINAL_FILES
		fi
		echo "remove $target" >> $PREVIOUS_FILES
	fi

	if [ -e $target ]; then
		echo "Failed to save file $target"
		return 1
	fi

	if cp $1 $target; then
		md5sum $target >> $INSTALLED_FILES
		if [ -f $CURRENT_FILES ]; then
			rm -f $CURRENT_FILES.tmp
			grep -v $target $CURRENT_FILES > $CURRENT_FILES.tmp
			mv -f $CURRENT_FILES.tmp $CURRENT_FILES
		fi
		md5sum $target >> $CURRENT_FILES
	else
		echo "Failed to install file $target"
		return 1
	fi

	return 0
}

copy_files() {

	#####################################################################
	# This function copies files to their destination directories.
	#####################################################################
	
	msg "Installing files:"
	
	INSTALL_STAMP=`date "+%Y%m%d%H%M%S"`
	SUFFIX="$SAVE_SUFFIX-$INSTALL_STAMP"
	INSTALLED_FILES="$INSTALLED_FILES-$INSTALL_STAMP"
	UNINSTALL_SCRIPT="$UNINSTALL_SCRIPT-$INSTALL_STAMP"
	PREVIOUS_FILES="$PREVIOUS_FILES-$INSTALL_STAMP"

	msg -n "	DRI XFree86 modules..."
	cd $DRV_NAME
	for FILE in *drv.o
	do
		if install_file $FILE $XF86_DRV_DIR; then
			: OK
		else
			INSTALL_FAILED=1
		fi
	done
	for FILE in *dri.so
	do
		if install_file $FILE $XF86_DRI_DIR; then
			: OK
		else
			INSTALL_FAILED=1
		fi
	done	
	cd ..
	msg "done"
	
	msg -n "	kernel modules..."
	if [ "$USE_EXISTING_KERNEL_MODULE" == "1" ]; then
	   msg "using existing module"
	else
		mkdir -p $KERNEL_DRM_DIR
		if install_file drm/$DRV_MODULE.o $KERNEL_DRM_DIR; then
			: OK
		else
			INSTALL_FAILED=1
		fi
		if [ "$INSTALL_AGPGART" == "1" ]; then
			mkdir -p $KERNEL_AGPGART_DIR
			if install_file agpgart/agpgart.o $KERNEL_AGPGART_DIR;
			then
				: OK
			else
				INSTALL_FAILED=1
			fi
		fi
		msg "done"
	fi
	
	if [ -d GL ]; then
		msg -n "	GL libraries..."
		for FILE in GL/*; do
			if [ -e $FILE ]; then
				if install_file $FILE $XF86_GL_DIR; then
					: OK
				else
					INSTALL_FAILED=1
				fi
			fi
		done
		msg "done"
	fi
			
	if [ -d core/extensions ]; then
		msg -n "	extension modules..."
		for FILE in core/extensions/*; do
			if [ -e $FILE ]; then
				if install_file $FILE $XF86_EXT_DIR; then
					: OK
				else
					INSTALL_FAILED=1
				fi
			fi
		done
		msg "done"
	fi

	if [ -d core/os ]; then
		msg -n "	OS-specific server modules..."
		for FILE in core/os/*; do
			if [ -e $FILE ]; then
				if install_file $FILE $XF86_OS_DIR; then
					: OK
				else
					INSTALL_FAILED=1
				fi
			fi
		done
		msg "done"
	fi

	if [ -d core/misc ]; then
		msg -n "	Misc server modules..."
		for FILE in core/misc/*; do
			if [ -e $FILE ]; then
				if install_file $FILE $XF86_MOD_DIR; then
					: OK
				else
					INSTALL_FAILED=1
				fi
			fi
		done
		msg "done"
	fi

	if [ "$INSTALL_FAILED" = 1 ]; then
		errmsg "The installation of one or more files failed"
	else
		msg "done"
	fi

	if [ -e extras/extras.sh ]; then
		msg -n "	Executing extras script..."
		extras/extras.sh $XF86_DIR
	fi
}

restore_files () {

	# Restoring to the original state is handled a little differently
	# from the other cases in the interests of maximising the chance
	# of getting back to that state.

	if [ $1 = orig ]; then
		msg "Restoring original files..."
		FILES="`awk '{ print $2 }' < $ORIGINAL_FILES`"
		for FILE in $FILES; do
			dir=`dirname $FILE`
			base=`basename $FILE`
			save=$dir/$SAVE_PREFIX$base$SAVE_SUFFIX
			if [ -e $save ]; then
				#msg "Restoring $FILE"
				mv -f $save $FILE
			else
				if [ -e $save-remove ]; then
					#msg "Removing $FILE"
					rm -f $FILE
				fi
			fi
			#msg Removing $save-*
			rm -f $save-*
		done
		rm -f $CURRENT_FILES
		rm -f $ORIGINAL_FILES
		rm -f $PREVIOUS_FILES*
		rm -f $INSTALLED_FILES*
		rm -f $INSTALLATIONS
		rm -f $UNINSTALL_SCRIPT*
	else
		msg "Uninstalling installation $1..."
		if [ -f $UNINSTALL_SCRIPT-$1 ]; then
			$SHELL $UNINSTALL_SCRIPT-$1
			rm -f $PREVIOUS_FILES-$1
			rm -f $UNINSTALL_SCRIPT-$1
			FILES="`awk '{ print $2 }' < $INSTALLED_FILES-$1`"
			for FILE in $FILES; do
				dir=`dirname $FILE`
				base=`basename $FILE`
				orig=$dir/$SAVE_PREFIX$base$SAVE_SUFFIX
				if [ -f $CURRENT_FILES ]; then
					rm -f $CURRENT_FILES.tmp
					grep -v $FILE $CURRENT_FILES > \
						$CURRENT_FILES.tmp
					mv -f $CURRENT_FILES.tmp $CURRENT_FILES
				fi
				if [ -e $FILE ]; then
					md5sum $FILE >> $CURRENT_FILES
				fi
				if [ ! -e $orig -a ! -e $orig-remove ]; then
					rm -f $ORIGINAL_FILES.tmp
					grep -v $orig $ORIGINAL_FILES > \
						$ORIGINAL_FILES.tmp
					mv -f $ORIGINAL_FILES.tmp \
						$ORIGINAL_FILES
				fi
			done
			rm -f $INSTALLED_FILES-$1
			if [ -f $INSTALLATIONS ]; then
				rm -f $INSTALLATIONS.tmp
				grep -v $1 $INSTALLATIONS > $INSTALLATIONS.tmp
				mv -f $INSTALLATIONS.tmp $INSTALLATIONS
			fi
		else
			errmsg "Cannot find the uninstall script $UNINSTALL_SCRIPT-$1"
			return 1
		fi
	fi
}

update_config() {

	#####################################################################
	# This function updates the system configuration.
	#####################################################################
	
	msg ""
	msg "Updating configuration:"

	#echo -n "	Running ldconfig..."
	#grep "$XF86_DIR" /etc/ld.so.conf >& $LOGFILE_TMP
	#if [ $? != 0 ]; then
	#	echo "$XF86_DIR/lib" >> /etc/ld.so.conf
	#fi
	#/sbin/ldconfig >& $LOGFILE_TMP
	#if [ $? != 0 ]; then
	#	echo "ERROR"
	#	ERR=1
	#else
	#	echo "done"
	#fi
	#cat $LOGFILE_TMP >> $LOGFILE

	if [ ! "$X_RUNNING" == "1" ]; then
	
		# Only try and update kernel modules if X server is not running

		for MOD in $DRV_MODULE
		do
			msg -n "	Removing old kernel module \"$MOD\"..."
			/sbin/modprobe -r $MOD >& $LOGFILE_TMP
			if [ $? != 0 ]; then
				msg "ERROR"
				ERR=1
			else
				msg "done"
			fi
	
			msg -n "	Removing old kernel module \"agpgart\"..."
			/sbin/modprobe -r agpgart >& $LOGFILE_TMP

			if [ $? != 0 ]; then
				msg "ERROR"
				ERR=1
			else
				msg "done"
			fi
	
			cat $LOGFILE_TMP >> $LOGFILE
		
			msg -n "	Inserting new kernel module \"agpgart\"..."
			/sbin/modprobe agpgart >& $LOGFILE_TMP
			if [ $? != 0 ]; then
				msg "ERROR"
				ERR=1
			else
				msg "done"
			fi
	
			msg -n "	Inserting new kernel module \"$MOD\"..."
			/sbin/modprobe $MOD >& $LOGFILE_TMP
		
			if [ $? != 0 ]; then
				msg "ERROR"
				ERR=1
			else
				msg "done"
			fi
	
			cat $LOGFILE_TMP >> $LOGFILE
		done
	fi
	
	if [ "$ERR" == "1" ]; then
		errmsg ""
		errmsg "There were errors updating the system configuration."
		errmsg "See the $LOGFILE file for more information."
	fi

	rm -f $LOGFILE_TMP
}

check_config() {

	#####################################################################
	# This function checks the system configuration.
	#####################################################################
	
	msg ""
	msg -n "Checking configuration..."
	
	# Fix potential libGL problems
	#LIBGL1=`ldd $XF86_DIR/bin/glxinfo | grep libGL.so.1 | awk -F" " '{ printf "%s",$3 }'`
	#LIBGL=`echo $LIBGL1 | sed -e 's/\.1//'`
	#if [ "$LIBGL" != "$XF86_GL_DIR/libGL.so" ]; then
	#	echo ""
	#	echo -n "	second copy of DRI libraries found in "
	#	echo `echo $LIBGL1 | sed -e 's/libGL.so.1//'`
	#	
	#	echo -n "	libraries have been backed up to old.* in "
	#	echo `echo $LIBGL1 | sed -e 's/libGL.so.1//'`
	#	echo ""
	#	
	#	mv $LIBGL1 `echo $LIBGL1 | sed -e 's/libGL.so.1/old.libGL.so.1/'`
	#	mv $LIBGL  `echo $LIBGL | sed -e 's/libGL.so/old.libGL.so/'`
	#	ln -s $XF86_GL_DIR/libGL.so.1 $LIBGL1
	#	ln -s $XF86_GL_DIR/libGL.so $LIBGL
	#fi

	# Make sure libGL and libGLU have correct links
	#rm -f $XF86_GL_DIR/libGL.so
	#rm -f $XF86_GL_DIR/libGL.so.1
	#ln -s $XF86_GL_DIR/libGL.so.1.2 $XF86_GL_DIR/libGL.so
	#ln -s $XF86_GL_DIR/libGL.so.1.2 $XF86_GL_DIR/libGL.so.1

	#rm -f $XF86_GL_DIR/libGLU.so
	#rm -f $XF86_GL_DIR/libGLU.so.1
	#ln -s $XF86_GL_DIR/libGLU.so.1.3 $XF86_GL_DIR/libGLU.so
	#ln -s $XF86_GL_DIR/libGLU.so.1.3 $XF86_GL_DIR/libGLU.so.1
	
	msg "done"
}

verify_files() {

	if [ $1 = orig ]; then
		PREVLIST="$ORIGINAL_FILES"
		SUFFIX="$SAVE_SUFFIX"
		FILELIST="$CURRENT_FILES"
	else
		FILELIST="$INSTALLED_FILES-$1"
		SUFFIX="$SAVE_SUFFIX-$1"
		PREVLIST="$PREVIOUS_FILES-$1"
	fi

	# First, verify that the saved copies match
	FILES="`awk '{ print $2 }' < $PREVLIST`"
	NOEXIST=""
	NOMATCH=""
	for FILE in $FILES; do
		SAVESUM="`grep $FILE $PREVLIST | awk '{ print $1 }'`"
		dir=`dirname $FILE`
		base=`basename $FILE`
		save=$dir/$SAVE_PREFIX$base$SUFFIX
		if [ -e $save ]; then
			CURSUM="`md5sum $save | awk '{ print $1 }'`"
		else
			CURSUM="remove"
		fi
		if [ X"$CURSUM" != X"$SAVESUM" ]; then
			if [ X"$CURSUM" = Xremove ]; then
				NOEXIST="$NOEXIST $save"
			else
				NOMATCH="$NOMATCH $save"
			fi
		fi
	done
	if [ X"$NOMATCH" != X ]; then
		errmsg ""
		errmsg "The following saved files don't match the recorded md5 signatures:"
		for f in $NOMATCH; do
			errmsg "	$f"
		done
	fi
	if [ X"$NOEXIST" != X ]; then
		errmsg ""
		errmsg "The following saved files are missing:"
		for f in $NOEXIST; do
			errmsg "	$f"
		done
	fi
	msg ""
	if [ X"$NOMATCH$NOEXIST" != X ]; then
		return 2
	fi

	# Now verify that the installed file matches
	NOMATCH=""
	for FILE in $FILES; do
		SAVESUM="`grep $FILE $FILELIST | awk '{ print $1 }'`"
		CURSUM="`md5sum $FILE | awk '{ print $1 }'`"
		if [ X"$CURSUM" != X"$SAVESUM" ]; then
			NOMATCH="$NOMATCH $FILE"
		fi
	done
	if [ X"$NOMATCH" != X ]; then
		errmsg ""
		errmsg "The following saved files don't match the recorded md5 signatures:"
		for f in $NOMATCH; do
			errmsg "	$f"
		done
		errmsg ""
		return 1
	fi
	return 0
}


do_uninstall() {
	print_logo

	if [ ! -f $INSTALLATIONS ]; then
		errmsg "No previous installations were found."
		exit 0
	fi
		
	INST_LIST="`sort $INSTALLATIONS`"
	if [ X"$INST_LIST" = X ]; then
		errmsg "No previous installations were found."
		exit 0
	fi

	msg "The following installation states have been found:"
	msg ""
	i=0
	msg -n " 0.	Original files"
	for INST in $INST_LIST; do
		i=`expr $i + 1`
		msg ""
		msg -n " $i.	$INST"
	done
	msg "	(current state)"
	msg ""
	msg "This script can attempt to revert to any of the previous"
	msg "states listed.  In doing so, it will remove all versions"
	msg "installed after the specified state.  Selecting 0 will remove"
	msg "all previous installations.  Selecting $i will do nothing since"
	msg "that is the current state.  The default is `expr $i - 1`, which"
	msg "will back out the most recent installation."
	msg ""
	msg -n "Enter the state you wish to revert to [0-$i]: "

	KEY=""
	while [ X$KEY = X ]; do
		if [ "$batch" = 0 ]; then
			read KEY
		fi
		case "$KEY" in
		"")
			KEY=`expr $i - 1`
			;;
		[0-9]|[0-9][0-9]|[0-9][0-9][0-9])
			if [ $KEY -lt 0 -o $KEY -gt $i ]; then
				KEY=""
			fi
			;;
		*)
			KEY=""
			;;
		esac
		if [ X$KEY = X ]; then
			msg "Bad response."
			msg -n "Please enter a number in the range [0-$i]: "
		fi
	done

	case "$KEY" in
	0)
		msg "All previous installations will be removed."
		REMOVE_ALL=1
		;;
	$i)
		msg "The current state was selected.  Nothing to do."
		exit 0
		;;
	*)
		j=0
		REMOVE_LIST=""
		msg ""
		msg "The following installations will be removed:"
		for INST in $INST_LIST; do
			j=`expr $j + 1`
			if [ $j -gt $KEY ]; then
				REMOVE_LIST="$REMOVE_LIST $INST"
				msg "	$INST"
			fi
		done
	esac

	if [ "$batch" = 0 ]; then
		msg ""
		msg "Press ENTER to continue or CTRL-C to exit"
		read KEY
	fi
	
	set_values

	if [ "$REMOVE_ALL" = 1 ]; then
		verify_files orig
		case $? in
		0)
			if restore_files orig; then
				msg "Successfully restored original state."
			fi
			;;
		1)
			msg "One of more of the files doesn't match the most recently installed"
			msg "version.  This usually means that some other mechanism has installed"
			msg "a new version since this installation script was last used."

			if [ "$batch" = 1 -a "$force" = 0 ]; then
				errmsg "Aborting"
				exit 1
			fi

			msg "It is possible to continue with this uninstallation process, but"
			msg "be aware that it may interfere with some other packaging mechanism."
			if [ "$batch" = 0 ]; then
				msg ""
				msg "Press ENTER to continue or CTRL-C to exit"
				read KEY
			fi
			if restore_files orig; then
				msg "Successfully restored original state."
			fi
			;;
		*)
			errmsg "One or more of the files required to restore the original version"
			errmsg "is either missing or doesn't match the recorded md5 signatures."
			errmsg ""
			if [ "$force" = 0 ]; then
				errmsg "Aborting"
				exit 1
			fi
			if restore_files orig; then
				msg "Successfully restored original state."
			fi
			;;
		esac
	else
		n=`echo $REMOVE_LIST | wc -w | sed 's/ *//'`
		while [ $n -ne 0 ]; do
			REMOVEID=`echo $REMOVE_LIST | awk '{ print $'$n' }'`
			msg "Removing $REMOVEID..."
			verify_files $REMOVEID
			case $? in
			0)
				if restore_files $REMOVEID; then
					msg "Successfully uninstalled the $REMOVEID installation"
				else
					errmsg "Uninstalling the $REMOVEID installation failed."
					msg "Either try to revert to a later installation, or back to the original version."
					exit 1
				fi
				;;
			1)
				errmsg "One or more of the files doesn't match the installed version about to be"
				errmsg "removed.  This usually means that some other mechanism has installed"
				errmsg "a new version since this then."

				if [ "$batch" = 1 -a "$force" = 0 ]; then
					errmsg "Aborting"
					exit 1
				fi

				msg "It is possible to continue with this uninstallation process, but"
				msg "be aware that it may interfere with some other packaging mechanism."
				if [ "$batch" = 0 ]; then
					msg ""
					msg "Press ENTER to continue or CTRL-C to exit"
					read KEY
				fi
				if restore_files $REMOVEID; then
					msg "Successfully uninstalled the $REMOVEID installation."
				else
					errmsg "Uninstalling the $REMOVEID installation failed."
					msg "Either try to revert to a later installation, or back to the original version."
					exit 1
				fi
				;;
			*)
				errmsg "One or more of the files required to uninstall the $REMOVEID installation."
				errmsg "is either missing or doesn't match the recorded signatures."
				errmsg ""
				errmsg "Uninstalling the $REMOVEID installation failed."
				msg "Either try to revert to a later installation, or back to the original version."
				msg ""
				errmsg "Aborting"
				exit 1
				;;
			esac
			n=`expr $n - 1`
		done
	fi
	msg ""
}

###########################################################
# MAIN SCRIPT
###########################################################

# Check if pkginfo file exists
if [ ! -e "pkginfo" ]; then
	errmsg "Could not locate 'pkginfo' file. Aborting."
	exit 127
fi

# Check if we are running as root
if [ `id -u` != "0" ]; then
	errmsg "You must be root to install the DRI drivers."
	exit 127
fi

if [ ! -d $DB_DIR ]; then
	mkdir -p $DB_DIR
	if [ ! -d $DB_DIR ]; then
		errmsg "Cannot create directory $DB_DIR"
		exit 127
	fi
fi

### FIXME: We should check for matching architectures here!!! ###


# Print X is running message
ps -C X > /dev/null
if [ $? == 0 ]; then
	X_RUNNING=1
	print_logo
	errmsg "WARNING: YOUR X SERVER SEEMS TO BE RUNNING!"
	errmsg ""
	errmsg "The script can not update your kernel modules while the X server is running."
	errmsg ""
	errmsg "You can do one of the following:"
	errmsg ""
	errmsg "1. Log out of your X session now and then run this script."
	errmsg ""
	errmsg "2. Run the script now and restart your computer after the installation."
	errmsg "   If you exit your X session now you will not have to restart."

	if [ "$batch" = 1 -a "$force" = 0 ]; then
		errmsg "Aborting"
		exit 1
	fi

	if [ "$batch" = 0 ]; then
		msg ""
		msg "Press ENTER to continue or CTRL-C to exit."
		read KEY
	fi
else
	X_RUNNING=0
fi

# Do misc stuff
rm -f $LOGFILE
touch $LOGFILE

# Setup the defaults values
set_values

# Print a welcome message
print_logo
msg "Welcome to the DRI Driver Installation Script"
msg ""
msg "The package you downloaded is for the following driver: "
msg ""
msg "Driver Name    : $DRV_NAME"
msg "Description    : $DRV_DESC"
msg "Architecture   : $DRV_ARCH"
msg "Build Date     : $DRV_DATE"
msg "Kernel Module  : $DRV_MODULE"
msg ""
msg "Optional Information"
msg ""
msg "Driver Version      : $DRV_VERSION"
msg "Special Description : $DRV_BUILD_DESC"
msg ""
if [ "$batch" = 0 ]; then
	msg "Press ENTER to continue or CTRL-C to exit."
	read KEY
fi

case "$OPTION" in
restore|uninstall|remove|install|"")
	;;
*)
	if [ "$batch" = 1 ]; then
		errmsg "Unrecognised command line option '$OPTION' (aborting)"
		exit 1
	fi
	print_logo
	msg "Unrecognised command line option '$OPTION' (ignoring)"
	msg ""
	msg "Press ENTER to continue or CTRL-C to exit."
	read KEY
	OPTION=""
	;;
esac

while [ X"$OPTION" = X ]; do
	print_logo
	msg "Select from the following options:"
	msg ""
	msg "  1. Install driver (default)"
	msg "  2. Uninstall driver"
	msg ""
	msg "Enter reponse [1-2], or CTRL-C to exit"
	read KEY
	case $KEY in
	2)
		OPTION="restore"
		;;
	*)
		OPTION="install"
		;;
	esac
done

case $OPTION in
restore|uninstall|remove)
	do_uninstall
	exit 0
	;;
esac

# Determine XFree86 Directory and Kernel Module directory from user input
if [ "$batch" = 1 ]; then
	set_values
	check_values
	ERR=0
else
	ERR=-1
fi

while [ "$ERR" == "-1" ]; do

	# Determine XFree86 directory
	print_logo
	msg "The script will need to copy the DRI XFree86 driver modules to"
	msg "your XFree86 directory."
	msg ""
	msg "The script will use the following XFree86 directory:"
	msg ""
	msg " $XF86_DIR"
	msg ""
	msg "If this is correct press ENTER, press C to change or CTRL-C to exit."
	read KEY

	if [ "$KEY" == "C" ] || [ "$KEY" == "c" ]; then
		print_logo
		msg "Please enter the XFree86 directory you would like to use."
		msg ""
		msg "Press ENTER when done or CTRL-C to abort."
		msg ""
		msg -n "Directory: "
		read XF86_DIR
	fi

	# Determine Kernel module directory
	print_logo
	msg "The script also needs to copy the DRM kernel modules to your"
	msg "kernel module directory."
	msg ""
	msg "Please note that only 2.4.x kernels are supported right now."
	msg ""
	msg "Kernel Version   : $KERNEL_VER"
	msg "Module Directory : $KERNEL_LIB_DIR"
	msg ""
	msg "If this is correct press ENTER, press C to change or CTRL-C to exit."
	read KEY

	if [ "$KEY" == "C" ] || [ "$KEY" == "c" ]; then
		print_logo
		msg "Please enter your kernel module directory."
		msg ""
		msg "Press ENTER when done or CTRL-C to abort."
		msg ""
		msg -n "Directory: "
		read KERNEL_LIB_DIR
	fi

	# Determine paths from user input or defaults
	set_values
	
	# Check if values are good
	check_values
	ERR="$?"
done

if [ X"$INSTALL_STAMP" != X ]; then
	msg "This package is already installed"
	exit 0
fi

get_osname
if [ -d kernel-modules/$DIST_NAME-$KERNEL_VER ]; then
	PRECOMP_MODULES=kernel-modules/$DIST_NAME-$KERNEL_VER
	msg "There are pre-compiled modules for your kernel"
	if [ "$batch" = 0 ]; then
		msg ""
		msg "Press ENTER to continue or CTRL-C to exit."
		read KEY
	fi
	if [ "$INSTALL_AGPGART" == "1" ]; then
		cp -p -f $PRECOMP_MODULES/agpgart.o agpgart/agpgart.o
	fi
	cp -p -f $PRECOMP_MODULES/$DRV_MODULE.o drm/$DRV_MODULE.o
else
	# Compile the kernel modules
	print_logo
	msg -n "The script will now compile the "
	if [ "$INSTALL_AGPGART" == "1" ]; then
		msg -n "agpgart module and "
	fi
	msg "DRM kernel modules"
	msg "for your machine."
	if [ "$batch" = 0 ]; then
		msg ""
		msg "Press ENTER to continue or CTRL-C to exit."
		read KEY
	fi

	msg ""
	if [ "$INSTALL_AGPGART" == "1" ]; then
		msg -n "Compiling new agpgart module..."
		cd agpgart
		make clean agpgart.o >& ../$LOGFILE_TMP
		if [ $? != 0 ]; then
			cat ../$LOGFILE_TMP >> ../$LOGFILE
			errmsg ""
			errmsg ""
			errmsg "ERROR: AGPGART module did not compile"
		fi
		cd ..

	fi
	msg ""
	msg -n "Compiling DRM module..."
	cd drm
	make -f Makefile.linux clean $DRV_MODULE.o >& ../$LOGFILE_TMP
	if [ $? != 0 ]; then
		cat ../$LOGFILE_TMP >> ../$LOGFILE
		errmsg ""
		errmsg ""
		errmsg "ERROR: Kernel modules did not compile"

        	# Check to see if any older DRM module already exists
		if [ -e $KERNEL_DRM_DIR/$DRV_MODULE.o ]; then
			msg ""
			msg "WARNING: The existing DRM kernel module available on your system might"
			msg "work; however, using an older module may severly limit functionality"
			msg "or performance!!!  It is recommended that you upgrade to the latest"
			msg "DRM kernel module.  For further details see:"
			msg ""
			msg "	http://dri.sourceforge.net/doc/DRIcompile.html#s4"
			if [ "$batch" = 0 ]; then
				msg ""
				msg "Press ENTER to continue with the existing kernel module or CTRL-C to exit."
				read KEY
			fi
			USE_EXISTING_KERNEL_MODULE="1"
        	else
			cd ..
			errmsg ""
			errmsg "The DRI drivers can not be installed without the latest kernel modules."
			errmsg "Installation will be aborted. See the $LOGFILE file for information on"
			errmsg "what went wrong."
			errmsg ""
			exit 127
        	fi
	else
		msg "done"
	fi
	cat ../$LOGFILE_TMP >> ../$LOGFILE
	cd ..
fi

# Print a last message
print_logo
msg "The script is now ready to complete the installation."
if [ "$batch" = 0 ]; then
	msg ""
	msg "Press ENTER to continue or CTRL-C to exit."
	read KEY
fi

# Copy files, update config, check config and fix problems
print_logo

copy_files
update_config
check_config

if [ "$batch" = 0 ]; then
	msg ""
	msg "Press ENTER to continue."
	read KEY
fi

# Print last message
print_logo
msg "The DRI installation is complete."
msg ""
msg "Restart your X server to try the new DRI drivers."
msg ""
msg "If you have problems with the DRI after upgrading your drivers"
msg "please visit the DRI website and read the Help and FAQ section."
msg "The FAQ contains solutions to many common problems."
msg ""
msg "Report any bugs, problems and comments on the dri-devel mailing list."
msg ""
msg "Thank you for using the DRI."
msg ""
