#include "windows.h"
#include "brigade.h"
#include "splash.h"
#include "window.h"
#include "Graphics.h"
#include "bitmap.h"
#include "system.h"
using namespace B2;
int2 ScreenResolution( 1024, 640 );
int2 RenderResolution( ScreenResolution );
int samplecount = 4;

PathTracer* pathtracer;
Scene* scene;

bool Loading = true;
bool ShowHUD = false;

void SaveCam( int _Idx );
void LoadCam( int _Idx );
float dtime;

Timer Clock;
void LoadScene();
void Update();
class LoaderThread : public Thread
{
	void run()
	{
		// init path tracer
		pathtracer = new PathTracer( RenderResolution, samplecount );
		pathtracer->SetLoadBalancer(PathTracer::RobinHood);
		// init scene
		scene = new Scene();
		pathtracer->SetScene( scene );
		
		LoadScene();
	
		scene->ForceUpdate(); // forcefully update the scenegraph, so the first game-frame doesnt take longer than usual ( it will still be a bit longer due to GPU transfers etc. )
		Loading = false;
	}
}
loader;

void SaveCam( int _Idx )
{
	char fname[128];
	sprintf( fname, "%i.cam", _Idx );
	FILE* file = fopen( fname, "wb" );
	if (!file) return;
	float4x4 view = scene->GetCamera()->GetView();
	fwrite( &view, sizeof( float4x4 ), 1, file );
	fclose( file );
}

void LoadCam( int _Idx )
{
	char fname[128];
	sprintf( fname, "%i.cam", _Idx );
	FILE* file = fopen( fname, "rb" );
	if (!file) return;
	float4x4 view;
	fread( &view, sizeof( float4x4 ), 1, file );
	scene->GetCamera()->SetView(view);
	fclose( file );
}

void OutlinerNodeRecurse( Bitmap& _dst, int indent, int& line,Node* _Node, float3 _center )
{
	float3 center = _Node->GetWorldTransform().W.XYZ;
	int2 pos( 4 + indent * 8, 4 + line * 8);
	_dst.Print( pos, byte4( 255, 255, 255, 255 ), "%s", _Node->name.data, center.X, center.Y, center.Z );
	float3 from = pathtracer->GetPixelCoord( _center );
	float3 to   = pathtracer->GetPixelCoord( center );
	if (from.Z > 0 && to.Z > 0)
	{
		int2 f( (int)from.X, (int)from.Y ); f *= ScreenResolution; f /= RenderResolution;
		int2 t( (int)to  .X, (int)to  .Y ); t *= ScreenResolution; t /= RenderResolution;
		_dst.Line( f, t, byte4( 255, 255, 255, 255 ) );
	}
	int hl = MAX( 0, pos.X - 4 );
	line++;
	int vf = pos.Y;
	for( LNode<Object*>* n = _Node->GetObjects().GetFirst(); n; n = n->GetNext() )
	{
		Object* obj = n->data;
		int c = obj->type + 1;
		byte4 color(((c >> 0) & 1) * 255, ((c >> 1) & 1) * 255, ((c >> 2) & 1) * 255, 255);
		//if(obj->name.length==0)
		//	_dst.Print( int2( 4 + indent * 8, 4 + line * 8), color, "%s(%i)", obj->GetMesh()->name.data, obj->GetTriangleCount() );
		//else
			_dst.Print( int2( 4 + indent * 8, 4 + line * 8), color, "%s(%i)", obj->name.data, obj->GetTriangleCount() );
		line++;
	}
	for( LNode<Node*>* n = _Node->GetNodes().GetFirst(); n; n = n->GetNext() )
	{
		OutlinerNodeRecurse( _dst, indent + 1, line, n->data, center );
	}
	int vt = line * 8;
}

bool DefaultKeyboardInput()
{
	bool moved = false;
// get the current transform of the camera
	float4x4 view = scene->GetCamera()->GetView();

	float movespeed = 6;
	float rotspeed = 0.8f;
	if (GetAsyncKeyState( VK_SHIFT )) movespeed /= 4, rotspeed /= 4;
	// keyboard input
	if (GetAsyncKeyState('D'))		view = float4x4::Translate( float3(-movespeed, 0, 0 ) * dtime ) * view, moved = true;
	if (GetAsyncKeyState('W'))		view = float4x4::Translate( float3( 0, 0, movespeed ) * dtime ) * view, moved = true;
	if (GetAsyncKeyState('A'))		view = float4x4::Translate( float3( movespeed, 0, 0 ) * dtime ) * view, moved = true;
	if (GetAsyncKeyState('R'))		view = float4x4::Translate( float3( 0, movespeed, 0 ) * dtime ) * view, moved = true;
	if (GetAsyncKeyState('F'))		view = float4x4::Translate( float3( 0,-movespeed, 0 ) * dtime ) * view, moved = true;
	if (GetAsyncKeyState('S'))		view = float4x4::Translate( float3( 0, 0,-movespeed ) * dtime ) * view, moved = true;

	if (GetAsyncKeyState('Q'))		view = float4x4::RotateZ(-rotspeed * dtime ) * view, moved = true;
	if (GetAsyncKeyState('E'))		view = float4x4::RotateZ( rotspeed * dtime ) * view, moved = true;
	if (GetAsyncKeyState(VK_RIGHT))	view = float4x4::RotateY(-rotspeed * dtime ) * view, moved = true;
	if (GetAsyncKeyState(VK_LEFT))	view = float4x4::RotateY( rotspeed * dtime ) * view, moved = true;
	if (GetAsyncKeyState(VK_DOWN))	view = float4x4::RotateX(-rotspeed * dtime ) * view, moved = true;
	if (GetAsyncKeyState(VK_UP))	view = float4x4::RotateX( rotspeed * dtime ) * view, moved = true;

#if 1 // disable rolling
	view = float4x4::LookAt( view.W.XYZ, (view * float4( 0, 0, 1, 1 )).XYZ, float3( 0, 1, 0 ) );
#endif
	view.OrthoNormalize();

	// assign view to scene
	scene->GetCamera()->SetView( view );

	return moved;
}

int main( int argc, char* argv[] )
{
	//set working directory
	SetCurrentDirectory( System::GetDirectory(argv[0]).data );
	SplashScreen* splashscreen = new SplashScreen();
	Log::Console::SetTitle( "Brigade 2 - Console" );
	
	loader.start();	//start loader thread ( this demo application loads its scene async )
	while (Loading)
	{
		splashscreen->Update();
		Sleep( 20 );
	}
	delete splashscreen;

	WindowedWindow* window = new WindowedWindow( ScreenResolution, "Brigade 2", true );
	int2 screensize = int2( GetSystemMetrics( SM_CXSCREEN ), GetSystemMetrics( SM_CYSCREEN ) );
	window->SetPosition(screensize / 2, true);
	GLStreamingTexture* ScreenBuffer = new GLStreamingTexture( ScreenResolution );
	GLStreamingTexture* RenderBuffer = new GLStreamingTexture( RenderResolution );
	// done loading, give window the 'ingame' size
	window->SetPosition( screensize / 2, true );
	//try to create a gamepad
	LoadCam( 0 );	//load last used cam
	dtime = 0.01f;
	Clock.Start();
	while (true)
	{
		Clock.Stop();
		float time = (float)Clock.ElapsedMs() / 1000.0f;
		Timer timer;
		timer.Start();
		
		Update();

		// render image!
		pathtracer->RenderBegin();
		pathtracer->RenderSynchronize();
		pathtracer->RenderEnd();
		// present image
		window->Begin();
		Bitmap screen( ScreenResolution, (byte4*)ScreenBuffer->Lock() );	//wrap a Bitmap around the ScreenBuffer ( so we can print text etc.. )
		memset( screen.GetBuffer(), 0, ScreenResolution.X * ScreenResolution.Y * sizeof( byte4 ) );
		
		Bitmap finalimage = pathtracer->GetFinalImage();
		if (window->Key['C']) finalimage.Save("screenshot.png");
		memcpy(RenderBuffer->Lock(),finalimage.GetBuffer(),RenderResolution.X*RenderResolution.Y*sizeof(byte4));
		RenderBuffer->Unlock();
float rayssec = (float)pathtracer->GetRaysPerSecond() / 1000000.0f;
		float maxrayssec = (float)pathtracer->GetMaxRaysPerSecond() / 1000000.0f;
		// HUD
		if (ShowHUD)
		{
			int lines = 5;
			//draw dark bar
		#if 1
			for( int Y = 0; Y < MIN( ScreenResolution.Y, lines * 8 + 8 ); Y++ )
				for( int X = 0; X < ScreenResolution.X; X++ )
					screen.GetBuffer()[X + Y * ScreenResolution.X].W = 128;
		#endif
			// print performance stats
			int line = 0;
			double frametime = (double)pathtracer->GetFrameTimeUs() / 1000000.0;
			screen.Print( int2( 4, 4 + line * 8 ), byte4( 255, 255, 255, 255 ), "pathtracer:" ), line++;
			if( maxrayssec != 0 && rayssec != 0 )
				screen.Print( int2( 4, 4 + line * 8 ), byte4( 255, 255, 255, 255 ), "%.3f(%.2f)m/s %.2f%%", maxrayssec, rayssec, (rayssec/maxrayssec)*100.0f ), line++;
			screen.Print( int2( 4, 4 + line * 8 ), byte4( 255, 255, 255, 255 ), "%i x %i spp%i", pathtracer->GetResolution().X, pathtracer->GetResolution().Y, pathtracer->GetSamplesDone()); line++;
			screen.Print( int2( 4, 4 + line * 8 ), byte4( 255, 255, 255, 255 ), "render %ims %.1f FPS", pathtracer->GetFrameTimeUs() / 1000, 1.0f / (float)frametime  ); line++;

			line = 0;
			screen.Print( int2( 200, 4 + line * 8 ), byte4( 255, 255, 255, 255 ), "scene:" ); line++;
			screen.Print( int2( 200, 4 + line * 8 ), byte4( 255, 255, 255, 255 ), "update %ius", scene->GetUpdateTimeUs() ); line++;
			screen.Print( int2( 200, 4 + line * 8 ), byte4( 255, 255, 255, 255 ), "lights %i area %.0f", scene->GetLightCount(), scene->GetTotalarea() ); line++;
			screen.Print( int2( 200, 4 + line * 8 ), byte4( 255, 255, 255, 255 ), "objects %i prims %i", scene->GetObjectCount(), scene->GetTriangleCount() ); line++;
			// more text here...
			line = 0;
			int index = 0;
			int right = ScreenResolution.X - 300;
			for( int i = 0; i < pathtracer->GetTracerCount(); i++ )
			{
				Tracer* tracer = pathtracer->GetTracers()[i];
				screen.Print( int2(right, 4 + line * 8), byte4(255, 255, 255, 255), "%s", tracer->GetName().data); line++;
				for(LNode<Tracer::Worker*>*e = tracer->workers.GetFirst(); e; e=e->GetNext())
				{
					Tracer::Worker*worker = e->data;
					int c = index + 1; index++;
					// create color for tracer
					byte4 color(((c >> 0) & 1) * 255, ((c >> 1) & 1) * 255, ((c >> 2) & 1) * 255, 255);
					
					screen.Line( int2( right + 2, 6 + line * 8), int2( right + 6, 6 + line * 8), byte4(255, 255, 255, 255));
					if(e==tracer->workers.GetFirst())
						screen.Line( int2( right + 2, 2 + line * 8), int2( right + 2, 6 + line * 8), byte4(255, 255, 255, 255));
					else
						screen.Line( int2( right + 2, -2 + line * 8), int2( right + 2, 6 + line * 8), byte4(255, 255, 255, 255));
					
					// print tracer stats
					if (!worker->IsEnabled()) color = byte4( 128, 128, 128, 255 );
					if (worker->IsEnabled())
						screen.Print( int2(right + 8, 4 + line * 8), color, "%s(%iMB)", worker->GetName().data, worker->GetRemoteMemoryUsage() / 1024 / 1024), line++;
					else
						screen.Print( int2(right + 8, 4 + line * 8), color, "%s", worker->GetName().data), line++;
					
					if (!worker->IsEnabled()) continue;
					// draw tracer indicator lines
					int wfirstline = worker->GetFirstLine();
					int wlastline = wfirstline + worker->GetLineCount();
					for( int y = (wfirstline * ScreenResolution.Y) / RenderResolution.Y; y < (wlastline * ScreenResolution.Y) / RenderResolution.Y; y++)
					{
						screen.Set( int2( ScreenResolution.X - 2, y ), color );
						screen.Set( int2( ScreenResolution.X - 3, y ), color );
					}
				}
			}
			// scene Outline
			int l = lines + 1;
			OutlinerNodeRecurse( screen, 0, l, scene->GetRoot(), float3( 0, 0, 0 ) );
		}
		if (window->Key[VK_F1]) ShowHUD = window->Key[VK_CONTROL] ? false : true;
		if (window->Key['P']) //Probe
		{
			float4x4 view = scene->GetCamera()->GetView();
			float3 RO = view.W.XYZ;
			float3 RD = view.Z.XYZ;
			float dist = RAYLENGTH;
			Triangle* tri = 0;
			float3 N;
			scene->Trace( RO, RD, dist, N, tri );
			screen.Print( ScreenResolution / 2 + int2( 2, 2 ), byte4( 255, 255, 255,255 ), "dist:%f", dist );
			if (tri)
			{
				screen.Print( ScreenResolution / 2 + int2( 2, 10 ), byte4( 255, 255, 255, 255 ), "mat:%s", tri->material->name );
			}
			for( int i = -2; i <= 2; i++ )
			{
				if (!i) continue;
				screen.Set( ScreenResolution / 2 + int2( i, 0 ), byte4( 255, 255, 255, 255 ) );
				screen.Set( ScreenResolution / 2 + int2( 0, i ), byte4( 255, 255, 255, 255 ) );
			}
			scene->GetCamera()->SetFocalDistance(dist);
		}
		
		// upload ScreenBuffer to texture
		ScreenBuffer->Unlock();
		// present rendered image
		glViewport( 0, 0, ScreenResolution.X, ScreenResolution.Y );
		glEnable( GL_TEXTURE_2D );
		RenderBuffer->Bind();
		glBegin( GL_QUADS );
		glTexCoord2f( 0, 1 ); glVertex2f( -1, -1 );
		glTexCoord2f( 1, 1 ); glVertex2f(  1, -1 );
		glTexCoord2f( 1, 0 ); glVertex2f(  1,  1 );
		glTexCoord2f( 0, 0 ); glVertex2f( -1,  1 );
		glEnd();
		glEnable( GL_BLEND );
		glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
		ScreenBuffer->Bind();
		glColor3f( 1, 1, 1 );
		glBegin( GL_QUADS );
		glTexCoord2f( 0, 1 ); glVertex2f( -1, -1 );
		glTexCoord2f( 1, 1 ); glVertex2f(  1, -1 );
		glTexCoord2f( 1, 0 ); glVertex2f(  1,  1 );
		glTexCoord2f( 0, 0 ); glVertex2f( -1,  1 );
		glEnd();
		glDisable( GL_BLEND );
		glDisable( GL_TEXTURE_2D );

		window->End();
		window->UpdateWindow();
		// resize buffers if needed
		if (window->GetClientSize() != ScreenResolution)
		{
			ScreenResolution = window->GetClientSize();
			window->ResizeClient(ScreenResolution);
			if (ScreenBuffer) delete ScreenBuffer;
			ScreenBuffer = new GLStreamingTexture( ScreenResolution );
		#if 0
			RenderResolution = ScreenResolution;
		#endif

		}
		if (RenderResolution.X > ScreenResolution.X) RenderResolution.X = ScreenResolution.X;
		if (RenderResolution.Y > ScreenResolution.Y) RenderResolution.Y = ScreenResolution.Y;
		if (pathtracer->GetResolution() != RenderResolution)
		{
			pathtracer->SetResolution( RenderResolution );
			RenderResolution = pathtracer->GetResolution();
			if (RenderBuffer) delete RenderBuffer;
			RenderBuffer = new GLStreamingTexture( RenderResolution );
		}
		if (window->Key[ 27 ]) break;
		timer.Stop();
		dtime = timer.ElapsedMs() / 1000.0f;
	}
	// end of application -  save last camera position
	SaveCam( 0 );
	//throw away some things
	delete ScreenBuffer;
	delete window;
	delete scene;
	delete pathtracer;
}
