/*
 * COPYRIGHT:        See COPYRIGHT.TXT
 * PROJECT:          Ext2 File System Driver for WinNT/2K/XP
 * FILE:             init.c
 * PROGRAMMER:       Matt Wu <mattwu@163.com>
 * HOMEPAGE:         http://ext2.yeah.net
 * UPDATE HISTORY: 
 */

/* INCLUDES *****************************************************************/

#include "ntifs.h"
#include "ext2fs.h"

/* GLOBALS ***************************************************************/

PEXT2_GLOBAL    Ext2Global = NULL;

/* DEFINITIONS ***********************************************************/

NTSTATUS
DriverEntry(
    IN PDRIVER_OBJECT DriverObject,
    IN PUNICODE_STRING RegistryPath   );

#ifdef ALLOC_PRAGMA
#pragma alloc_text(INIT, Ext2QueryRegistry)
#pragma alloc_text(INIT, DriverEntry)
#if EXT2_UNLOAD
#pragma alloc_text(PAGE, DriverUnload)
#endif
#endif

/* FUNCTIONS ***************************************************************/

#if EXT2_UNLOAD

VOID
DriverUnload (IN PDRIVER_OBJECT DriverObject)
/*
 * FUNCTION: Called by the system to unload the driver
 * ARGUMENTS:
 *           DriverObject = object describing this driver
 * RETURNS:  None
 */
{
    UNICODE_STRING  DosDeviceName;

    KdPrint(("Ext2Fsd: Unloading routine.\n"));

    RtlInitUnicodeString(&DosDeviceName, DOS_DEVICE_NAME);
    IoDeleteSymbolicLink(&DosDeviceName);

    ExDeleteResourceLite(&Ext2Global->Resource);

    ExDeletePagedLookasideList(&(Ext2Global->Ext2McbLookasideList));
    ExDeleteNPagedLookasideList(&(Ext2Global->Ext2CcbLookasideList));
    ExDeleteNPagedLookasideList(&(Ext2Global->Ext2FcbLookasideList));
    ExDeleteNPagedLookasideList(&(Ext2Global->Ext2IrpContextLookasideList));
    
    IoDeleteDevice(Ext2Global->DeviceObject);
}

#endif

BOOLEAN
Ext2QueryRegistry( IN PUNICODE_STRING  RegistryPath, PULONG WritingSupport)
{
    NTSTATUS                    Status;
    UNICODE_STRING              ParameterPath;
    RTL_QUERY_REGISTRY_TABLE    QueryTable[2];

    ParameterPath.Length = 0;

    ParameterPath.MaximumLength =
        RegistryPath->Length + sizeof(PARAMETERS_KEY) + sizeof(WCHAR);

    ParameterPath.Buffer =
        (PWSTR) ExAllocatePool(PagedPool, ParameterPath.MaximumLength);

    if (!ParameterPath.Buffer)
    {
        return FALSE;
    }

    RtlCopyUnicodeString(&ParameterPath, RegistryPath);

    RtlAppendUnicodeToString(&ParameterPath, PARAMETERS_KEY);

    RtlZeroMemory(&QueryTable[0], sizeof(QueryTable));

    QueryTable[0].Flags = RTL_QUERY_REGISTRY_DIRECT | RTL_QUERY_REGISTRY_REQUIRED;
    QueryTable[0].Name = WRITING_SUPPORT;
    QueryTable[0].EntryContext = WritingSupport;

    Status = RtlQueryRegistryValues(
        RTL_REGISTRY_ABSOLUTE,
        ParameterPath.Buffer,
        &QueryTable[0],
        NULL,
        NULL        );

    ExFreePool(ParameterPath.Buffer);

    if (NT_SUCCESS(Status))
        return TRUE;

    return FALSE;
}

NTSTATUS
DriverEntry (
         IN PDRIVER_OBJECT   DriverObject,
         IN PUNICODE_STRING  RegistryPath
         )
/*
 * FUNCTION: Called by the system to initalize the driver
 * ARGUMENTS:
 *           DriverObject = object describing this driver
 *           RegistryPath = path to our configuration entries
 * RETURNS: Success or failure
 */
{
    PDEVICE_OBJECT              DeviceObject;
    PFAST_IO_DISPATCH           FastIoDispatch;
    PCACHE_MANAGER_CALLBACKS    CacheManagerCallbacks;
    PEXT2FS_EXT                 DeviceExt;

    UNICODE_STRING              DeviceName;
#if EXT2_UNLOAD
    UNICODE_STRING              DosDeviceName;
#endif
    NTSTATUS                    Status;

    LARGE_INTEGER               CurrentTime;
    LARGE_INTEGER               LocalTime;
    TIME_FIELDS                 TimeFields;

    ULONG                       WritingSupport;

    KeQuerySystemTime( &CurrentTime);
    RtlTimeToTimeFields(&CurrentTime, &TimeFields);

    RtlZeroMemory(&TimeFields, sizeof(TIME_FIELDS));
    TimeFields.Year = 1970;
    TimeFields.Month = 1;
    TimeFields.Day = 1;
    TimeFields.Hour = 0;
    TimeFields.Minute = 0;
    TimeFields.Second = 0;
    TimeFields.Milliseconds = 0;

    if (RtlTimeFieldsToTime(
        &TimeFields,
        &(LocalTime)))
    {
        KdPrint(("LocalTime: %I64xh\n", LocalTime.QuadPart));
    }
    else
        KdPrint(("FiledstoTime error.\n"));

  
    KdPrint(("Ext2 File System Driver by Matt Wu.\n"));
    KdPrint(("Ext2 DriverEntry...\n"));

    //
    // Print some info about the driver
    //

    DbgPrint(DRIVER_NAME ": " __DATE__ " " __TIME__

#if EXT2_READ_ONLY
    ", ReadOnly"
#endif

#if DBG
    ", Checked"
#endif
    ", _WIN32_WINNT=%#x.\n", _WIN32_WINNT);


    RtlInitUnicodeString(&DeviceName, DEVICE_NAME);
    
    Status = IoCreateDevice(
        DriverObject,
        sizeof(EXT2FS_EXT),
        &DeviceName,
        FILE_DEVICE_DISK_FILE_SYSTEM,
        0,
        FALSE,
        &DeviceObject );
    
    if (!NT_SUCCESS(Status))
    {
        KdPrint(("Ext2 IoCreateDevice error.\n"));
        return Status;
    }

    DeviceExt = (PEXT2FS_EXT) DeviceObject->DeviceExtension;
    RtlZeroMemory(DeviceExt, sizeof(EXT2FS_EXT));
    
    Ext2Global = &(DeviceExt->Ext2Global);
    Ext2Global->TimeZone.QuadPart = LocalTime.QuadPart;

    KdPrint(("TimeZone: %I64xh\n", Ext2Global->TimeZone.QuadPart));


    Ext2Global->Identifier.Type = EXT2FGD;
    Ext2Global->Identifier.Size = sizeof(EXT2_GLOBAL);
    Ext2Global->DeviceObject = DeviceObject;
    Ext2Global->DriverObject = DriverObject;

    if(Ext2QueryRegistry(RegistryPath, &WritingSupport))
    {
        if (WritingSupport)
            SetFlag(Ext2Global->Flags, EXT2_SUPPORT_WRITING);
    }
        
    DriverObject->MajorFunction[IRP_MJ_CREATE]              = Ext2BuildRequest;
    DriverObject->MajorFunction[IRP_MJ_CLOSE]               = Ext2BuildRequest;
    DriverObject->MajorFunction[IRP_MJ_READ]                = Ext2BuildRequest;
    DriverObject->MajorFunction[IRP_MJ_WRITE]               = Ext2BuildRequest;

    DriverObject->MajorFunction[IRP_MJ_FLUSH_BUFFERS]       = Ext2BuildRequest;
    DriverObject->MajorFunction[IRP_MJ_SHUTDOWN]	        = Ext2BuildRequest;

    DriverObject->MajorFunction[IRP_MJ_QUERY_INFORMATION]   = Ext2BuildRequest;
    DriverObject->MajorFunction[IRP_MJ_SET_INFORMATION]     = Ext2BuildRequest;

    DriverObject->MajorFunction[IRP_MJ_QUERY_VOLUME_INFORMATION]    = Ext2BuildRequest;
    DriverObject->MajorFunction[IRP_MJ_SET_VOLUME_INFORMATION]      = Ext2BuildRequest;

    DriverObject->MajorFunction[IRP_MJ_DIRECTORY_CONTROL]   = Ext2BuildRequest;
    DriverObject->MajorFunction[IRP_MJ_FILE_SYSTEM_CONTROL] = Ext2BuildRequest;
    DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL]      = Ext2BuildRequest;
    DriverObject->MajorFunction[IRP_MJ_LOCK_CONTROL]        = Ext2BuildRequest;

    DriverObject->MajorFunction[IRP_MJ_CLEANUP]             = Ext2BuildRequest;
    DriverObject->DriverUnload                              = NULL;

    //
    // Initialize the fast I/O entry points
    //
    
    FastIoDispatch = &(Ext2Global->FastIoDispatch);
    
    FastIoDispatch->SizeOfFastIoDispatch        = sizeof(FAST_IO_DISPATCH);
    FastIoDispatch->FastIoCheckIfPossible       = Ext2FastIoCheckIfPossible;
#if DBG
    FastIoDispatch->FastIoRead                  = Ext2FastIoRead;
    FastIoDispatch->FastIoWrite                 = Ext2FastIoWrite;
#else
    FastIoDispatch->FastIoRead                  = FsRtlCopyRead;
    FastIoDispatch->FastIoWrite                 = FsRtlCopyWrite;
#endif
    FastIoDispatch->FastIoQueryBasicInfo        = Ext2FastIoQueryBasicInfo;
    FastIoDispatch->FastIoQueryStandardInfo     = Ext2FastIoQueryStandardInfo;
    FastIoDispatch->FastIoLock                  = Ext2FastIoLock;
    FastIoDispatch->FastIoUnlockSingle          = Ext2FastIoUnlockSingle;
    FastIoDispatch->FastIoUnlockAll             = Ext2FastIoUnlockAll;
    FastIoDispatch->FastIoUnlockAllByKey        = Ext2FastIoUnlockAllByKey;
    FastIoDispatch->FastIoQueryNetworkOpenInfo  = Ext2FastIoQueryNetworkOpenInfo;

    DriverObject->FastIoDispatch = FastIoDispatch;

    switch ( MmQuerySystemSize() ) {

    case MmSmallSystem:

        Ext2Global->MaxDepth = 16;
        break;

    case MmMediumSystem:

        Ext2Global->MaxDepth = 64;
        break;

    case MmLargeSystem:

        Ext2Global->MaxDepth = 256;
        break;
    }

    //
    // Initialize the Cache Manager callbacks
    //
    
    CacheManagerCallbacks = &(Ext2Global->CacheManagerCallbacks);
    CacheManagerCallbacks->AcquireForLazyWrite  = Ext2AcquireForLazyWrite;
    CacheManagerCallbacks->ReleaseFromLazyWrite = Ext2ReleaseFromLazyWrite;
    CacheManagerCallbacks->AcquireForReadAhead  = Ext2AcquireForReadAhead;
    CacheManagerCallbacks->ReleaseFromReadAhead = Ext2ReleaseFromReadAhead;

    Ext2Global->CacheManagerNoOpCallbacks.AcquireForLazyWrite  = Ext2NoOpAcquire;
    Ext2Global->CacheManagerNoOpCallbacks.ReleaseFromLazyWrite = Ext2NoOpRelease;
    Ext2Global->CacheManagerNoOpCallbacks.AcquireForReadAhead  = Ext2NoOpAcquire;
    Ext2Global->CacheManagerNoOpCallbacks.ReleaseFromReadAhead = Ext2NoOpRelease;


    //
    // Initialize the global data
    //

    InitializeListHead(&(Ext2Global->VcbList));
    ExInitializeResourceLite(&(Ext2Global->Resource));

    ExInitializeNPagedLookasideList( &(Ext2Global->Ext2IrpContextLookasideList),
                                     NULL,
                                     NULL,
                                     0,
                                     sizeof(EXT2_IRP_CONTEXT),
                                     '2TXE',
                                     Ext2Global->MaxDepth);

    ExInitializeNPagedLookasideList( &(Ext2Global->Ext2FcbLookasideList),
                                     NULL,
                                     NULL,
                                     0,
                                     sizeof(EXT2_FCB),
                                     '2TXE',
                                     Ext2Global->MaxDepth);

    ExInitializeNPagedLookasideList( &(Ext2Global->Ext2CcbLookasideList),
                                     NULL,
                                     NULL,
                                     0,
                                     sizeof(EXT2_CCB),
                                     '2TXE',
                                     (USHORT)(Ext2Global->MaxDepth << 1));

    ExInitializePagedLookasideList( &(Ext2Global->Ext2McbLookasideList),
                                     NULL,
                                     NULL,
                                     0,
                                     sizeof(EXT2_MCB),
                                     '2TXE',
                                     (USHORT)(Ext2Global->MaxDepth << 1));

#if EXT2_UNLOAD
    RtlInitUnicodeString(&DosDeviceName, DOS_DEVICE_NAME);
    IoCreateSymbolicLink(&DosDeviceName, &DeviceName);
#endif

#if DBG
    ProcessNameOffset = Ext2GetProcessNameOffset();
#endif

    IoRegisterFileSystem(DeviceObject);
    
    return Status;
}
