//////////////////////////////////////////////////////////////////////////
//
//  IGATOR Copyright (C) 1997-98 RIT Research Labs
//
//  This programs is free for commercial and non-commercial use as long as
//  the following conditions are aheared to.
//
//  Copyright remains RIT Research Labs, and as such any Copyright notices
//  in the code are not to be removed. If this package is used in a
//  product, RIT Research Labs should be given attribution as the RIT Research
//  Labs of the parts of the library used. This can be in the form of a textual
//  message at program startup or in documentation (online or textual)
//  provided with the package.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions are
//  met:
//
//  1. Redistributions of source code must retain the copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. All advertising materials mentioning features or use of this software
//     must display the following acknowledgement:
//     "Based on IGATOR by RIT Research Labs."
//
//  THIS SOFTWARE IS PROVIDED BY RIT RESEARCH LABS "AS IS" AND ANY EXPRESS
//  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//  DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
//  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
//  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
//  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
//  IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
//  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
//  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//  The licence and distribution terms for any publically available
//  version or derivative of this code cannot be changed. i.e. this code
//  cannot simply be copied and put under another distribution licence
//  (including the GNU Public Licence).
//
//////////////////////////////////////////////////////////////////////////

unit WSocket;

interface
uses Classes, Windows;


type

    TSocketOption = (soBroadcast, soDebug, soDontLinger,
                     soDontRoute, soKeepAlive, soOOBInLine,
                     soReuseAddr, soNoDelay, soBlocking, soAcceptConn);

    TSocketOptions = Set of TSocketOption;

    TSocket = class(TStream)
    public
      VPos: Integer;
      Connected: Boolean;
      Handle: Integer;
      Status: Integer;
      Options: TSocketOptions;
      Linger: Boolean;
      LingerTimeout: Integer;

      constructor Create(Protocol: Integer);
      constructor InitHandle(AHandle: Integer);
      destructor Destroy; override;

      function Bind(Port: Integer; Address: Integer): Boolean;
      function Connect(const Addr: String; APort: Integer): Boolean;
//      procedure Shutdown(ForRead, ForWrite: Boolean);

      function Seek(Offset: Longint; Origin: Word): Longint; override;
      function Read(var Buffer; Count: Longint): Longint; override;
      function Write(const Buffer; Count: Longint): Longint; override;

      procedure WriteStr(const S: String);

//      function Peek(var B; Size: Integer): Integer;

      function _Write(var B; Size, Flags: Integer): Integer;
      function _Read(var B; Size, Flags: Integer): Integer;

//      procedure SetOptions(O: TSocketOptions; OnOff: Boolean);
{$IFDEF SOCKOPT}
      procedure SetLinger(OnOff: Boolean; Timeout: Integer);
{$ENDIF}
//      procedure SetInSize(NewSize: Integer);
//      procedure SetOutSize(NewSize: Integer);
//      function  InSize: Integer;
//      function  OutSize: Integer;
{$IFDEF SOCKOPT}
    procedure UpdateOptions;
{$ENDIF}
//      procedure InitBuffer(ABufSize: Integer);
      procedure StartRead;
//      procedure ResizeBuffer(NewSize: Integer);
    end;

procedure FinalizeWSock;

implementation
uses WinSock;

var
  SocketsCS: TRTLCriticalSection;
  Sockets: TList;

const
    SockOptionNames: Array [TSocketOption] of Integer =
                     (SO_BROADCAST,  SO_DEBUG, SO_DONTLINGER, SO_DONTROUTE,
                      SO_KEEPALIVE, SO_OOBINLINE, SO_REUSEADDR, TCP_NODELAY,
                      0, SO_ACCEPTCONN);

    Sock_NotReady: Boolean = True;


constructor TSocket.Create;
  var WData: TWSAData;
begin
  inherited Create;
  if Sock_NotReady then WSAStartup(MakeWord(1,1), WData);
  Sock_NotReady := False;
  Handle := WinSock.Socket(AF_INET, Sock_Stream, 0);
  if Handle = INVALID_SOCKET then Status := WSAGetLastError else Status := 0;
//  if Status = 0 then UpdateOptions;
//  InitBuffer(ABufSize);
  EnterCriticalSection(SocketsCS);
  Sockets.Add(Self);
  LeaveCriticalSection(SocketsCS);
end;

constructor TSocket.InitHandle;
begin
  Handle := AHandle;
//  UpdateOptions;
  Status := WSAGetLastError;
//  InitBuffer(ABufSize);
end;

{procedure TSocket.InitBuffer;
begin
  BufSize := 0; InBuf := nil;
  ResizeBuffer(ABufSize);
end;}

{procedure TSocket.ResizeBuffer(NewSize: Integer);
  var P: PChar;
      I: Integer;
begin
  if BufSize > NewSize then Exit;
  I := BufSize;
  BufSize := NewSize;
  if BufSize < 10 then BufSize := 10;
  GetMem(P, BufSize);
  if InBuf <> nil then
    begin
      Move(InBuf[0], P[0], I);
      FreeMem(InBuf, I);
    end;
  InBuf := P;
end;
}

destructor TSocket.Destroy;
begin
//  Shutdown(True, True);
//  FreeMem(InBuf, BufSize);
  EnterCriticalSection(SocketsCS);
  if Handle <> INVALID_HANDLE_VALUE then
  begin
    CloseSocket(Handle); Handle := INVALID_HANDLE_VALUE;
  end;
  Sockets.Remove(Self);
  LeaveCriticalSection(SocketsCS);
end;

function TSocket.Bind(Port: Integer; Address: Integer): Boolean;
  var H: TSockAddrIn;
begin
  Result := False;
  if Handle = INVALID_HANDLE_VALUE then Exit;
  H.sin_family := AF_INET;
  H.sin_port := htons(Port);
  H.sin_addr.s_addr := Address;
  Result := WinSock.Bind(Handle, H, SizeOf(H)) = 0;
  if not Result then Status := WSAGetLastError;
end;

(*
function TSocket.Accept;
  var H: Integer;
      addr: TSockAddr;
      addrlen: Integer;
begin
  Result := nil;
  if Handle = INVALID_HANDLE_VALUE then Exit;
  H := WinSock.Accept(Handle, addr, addrlen);
  if H <> Invalid_Socket then Accept := TSocket.InitHandle(H) else Accept := nil;
  Status := WSAGetLastError;
end;
*)

function TSocket.Connect(const Addr: String; APort: Integer): Boolean;
  var H: PHostEnt;
      A: TSockAddrIn;
begin
  Result := False;
  if Handle = INVALID_HANDLE_VALUE then Exit;
  FillChar(A, SizeOf(A), 0);
  A.sin_Addr.S_Addr := inet_addr(PChar(Addr));
  if A.sin_Addr.S_Addr = INADDR_NONE then
    begin
      H := GetHostByName(PChar(Addr));
      A.sin_Addr.S_Addr := 0;
      if H = nil then
        begin
          Status := WSAEADDRNOTAVAIL;
          Exit;
        end;
      Move(H^.h_addr_list^[0], A.sin_Addr.S_Addr, SizeOf(A.sin_Addr.S_Addr)); 
    end;
  A.sin_Port := htons(APort);
  A.sin_family := AF_INET;
  if WinSock.Connect(Handle, A, SizeOf(A)) = 0 then begin Connected := True; Result := True end
    else Status := WSAGetLastError;
end;

procedure TSocket.StartRead;
begin
end;

{
function TSocket.Listen(Max: Integer): Boolean;
begin
  Result := False;
  if Handle = INVALID_HANDLE_VALUE then Exit;
  Max := WinSock.Listen(Handle, Max);
  Status := Max;
  Listen := Max = 0;
end;
}

function ItoS(L: Integer): String;
  var S: String;
begin
  Str(L, S);
  ItoS := S;
end;

{
function TSocket.PeerName: String;
  var H: TSockAddr;
      L: Integer;
begin
  Result := '';
  if Handle = INVALID_HANDLE_VALUE then Exit;
  L := SizeOf(H);
  if GetPeerName(Handle, H, L) = 0 then
     begin
       PeerName := ItoS(H.sin_addr.ClassA)+'.' + ItoS(H.sin_addr.ClassB) + '.' + ItoS(H.sin_addr.ClassC) + '.' + ItoS(H.sin_addr.ClassD);
     end else PeerName := '';
end;

function TSocket.DataReady: Integer;
begin
  DataReady := BufPtr;
end;}

function TSocket.Read;
begin
  Result := _Read(Buffer, Count, 0);
  Inc(VPos, Result);
end;

function TSocket.Seek(Offset: Longint; Origin: Word): Longint;
begin
  case Origin of
    FILE_BEGIN : VPos := Offset;
    FILE_CURRENT: Inc(VPos, Offset);
    else VPos := MaxInt+Offset;
  end;
  Result := VPos;
end;


function TSocket.Write;
  var I: Integer;
begin
  Result := 0;
  if Size = 0 then Exit;
  if Handle = INVALID_HANDLE_VALUE then Exit;
  I := WinSock.Send(Handle, Buffer, Count, 0);
  if I = SOCKET_ERROR then Status := WSAGetLastError
                      else Status := 0;
  Result := I;
end;

procedure TSocket.WriteStr;
begin
  if Length(S) > 0 then Write(S[1], Length(S));
end;


{function TSocket.Peek(var B; Size: Integer): Integer;
begin
  Peek := 0;
  if Size > BufPtr then Size := BufPtr;
  if (Size = 0) then Exit;
  Move(InBuf[0], B, Size);
  Peek := Size;
end;}

function TSocket._Write(var B; Size, Flags: Integer): Integer;
begin
  Result := 0;
  if Handle = INVALID_HANDLE_VALUE then Exit;
  Result := WinSock.Send(Handle, B, Size, Flags);
  if Result = SOCKET_ERROR then begin Status := WSAGetLastError; Result := 0 end else Status := 0;
end;

function TSocket._Read(var B; Size, Flags: Integer): Integer;
begin
  Result := 0;
  if Handle = INVALID_HANDLE_VALUE then Exit;
  Result := WinSock.Recv(Handle, B, Size, Flags);
  if Result = SOCKET_ERROR then begin Status := WSAGetLastError; Result := 0 end else Status := 0;
end;

{
procedure TSocket.SetOptions(O: TSocketOptions; OnOff: Boolean);
  var I: TSocketOption;
      V: LongInt;
begin
  if Handle = INVALID_HANDLE_VALUE then Exit;
  V := Integer(OnOff);
  for I := soBroadcast to soAcceptConn do
    if I in O then
     begin
       case I of
         soNoDelay: Status := WinSock.SetSockOpt(Handle, IPPROTO_TCP, SockOptionNames[I], V, SizeOf(V));
         soBlocking: begin
                       V := Integer(not OnOff);
                       Status := WinSock.IoctlSocket(Handle, FIONBIO, V);
                     end;
             else Status := WinSock.SetSockOpt(Handle, SOL_SOCKET, SockOptionNames[I], V, SizeOf(V));
       end;
       if Status <> 0 then Break;
       if OnOff then Options := Options + [I]
                else Options := Options - [I];
     end;
end;
}

{$IFDEF SOCKOPT}
type
  PLinger = ^TLinger;
  TLinger = packed record
    l_onoff: U_Short;
    l_linger: U_Short;
  end;


procedure TSocket.SetLinger(OnOff: Boolean; Timeout: Integer);
  var V: TLinger;
      I: Integer;
begin
  if Handle = INVALID_HANDLE_VALUE then Exit;
  V.l_onoff := Integer(OnOff); I := Integer(OnOff);
  V.l_linger := Timeout;
  if OnOff then Status := WinSock.SetSockOpt(Handle, SOL_SOCKET, SO_LINGER, V, SizeOf(V))
           else Status := WinSock.SetSockOpt(Handle, SOL_SOCKET, SO_DONTLINGER, I, SizeOf(I));
  if Status = 0 then
     begin
       Linger := OnOff;
       LingerTimeout := Timeout;
     end;
end;
{$ENDIF}

{
procedure TSocket.SetInSize(NewSize: Integer);
begin
  if Handle = INVALID_HANDLE_VALUE then Exit;
  Status := WinSock.SetSockOpt(Handle, SOL_SOCKET, SO_RCVBUF, NewSize, SizeOf(NewSize))
end;

procedure TSocket.SetOutSize(NewSize: Integer);
begin
  if Handle = INVALID_HANDLE_VALUE then Exit;
  Status := WinSock.SetSockOpt(Handle, SOL_SOCKET, SO_SNDBUF, NewSize, SizeOf(NewSize))
end;

function  TSocket.InSize: Integer;
  var N: Integer;
begin
  Result := 0;
  if Handle = INVALID_HANDLE_VALUE then Exit;
  Status := WinSock.SetSockOpt(Handle, SOL_SOCKET, SO_RCVBUF, N, SizeOf(N));
  Result := N;
end;

function  TSocket.OutSize: Integer;
  var N: Integer;
begin
  Result := 0;
  if Handle = INVALID_HANDLE_VALUE then Exit;
  Status := WinSock.SetSockOpt(Handle, SOL_SOCKET, SO_SNDBUF, N, SizeOf(N));
  Result := N;
end;
}
{$IFDEF SOCKOPT}

procedure TSocket.UpdateOptions;
  var I: TSocketOption;
      V: Integer;
      L: TLinger;
      LL: Integer;
begin
  if Handle = INVALID_HANDLE_VALUE then Exit;
  Options := [];
  for I := soBroadcast to soAcceptConn do
     begin
       LL := SizeOf(V);
       if I = soNoDelay then WinSock.GetSockOpt(Handle, IPPROTO_TCP, SockOptionNames[I], V, LL)
                        else WinSock.GetSockOpt(Handle, SOL_SOCKET, SockOptionNames[I], V, LL);
       if V <> 0 then Options := Options + [I];
     end;
  LL := SizeOf(L);
  WinSock.GetSockOpt(Handle, SOL_SOCKET, SO_LINGER, L, LL);
  Linger := L.l_onoff <> 0;
  LingerTimeout := L.l_linger;
end;
{$ENDIF}

{
procedure TSocket.Shutdown;
  var I: Integer;
begin
  if Handle = INVALID_HANDLE_VALUE then Exit;
  Status := 0;
  if ForRead then
    if ForWrite then I := 2
                else I := 0
      else if ForWrite then I := 1
                       else Exit;
  Status := WinSock.Shutdown(Handle, I);
end;}

procedure FinalizeWSock;
var
  j, i: Integer;
  s: TSocket;
begin
  if Sockets = nil then Exit;
  EnterCriticalSection(SocketsCS);
  for i := 0 to Sockets.Count-1 do
  begin
    s := Sockets[i];
    if s.Handle <> INVALID_HANDLE_VALUE then
    begin
      closesocket(s.Handle);
      s.Handle := INVALID_HANDLE_VALUE;
    end;
  end;
  LeaveCriticalSection(SocketsCS);
  i := 0;
  while Sockets.Count > 0 do
  begin
    Sleep(100);
    Inc(i);
    if i > 30 then Break;
  end;
  if (not Sock_NotReady) and (not (i > 30)) then WSACleanup;
  Sockets.Free; Sockets := nil;
  DeleteCriticalSection(SocketsCS);
end;

initialization
  InitializeCriticalSection(SocketsCS);
  Sockets := TList.Create;
end.


