//////////////////////////////////////////////////////////////////////////
//
//  IGATOR Copyright (C) 1997-98 RIT Research Labs
//
//  This programs is free for commercial and non-commercial use as long as
//  the following conditions are aheared to.
//
//  Copyright remains RIT Research Labs, and as such any Copyright notices
//  in the code are not to be removed. If this package is used in a
//  product, RIT Research Labs should be given attribution as the RIT Research
//  Labs of the parts of the library used. This can be in the form of a textual
//  message at program startup or in documentation (online or textual)
//  provided with the package.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions are
//  met:
//
//  1. Redistributions of source code must retain the copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. All advertising materials mentioning features or use of this software
//     must display the following acknowledgement:
//     "Based on IGATOR by RIT Research Labs."
//
//  THIS SOFTWARE IS PROVIDED BY RIT RESEARCH LABS "AS IS" AND ANY EXPRESS
//  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//  DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
//  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
//  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
//  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
//  IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
//  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
//  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//  The licence and distribution terms for any publically available
//  version or derivative of this code cannot be changed. i.e. this code
//  cannot simply be copied and put under another distribution licence
//  (including the GNU Public Licence).
//
//////////////////////////////////////////////////////////////////////////

unit Utils;

interface
uses Windows, SysUtils;

type
    SmallWord = Word;
    Str255    = String[255];

procedure ClrIO;

function IsInteger(const S: String): Boolean;
function StoI(const S: String): LongInt;
function ItoS(N: LongInt): String;
function SStr(N: LongInt; P: Byte; C: Char): String;

function PosChar(C: Char; const S: String): Integer;
procedure AddStr(var S: String; C: Char);

procedure ExtractAddrInfo(S: String; var Name, Addr: String);
function ParseAddress(const Address: String; var Zone, Net, Node, Point: Word): Boolean;
function MakeAddress(Zone, Net, Node, Point: Word): String;

function Hex2(B: Byte): String;
function Hex4(W: Word): String;
function Hex8(L: LongInt): String;

function Max(A,B: LongInt): LongInt;
function Min(A,B: LongInt): LongInt;


const
     stOK     = 0;    
     stOpen   = 1;
     stRead   = 2;
     stWrite  = 3;
     stCreate = 4;


type
    TDosStream = class
      F: hFile;
      Status: Integer;
      constructor Create(const Fn: String; Mode: Word);
      function Read(var B; N: Cardinal): Cardinal;
      function Write(var B; N: Cardinal): Cardinal;
      procedure Seek(N: Cardinal);
      function GetPos: Cardinal;
      function GetSize: Cardinal;
      procedure SetSize(N: Cardinal);
      procedure CopyFrom(S: TDosStream; N: Cardinal);
      procedure Truncate;
      destructor Destroy; override;
      property Position: Cardinal read GetPos write Seek;
      property Size: Cardinal read GetSize;
    end;


implementation

function Max(A,B: LongInt): LongInt; begin if A>B then Max:=A else Max := B end;
function Min(A,B: LongInt): LongInt; begin if A<B then Min:=A else Min := B end;


function PosChar(C: Char; const S: String): Integer;
begin
  PosChar := Pos(C, S);
end;

procedure AddStr(var S: String; C: Char);
  var L: Integer;
begin
  L := Length(S)+1;
  SetLength(S, L);
  S[L] := C;
end;

function IsInteger(const S: String): Boolean;
  var I: Integer;
begin
  Result := False; if S = '' then Exit;
  for I := 1 to Length(S) do
    if (S[I] < '0') or (S[I] > '9') then Exit;
  Result := True;
end;

function StoI(const S: String): LongInt;
  var J: Integer;
      I: LongInt;
      B: Boolean;
begin
  Result := 0;
  if not IsInteger(Trim(S)) then Exit;
  Val(S, I, J);
  StoI := I;
end;

function SStr(N: LongInt; P: Byte; C: Char): String;
  var S: String;
      I: Integer;
begin
  Str(N:P, S);
  if C <> ' ' then
    begin
      I := 1;
      while (S[I] = ' ') do begin S[I] := C; Inc(I); end;
    end;
  SStr := S;
end;

function ItoS(N: LongInt): String;
  var S: String;
begin
  Str(N, S);
  ItoS := S;
end;

function ParseAddress(const Address: String; var Zone, Net, Node, Point: Word): Boolean;
  const Signs = ':/.';
  var S1, S2: String[6];
      I,J,N,K: Integer;
      KK: Integer;
begin
  ParseAddress := False; Point := 0;
  I := 1; N := 0;
  KK := PosChar('@', Address);
  if KK = 0 then KK := Length(Address)+1;
  while I < KK do
    begin
      S1 := '';
      while (I < KK) and (Length(S1) < 5) and
            (Address[I] >= '0') and (Address[I] <= '9') do
             begin S1 := S1+Address[I]; Inc(I); end;
      {if I <= Length(Address) then}
        begin
          J := PosChar(Address[I], Signs); K := StoI(S1);
          if (J <> 0) and (N <> 0) and (J-1 <> N) then Exit;
          case J of
            0: if N = 3 then Point := K
                 else if (N = 0) or (N = 2) then
                  begin if S1 <> '' then Node := K end
                   else Exit;
            1: begin if S1 <> '' then Zone := K; Net := 0; Node := 0; end;
            2: begin if S1 <> '' then Net := K; Node := 0; end;
            3: if S1 <> '' then Node := K;
          end;
          N := J;
          Inc(I);
        end;
    end;
  ParseAddress := True;
end;

function MakeAddress(Zone, Net, Node, Point: Word): String;
begin
  if Point = 0 then
    MakeAddress := Format('%d:%d/%d', [Zone, Net, Node]) else
    MakeAddress := Format('%d:%d/%d.%d', [Zone, Net, Node, Point])
end;

procedure UpStr(var S: String);
  var I: Integer;
begin
  for I := 1 to Length(S) do S[I] := UpCase(S[I]);
end;

function UpStrg(S: String): String;
begin
  UpStr(S); UpStrg := S;
end;

procedure ClrIO;
  var I: Integer;
begin
  I := IoResult; I := IoResult;
end;


const HexChars: Array[0..15] of Char = ('0','1','2','3','4','5','6','7',
                                        '8','9','A','B','C','D','E','F');

function Hex2(B: Byte): String;
  var S: String[2];
begin
  S[0] := #2; S[1] := HexChars[B shr 4]; S[2] := HexChars[B and 15];
  Hex2 := S;
end;

function Hex4(W: Word): String;
begin
  Hex4 := Hex2(Hi(W))+Hex2(Lo(W));
end;

function Hex8(L: LongInt): String;
begin
  Hex8 := Hex4(L shr 16) + Hex4(L and $FFFF);
end;

procedure ExtractAddrInfo(S: String; var Name, Addr: String);
  var I, J: Integer;

  procedure Get(C: Char; var S1, S2: String);
  begin
    J := I+1;
    while (J <= Length(S)) and (S[J] <> C) do Inc(J);
    S1 := Copy(S, I+1, J-I-1);
    Delete(S, I, J-I+1); S2 := S; Trim(S2);
  end;

begin
  Delete(S, 1, Pos(':', S));
  I := Pos('<', S);
  if I > 0 then Get('>', Addr, Name) else
    begin
      I := Pos('"', S);
      if I > 0 then Get('"', Name, Addr) else
        begin
          I := Pos('(', S);
          if I > 0 then Get(')', Name, Addr) else begin Addr := S; Trim(Addr); end;
        end;
    end;
  if Addr = '' then Addr := Name;
  if Name = '' then Name := Addr;
end;

{ TDOSStream }

constructor TDOSStream.Create;
  var Access,
      CM,
      Attr,
      Share: Integer;
begin
  inherited Create;
  Share := 0; Access := Generic_Read;
  CM := Open_Existing;
  case Mode of
    stOpen: Access := GENERIC_READ or GENERIC_WRITE;
    stRead: Access := GENERIC_READ;
    stWrite: Access := GENERIC_WRITE;
    stCreate: begin
                CM := Create_Always;
                Access := GENERIC_READ or GENERIC_WRITE;
              end;  
  end;
  F := CreateFile(PChar(Fn), Access, Share, nil, CM, 0, 0 );
  if F = INVALID_HANDLE_VALUE then Status := GetLastError
    else Status := 0;
end;

destructor TDOSStream.Destroy;
begin
  CloseHandle(F);
  inherited Destroy;
end;

function TDOSStream.Read;
  var W: DWord;
begin
  if ReadFile(F, B, N, W, nil) then Status := 0
     else Status := GetLastError;
  Result := W;
end;

function TDOSStream.Write;
  var W: DWord;
begin
  if WriteFile(F, B, N, W, nil) then Status := 0
     else Status := GetLastError;
  Result := W;
end;

procedure TDOSStream.Seek(N: Cardinal);
begin
  if SetFilePointer(F, N, nil, FILE_BEGIN) <> $FFFFFFFF then Status := 0
    else Status := GetLastError;
end;

function TDOSStream.GetPos: Cardinal;
begin
  Result := SetFilePointer(F, 0, nil, FILE_CURRENT);
end;

function TDOSStream.GetSize: Cardinal;
begin
  Result := GetFileSize(F, nil);
end;

procedure TDOSStream.SetSize;
begin
  Seek(N);
  Truncate;
end;

procedure TDOSStream.CopyFrom;
  var B: Pointer;
      L: Integer;
begin
  GetMem(B, $4000);
  if N > S.Size-S.Position then N := S.Size-S.Position;
  while N > 0 do
    begin
      if N < $4000 then L := N else L := $4000;
      S.Read(B^, L);
      Write(B^, L);
      Dec(N, L);
    end;
  FreeMem(B, $4000);
end;


procedure TDOSStream.Truncate;
begin
  if SetEndOfFile(F) then Status := 0 else Status := GetLastError;
end;


end.
