/*
 *  ftnnode.c -- Handle our links
 *
 *  ftnnode.c is a part of binkd project
 *
 *  Copyright (C) 1996, 1997  Dima Maloff, 5047/13
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version. See COPYING.
 */

/*
 * $Id: ftnnode.c,v 1.3 1997/11/03 06:10:39 mff Exp mff $
 *
 * $Log: ftnnode.c,v $
 * Revision 1.3  1997/11/03  06:10:39  mff
 * +nodes_init()
 *
 * Revision 1.2  1997/10/23  04:07:55  mff
 * many changes to hide pNod int ftnnode.c
 *
 * Revision 1.1  1996/12/29  09:41:37  mff
 * Initial revision
 */

#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "ftnnode.h"
#include "tools.h"
#include "sem.h"

#if defined(HAVE_THREADS) || defined(AMIGA)

static MUTEXSEM LSem;

#endif

int nNod = 0;
FTN_NODE *pNod = 0;
int nNodSorted = 0;

/*
 * Call this before all others functions from this file.
 */
void nodes_init()
{
  InitSem (&LSem);
}

/*
 * Compares too nodes. 0 == don't match
 */
int node_cmp (FTN_NODE *a, FTN_NODE *b)
{
  return ftnaddress_cmp (&a->fa, &b->fa);
}

/*
 * Sorts pNod array. Must NOT be called if LSem is locked!
 */
void sort_nodes ()
{
  LockSem (&LSem);
  qsort (pNod, nNod, sizeof (FTN_NODE), (int (*) (const void *, const void *)) node_cmp);
  nNodSorted = 1;
  ReleaseSem (&LSem);
}

/*
 * Return up/downlink info by fidoaddress. 0 == node not found
 */
FTN_NODE *get_node_info (FTN_ADDR *fa)
{
  FTN_NODE n, *np;

  if (!nNodSorted)
    sort_nodes ();
  LockSem (&LSem);
  memcpy (&n.fa, fa, sizeof (FTN_ADDR));
  np = (FTN_NODE *) bsearch (&n, pNod, nNod, sizeof (FTN_NODE),
			   (int (*) (const void *, const void *)) node_cmp);
  ReleaseSem (&LSem);
  return np;
}

/*
 * Add a new node, or edit old settings for a node
 *
 * 1 -- ok, 0 -- error;
 */
int add_node (FTN_ADDR *fa, char *hosts, char *pwd, char obox_flvr,
	       char *obox, char *ibox, int NR_flag)
{
  int cn;

  LockSem (&LSem);
  for (cn = 0; cn < nNod; ++cn)
  {
    if (!ftnaddress_cmp (&pNod[cn].fa, fa))
      break;
  }
  /* Node not found, create new entry */
  if (cn >= nNod)
  {
    cn = nNod;

    pNod = realloc (pNod, sizeof (FTN_NODE) * ++nNod);
    assert (pNod != 0);

    memset (pNod + cn, 0, sizeof (FTN_NODE));
    memcpy (&pNod[cn].fa, fa, sizeof (FTN_ADDR));
    strcpy (pNod[cn].pwd, "-");
    pNod[cn].hosts = strdup ("-");
    pNod[cn].obox_flvr = 'f';
    pNod[cn].NR_flag = NR_OFF;

    /* We've broken the order... */
    nNodSorted = 0;
  }

  if (NR_flag != NR_USE_OLD)
    pNod[cn].NR_flag = NR_flag;

  if (hosts && *hosts)
  {
    free (pNod[cn].hosts);	       /* pNod[cn].hosts is never NULL */
    pNod[cn].hosts = strdup (hosts);
  }

  if (pwd && *pwd)
  {
    strnzcpy (pNod[cn].pwd, pwd, sizeof (pNod[cn].pwd));
  }

  if (obox_flvr != '-')
  {
    pNod[cn].obox_flvr = obox_flvr;
  }

  if (obox)
  {
    if (pNod[cn].obox)
      free (pNod[cn].obox);
    pNod[cn].obox = strdup (obox);
  }

  if (ibox)
  {
    if (pNod[cn].ibox)
      free (pNod[cn].ibox);
    pNod[cn].ibox = strdup (ibox);
  }

  ReleaseSem (&LSem);
  return 1;
}

/*
 * Iterates through nodes while func() == 0.
 */
int foreach_node (int (*func) (FTN_NODE *, void *), void *arg)
{
  int i, rc = 0;

  LockSem (&LSem);
  for (i = 0; i < nNod; ++i)
  {
    rc = func (pNod + i, arg);
    if (rc != 0)
      break;
  }
  ReleaseSem (&LSem);
  return rc;
}

static int print_node_info_1 (FTN_NODE *fn, void *arg)
{
  char szfa[FTN_ADDR_SZ + 1];

  ftnaddress_to_str (szfa, &fn->fa);
  fprintf ((FILE *) arg, "%-20.20s %s %s %c %s %s %s\n",
	   szfa, fn->hosts, fn->pwd,
	   fn->obox_flvr, fn->obox ? fn->obox : "-",
	   fn->ibox ? fn->ibox : "-",
	   (fn->NR_flag == NR_ON) ? "-NR" : "");
  return 0;
}

void print_node_info (FILE *out)
{
  foreach_node (print_node_info_1, out);
}
