/*
 * GSysInfo Applet
 *  - A GNOME panel applet to display various system information.
 *  Copyright (C) 1999 Jason D. Hildebrand
 *  - jdhildeb@undergrad.math.uwaterloo.ca
 *  - http://www.undergrad.math.uwaterloo.ca/~jdhildeb/gsysinfo
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>
#include <time.h>
/* #include <config.h> */
#include <gnome.h>
#include <gdk/gdkx.h>

#include <applet-widget.h>

#include "gsysinfo.h"
#include "session.h"
#include "properties.h"
#include "sysinfo.h"

a_color AppColors[NUM_COLORS] = {
    // colors for load average bar
    { "loadavgbar0", "#FFC1C1", 0 },	// RosyBrown1
    { "loadavgbar1", "#FF6A6A", 0 },	// IndianRed1
    { "loadavgbar2", "#FF4500", 0 },	// OrangeRed1
    { "loadavgbar3", "#FF3838", 0 },	// firebrick1
    { "loadavgbar4", "#FFB5C5", 0 },	// pink1
    { "loadavgbar5", "#FF6EB4", 0 },	// HotPink1
    { "loadavgbar6", "#FF1493", 0 },	// DeepPink2
    { "loadavgbar7", "#B03060", 0 },	// maroon1

    // colors for current load bar
    { "loadbar", "#00EEEE", 0 },   // cyan2
    { "loadbar1", "#5CACEE", 0 },  // SteelBlue2
    { "loadbar2", "#0000FF", 0 },  // blue

    // colors for memory bar
    { "membar", "#00EE00", 0 },    // green2
    { "membar1", "#008B00", 0 },   // green4

    // colors for swap space bar
    { "swapbar", "#FF6EB4", 0 },   // hotpink1
    { "swapbar1", "#CD6090", 0 },  // hotpink3
    { "swapbar2", "#8B3A62", 0 },  // hotpink4

    // bar background color
    { "bargauge_bg", "#FFFFFF", 0 }	// white
};


int
main (int argc, char ** argv)
{
  const gchar *goad_id;
  GtkWidget *applet;
  int	    rc;

  /* Initialize i18n */
  bindtextdomain (PACKAGE, GNOMELOCALEDIR);
  textdomain (PACKAGE);

  applet_widget_init ("gsysinfo_applet", VERSION, argc, argv, NULL, 0, NULL);

  applet_factory_new ("gsysinfo_applet", NULL,
		     (AppletFactoryActivator) applet_start_new_applet);

  goad_id = goad_server_activation_id ();
  if (! goad_id) {
    return 1;
  }
  rc = sysinfo_init();
  if( rc != 0 ) {
    return 1;
  }

  /* Create the applet widget */
  applet = make_new_gsysinfo_applet (goad_id);
  if( applet == NULL ) {
      return 1;
  }

  /* Run... */

  applet_widget_gtk_main ();

  return 0;
} /* main */


void draw_border( GSysInfoData * gd, int x1, int y1, int x2, int y2 )
{
  GtkStyle *	style;

  style = gtk_widget_get_style( gd->applet );
  gdk_draw_line( gd->pixmap, style->dark_gc[0], x1, y1, x2, y1 );
  gdk_draw_line( gd->pixmap, style->dark_gc[0], x1, y1, x1, y2 );
  gdk_draw_line( gd->pixmap, style->black_gc, x1 + 1, y1 + 1, x2 - 1, y1 + 1 );
  gdk_draw_line( gd->pixmap, style->black_gc, x1 + 1, y1 + 1, x1 + 1, y2 - 1 );

  gdk_draw_line( gd->pixmap, style->white_gc, x1 + 1, y2, x2, y2 );
  gdk_draw_line( gd->pixmap, style->white_gc, x2, y2, x2, y1 + 1 );
  gdk_draw_line( gd->pixmap, style->mid_gc[0], x1 + 2, y2 - 1, x2 - 1, y2 - 1 );
  gdk_draw_line( gd->pixmap, style->mid_gc[0], x2 - 1, y2 - 1, x2 - 1, y1 + 2 );
}

// barx, bary, bar_breadth, and bar_depth have already been corrected for the 
// size of the border bevelling
void draw_loadavg( GSysInfoData * gd, int barx, int bary, int bar_breadth, int bar_depth ) {

  double loadavg;
  int segments;
  int segsize;
  int segstart;
  int segend;
  int n;

  loadavg = get_loadavg();
  segments = ((int)(loadavg)) + 1;
  segsize = bar_breadth / segments;

  /* Clear the graph pixmap */
  gdk_gc_set_foreground (gd->gc, & (AppColors[ BARGAUGEBG ].val) );
  gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, barx, bary, bar_breadth, bar_depth );

  segend = 0;
  for( n = 0; n < segments - 1; n++ ) {
      segstart = segend;
      segend = (( bar_breadth * 10 ) / segments ) * ( n + 1 ) / 10;
      segsize = segend - segstart;
      gdk_gc_set_foreground (gd->gc, & (AppColors[ LOADAVGBAR0 + n ].val) );
      gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, barx + segstart, bary, segsize - 1, bar_depth );
  }

  segstart = segend;
  segend = bar_breadth - 1;
  segsize = (loadavg * (double)10 - ( segments - 1 ) * 10 ) * (segend - segstart + 1 ) / 10 + 1;
  if( segstart + segsize - 1 > segend ) {
      segsize = segend - segstart + 1;
  }
  gdk_gc_set_foreground (gd->gc, & (AppColors[ LOADAVGBAR0 + segments - 1 ].val) );
  gdk_draw_rectangle (gd->pixmap, 
		      gd->gc, 
		      TRUE, 
		      barx + segstart,
		      bary, 
		      segsize,
		      bar_depth );
}


// barx, bary, bar_breadth, and bar_depth have already been corrected for the 
// size of the border bevelling
void draw_load( GSysInfoData * gd, int barx, int bary, int bar_breadth, int bar_depth ) {

  int part1, part2, part3;
  struct load load;

  get_load ( &load );

  /* Calculate the bounds */
  part1 = ( load.user * bar_breadth + 1 ) / load.total;
  part2 = ( load.system * bar_breadth + 1 ) / load.total;
  part3 = ( load.nice * bar_breadth + 1 ) / load.total;
  
  /* Clear the graph pixmap */
  gdk_gc_set_foreground (gd->gc, & (AppColors[ BARGAUGEBG ].val) );
  gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, barx, bary, bar_breadth, bar_depth );

  /* Draw the load bar graph */
  gdk_gc_set_foreground (gd->gc, &(AppColors[ LOADBAR ].val ) );
  gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, barx, bary, part1, bar_depth );

  gdk_gc_set_foreground (gd->gc, &(AppColors[ LOADBAR1 ].val ) );
  gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, barx + part1, bary, part2, bar_depth );

  gdk_gc_set_foreground (gd->gc, &(AppColors[ LOADBAR2 ].val ) );
  gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, barx + part1 + part2, bary, part3, bar_depth );
}


void draw_mem( GSysInfoData * gd, int barx, int bary, int bar_breadth, int bar_depth ) {
    int normalmem, buffmem;
    struct meminfo * mem;
    int segstart;
    int	segend;
    int segsize;

    mem = &(gd->memstats[0]);
    buffmem = mem->cache + mem->buffers;
    normalmem = mem->total - mem->free - buffmem;

    // Clear the graph pixmap 
    gdk_gc_set_foreground (gd->gc, & (AppColors[ BARGAUGEBG ].val) );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, barx, bary, bar_breadth, bar_depth );

    // Draw the "normal used memory" bar
    segend = normalmem * bar_breadth / mem->total;
    segsize = segend - 1;
    gdk_gc_set_foreground (gd->gc, &(AppColors[ MEMBAR0 ].val ) );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, barx, bary, segsize, bar_depth );

    // Draw the buffered + cached memory bar
    segstart = segend;
    segend = ( buffmem + normalmem ) * bar_breadth / mem->total;
    segsize = segend - segstart + 1;
    gdk_gc_set_foreground (gd->gc, &(AppColors[ MEMBAR1 ].val ) );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, barx + segstart, bary, segsize, bar_depth );
}

void draw_swap( GSysInfoData * gd, int barx, int bary, int bar_breadth, int bar_depth ) {
    int totalspace = 0;
    int totalfree = 0;
    int totalused;
    int n;
    int segsize;
    
    for( n = 0; n < gd->numswapfiles ; n++ ) {
	totalspace += gd->memstats[ n + 1 ].total;
	totalfree += gd->memstats[ n + 1 ].free;
    }
    totalused = totalspace - totalfree;

    // Clear the graph pixmap 
    gdk_gc_set_foreground (gd->gc, & (AppColors[ BARGAUGEBG ].val) );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, barx, bary, bar_breadth, bar_depth );

    segsize = totalused * bar_breadth / totalspace + 1;
    gdk_gc_set_foreground (gd->gc, &(AppColors[ SWAPBAR ].val ) );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, barx, bary, segsize, bar_depth );
}

/*
 * This function, gsysinfo_update, gets the new stats and updates the
 * pixmap.
 *
 */
gint
gsysinfo_update (gpointer data)
{
  GSysInfoData * gd = data;
  int cury;
  int n;

  cury = 0; 

  if( gd->loadavg_on ) {
      draw_loadavg( gd, 2, cury + 2, gd->bar_breadth - 4, gd->bar_depth - 4 );
      cury += gd->bar_depth;
  }


  if( gd->load_on ) {
      draw_load( gd, 2, cury + 2, gd->bar_breadth - 4, gd->bar_depth - 4 );
      cury += gd->bar_depth;
  }

  // get mem and swap information
  if( gd->mem_on || gd->swap_on ) {
     get_meminfo( &gd->numswapfiles, gd->memstats );
  }

  if( gd->mem_on ) {
      draw_mem( gd, 2, cury + 2, gd->bar_breadth - 4, gd->bar_depth - 4 );
      cury += gd->bar_depth;
  }

  if( gd->swap_on ) {
      draw_swap( gd, 2, cury + 2, gd->bar_breadth - 4, gd->bar_depth - 4 );
  }


  /* Update the display. */
  expose_handler (gd->loadavg_area, NULL, gd);

  return TRUE;
} /* gsysinfo_update */


/*
 * This function is called whenever a portion of the
 * applet window has been exposed and needs to be redrawn.  In this
 * function, we just blit the appropriate portion of the pixmap onto the window.
 *
 */
gint
expose_handler (GtkWidget * ignored, GdkEventExpose * expose,
			gpointer data)
{
  GSysInfoData * gd = data;
  int cury;

  if (!gd->setup) {
    return FALSE;
  }

    cury = 0;
    if( gd->loadavg_on ) {
	gdk_draw_pixmap( gd->loadavg_area->window, 
			 gd->loadavg_area->style->fg_gc[GTK_WIDGET_STATE (gd->loadavg_area)],
			 gd->pixmap, 
			 0, 0, 
			 0, 0, 
			 gd->bar_breadth, gd->bar_depth);
	cury += gd->bar_depth;
    }
    if( gd->load_on ) {
	gdk_draw_pixmap( gd->load_area->window, 
			 gd->load_area->style->fg_gc[GTK_WIDGET_STATE (gd->load_area)],
			 gd->pixmap, 
			 0, cury, 
			 0, 0, 
			 gd->bar_breadth, gd->bar_depth);
	cury += gd->bar_depth;
    }
    if( gd->mem_on ) {
	gdk_draw_pixmap( gd->mem_area->window, 
			 gd->mem_area->style->fg_gc[GTK_WIDGET_STATE (gd->mem_area)],
			 gd->pixmap, 
			 0, cury, 
			 0, 0, 
			 gd->bar_breadth, gd->bar_depth);
	cury += gd->bar_depth;
    }
    if( gd->swap_on ) {
	gdk_draw_pixmap( gd->swap_area->window, 
			 gd->swap_area->style->fg_gc[GTK_WIDGET_STATE (gd->swap_area)],
			 gd->pixmap, 
			 0, cury, 
			 0, 0, 
			 gd->bar_breadth, gd->bar_depth);
	cury += gd->bar_depth;
    }
  
  return FALSE; 
} /* expose_handler */

/* This handler gets called whenever the panel changes orientations.
   When the applet is set up, we get an initial call, too.  We don't
   actually do anything differently for vertical displays.  Not yet,
   anyways. */
gint
orient_handler (GtkWidget * w, PanelOrientType o, gpointer data)
{
  GSysInfoData * gd = data;
  gboolean vertical = (o == ORIENT_UP) || (o == ORIENT_DOWN);

  if (vertical != gd->vertical) {
    gd->vertical = vertical;
  }

  return FALSE;
} /* orient_handler */

void configure_layout( GSysInfoData * gd ) {
  int cury;

  
  cury = 0; 
  if( gd->loadavg_on ) {
      gtk_fixed_move( gd->fixed, gd->loadavg_area, 0, cury );
      gtk_widget_show( gd->loadavg_area );
      cury += gd->bar_depth + gd->gap_depth;
  } else {
      gtk_widget_hide( gd->loadavg_area );
  }
  if( gd->load_on ) {
      gtk_fixed_move( gd->fixed, gd->load_area, 0, cury );
      gtk_widget_show( gd->load_area );
      cury += gd->bar_depth + gd->gap_depth;
  } else {
      gtk_widget_hide( gd->load_area );
  }
  if( gd->mem_on ) {
      gtk_fixed_move( gd->fixed, gd->mem_area, 0, cury );
      gtk_widget_show( gd->mem_area );
      cury += gd->bar_depth + gd->gap_depth;
  } else {
      gtk_widget_hide( gd->mem_area );
  }
  if( gd->swap_on ) {
      gtk_fixed_move( gd->fixed, gd->swap_area, 0, cury );
      gtk_widget_show( gd->swap_area );
      cury += gd->bar_depth + gd->gap_depth;
  } else {
      gtk_widget_hide( gd->swap_area );
  }
}

GtkWidget *
applet_start_new_applet (const gchar *goad_id, const char **params,
			 int nparams)
{
  return make_new_gsysinfo_applet (goad_id);
} /* applet_start_new_applet */

/* This is the function that actually creates the display widgets */
GtkWidget *
make_new_gsysinfo_applet (const gchar *goad_id) {
  GSysInfoData * gd;
  gchar * param = "gsysinfo_applet";

  gd = g_new0 (GSysInfoData, 1);

  gd->applet = applet_widget_new (goad_id);
  gtk_widget_set_usize (gd->applet, gd->bar_breadth, gd->bar_depth * 4 + gd->gap_depth * 4 ); 

  if (gd->applet == NULL) {
      return( NULL );
  }

  gd->setup = FALSE;

  /*
   * Load all the saved session parameters (or the defaults if none
   * exist).
   */
  // first load all of the default values
    gsysinfo_session_defaults (gd);


    // load values from config file if they exist
    if ( (APPLET_WIDGET (gd->applet)->privcfgpath) && *(APPLET_WIDGET (gd->applet)->privcfgpath)) {
      gsysinfo_session_load (APPLET_WIDGET (gd->applet)->privcfgpath, gd);
    }

  /*
   * area is the drawing area into which each gauge gets drawn
   */
  gd->fixed = (GtkFixed *) gtk_fixed_new();
  gd->loadavg_area = gtk_drawing_area_new ();
  gd->load_area = gtk_drawing_area_new ();
  gd->mem_area = gtk_drawing_area_new ();
  gd->swap_area = gtk_drawing_area_new ();

  gtk_widget_set_usize (gd->loadavg_area, gd->bar_breadth, gd->bar_depth ); 
  gtk_widget_set_usize (gd->load_area, gd->bar_breadth, gd->bar_depth );
  gtk_widget_set_usize (gd->mem_area, gd->bar_breadth, gd->bar_depth );
  gtk_widget_set_usize (gd->swap_area, gd->bar_breadth, gd->bar_depth );

  gtk_fixed_put( gd->fixed, gd->loadavg_area, 0, 0 );
  gtk_fixed_put( gd->fixed, gd->load_area, 0, 0 );
  gtk_fixed_put( gd->fixed, gd->mem_area, 0, 0 );
  gtk_fixed_put( gd->fixed, gd->swap_area, 0, 0 );

  /* Set up the event callbacks for the area. */
  gtk_signal_connect (GTK_OBJECT (gd->loadavg_area), "expose_event",
		      (GtkSignalFunc)expose_handler, gd);
  gtk_signal_connect (GTK_OBJECT (gd->load_area), "expose_event",
		      (GtkSignalFunc)expose_handler, gd);
  gtk_signal_connect (GTK_OBJECT (gd->mem_area), "expose_event",
		      (GtkSignalFunc)expose_handler, gd);
  gtk_signal_connect (GTK_OBJECT (gd->swap_area), "expose_event",
		      (GtkSignalFunc)expose_handler, gd);

  gtk_widget_set_events (gd->loadavg_area, GDK_EXPOSURE_MASK );
  gtk_widget_set_events (gd->load_area, GDK_EXPOSURE_MASK );
  gtk_widget_set_events (gd->mem_area, GDK_EXPOSURE_MASK );
  gtk_widget_set_events (gd->swap_area, GDK_EXPOSURE_MASK );

  /* This will let us know when the panel changes orientation */
  gtk_signal_connect (GTK_OBJECT (gd->applet), "change_orient",
		      GTK_SIGNAL_FUNC (orient_handler),
		      gd);


  applet_widget_add (APPLET_WIDGET (gd->applet), GTK_WIDGET( gd->fixed ) );

  gtk_signal_connect (GTK_OBJECT (gd->applet), "save_session",
		      GTK_SIGNAL_FUNC (gsysinfo_session_save),
		      gd);

  applet_widget_register_stock_callback (APPLET_WIDGET (gd->applet),
					 "about",
					 GNOME_STOCK_MENU_ABOUT,
					 _("About..."),
					 about_cb,
					 gd);

  applet_widget_register_stock_callback (APPLET_WIDGET (gd->applet),
					 "properties",
					 GNOME_STOCK_MENU_PROP,
					 ("Properties..."),
					 properties_window,
					 gd);

  gtk_widget_show_all (gd->applet);

  configure_layout( gd );
  gsysinfo_set_size (gd);

  gsysinfo_create_gc (gd);
  gsysinfo_setup_colors (gd);

  /* Nothing is drawn until this is set. */
  gd->setup = TRUE;

  /* Will schedule a timeout */
  set_timeout( gd );

  return gd->applet;
} /* make_new_gsysinfo_applet */

void set_timeout (GSysInfoData *gd) { 
    // remove the old timeout if there was one
    if (gd->timeout != 0 ) {
      gtk_timeout_remove (gd->timeout);
    }
    gd->timeout = gtk_timeout_add (gd->timeout_t, (GtkFunction) gsysinfo_update, gd);
}

void
destroy_about (GtkWidget *w, gpointer data)
{
  GSysInfoData *gd = data;
} /* destroy_about */

void
about_cb (AppletWidget *widget, gpointer data)
{
  GSysInfoData *gd = data;
  char *authors[2];
  
  authors[0] = "Jason Hildebrand <jdhildeb@undergrad.math.uwaterloo.ca>";
  authors[1] = NULL;

  gd->about_box =
    gnome_about_new (_("GSysInfo Applet"), VERSION,
		     _("Copyright (C) 1999 Jason D. Hildebrand"),
		     (const char **) authors,
	     _("This applet displays the system load average, "
	       "current CPU usage, memory usage, and swap file usage, in the "
	       "same style as the program xsysinfo by Gabor Herr.\n"

	       " \nLOAD AVERAGE:\n"
	       "The load average gauge is subdivided into n sections, where "
	       "n is the next largest whole number greater than the load average.  "
	       "For example, a load of 2.3 would be represented with three sections.  "
	       "The first two are coloured in completely, and the third section is 3/10 "
	       "filled in.\n"

	       " \nCPU USAGE:\n"
	       "The CPU usage gauge is divided into user time, system time, and nice time "
	       "(in that order).  The types of usage are distinguished by colour.\n"

	       " \nMEMORY USAGE:\n"

	       "The memory gauge has two sections: normal memory, and memory used for "
	       "cache and buffers.\n"

	       " \nSWAP USAGE:\n"

	       "The swap space gauge shows how much of the swap file(s) are currently "
	       "in use.\n"

               "\n \nThis applet comes with ABSOLUTELY NO WARRANTY.\n"
               "This is free software, and you are welcome to redistribute it "
               "under certain conditions.  "
               "See the LICENSE file for details.\n" ),
		     NULL);

  gtk_signal_connect (GTK_OBJECT (gd->about_box), "destroy",
		      GTK_SIGNAL_FUNC (destroy_about), gd);

  gtk_widget_show (gd->about_box);
} /* about_cb */


void
gsysinfo_set_size (GSysInfoData * gd)
{
  int n;

  gd->numbars = gd->loadavg_on + gd->load_on + gd->mem_on + gd->swap_on;


  gd->depth = gd->numbars * gd->bar_depth + ( gd->numbars - 1 ) * gd->gap_depth;

  /*
   * If pixmaps have already been allocated, then free them here
   * before creating new ones.  */
  if (gd->pixmap != NULL) {
    gdk_pixmap_unref (gd->pixmap);
  }

  gd->pixmap = gdk_pixmap_new( gd->loadavg_area->window, 
			       gd->bar_breadth, 
			       gd->bar_depth * gd->numbars,
                               gtk_widget_get_visual (gd->loadavg_area)->depth);

  // clear drawing area
  gdk_gc_set_foreground (gd->gc, & (AppColors[ BARGAUGEBG ].val) );
  gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, 0, 0, gd->bar_breadth, gd->bar_depth * gd->numbars);

  if (gd->image != NULL) {
    gdk_image_destroy (gd->image);
  }

  for( n = 0; n < gd->numbars; n++ ) {
      draw_border( gd, 0, n * gd->bar_depth, gd->bar_breadth - 1, ( n + 1 ) * gd->bar_depth - 1 );
  }

} /* gsysinfo_set_size */

void
gsysinfo_create_gc (GSysInfoData * gd)
{
  gd->gc = gdk_gc_new ( gd->applet->window );
  gdk_gc_copy (gd->gc, gd->loadavg_area->style->white_gc);

  gdk_gc_set_function (gd->gc, GDK_COPY);
} /* gsysinfo_create_gc */

void 
gsysinfo_setup_colors(GSysInfoData * gd)
{
  GdkColormap *colormap;
  gint strl, strr, strw, stra, strd;
  int	n;

  /*
   * FIXME: We should use gdk_color_change if we've already set up the
   * colors. 
   */

  colormap = gtk_widget_get_colormap (gd->loadavg_area);

  for( n = 0; n < NUM_COLORS; n++ ) {
      gdk_color_parse( AppColors[ n ].stringval, &( AppColors[n].val ) );
      gdk_color_alloc( colormap, &( AppColors[n].val ) );
  }

} 

