#include "alarm_applet.h"

#define free_alarm_mem(alarm) \
	 if((alarm)->msg_set && ((alarm)->msg_txt != NULL))\
		g_string_free((alarm)->msg_txt, TRUE);\
	 if((alarm)->command_set && ((alarm)->command != NULL))\
		g_string_free((alarm)->command, TRUE);\
	 g_free((alarm));


typedef struct _alarm_store AlarmStore;

struct _alarm_store
{
   GList* alarms;
   guint num_alarms;
   gchar* sound_file;
   gchar* anim_file;
   guint num_persistant;
};


AlarmStore* store = NULL;




Alarm* new_alarm(void)
{
   static guint num = 0;
   Alarm* a = g_new0(Alarm, 1);
   a->msg_txt = NULL;
   a->command = NULL;
   a->hour = -1;
   a->min = -1;
   a->alarm_type = ALARM_DIALOG;
   a->msg_set = FALSE;
   a->command_set = FALSE;
   a->modal_dialog = FALSE;
   a->ref_count = num++;
   a->persistant = FALSE;

   return a;
}


void init_alarm_store(void)
{
   store = g_new(AlarmStore, 1);
   store->alarms = NULL;
   store->num_alarms = 0;
   store->num_persistant = 0;
}


void set_sound_file(gchar* filename)
{
   store->sound_file = filename;
}
   

void show_msg_dialog(gchar* msg, gboolean modal)
{
   GtkWidget* dialog;
   
   dialog = gnome_message_box_new(_(msg), GNOME_MESSAGE_BOX_INFO,
								  GNOME_STOCK_BUTTON_OK, NULL);
   if(modal)
	 gnome_message_box_set_modal(GNOME_MESSAGE_BOX(dialog));

   gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0,
									  GTK_SIGNAL_FUNC(gtk_widget_destroy),
									  GTK_OBJECT(dialog));
   gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
   gtk_widget_show(dialog);
}



gint compare_alarms(gconstpointer alarm1, gconstpointer alarm2)
{
   Alarm* a1;
   Alarm* a2;
   
   a1 = (Alarm*)alarm1;
   a2 = (Alarm*)alarm2;
 
   if(!a1 || !a2)
	 return 0;
   
   if(a1->hour > a2->hour)
	 return 1;
   if(a2->hour > a1->hour)
	 return -1;
   if(a2->hour == a1->hour)
   {
	  if(a1->min > a2->min)
		return 1;
	  else if(a2->min > a1->min)
		return -1;
	  else if(a1->min == a2->min)
		return 0;
   }
   
   return 0;
}


void destroy_alarms(void)
{
   GList* list;
   gint i = 0;
   for(list = store->alarms; list; list = g_list_next(list))
   {
	  Alarm* a = list->data;
	  dump_alarm("destroy_alarms", a);
	  free_alarm_mem(a);
	  i++;
   }
   
#ifdef MY_DEBUG
   if(i != store->num_alarms)
	 fprintf(stderr, "Failed to save all alarms: num_alarms = %d i = %d\n", store->num_alarms, i);
#endif
   g_list_free(store->alarms);
   g_free(store);
}



void save_persistant_alarms(gchar* cfgpath)
{
   gint i = 0;
   GList* list = store->alarms;
   
   gnome_config_push_prefix(cfgpath);
   gnome_config_set_int("persistant/num", store->num_persistant);
   
   /* We've no alarms to save so bail out here */
   if(store->num_persistant <= 0)
	 goto finish_up;
   
   while(list)
   {
	  Alarm* alarm = list->data;
	  if(alarm->persistant)
	  {
		 /* Kludgy way of doing this. There is probably a better
		  * way of 'realloc'ing the memory rather than creating/freeing
		  * all the time. Not totally sure i need to.
		  */
		 gchar* cfgdata;
		 gchar* prefix = g_strdup_printf("alarm%d/", i++);
		 cfgdata = g_strconcat(prefix, "hour", NULL);
		 gnome_config_set_int(cfgdata, alarm->hour);
		 g_free(cfgdata);
		 cfgdata = g_strconcat(prefix, "min", NULL);
		 gnome_config_set_int(cfgdata, alarm->min);
		 g_free(cfgdata);
		 cfgdata = g_strconcat(prefix, "type", NULL);
		 gnome_config_set_int(cfgdata, alarm->alarm_type);
		 g_free(cfgdata);
		 cfgdata = g_strconcat(prefix, "msg_set", NULL);
		 gnome_config_set_bool(cfgdata, alarm->msg_set);
		 g_free(cfgdata);
		 cfgdata = g_strconcat(prefix, "msg", NULL);
		 gnome_config_set_string(cfgdata, alarm->msg_set ? alarm->msg_txt->str : "");
		 g_free(cfgdata);
		 cfgdata = g_strconcat(prefix, "command_set", NULL);
		 gnome_config_set_bool(cfgdata, alarm->command_set);
		 g_free(cfgdata);
		 cfgdata = g_strconcat(prefix, "command", NULL);
		 gnome_config_set_string(cfgdata, alarm->command_set ? alarm->command->str : "");
		 g_free(cfgdata);
		 cfgdata = g_strconcat(prefix, "modal", NULL);
		 gnome_config_set_bool(cfgdata, alarm->modal_dialog);
		 g_free(cfgdata);
		 g_free(prefix);
	  }
	  list = list->next;
   }   
#ifdef MY_DEBUG
   if(i != store->num_persistant)
	 fprintf(stderr, "Failed to save all alarms num = %d persis = %d\n",
			 i, store->num_persistant);
#endif
   ddebug("Saved %d persistant alarms\n", i);
   /* 
	* Not sure i should do this here as
	* well as in the save_session_info function ??
	* Proably a good idea to keep everything synced.
	*/
finish_up:
   gnome_config_pop_prefix();
   gnome_config_sync();
   gnome_config_drop_all();
}


	  
void load_persistant_alarms(gchar* cfgpath)
{
   gint i;
   gint read;
   gchar* prefix;
   
   ddebug("Loading persistant alarms\n");
   gnome_config_push_prefix(cfgpath);
   read = gnome_config_get_int("persistant/num=0");
   if(read <= 0)
   {
	  gnome_config_pop_prefix();
	  return;
   }
   
   for(i = 0; i < read; i++)
   {
	  gchar* cfgdata;
	  gboolean bool;
	  gchar* tmp;
	  gchar* prefix;
	  Alarm* a = new_alarm();
	  a->persistant = TRUE;
	  prefix = g_strdup_printf("alarm%d/", i);
	  cfgdata = g_strconcat(prefix, "hour=0", NULL);
	  a->hour = gnome_config_get_int(cfgdata);
	  g_free(cfgdata);
	  cfgdata = g_strconcat(prefix, "min=0", NULL);
	  a->min = gnome_config_get_int(cfgdata);
	  g_free(cfgdata);
	  cfgdata = g_strconcat(prefix, "type=0", NULL);
	  a->alarm_type = gnome_config_get_int(cfgdata);
	  g_free(cfgdata);
	  cfgdata = g_strconcat(prefix, "msg_set=false", NULL);
	  bool = gnome_config_get_bool(cfgdata);
	  g_free(cfgdata);
	  cfgdata = g_strconcat(prefix, "msg=null", NULL);
	  tmp = gnome_config_get_string(cfgdata);
	  g_free(cfgdata);
	  if(bool)
		alarm_set_msg(a, tmp);
	  bool = FALSE;
	  cfgdata = g_strconcat(prefix, "command_set=false", NULL);
	  bool = gnome_config_get_bool(cfgdata);
	  g_free(cfgdata);
	  cfgdata = g_strconcat(prefix, "command=null", NULL);
	  tmp = gnome_config_get_string(cfgdata);
	  g_free(cfgdata);
	  if(bool)
		alarm_set_command(a, tmp);
	  cfgdata = g_strconcat(prefix, "modal=false", NULL);
	  a->modal_dialog = gnome_config_get_bool(cfgdata);
	  g_free(cfgdata);
	  g_free(prefix);
	  
	  add_alarm(a);
   }
   
   gnome_config_pop_prefix();
   ddebug("Loaded %d persistant alarms\n", i);
}


void add_alarm(Alarm* a)
{
   dump_alarm("add_alarm", a);
   if(store->num_alarms >= MAX_ALARMS)
   {
	  show_msg_dialog("You have reached the maximum number"
					  " of alarms allowed\nPlease delete some"
					  " of the current alarms or wait for one"
					  " of them to expire", TRUE);
	  return;
   }
   
   store->alarms = g_list_insert_sorted(store->alarms, a, compare_alarms);
   store->num_alarms++;
   if(a->persistant)
	 store->num_persistant++;
}



void delete_alarm(Alarm* a)
{
   dump_alarm("delete_alarm", a);
   store->alarms = g_list_remove(store->alarms, a);
   store->num_alarms--;
   if(a->persistant)
	 store->num_persistant--;
   free_alarm_mem(a);
}


void activate_alarm(Alarm* a)
{
   dump_alarm("activate_alarm", a);
   switch(a->alarm_type)
   {
	  case ALARM_SOUND:
	  {
		 set_alarm_state(ALARM_STATE_SOUND_RING);
		 play_sound_alarm();
	  }
	  break;
	  case ALARM_VISUAL:
	  {
		 set_alarm_state(ALARM_STATE_VISUAL_RING);
	  }
	  break;
	  case ALARM_DIALOG:
	  {
		 set_alarm_state(ALARM_STATE_NORMAL);
		 if(a->msg_set)
		   show_msg_dialog(a->msg_txt->str, a->modal_dialog);
		 else
		 {
			gchar* msg = g_strdup_printf("An alarm at %.2d:%.2d hrs has been called", a->hour, a->min);
			show_msg_dialog(msg, a->modal_dialog);
			g_free(msg);
		 }
	  }
	  break;
	  case ALARM_COMMAND:
	  {
		 set_alarm_state(ALARM_STATE_NORMAL);
		 if(a->command_set)
		 {
			if(!gnome_execute_shell(NULL, a->command->str))
			{
			   gchar* str = g_strdup_printf("Unable to run command: \"%s\"", a->command->str);
			   show_msg_dialog(str, FALSE);
			   g_free(str);
			}
		 }
	  }
   }
}


void alarm_set_msg(Alarm* a, gchar* msg)
{
   if((a == NULL) || (msg == NULL))
	 return;
   
   if(!strcmp(msg, ""))
	 return;
   
   if(a->msg_set)
   {
	  g_string_assign(a->msg_txt, msg);
	  return;
   }
   
   a->msg_txt = g_string_new(msg);
   a->msg_set = TRUE;
   dump_alarm("alarm_set_msg", a);
}


void alarm_set_command(Alarm* a, gchar* cmd)
{
   if((a == NULL) || (cmd == NULL))
	 return;
 
   if(!strcmp(cmd, ""))
	 return;
   
   if(a->command_set)
   {
	  g_string_assign(a->command, cmd);
	  return;
   }
   
   a->command = g_string_new(cmd);
   a->command_set = TRUE;
   dump_alarm("alarm_set_command", a);
}


void delete_alarm_with_id(gint id)
{
   GList* list = store->alarms;
   ddebug("Called delete_alarm_with_id\n");
   while(list)
   {
	  Alarm* a = list->data;
	  if(a->ref_count == id)
	  {
		 if(!a->persistant)
		   delete_alarm(a);
		 break;
	  }
	  list = list->next;
   }
}



Alarm* find_alarm_with_id(gint id)
{
   GList* list = store->alarms;
   ddebug("Called find_alarm_with_id\n");
   while(list)
   {
	  Alarm* a = list->data;
	  if(a->ref_count == id)
	  {
		 ddebug("Returning alarm with id %d\n", a->ref_count);
		 return (Alarm*)a;
	  }
	  list = list->next;
   }
   return NULL;
}



gint timeout_func(gpointer cbdata)
{
   static int last_hour = 0;
   static int last_min = 0;
   int hour;
   int min;
   struct tm* local;
   time_t now;
   /* Store a reference to the head of the list */
   GList* list = store->alarms;
   gint i = 0;
   Alarm* a;
   
   ddebug("Called timeout\n");
   if(list == NULL)
	 return TRUE;
   
   ddebug("Store->alarms != NULL\n");
   now = time(0);
   local = localtime(&now);
   
   hour = local->tm_hour;
   min = local->tm_min;
   
   /* Stop persistant alarms being executed multiple times
	* if we check the list more than once a minute
	*/
   if((hour == last_hour) && (min == last_min))
	 return TRUE;
   
   
   ddebug("Last hour (%d) != hour (%d) and last min (%d) != min (%d)\n",
		 last_hour, hour, last_min, min);
   
    
   /* Try and allow multiple alarms for the same
	* time to be called without causing a problem
	*/
  
   while(list)
   {
	  GList* link = list;
	  GList* next = list->next;
	  Alarm* a = list->data;
	  
	  ddebug("Pass no %d\n", ++i);
	  if((a->hour == hour) && (a->min == min))
	  {
		 activate_alarm(a);
		 if(!a->persistant)
		 {
			store->alarms = g_list_remove_link(store->alarms, link);
			free_alarm_mem(a);
			g_list_free(link);
			store->num_alarms--;
			list = next;
			continue;
		 }
	  }
	  list = list->next;
   }
   
   
   last_hour = hour;
   last_min = min;
  
   return TRUE;
}



GList* get_alarm_list(void)
{
   return store->alarms;
}
