/*
 * Relay calls helper routines
 *
 * Copyright 1993 Robert J. Amstadt
 * Copyright 1995 Martin von Loewis
 * Copyright 1995, 1996, 1997 Alexandre Julliard
 * Copyright 1997 Eric Youngdale
 * Copyright 1999 Ulrich Weigand
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"
#include "wine/port.h"

#include <ctype.h>

#include "thread.h"
#include "stackframe.h"

#include "build.h"

#if defined(__GNUC__) && !defined(__svr4__)
static const int use_stabs = 1;
#else
static const int use_stabs = 0;
#endif

#ifdef __i386__

static void function_header( FILE *outfile, const char *name )
{
    fprintf( outfile, "\n\t.align %d\n", get_alignment(4) );
    if (use_stabs)
        fprintf( outfile, "\t.stabs \"%s:F1\",36,0,0," __ASM_NAME("%s") "\n", name, name);
    fprintf( outfile, "\t" __ASM_FUNC("%s") "\n", name );
    fprintf( outfile, "\t.globl " __ASM_NAME("%s") "\n", name );
    fprintf( outfile, __ASM_NAME("%s") ":\n", name );
}


/*******************************************************************
 *         BuildCallFrom16Core
 *
 * This routine builds the core routines used in 16->32 thunks:
 * CallFrom16Word, CallFrom16Long, CallFrom16Register, and CallFrom16Thunk.
 *
 * These routines are intended to be called via a far call (with 32-bit
 * operand size) from 16-bit code.  The 16-bit code stub must push %bp,
 * the 32-bit entry point to be called, and the argument conversion
 * routine to be used (see stack layout below).
 *
 * The core routine completes the STACK16FRAME on the 16-bit stack and
 * switches to the 32-bit stack.  Then, the argument conversion routine
 * is called; it gets passed the 32-bit entry point and a pointer to the
 * 16-bit arguments (on the 16-bit stack) as parameters. (You can either
 * use conversion routines automatically generated by BuildCallFrom16,
 * or write your own for special purposes.)
 *
 * The conversion routine must call the 32-bit entry point, passing it
 * the converted arguments, and return its return value to the core.
 * After the conversion routine has returned, the core switches back
 * to the 16-bit stack, converts the return value to the DX:AX format
 * (CallFrom16Long), and returns to the 16-bit call stub.  All parameters,
 * including %bp, are popped off the stack.
 *
 * The 16-bit call stub now returns to the caller, popping the 16-bit
 * arguments if necessary (pascal calling convention).
 *
 * In the case of a 'register' function, CallFrom16Register fills a
 * CONTEXT86 structure with the values all registers had at the point
 * the first instruction of the 16-bit call stub was about to be
 * executed.  A pointer to this CONTEXT86 is passed as third parameter
 * to the argument conversion routine, which typically passes it on
 * to the called 32-bit entry point.
 *
 * CallFrom16Thunk is a special variant used by the implementation of
 * the Win95 16->32 thunk functions C16ThkSL and C16ThkSL01 and is
 * implemented as follows:
 * On entry, the EBX register is set up to contain a flat pointer to the
 * 16-bit stack such that EBX+22 points to the first argument.
 * Then, the entry point is called, while EBP is set up to point
 * to the return address (on the 32-bit stack).
 * The called function returns with CX set to the number of bytes
 * to be popped of the caller's stack.
 *
 * Stack layout upon entry to the core routine (STACK16FRAME):
 *  ...           ...
 * (sp+24) word   first 16-bit arg
 * (sp+22) word   cs
 * (sp+20) word   ip
 * (sp+18) word   bp
 * (sp+14) long   32-bit entry point (reused for Win16 mutex recursion count)
 * (sp+12) word   ip of actual entry point (necessary for relay debugging)
 * (sp+8)  long   relay (argument conversion) function entry point
 * (sp+4)  long   cs of 16-bit entry point
 * (sp)    long   ip of 16-bit entry point
 *
 * Added on the stack:
 * (sp-2)  word   saved gs
 * (sp-4)  word   saved fs
 * (sp-6)  word   saved es
 * (sp-8)  word   saved ds
 * (sp-12) long   saved ebp
 * (sp-16) long   saved ecx
 * (sp-20) long   saved edx
 * (sp-24) long   saved previous stack
 */
static void BuildCallFrom16Core( FILE *outfile, int reg_func, int thunk, int short_ret )
{
    char *name = thunk? "thunk" : reg_func? "regs" : short_ret? "word" : "long";

    /* Function header */
    if (thunk) function_header( outfile, "__wine_call_from_16_thunk" );
    else if (reg_func) function_header( outfile, "__wine_call_from_16_regs" );
    else if (short_ret) function_header( outfile, "__wine_call_from_16_word" );
    else function_header( outfile, "__wine_call_from_16_long" );

    /* Create STACK16FRAME (except STACK32FRAME link) */
    fprintf( outfile, "\tpushw %%gs\n" );
    fprintf( outfile, "\tpushw %%fs\n" );
    fprintf( outfile, "\tpushw %%es\n" );
    fprintf( outfile, "\tpushw %%ds\n" );
    fprintf( outfile, "\tpushl %%ebp\n" );
    fprintf( outfile, "\tpushl %%ecx\n" );
    fprintf( outfile, "\tpushl %%edx\n" );

    /* Save original EFlags register */
    fprintf( outfile, "\tpushfl\n" );

    if ( UsePIC )
    {
        /* Get Global Offset Table into %ecx */
        fprintf( outfile, "\tcall .L__wine_call_from_16_%s.getgot1\n", name );
        fprintf( outfile, ".L__wine_call_from_16_%s.getgot1:\n", name );
        fprintf( outfile, "\tpopl %%ecx\n" );
        fprintf( outfile, "\taddl $_GLOBAL_OFFSET_TABLE_+[.-.L__wine_call_from_16_%s.getgot1], %%ecx\n", name );
    }

    if (UsePIC)
    {
        fprintf( outfile, "\t.byte 0x2e\n\tmovl " __ASM_NAME("CallTo16_DataSelector@GOT") "(%%ecx), %%edx\n" );
        fprintf( outfile, "\t.byte 0x2e\n\tmovl (%%edx), %%edx\n" );
    }
    else
        fprintf( outfile, "\t.byte 0x2e\n\tmovl " __ASM_NAME("CallTo16_DataSelector") ",%%edx\n" );

    /* Load 32-bit segment registers */
#ifdef __svr4__
    fprintf( outfile, "\tdata16\n");
#endif
    fprintf( outfile, "\tmovw %%dx, %%ds\n" );
#ifdef __svr4__
    fprintf( outfile, "\tdata16\n");
#endif
    fprintf( outfile, "\tmovw %%dx, %%es\n" );

    if ( UsePIC )
    {
        fprintf( outfile, "\tmovl " __ASM_NAME("SYSLEVEL_Win16CurrentTeb@GOT") "(%%ecx), %%edx\n" );
        fprintf( outfile, "\tmovw (%%edx), %%fs\n" );
    }
    else
        fprintf( outfile, "\tmovw " __ASM_NAME("SYSLEVEL_Win16CurrentTeb") ", %%fs\n" );

    /* Get address of wine_ldt_copy array into %ecx */
    if ( UsePIC )
        fprintf( outfile, "\tmovl " __ASM_NAME("wine_ldt_copy@GOT") "(%%ecx), %%ecx\n" );
    else
        fprintf( outfile, "\tmovl $" __ASM_NAME("wine_ldt_copy") ", %%ecx\n" );

    /* Translate STACK16FRAME base to flat offset in %edx */
    fprintf( outfile, "\tmovw %%ss, %%dx\n" );
    fprintf( outfile, "\tandl $0xfff8, %%edx\n" );
    fprintf( outfile, "\tshrl $1, %%edx\n" );
    fprintf( outfile, "\tmovl (%%ecx,%%edx), %%edx\n" );
    fprintf( outfile, "\tmovzwl %%sp, %%ebp\n" );
    fprintf( outfile, "\tleal (%%ebp,%%edx), %%edx\n" );

    /* Get saved flags into %ecx */
    fprintf( outfile, "\tpopl %%ecx\n" );

    /* Get the 32-bit stack pointer from the TEB and complete STACK16FRAME */
    fprintf( outfile, "\t.byte 0x64\n\tmovl (%d), %%ebp\n", STACKOFFSET );
    fprintf( outfile, "\tpushl %%ebp\n" );

    /* Switch stacks */
#ifdef __svr4__
    fprintf( outfile,"\tdata16\n");
#endif
    fprintf( outfile, "\t.byte 0x64\n\tmovw %%ss, (%d)\n", STACKOFFSET + 2 );
    fprintf( outfile, "\t.byte 0x64\n\tmovw %%sp, (%d)\n", STACKOFFSET );
    fprintf( outfile, "\tpushl %%ds\n" );
    fprintf( outfile, "\tpopl %%ss\n" );
    fprintf( outfile, "\tmovl %%ebp, %%esp\n" );
    fprintf( outfile, "\taddl $%d, %%ebp\n", STRUCTOFFSET(STACK32FRAME, ebp) );


    /* At this point:
       STACK16FRAME is completely set up
       DS, ES, SS: flat data segment
       FS: current TEB
       ESP: points to last STACK32FRAME
       EBP: points to ebp member of last STACK32FRAME
       EDX: points to current STACK16FRAME
       ECX: contains saved flags
       all other registers: unchanged */

    /* Special case: C16ThkSL stub */
    if ( thunk )
    {
        /* Set up registers as expected and call thunk */
        fprintf( outfile, "\tleal %d(%%edx), %%ebx\n", sizeof(STACK16FRAME)-22 );
        fprintf( outfile, "\tleal -4(%%esp), %%ebp\n" );

        fprintf( outfile, "\tcall *%d(%%edx)\n", STACK16OFFSET(entry_point) );

        /* Switch stack back */
        fprintf( outfile, "\t.byte 0x64\n\tmovw (%d), %%ss\n", STACKOFFSET+2 );
        fprintf( outfile, "\t.byte 0x64\n\tmovzwl (%d), %%esp\n", STACKOFFSET );
        fprintf( outfile, "\t.byte 0x64\n\tpopl (%d)\n", STACKOFFSET );

        /* Restore registers and return directly to caller */
        fprintf( outfile, "\taddl $8, %%esp\n" );
        fprintf( outfile, "\tpopl %%ebp\n" );
        fprintf( outfile, "\tpopw %%ds\n" );
        fprintf( outfile, "\tpopw %%es\n" );
        fprintf( outfile, "\tpopw %%fs\n" );
        fprintf( outfile, "\tpopw %%gs\n" );
        fprintf( outfile, "\taddl $20, %%esp\n" );

        fprintf( outfile, "\txorb %%ch, %%ch\n" );
        fprintf( outfile, "\tpopl %%ebx\n" );
        fprintf( outfile, "\taddw %%cx, %%sp\n" );
        fprintf( outfile, "\tpush %%ebx\n" );

        fprintf( outfile, "\t.byte 0x66\n" );
        fprintf( outfile, "\tlret\n" );

        return;
    }


    /* Build register CONTEXT */
    if ( reg_func )
    {
        fprintf( outfile, "\tsubl $%d, %%esp\n", sizeof(CONTEXT86) );

        fprintf( outfile, "\tmovl %%ecx, %d(%%esp)\n", CONTEXTOFFSET(EFlags) );

        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(Eax) );
        fprintf( outfile, "\tmovl %%ebx, %d(%%esp)\n", CONTEXTOFFSET(Ebx) );
        fprintf( outfile, "\tmovl %%esi, %d(%%esp)\n", CONTEXTOFFSET(Esi) );
        fprintf( outfile, "\tmovl %%edi, %d(%%esp)\n", CONTEXTOFFSET(Edi) );

        fprintf( outfile, "\tmovl %d(%%edx), %%eax\n", STACK16OFFSET(ebp) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(Ebp) );
        fprintf( outfile, "\tmovl %d(%%edx), %%eax\n", STACK16OFFSET(ecx) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(Ecx) );
        fprintf( outfile, "\tmovl %d(%%edx), %%eax\n", STACK16OFFSET(edx) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(Edx) );

        fprintf( outfile, "\tmovzwl %d(%%edx), %%eax\n", STACK16OFFSET(ds) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(SegDs) );
        fprintf( outfile, "\tmovzwl %d(%%edx), %%eax\n", STACK16OFFSET(es) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(SegEs) );
        fprintf( outfile, "\tmovzwl %d(%%edx), %%eax\n", STACK16OFFSET(fs) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(SegFs) );
        fprintf( outfile, "\tmovzwl %d(%%edx), %%eax\n", STACK16OFFSET(gs) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(SegGs) );

        fprintf( outfile, "\tmovzwl %d(%%edx), %%eax\n", STACK16OFFSET(cs) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(SegCs) );
        fprintf( outfile, "\tmovzwl %d(%%edx), %%eax\n", STACK16OFFSET(ip) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(Eip) );

        fprintf( outfile, "\t.byte 0x64\n\tmovzwl (%d), %%eax\n", STACKOFFSET+2 );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(SegSs) );
        fprintf( outfile, "\t.byte 0x64\n\tmovzwl (%d), %%eax\n", STACKOFFSET );
        fprintf( outfile, "\taddl $%d, %%eax\n", STACK16OFFSET(ip) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(Esp) );
#if 0
        fprintf( outfile, "\tfsave %d(%%esp)\n", CONTEXTOFFSET(FloatSave) );
#endif

        /* Push address of CONTEXT86 structure -- popped by the relay routine */
        fprintf( outfile, "\tpushl %%esp\n" );
    }


    /* Print debug info before call */
    if ( debugging )
    {
        if ( UsePIC )
        {
            fprintf( outfile, "\tpushl %%ebx\n" );

            /* Get Global Offset Table into %ebx (for PLT call) */
            fprintf( outfile, "\tcall .L__wine_call_from_16_%s.getgot2\n", name );
            fprintf( outfile, ".L__wine_call_from_16_%s.getgot2:\n", name );
            fprintf( outfile, "\tpopl %%ebx\n" );
            fprintf( outfile, "\taddl $_GLOBAL_OFFSET_TABLE_+[.-.L__wine_call_from_16_%s.getgot2], %%ebx\n", name );
        }

        fprintf( outfile, "\tpushl %%edx\n" );
        if ( reg_func )
            fprintf( outfile, "\tleal -%d(%%ebp), %%eax\n\tpushl %%eax\n",
                              sizeof(CONTEXT) + STRUCTOFFSET(STACK32FRAME, ebp) );
        else
            fprintf( outfile, "\tpushl $0\n" );

        if ( UsePIC )
            fprintf( outfile, "\tcall " __ASM_NAME("RELAY_DebugCallFrom16@PLT") "\n ");
        else
            fprintf( outfile, "\tcall " __ASM_NAME("RELAY_DebugCallFrom16") "\n ");

        fprintf( outfile, "\tpopl %%edx\n" );
        fprintf( outfile, "\tpopl %%edx\n" );

        if ( UsePIC )
            fprintf( outfile, "\tpopl %%ebx\n" );
    }

    /* Call relay routine (which will call the API entry point) */
    fprintf( outfile, "\tleal %d(%%edx), %%eax\n", sizeof(STACK16FRAME) );
    fprintf( outfile, "\tpushl %%eax\n" );
    fprintf( outfile, "\tpushl %d(%%edx)\n", STACK16OFFSET(entry_point) );
    fprintf( outfile, "\tcall *%d(%%edx)\n", STACK16OFFSET(relay) );

    /* Print debug info after call */
    if ( debugging )
    {
        if ( UsePIC )
        {
            fprintf( outfile, "\tpushl %%ebx\n" );

            /* Get Global Offset Table into %ebx (for PLT call) */
            fprintf( outfile, "\tcall .L__wine_call_from_16_%s.getgot3\n", name );
            fprintf( outfile, ".L__wine_call_from_16_%s.getgot3:\n", name );
            fprintf( outfile, "\tpopl %%ebx\n" );
            fprintf( outfile, "\taddl $_GLOBAL_OFFSET_TABLE_+[.-.L__wine_call_from_16_%s.getgot3], %%ebx\n", name );
        }

        fprintf( outfile, "\tpushl %%eax\n" );
        if ( reg_func )
            fprintf( outfile, "\tleal -%d(%%ebp), %%eax\n\tpushl %%eax\n",
                              sizeof(CONTEXT) + STRUCTOFFSET(STACK32FRAME, ebp) );
        else
            fprintf( outfile, "\tpushl $0\n" );

        if ( UsePIC )
            fprintf( outfile, "\tcall " __ASM_NAME("RELAY_DebugCallFrom16Ret@PLT") "\n ");
        else
            fprintf( outfile, "\tcall " __ASM_NAME("RELAY_DebugCallFrom16Ret") "\n ");

        fprintf( outfile, "\tpopl %%eax\n" );
        fprintf( outfile, "\tpopl %%eax\n" );

        if ( UsePIC )
            fprintf( outfile, "\tpopl %%ebx\n" );
    }


    if ( reg_func )
    {
        fprintf( outfile, "\tmovl %%esp, %%ebx\n" );

        /* Switch stack back */
        fprintf( outfile, "\t.byte 0x64\n\tmovw (%d), %%ss\n", STACKOFFSET+2 );
        fprintf( outfile, "\t.byte 0x64\n\tmovzwl (%d), %%esp\n", STACKOFFSET );
        fprintf( outfile, "\t.byte 0x64\n\tpopl (%d)\n", STACKOFFSET );

        /* Get return address to CallFrom16 stub */
        fprintf( outfile, "\taddw $%d, %%sp\n", STACK16OFFSET(callfrom_ip)-4 );
        fprintf( outfile, "\tpopl %%eax\n" );
        fprintf( outfile, "\tpopl %%edx\n" );

        /* Restore all registers from CONTEXT */
        fprintf( outfile, "\tmovw %d(%%ebx), %%ss\n", CONTEXTOFFSET(SegSs) );
        fprintf( outfile, "\tmovl %d(%%ebx), %%esp\n", CONTEXTOFFSET(Esp) );
        fprintf( outfile, "\taddl $4, %%esp\n" );  /* room for final return address */

        fprintf( outfile, "\tpushw %d(%%ebx)\n", CONTEXTOFFSET(SegCs) );
        fprintf( outfile, "\tpushw %d(%%ebx)\n", CONTEXTOFFSET(Eip) );
        fprintf( outfile, "\tpushl %%edx\n" );
        fprintf( outfile, "\tpushl %%eax\n" );
        fprintf( outfile, "\tpushl %d(%%ebx)\n", CONTEXTOFFSET(EFlags) );
        fprintf( outfile, "\tpushl %d(%%ebx)\n", CONTEXTOFFSET(SegDs) );

        fprintf( outfile, "\tmovw %d(%%ebx), %%es\n", CONTEXTOFFSET(SegEs) );
        fprintf( outfile, "\tmovw %d(%%ebx), %%fs\n", CONTEXTOFFSET(SegFs) );
        fprintf( outfile, "\tmovw %d(%%ebx), %%gs\n", CONTEXTOFFSET(SegGs) );

        fprintf( outfile, "\tmovl %d(%%ebx), %%ebp\n", CONTEXTOFFSET(Ebp) );
        fprintf( outfile, "\tmovl %d(%%ebx), %%esi\n", CONTEXTOFFSET(Esi) );
        fprintf( outfile, "\tmovl %d(%%ebx), %%edi\n", CONTEXTOFFSET(Edi) );
        fprintf( outfile, "\tmovl %d(%%ebx), %%eax\n", CONTEXTOFFSET(Eax) );
        fprintf( outfile, "\tmovl %d(%%ebx), %%edx\n", CONTEXTOFFSET(Edx) );
        fprintf( outfile, "\tmovl %d(%%ebx), %%ecx\n", CONTEXTOFFSET(Ecx) );
        fprintf( outfile, "\tmovl %d(%%ebx), %%ebx\n", CONTEXTOFFSET(Ebx) );

        fprintf( outfile, "\tpopl %%ds\n" );
        fprintf( outfile, "\tpopfl\n" );
        fprintf( outfile, "\tlret\n" );
    }
    else
    {
        /* Switch stack back */
        fprintf( outfile, "\t.byte 0x64\n\tmovw (%d), %%ss\n", STACKOFFSET+2 );
        fprintf( outfile, "\t.byte 0x64\n\tmovzwl (%d), %%esp\n", STACKOFFSET );
        fprintf( outfile, "\t.byte 0x64\n\tpopl (%d)\n", STACKOFFSET );

        /* Restore registers */
        fprintf( outfile, "\tpopl %%edx\n" );
        fprintf( outfile, "\tpopl %%ecx\n" );
        fprintf( outfile, "\tpopl %%ebp\n" );
        fprintf( outfile, "\tpopw %%ds\n" );
        fprintf( outfile, "\tpopw %%es\n" );
        fprintf( outfile, "\tpopw %%fs\n" );
        fprintf( outfile, "\tpopw %%gs\n" );

        /* Prepare return value and set flags accordingly */
        if ( !short_ret )
            fprintf( outfile, "\tshldl $16, %%eax, %%edx\n" );
        fprintf( outfile, "\torl %%eax, %%eax\n" );

        /* Return to return stub which will return to caller */
        fprintf( outfile, "\tlret $12\n" );
    }
}


/*******************************************************************
 *         BuildCallTo16Core
 *
 * This routine builds the core routines used in 32->16 thunks:
 *
 *   extern void WINAPI wine_call_to_16_word( SEGPTR target, int nb_args );
 *   extern void WINAPI wine_call_to_16_long( SEGPTR target, int nb_args );
 *   extern void WINAPI wine_call_to_16_regs_short( const CONTEXT86 *context, int nb_args );
 *   extern void WINAPI wine_call_to_16_regs_long ( const CONTEXT86 *context, int nb_args );
 *
 * These routines can be called directly from 32-bit code.
 *
 * All routines expect that the 16-bit stack contents (arguments) were
 * already set up by the caller; nb_args must contain the number of bytes
 * to be conserved.  The 16-bit SS:SP will be set accordinly.
 *
 * All other registers are either taken from the CONTEXT86 structure
 * or else set to default values.  The target routine address is either
 * given directly or taken from the CONTEXT86.
 *
 * If you want to call a 16-bit routine taking only standard argument types
 * (WORD and LONG), you can also have an appropriate argument conversion
 * stub automatically generated (see BuildCallTo16); you'd then call this
 * stub, which in turn would prepare the 16-bit stack and call the appropiate
 * core routine.
 *
 */
static void BuildCallTo16Core( FILE *outfile, int short_ret, int reg_func )
{
    char *name = reg_func == 2 ? "regs_long" :
                 reg_func == 1 ? "regs_short" :
                 short_ret? "word" : "long";

    /* Function header */
    if (reg_func == 2) function_header( outfile, "wine_call_to_16_regs_long" );
    else if (reg_func == 1) function_header( outfile, "wine_call_to_16_regs_short" );
    else if (short_ret) function_header( outfile, "wine_call_to_16_word" );
    else function_header( outfile, "wine_call_to_16_long" );

    /* Function entry sequence */
    fprintf( outfile, "\tpushl %%ebp\n" );
    fprintf( outfile, "\tmovl %%esp, %%ebp\n" );

    /* Save the 32-bit registers */
    fprintf( outfile, "\tpushl %%ebx\n" );
    fprintf( outfile, "\tpushl %%ecx\n" );
    fprintf( outfile, "\tpushl %%edx\n" );
    fprintf( outfile, "\tpushl %%esi\n" );
    fprintf( outfile, "\tpushl %%edi\n" );

    if ( UsePIC )
    {
        /* Get Global Offset Table into %ebx */
        fprintf( outfile, "\tcall .Lwine_call_to_16_%s.getgot1\n", name );
        fprintf( outfile, ".Lwine_call_to_16_%s.getgot1:\n", name );
        fprintf( outfile, "\tpopl %%ebx\n" );
        fprintf( outfile, "\taddl $_GLOBAL_OFFSET_TABLE_+[.-.Lwine_call_to_16_%s.getgot1], %%ebx\n", name );
    }

    /* Print debugging info */
    if (debugging)
    {
        /* Push flags, number of arguments, and target */
        fprintf( outfile, "\tpushl $%d\n", reg_func );
        fprintf( outfile, "\tpushl 12(%%ebp)\n" );
        fprintf( outfile, "\tpushl  8(%%ebp)\n" );

        if ( UsePIC )
            fprintf( outfile, "\tcall " __ASM_NAME("RELAY_DebugCallTo16@PLT") "\n" );
        else
            fprintf( outfile, "\tcall " __ASM_NAME("RELAY_DebugCallTo16") "\n" );

        fprintf( outfile, "\taddl $12, %%esp\n" );
    }

    /* Enter Win16 Mutex */
    if ( UsePIC )
        fprintf( outfile, "\tcall " __ASM_NAME("_EnterWin16Lock@PLT") "\n" );
    else
        fprintf( outfile, "\tcall " __ASM_NAME("_EnterWin16Lock") "\n" );

    /* Setup exception frame */
    fprintf( outfile, "\t.byte 0x64\n\tpushl (%d)\n", STACKOFFSET );
    if (UsePIC)
        fprintf( outfile, "\tpushl " __ASM_NAME("__wine_callto16_handler@GOT") "(%%ebx)\n" );
    else
        fprintf( outfile, "\tpushl $" __ASM_NAME("__wine_callto16_handler") "\n" );
    fprintf( outfile, "\t.byte 0x64\n\tpushl (%d)\n", STRUCTOFFSET(TEB,except) );
    fprintf( outfile, "\t.byte 0x64\n\tmovl %%esp,(%d)\n", STRUCTOFFSET(TEB,except) );

    /* Get return address */
    if ( UsePIC )
    {
        fprintf( outfile, "\tmovl " __ASM_NAME("CallTo16_RetAddr@GOT") "(%%ebx), %%ecx\n" );
        fprintf( outfile, "\tmovl (%%ecx), %%ecx\n" );
    }
    else
        fprintf( outfile, "\tmovl " __ASM_NAME("CallTo16_RetAddr") ", %%ecx\n" );

    /* Call the actual CallTo16 routine (simulate a lcall) */
    fprintf( outfile, "\tpushl %%cs\n" );
    fprintf( outfile, "\tcall .Lwine_call_to_16_%s\n", reg_func ? name : "long" );

    /* Remove exception frame */
    fprintf( outfile, "\t.byte 0x64\n\tpopl (%d)\n", STRUCTOFFSET(TEB,except) );
    fprintf( outfile, "\taddl $4, %%esp\n" );
    fprintf( outfile, "\t.byte 0x64\n\tpopl (%d)\n", STACKOFFSET );

    if ( !reg_func )
    {
        /* Convert and push return value */
        if ( short_ret )
        {
            fprintf( outfile, "\tmovzwl %%ax, %%eax\n" );
            fprintf( outfile, "\tpushl %%eax\n" );
        }
        else
        {
            fprintf( outfile, "\tshll $16,%%edx\n" );
            fprintf( outfile, "\tmovw %%ax,%%dx\n" );
            fprintf( outfile, "\tpushl %%edx\n" );
        }
    }
    else
    {
        /*
         * Modify CONTEXT86 structure to contain new values
         *
         * NOTE:  We restore only EAX, EBX, EDX, EDX, EBP, and ESP.
         *        The segment registers as well as ESI and EDI should
         *        not be modified by a well-behaved 16-bit routine in
         *        any case.  [If necessary, we could restore them as well,
         *        at the cost of a somewhat less efficient return path.]
         */

        fprintf( outfile, "\tmovl %d(%%esp), %%edi\n", STACK32OFFSET(target) - STACK32OFFSET(edi));
                /* everything above edi has been popped already */

        fprintf( outfile, "\tmovl %%eax, %d(%%edi)\n", CONTEXTOFFSET(Eax) );
        fprintf( outfile, "\tmovl %%ebx, %d(%%edi)\n", CONTEXTOFFSET(Ebx) );
        fprintf( outfile, "\tmovl %%ecx, %d(%%edi)\n", CONTEXTOFFSET(Ecx) );
        fprintf( outfile, "\tmovl %%edx, %d(%%edi)\n", CONTEXTOFFSET(Edx) );
        fprintf( outfile, "\tmovl %%ebp, %d(%%edi)\n", CONTEXTOFFSET(Ebp) );
        fprintf( outfile, "\tmovl %%esi, %d(%%edi)\n", CONTEXTOFFSET(Esp) );
                 /* The return glue code saved %esp into %esi */

        fprintf( outfile, "\tpushl %%edi\n" );
    }

    if ( UsePIC )
    {
        /* Get Global Offset Table into %ebx (might have been overwritten) */
        fprintf( outfile, "\tcall .Lwine_call_to_16_%s.getgot2\n", name );
        fprintf( outfile, ".Lwine_call_to_16_%s.getgot2:\n", name );
        fprintf( outfile, "\tpopl %%ebx\n" );
        fprintf( outfile, "\taddl $_GLOBAL_OFFSET_TABLE_+[.-.Lwine_call_to_16_%s.getgot2], %%ebx\n", name );
    }

    /* Leave Win16 Mutex */
    if ( UsePIC )
        fprintf( outfile, "\tcall " __ASM_NAME("_LeaveWin16Lock@PLT") "\n" );
    else
        fprintf( outfile, "\tcall " __ASM_NAME("_LeaveWin16Lock") "\n" );

    /* Print debugging info */
    if (debugging)
    {
        fprintf( outfile, "\tpushl $%d\n", reg_func );

        if ( UsePIC )
            fprintf( outfile, "\tcall " __ASM_NAME("RELAY_DebugCallTo16Ret@PLT") "\n" );
        else
            fprintf( outfile, "\tcall " __ASM_NAME("RELAY_DebugCallTo16Ret") "\n" );

        fprintf( outfile, "\taddl $4, %%esp\n" );
    }

    /* Get return value */
    fprintf( outfile, "\tpopl %%eax\n" );

    /* Restore the 32-bit registers */
    fprintf( outfile, "\tpopl %%edi\n" );
    fprintf( outfile, "\tpopl %%esi\n" );
    fprintf( outfile, "\tpopl %%edx\n" );
    fprintf( outfile, "\tpopl %%ecx\n" );
    fprintf( outfile, "\tpopl %%ebx\n" );

    /* Function exit sequence */
    fprintf( outfile, "\tpopl %%ebp\n" );
    fprintf( outfile, "\tret $8\n" );


    /* Start of the actual CallTo16 routine */

    if (!reg_func && short_ret) return;  /* call_to_16_word uses call_to_16_long backend routine */

    fprintf( outfile, ".Lwine_call_to_16_%s:\n", name );

    /* Switch to the 16-bit stack */
    fprintf( outfile, "\tmovl %%esp,%%edx\n" );
#ifdef __svr4__
    fprintf( outfile,"\tdata16\n");
#endif
    fprintf( outfile, "\t.byte 0x64\n\tmovw (%d),%%ss\n", STACKOFFSET + 2);
    fprintf( outfile, "\t.byte 0x64\n\tmovw (%d),%%sp\n", STACKOFFSET );
    fprintf( outfile, "\t.byte 0x64\n\tmovl %%edx,(%d)\n", STACKOFFSET );

    /* Make %bp point to the previous stackframe (built by CallFrom16) */
    fprintf( outfile, "\tmovzwl %%sp,%%ebp\n" );
    fprintf( outfile, "\tleal %d(%%ebp),%%ebp\n", STACK16OFFSET(bp) );

    /* Add the specified offset to the new sp */
    fprintf( outfile, "\tsubw %d(%%edx), %%sp\n", STACK32OFFSET(nb_args) );

    /* Push the return address
     * With sreg suffix, we push 16:16 address (normal lret)
     * With lreg suffix, we push 16:32 address (0x66 lret, for KERNEL32_45)
     */
    if (reg_func != 2)
        fprintf( outfile, "\tpushl %%ecx\n" );
    else
    {
        fprintf( outfile, "\tshldl $16, %%ecx, %%eax\n" );
        fprintf( outfile, "\tpushw $0\n" );
        fprintf( outfile, "\tpushw %%ax\n" );
        fprintf( outfile, "\tpushw $0\n" );
        fprintf( outfile, "\tpushw %%cx\n" );
    }

    if (reg_func)
    {
        /* Push the called routine address */
        fprintf( outfile, "\tmovl %d(%%edx),%%edx\n", STACK32OFFSET(target) );
        fprintf( outfile, "\tpushw %d(%%edx)\n", CONTEXTOFFSET(SegCs) );
        fprintf( outfile, "\tpushw %d(%%edx)\n", CONTEXTOFFSET(Eip) );

        /* Get the registers */
        fprintf( outfile, "\tpushw %d(%%edx)\n", CONTEXTOFFSET(SegDs) );
        fprintf( outfile, "\tmovl %d(%%edx),%%eax\n", CONTEXTOFFSET(SegEs) );
        fprintf( outfile, "\tmovw %%ax,%%es\n" );
        fprintf( outfile, "\tmovl %d(%%edx),%%eax\n", CONTEXTOFFSET(SegFs) );
        fprintf( outfile, "\tmovw %%ax,%%fs\n" );
        fprintf( outfile, "\tmovl %d(%%edx),%%ebp\n", CONTEXTOFFSET(Ebp) );
        fprintf( outfile, "\tmovl %d(%%edx),%%esi\n", CONTEXTOFFSET(Esi) );
        fprintf( outfile, "\tmovl %d(%%edx),%%edi\n", CONTEXTOFFSET(Edi) );
        fprintf( outfile, "\tmovl %d(%%edx),%%eax\n", CONTEXTOFFSET(Eax) );
        fprintf( outfile, "\tmovl %d(%%edx),%%ebx\n", CONTEXTOFFSET(Ebx) );
        fprintf( outfile, "\tmovl %d(%%edx),%%ecx\n", CONTEXTOFFSET(Ecx) );
        fprintf( outfile, "\tmovl %d(%%edx),%%edx\n", CONTEXTOFFSET(Edx) );

        /* Get the 16-bit ds */
        fprintf( outfile, "\tpopw %%ds\n" );
    }
    else  /* not a register function */
    {
        /* Push the called routine address */
        fprintf( outfile, "\tpushl %d(%%edx)\n", STACK32OFFSET(target) );

        /* Set %fs to the value saved by the last CallFrom16 */
        fprintf( outfile, "\tmovw %d(%%ebp),%%ax\n", STACK16OFFSET(fs)-STACK16OFFSET(bp) );
        fprintf( outfile, "\tmovw %%ax,%%fs\n" );

        /* Set %ds and %es (and %ax just in case) equal to %ss */
        fprintf( outfile, "\tmovw %%ss,%%ax\n" );
        fprintf( outfile, "\tmovw %%ax,%%ds\n" );
        fprintf( outfile, "\tmovw %%ax,%%es\n" );
    }

    /* Jump to the called routine */
    fprintf( outfile, "\t.byte 0x66\n" );
    fprintf( outfile, "\tlret\n" );
}


/*******************************************************************
 *         BuildRet16Func
 *
 * Build the return code for 16-bit callbacks
 */
static void BuildRet16Func( FILE *outfile )
{
    /*
     *  Note: This must reside in the .data section to allow
     *        run-time relocation of the SYSLEVEL_Win16CurrentTeb symbol
     */

    function_header( outfile, "CallTo16_Ret" );

    /* Save %esp into %esi */
    fprintf( outfile, "\tmovl %%esp,%%esi\n" );

    /* Restore 32-bit segment registers */

    fprintf( outfile, "\t.byte 0x2e\n\tmovl " __ASM_NAME("CallTo16_DataSelector") "-" __ASM_NAME("Call16_Ret_Start") ",%%edi\n" );
#ifdef __svr4__
    fprintf( outfile, "\tdata16\n");
#endif
    fprintf( outfile, "\tmovw %%di,%%ds\n" );
#ifdef __svr4__
    fprintf( outfile, "\tdata16\n");
#endif
    fprintf( outfile, "\tmovw %%di,%%es\n" );

    fprintf( outfile, "\tmovw " __ASM_NAME("SYSLEVEL_Win16CurrentTeb") ",%%fs\n" );

    /* Restore the 32-bit stack */

#ifdef __svr4__
    fprintf( outfile, "\tdata16\n");
#endif
    fprintf( outfile, "\tmovw %%di,%%ss\n" );
    fprintf( outfile, "\t.byte 0x64\n\tmovl (%d),%%esp\n", STACKOFFSET );

    /* Return to caller */

    fprintf( outfile, "\tlret\n" );

    /* Declare the return address and data selector variables */

    fprintf( outfile, "\n\t.align %d\n", get_alignment(4) );
    fprintf( outfile, "\t.globl " __ASM_NAME("CallTo16_DataSelector") "\n" );
    fprintf( outfile, __ASM_NAME("CallTo16_DataSelector") ":\t.long 0\n" );
    fprintf( outfile, "\t.globl " __ASM_NAME("CallTo16_RetAddr") "\n" );
    fprintf( outfile, __ASM_NAME("CallTo16_RetAddr") ":\t.long 0\n" );
}


/*******************************************************************
 *         BuildCallTo32CBClient
 *
 * Call a CBClient relay stub from 32-bit code (KERNEL.620).
 *
 * Since the relay stub is itself 32-bit, this should not be a problem;
 * unfortunately, the relay stubs are expected to switch back to a
 * 16-bit stack (and 16-bit code) after completion :-(
 *
 * This would conflict with our 16- vs. 32-bit stack handling, so
 * we simply switch *back* to our 32-bit stack before returning to
 * the caller ...
 *
 * The CBClient relay stub expects to be called with the following
 * 16-bit stack layout, and with ebp and ebx pointing into the 16-bit
 * stack at the designated places:
 *
 *    ...
 *  (ebp+14) original arguments to the callback routine
 *  (ebp+10) far return address to original caller
 *  (ebp+6)  Thunklet target address
 *  (ebp+2)  Thunklet relay ID code
 *  (ebp)    BP (saved by CBClientGlueSL)
 *  (ebp-2)  SI (saved by CBClientGlueSL)
 *  (ebp-4)  DI (saved by CBClientGlueSL)
 *  (ebp-6)  DS (saved by CBClientGlueSL)
 *
 *   ...     buffer space used by the 16-bit side glue for temp copies
 *
 *  (ebx+4)  far return address to 16-bit side glue code
 *  (ebx)    saved 16-bit ss:sp (pointing to ebx+4)
 *
 * The 32-bit side glue code accesses both the original arguments (via ebp)
 * and the temporary copies prepared by the 16-bit side glue (via ebx).
 * After completion, the stub will load ss:sp from the buffer at ebx
 * and perform a far return to 16-bit code.
 *
 * To trick the relay stub into returning to us, we replace the 16-bit
 * return address to the glue code by a cs:ip pair pointing to our
 * return entry point (the original return address is saved first).
 * Our return stub thus called will then reload the 32-bit ss:esp and
 * return to 32-bit code (by using and ss:esp value that we have also
 * pushed onto the 16-bit stack before and a cs:eip values found at
 * that position on the 32-bit stack).  The ss:esp to be restored is
 * found relative to the 16-bit stack pointer at:
 *
 *  (ebx-4)   ss  (flat)
 *  (ebx-8)   sp  (32-bit stack pointer)
 *
 * The second variant of this routine, CALL32_CBClientEx, which is used
 * to implement KERNEL.621, has to cope with yet another problem: Here,
 * the 32-bit side directly returns to the caller of the CBClient thunklet,
 * restoring registers saved by CBClientGlueSL and cleaning up the stack.
 * As we have to return to our 32-bit code first, we have to adapt the
 * layout of our temporary area so as to include values for the registers
 * that are to be restored, and later (in the implementation of KERNEL.621)
 * we *really* restore them. The return stub restores DS, DI, SI, and BP
 * from the stack, skips the next 8 bytes (CBClient relay code / target),
 * and then performs a lret NN, where NN is the number of arguments to be
 * removed. Thus, we prepare our temporary area as follows:
 *
 *     (ebx+22) 16-bit cs  (this segment)
 *     (ebx+20) 16-bit ip  ('16-bit' return entry point)
 *     (ebx+16) 32-bit ss  (flat)
 *     (ebx+12) 32-bit sp  (32-bit stack pointer)
 *     (ebx+10) 16-bit bp  (points to ebx+24)
 *     (ebx+8)  16-bit si  (ignored)
 *     (ebx+6)  16-bit di  (ignored)
 *     (ebx+4)  16-bit ds  (we actually use the flat DS here)
 *     (ebx+2)  16-bit ss  (16-bit stack segment)
 *     (ebx+0)  16-bit sp  (points to ebx+4)
 *
 * Note that we ensure that DS is not changed and remains the flat segment,
 * and the 32-bit stack pointer our own return stub needs fits just
 * perfectly into the 8 bytes that are skipped by the Windows stub.
 * One problem is that we have to determine the number of removed arguments,
 * as these have to be really removed in KERNEL.621. Thus, the BP value
 * that we place in the temporary area to be restored, contains the value
 * that SP would have if no arguments were removed. By comparing the actual
 * value of SP with this value in our return stub we can compute the number
 * of removed arguments. This is then returned to KERNEL.621.
 *
 * The stack layout of this function:
 * (ebp+20)  nArgs     pointer to variable receiving nr. of args (Ex only)
 * (ebp+16)  esi       pointer to caller's esi value
 * (ebp+12)  arg       ebp value to be set for relay stub
 * (ebp+8)   func      CBClient relay stub address
 * (ebp+4)   ret addr
 * (ebp)     ebp
 */
static void BuildCallTo32CBClient( FILE *outfile, BOOL isEx )
{
    char *name = isEx? "CBClientEx" : "CBClient";
    int size = isEx? 24 : 12;

    /* Function header */

    fprintf( outfile, "\n\t.align %d\n", get_alignment(4) );
    if (use_stabs)
        fprintf( outfile, ".stabs \"CALL32_%s:F1\",36,0,0," __ASM_NAME("CALL32_%s") "\n",
                 name, name );
    fprintf( outfile, "\t.globl " __ASM_NAME("CALL32_%s") "\n", name );
    fprintf( outfile, __ASM_NAME("CALL32_%s") ":\n", name );

    /* Entry code */

    fprintf( outfile, "\tpushl %%ebp\n" );
    fprintf( outfile, "\tmovl %%esp,%%ebp\n" );
    fprintf( outfile, "\tpushl %%edi\n" );
    fprintf( outfile, "\tpushl %%esi\n" );
    fprintf( outfile, "\tpushl %%ebx\n" );

    /* Get the 16-bit stack */

    fprintf( outfile, "\t.byte 0x64\n\tmovl (%d),%%ebx\n", STACKOFFSET);

    /* Convert it to a flat address */

    fprintf( outfile, "\tshldl $16,%%ebx,%%eax\n" );
    fprintf( outfile, "\tandl $0xfff8,%%eax\n" );
    fprintf( outfile, "\tshrl $1,%%eax\n" );
    fprintf( outfile, "\tmovl " __ASM_NAME("wine_ldt_copy") "(%%eax),%%esi\n" );
    fprintf( outfile, "\tmovw %%bx,%%ax\n" );
    fprintf( outfile, "\taddl %%eax,%%esi\n" );

    /* Allocate temporary area (simulate STACK16_PUSH) */

    fprintf( outfile, "\tpushf\n" );
    fprintf( outfile, "\tcld\n" );
    fprintf( outfile, "\tleal -%d(%%esi), %%edi\n", size );
    fprintf( outfile, "\tmovl $%d, %%ecx\n", sizeof(STACK16FRAME) );
    fprintf( outfile, "\trep\n\tmovsb\n" );
    fprintf( outfile, "\tpopf\n" );

    fprintf( outfile, "\t.byte 0x64\n\tsubw $%d,(%d)\n", size, STACKOFFSET );

    fprintf( outfile, "\tpushl %%edi\n" );  /* remember address */

    /* Set up temporary area */

    if ( !isEx )
    {
        fprintf( outfile, "\tleal 4(%%edi), %%edi\n" );

        fprintf( outfile, "\tleal -8(%%esp), %%eax\n" );
        fprintf( outfile, "\tmovl %%eax, -8(%%edi)\n" );    /* 32-bit sp */

        fprintf( outfile, "\tmovw %%ss, %%ax\n" );
        fprintf( outfile, "\tandl $0x0000ffff, %%eax\n" );
        fprintf( outfile, "\tmovl %%eax, -4(%%edi)\n" );    /* 32-bit ss */

        fprintf( outfile, "\taddl $%d, %%ebx\n", sizeof(STACK16FRAME)-size+4 + 4 );
        fprintf( outfile, "\tmovl %%ebx, 0(%%edi)\n" );    /* 16-bit ss:sp */

        fprintf( outfile, "\tmovl " __ASM_NAME("CALL32_%s_RetAddr") ", %%eax\n", name );
        fprintf( outfile, "\tmovl %%eax, 4(%%edi)\n" );   /* overwrite return address */
    }
    else
    {
        fprintf( outfile, "\taddl $%d, %%ebx\n", sizeof(STACK16FRAME)-size+4 );
        fprintf( outfile, "\tmovl %%ebx, 0(%%edi)\n" );

        fprintf( outfile, "\tmovw %%ds, %%ax\n" );
        fprintf( outfile, "\tmovw %%ax, 4(%%edi)\n" );

        fprintf( outfile, "\taddl $20, %%ebx\n" );
        fprintf( outfile, "\tmovw %%bx, 10(%%edi)\n" );

        fprintf( outfile, "\tleal -8(%%esp), %%eax\n" );
        fprintf( outfile, "\tmovl %%eax, 12(%%edi)\n" );

        fprintf( outfile, "\tmovw %%ss, %%ax\n" );
        fprintf( outfile, "\tandl $0x0000ffff, %%eax\n" );
        fprintf( outfile, "\tmovl %%eax, 16(%%edi)\n" );

        fprintf( outfile, "\tmovl " __ASM_NAME("CALL32_%s_RetAddr") ", %%eax\n", name );
        fprintf( outfile, "\tmovl %%eax, 20(%%edi)\n" );
    }

    /* Set up registers and call CBClient relay stub (simulating a far call) */

    fprintf( outfile, "\tmovl 16(%%ebp), %%esi\n" );
    fprintf( outfile, "\tmovl (%%esi), %%esi\n" );

    fprintf( outfile, "\tmovl %%edi, %%ebx\n" );
    fprintf( outfile, "\tmovl 8(%%ebp), %%eax\n" );
    fprintf( outfile, "\tmovl 12(%%ebp), %%ebp\n" );

    fprintf( outfile, "\tpushl %%cs\n" );
    fprintf( outfile, "\tcall *%%eax\n" );

    /* Return new esi value to caller */

    fprintf( outfile, "\tmovl 32(%%esp), %%edi\n" );
    fprintf( outfile, "\tmovl %%esi, (%%edi)\n" );

    /* Cleanup temporary area (simulate STACK16_POP) */

    fprintf( outfile, "\tpop %%esi\n" );

    fprintf( outfile, "\tpushf\n" );
    fprintf( outfile, "\tstd\n" );
    fprintf( outfile, "\tdec %%esi\n" );
    fprintf( outfile, "\tleal %d(%%esi), %%edi\n", size );
    fprintf( outfile, "\tmovl $%d, %%ecx\n", sizeof(STACK16FRAME) );
    fprintf( outfile, "\trep\n\tmovsb\n" );
    fprintf( outfile, "\tpopf\n" );

    fprintf( outfile, "\t.byte 0x64\n\taddw $%d,(%d)\n", size, STACKOFFSET );

    /* Return argument size to caller */
    if ( isEx )
    {
        fprintf( outfile, "\tmovl 32(%%esp), %%ebx\n" );
        fprintf( outfile, "\tmovl %%ebp, (%%ebx)\n" );
    }

    /* Restore registers and return */

    fprintf( outfile, "\tpopl %%ebx\n" );
    fprintf( outfile, "\tpopl %%esi\n" );
    fprintf( outfile, "\tpopl %%edi\n" );
    fprintf( outfile, "\tpopl %%ebp\n" );
    fprintf( outfile, "\tret\n" );
}

static void BuildCallTo32CBClientRet( FILE *outfile, BOOL isEx )
{
    char *name = isEx? "CBClientEx" : "CBClient";

    /* '16-bit' return stub */

    fprintf( outfile, "\n\t.globl " __ASM_NAME("CALL32_%s_Ret") "\n", name );
    fprintf( outfile, __ASM_NAME("CALL32_%s_Ret") ":\n", name );

    if ( !isEx )
    {
        fprintf( outfile, "\tmovzwl %%sp, %%ebx\n" );
        fprintf( outfile, "\tlssl %%ss:-16(%%ebx), %%esp\n" );
    }
    else
    {
        fprintf( outfile, "\tmovzwl %%bp, %%ebx\n" );
        fprintf( outfile, "\tsubw %%bp, %%sp\n" );
        fprintf( outfile, "\tmovzwl %%sp, %%ebp\n" );
        fprintf( outfile, "\tlssl %%ss:-12(%%ebx), %%esp\n" );
    }
    fprintf( outfile, "\tlret\n" );

    /* Declare the return address variable */

    fprintf( outfile, "\n\t.globl " __ASM_NAME("CALL32_%s_RetAddr") "\n", name );
    fprintf( outfile, __ASM_NAME("CALL32_%s_RetAddr") ":\t.long 0\n", name );
}


/*******************************************************************
 *         BuildCallFrom32Regs
 *
 * Build a 32-bit-to-Wine call-back function for a 'register' function.
 * 'args' is the number of dword arguments.
 *
 * Stack layout:
 *   ...
 * (ebp+12)  first arg
 * (ebp+8)   ret addr to user code
 * (ebp+4)   ret addr to relay code
 * (ebp+0)   saved ebp
 * (ebp-128) buffer area to allow stack frame manipulation
 * (ebp-332) CONTEXT86 struct
 * (ebp-336) CONTEXT86 *argument
 *  ....     other arguments copied from (ebp+12)
 *
 * The entry point routine is called with a CONTEXT* extra argument,
 * following the normal args. In this context structure, EIP_reg
 * contains the return address to user code, and ESP_reg the stack
 * pointer on return (with the return address and arguments already
 * removed).
 */
static void BuildCallFrom32Regs( FILE *outfile )
{
    static const int STACK_SPACE = 128 + sizeof(CONTEXT86);

    /* Function header */

    function_header( outfile, "__wine_call_from_32_regs" );

    /* Allocate some buffer space on the stack */

    fprintf( outfile, "\tpushl %%ebp\n" );
    fprintf( outfile, "\tmovl %%esp,%%ebp\n ");
    fprintf( outfile, "\tleal -%d(%%esp), %%esp\n", STACK_SPACE );

    /* Build the context structure */

    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(Eax) - STACK_SPACE );
    fprintf( outfile, "\tpushfl\n" );
    fprintf( outfile, "\tpopl %%eax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(EFlags) - STACK_SPACE );
    fprintf( outfile, "\tmovl 0(%%ebp),%%eax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(Ebp) - STACK_SPACE );
    fprintf( outfile, "\tmovl %%ebx,%d(%%ebp)\n", CONTEXTOFFSET(Ebx) - STACK_SPACE );
    fprintf( outfile, "\tmovl %%ecx,%d(%%ebp)\n", CONTEXTOFFSET(Ecx) - STACK_SPACE );
    fprintf( outfile, "\tmovl %%edx,%d(%%ebp)\n", CONTEXTOFFSET(Edx) - STACK_SPACE );
    fprintf( outfile, "\tmovl %%esi,%d(%%ebp)\n", CONTEXTOFFSET(Esi) - STACK_SPACE );
    fprintf( outfile, "\tmovl %%edi,%d(%%ebp)\n", CONTEXTOFFSET(Edi) - STACK_SPACE );

    fprintf( outfile, "\txorl %%eax,%%eax\n" );
    fprintf( outfile, "\tmovw %%cs,%%ax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(SegCs) - STACK_SPACE );
    fprintf( outfile, "\tmovw %%es,%%ax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(SegEs) - STACK_SPACE );
    fprintf( outfile, "\tmovw %%fs,%%ax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(SegFs) - STACK_SPACE );
    fprintf( outfile, "\tmovw %%gs,%%ax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(SegGs) - STACK_SPACE );
    fprintf( outfile, "\tmovw %%ss,%%ax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(SegSs) - STACK_SPACE );
    fprintf( outfile, "\tmovw %%ds,%%ax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(SegDs) - STACK_SPACE );
    fprintf( outfile, "\tmovw %%ax,%%es\n" );  /* set %es equal to %ds just in case */

    fprintf( outfile, "\tmovl $0x%x,%%eax\n", CONTEXT86_FULL );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(ContextFlags) - STACK_SPACE );

    fprintf( outfile, "\tmovl 8(%%ebp),%%eax\n" ); /* Get %eip at time of call */
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(Eip) - STACK_SPACE );

    /* Transfer the arguments */

    fprintf( outfile, "\tmovl 4(%%ebp),%%ebx\n" );   /* get relay code addr */
    fprintf( outfile, "\tpushl %%esp\n" );           /* push ptr to context struct */
    fprintf( outfile, "\tmovzbl 4(%%ebx),%%ecx\n" ); /* fetch number of args to copy */
    fprintf( outfile, "\tjecxz 1f\n" );
    fprintf( outfile, "\tsubl %%ecx,%%esp\n" );
    fprintf( outfile, "\tleal 12(%%ebp),%%esi\n" );  /* get %esp at time of call */
    fprintf( outfile, "\tmovl %%esp,%%edi\n" );
    fprintf( outfile, "\tshrl $2,%%ecx\n" );
    fprintf( outfile, "\tcld\n" );
    fprintf( outfile, "\trep\n\tmovsl\n" );  /* copy args */

    fprintf( outfile, "1:\tmovzbl 5(%%ebx),%%eax\n" ); /* fetch number of args to remove */
    fprintf( outfile, "\tleal 12(%%ebp,%%eax),%%eax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(Esp) - STACK_SPACE );

    /* Call the entry point */

    fprintf( outfile, "\tcall *0(%%ebx)\n" );

    /* Store %eip and %ebp onto the new stack */

    fprintf( outfile, "\tmovl %d(%%ebp),%%edx\n", CONTEXTOFFSET(Esp) - STACK_SPACE );
    fprintf( outfile, "\tmovl %d(%%ebp),%%eax\n", CONTEXTOFFSET(Eip) - STACK_SPACE );
    fprintf( outfile, "\tmovl %%eax,-4(%%edx)\n" );
    fprintf( outfile, "\tmovl %d(%%ebp),%%eax\n", CONTEXTOFFSET(Ebp) - STACK_SPACE );
    fprintf( outfile, "\tmovl %%eax,-8(%%edx)\n" );

    /* Restore the context structure */

    /* Note: we don't bother to restore %cs, %ds and %ss
     *       changing them in 32-bit code is a recipe for disaster anyway
     */
    fprintf( outfile, "\tmovl %d(%%ebp),%%eax\n", CONTEXTOFFSET(SegEs) - STACK_SPACE );
    fprintf( outfile, "\tmovw %%ax,%%es\n" );
    fprintf( outfile, "\tmovl %d(%%ebp),%%eax\n", CONTEXTOFFSET(SegFs) - STACK_SPACE );
    fprintf( outfile, "\tmovw %%ax,%%fs\n" );
    fprintf( outfile, "\tmovl %d(%%ebp),%%eax\n", CONTEXTOFFSET(SegGs) - STACK_SPACE );
    fprintf( outfile, "\tmovw %%ax,%%gs\n" );

    fprintf( outfile, "\tmovl %d(%%ebp),%%edi\n", CONTEXTOFFSET(Edi) - STACK_SPACE );
    fprintf( outfile, "\tmovl %d(%%ebp),%%esi\n", CONTEXTOFFSET(Esi) - STACK_SPACE );
    fprintf( outfile, "\tmovl %d(%%ebp),%%edx\n", CONTEXTOFFSET(Edx) - STACK_SPACE );
    fprintf( outfile, "\tmovl %d(%%ebp),%%ecx\n", CONTEXTOFFSET(Ecx) - STACK_SPACE );
    fprintf( outfile, "\tmovl %d(%%ebp),%%ebx\n", CONTEXTOFFSET(Ebx) - STACK_SPACE );

    fprintf( outfile, "\tmovl %d(%%ebp),%%eax\n", CONTEXTOFFSET(EFlags) - STACK_SPACE );
    fprintf( outfile, "\tpushl %%eax\n" );
    fprintf( outfile, "\tpopfl\n" );
    fprintf( outfile, "\tmovl %d(%%ebp),%%eax\n", CONTEXTOFFSET(Eax) - STACK_SPACE );

    fprintf( outfile, "\tmovl %d(%%ebp),%%ebp\n", CONTEXTOFFSET(Esp) - STACK_SPACE );
    fprintf( outfile, "\tleal -8(%%ebp),%%esp\n" );
    fprintf( outfile, "\tpopl %%ebp\n" );
    fprintf( outfile, "\tret\n" );
}


/*******************************************************************
 *         BuildRelays16
 *
 * Build all the 16-bit relay callbacks
 */
void BuildRelays16( FILE *outfile )
{
    char buffer[1024];

    /* File header */

    fprintf( outfile, "/* File generated automatically. Do not edit! */\n\n" );
    fprintf( outfile, "\t.text\n" );

    if (output_file_name && use_stabs)
    {
        getcwd(buffer, sizeof(buffer));
        fprintf( outfile, "\t.file\t\"%s\"\n", output_file_name );

        /*
         * The stabs help the internal debugger as they are an indication that it
         * is sensible to step into a thunk/trampoline.
         */
        fprintf( outfile, ".stabs \"%s/\",100,0,0,Code_Start\n", buffer);
        fprintf( outfile, ".stabs \"%s\",100,0,0,Code_Start\n", output_file_name );
        fprintf( outfile, "Code_Start:\n\n" );
    }

    fprintf( outfile, __ASM_NAME("Call16_Start") ":\n" );
    fprintf( outfile, "\t.globl " __ASM_NAME("Call16_Start") "\n" );
    fprintf( outfile, "\t.byte 0\n\n" );

    /* Standard CallFrom16 routine (WORD return) */
    BuildCallFrom16Core( outfile, FALSE, FALSE, TRUE );

    /* Standard CallFrom16 routine (DWORD return) */
    BuildCallFrom16Core( outfile, FALSE, FALSE, FALSE );

    /* Register CallFrom16 routine */
    BuildCallFrom16Core( outfile, TRUE, FALSE, FALSE );

    /* C16ThkSL CallFrom16 routine */
    BuildCallFrom16Core( outfile, FALSE, TRUE, FALSE );

    /* Standard CallTo16 routine (WORD return) */
    BuildCallTo16Core( outfile, TRUE, FALSE );

    /* Standard CallTo16 routine (DWORD return) */
    BuildCallTo16Core( outfile, FALSE, FALSE );

    /* Register CallTo16 routine (16:16 retf) */
    BuildCallTo16Core( outfile, FALSE, 1 );

    /* Register CallTo16 routine (16:32 retf) */
    BuildCallTo16Core( outfile, FALSE, 2 );

    /* CBClientThunkSL routine */
    BuildCallTo32CBClient( outfile, FALSE );

    /* CBClientThunkSLEx routine */
    BuildCallTo32CBClient( outfile, TRUE  );

    fprintf( outfile, __ASM_NAME("Call16_End") ":\n" );
    fprintf( outfile, "\t.globl " __ASM_NAME("Call16_End") "\n" );

    if (use_stabs)
    {
        fprintf( outfile, "\t.stabs \"\",100,0,0,.Letext\n");
        fprintf( outfile, ".Letext:\n");

        /* Some versions of gdb need to have the filename data for
           each section, so output it again for the data section. */
        if (output_file_name)
        {
            fprintf( outfile, ".stabs \"%s/\",100,0,0,Data_Start\n", buffer);
            fprintf( outfile, ".stabs \"%s\",100,0,0,Data_Start\n", output_file_name );
            fprintf( outfile, "Data_Start:\n\n" );
        }
    }

    /* The whole Call16_Ret segment must lie within the .data section */
    fprintf( outfile, "\n\t.data\n" );
    fprintf( outfile, "\t.globl " __ASM_NAME("Call16_Ret_Start") "\n" );
    fprintf( outfile, __ASM_NAME("Call16_Ret_Start") ":\n" );

    /* Standard CallTo16 return stub */
    BuildRet16Func( outfile );

    /* CBClientThunkSL return stub */
    BuildCallTo32CBClientRet( outfile, FALSE );

    /* CBClientThunkSLEx return stub */
    BuildCallTo32CBClientRet( outfile, TRUE  );

    /* End of Call16_Ret segment */
    fprintf( outfile, "\n\t.globl " __ASM_NAME("Call16_Ret_End") "\n" );
    fprintf( outfile, __ASM_NAME("Call16_Ret_End") ":\n" );
}

/*******************************************************************
 *         BuildRelays32
 *
 * Build all the 32-bit relay callbacks
 */
void BuildRelays32( FILE *outfile )
{
    /* File header */

    fprintf( outfile, "/* File generated automatically. Do not edit! */\n\n" );
    fprintf( outfile, "\t.text\n" );

    if (output_file_name && use_stabs)
    {
        char buffer[1024];
        getcwd(buffer, sizeof(buffer));
        fprintf( outfile, "\t.file\t\"%s\"\n", output_file_name );

        /*
         * The stabs help the internal debugger as they are an indication that it
         * is sensible to step into a thunk/trampoline.
         */
        fprintf( outfile, ".stabs \"%s/\",100,0,0,Code_Start\n", buffer);
        fprintf( outfile, ".stabs \"%s\",100,0,0,Code_Start\n", output_file_name );
        fprintf( outfile, "Code_Start:\n\n" );
    }

    /* 32-bit register entry point */
    BuildCallFrom32Regs( outfile );

    if (use_stabs)
    {
        fprintf( outfile, "\t.stabs \"\",100,0,0,.Letext\n");
        fprintf( outfile, ".Letext:\n");
    }
}

#else /* __i386__ */

void BuildRelays16( FILE *outfile )
{
    fprintf( outfile, "/* File not used with this architecture. Do not edit! */\n\n" );
}

void BuildRelays32( FILE *outfile )
{
    fprintf( outfile, "/* File not used with this architecture. Do not edit! */\n\n" );
}

#endif  /* __i386__ */
