# Module: moduleRhostsCheck
# Author: Craig H. Rowland <crowland@psionic.com>
# Created: 10-6-98
#
# Send all changes/modifications/bugfixes to the above address.  
# 
# This software is Copyright(c) 1997-98 Craig H. Rowland
# 
# Disclaimer:
# 
# All software distributed by Craig H. Rowland ("the author") and
# Psionic Systems is distributed AS IS and carries NO WARRANTY or
# GUARANTEE OF ANY KIND. End users of the software acknowledge that         
# they will not hold the author, Psionic Systems, and any employer of    
# the author liable for failure or non-function of a software
# product. YOU ARE USING THIS PRODUCT AT YOUR OWN RISK
# 
# Licensing restrictions apply. See the license that came with this
# file for more information or visit http://www.psionic.com for more
# information.
# 
# This software is NOT GPL NOR PUBLIC DOMAIN so please read the license   
# before modifying or distributing. Contact the above address if you have
# any questions.
# 
# Description:
# 
# This module issues and alert on logout if the user's .rhosts file contains
# any dangerous entries ('+ +').
# 
# $Id: moduleRhostsCheck.py,v 1.3 1999/03/22 04:57:14 crowland Exp crowland $


from hostSentryCore import *
import hostSentryConfig
import hostSentryUser
import hostSentryLog
import hostSentryDB

import sys
import string
import re
import pwd

class moduleRhostsCheck(hostSentryCore):

	def __init__(self):
		self.setLogLevel()
		self.__result = None

	def login(self, userObj, loginStamp):
		if self.getLogLevel() > 0:
			hostSentryLog.log('debug: moduleRhostsCheck: login: processing user: ' + userObj.getUsername())		

	def logout(self, userObj, logoutStamp):
		if self.getLogLevel() > 0:
			hostSentryLog.log('debug: moduleRhostsCheck: logout: processing user: ' + userObj.getUsername())		

		loginIP, loginHostname, loginTTY, loginTime, logoutTime =  string.split(logoutStamp, '@')

		try:
			pwInfo = pwd.getpwnam(userObj.getUsername())
		except:
			hostSentryLog.log('adminalert: moduleRhostsCheck: logout: Cannot find user: ' + userObj.getUsername() + ' in passwd database: ' + sys.exc_value[0])
			raise hostSentryError('adminalert: moduleRhostsCheck: logout: Cannot find user: ' + userObj.getUsername() + ' in passwd database: ' + sys.exc_value[0])

		try:
			userPath = pwInfo[5]
			if self.getLogLevel() > 0:
				hostSentryLog.log('debug: moduleRhostsCheck: logout: looking for .rhosts file in: %s' % userPath)
			rhosts = open(userPath + '/.rhosts')
		except:
			return

		try:
			rhostsData = None
			while 1:
				rhostsData = rhosts.readline()[:-1]
				if self.getLogLevel() > 0:
					hostSentryLog.log('debug: moduleRhostsCheck: logout: processing rhosts entry: %s' % rhostsData)

				if len(rhostsData) < 1:
					break
				elif re.search('\+', rhostsData) != None:
					hostSentryLog.log('securityalert: moduleRhostsCheck: Dangerous .rhosts entry found for user: ' + userObj.getUsername() + ' entry: ' + rhostsData)
					self.setResult('securityalert: moduleRhostsCheck: Dangerous .rhosts entry found for user: ' + userObj.getUsername() + ' entry: ' + rhostsData)
					rhosts.close()
					return
			rhosts.close()
		except:
			hostSentryLog.log('adminalert: moduleRhostsCheck: logout: Error reading .rhosts file for processing: ' + sys.exc_value[0])
			raise hostSentryError('adminalert: moduleRhostsCheck: logout: Error reading .rhosts file for processing: ' + sys.exc_value[0])

	def setResult(self, result):
		self.__result = result

	def getResult(self):
		return self.__result


if __name__ == '__main__':

	test = moduleRhostsCheck()
	test.setLogLevel(99)
	user = hostSentryUser.hostSentryUser()
	user.setUsername('test')
#	user.setUsername('hostSentrytest')
	user.setTotalLogins(1)

	config = hostSentryConfig.hostSentryConfig()
	config.configInit()
	dbFile = config.parseToken('DB_FILE')

	db = hostSentryDB.hostSentryDB(dbFile)
	db.store(user)
	db.close()

	test.login(user, '192.168.2.1@somewhere.com@tty0001@999999999@')
	test.logout(user, '192.168.2.1@somewhere.com@tty0001@999999999@98989898')

	db = hostSentryDB.hostSentryDB(dbFile)
	db.delete(user.getUsername())
	db.close()
