# Module: moduleForeignDomain
# Author: Craig H. Rowland <crowland@psionic.com>
# Created: 10-6-98
#
# Send all changes/modifications/bugfixes to the above address.  
# 
# This software is Copyright(c) 1997-98 Craig H. Rowland
# 
# Disclaimer:
# 
# All software distributed by Craig H. Rowland ("the author") and
# Psionic Systems is distributed AS IS and carries NO WARRANTY or
# GUARANTEE OF ANY KIND. End users of the software acknowledge that         
# they will not hold the author, Psionic Systems, and any employer of    
# the author liable for failure or non-function of a software
# product. YOU ARE USING THIS PRODUCT AT YOUR OWN RISK
# 
# Licensing restrictions apply. See the license that came with this
# file for more information or visit http://www.psionic.com for more
# information.
# 
# This software is NOT GPL NOR PUBLIC DOMAIN so please read the license   
# before modifying or distributing. Contact the above address if you have
# any questions.
# 
# Description:
# 
# This module will issue an alert if the login is occurring from a 
# domain not listed in moduleForeignDomain.allow. This is useful for
# spotting users coming in from oddball sites when they shouldn't be.
# 
# $Id: moduleForeignDomain.py,v 1.6 1999/03/22 04:57:08 crowland Exp crowland $


from hostSentryCore import *
import hostSentryConfig
import hostSentryUser
import hostSentryLog
import hostSentryDB

import sys
import string
import re

ALLOW_FILE = '/moduleForeignDomain.allow'

class moduleForeignDomain(hostSentryCore):

	def __init__(self):
		self.setLogLevel()
		self.__result = None

	def login(self, userObj, loginStamp):
		if self.getLogLevel() > 0:
			hostSentryLog.log('debug: moduleForeignDomain: login: processing user: ' + userObj.getUsername())		

		loginIP, loginHostname, loginTTY, loginTime, logoutTime =  string.split(loginStamp, '@')


		try:
			config = hostSentryConfig.hostSentryConfig()
			config.configInit()
			allowPath = config.parseToken('MODULE_PATH') 
                        if allowPath == None:
				hostSentryLog.log('adminalert: moduleForeignDomain: MODULE_PATH token not found in config.')
				raise hostSentryError('adminalert: moduleForeignDomain: MODULE_PATH token not found in config.')
			else:
				allowFilename = allowPath + ALLOW_FILE

			if self.getLogLevel() > 0:
				hostSentryLog.log('debug: moduleForeignDomain: login: looking for allow file: %s' % allowFilename)

			allow = open(allowFilename)
		except:
			hostSentryLog.log('adminalert: moduleForeignDomain: login: ERROR: Allow file open failed: %s' % sys.exc_value[0])
			raise hostSentryError('adminalert: moduleForeignDomain: login: ERROR: Allow file open failed: %s' % sys.exc_value[0])

		try:
			allowData = None
			while 1:
				allowData = allow.readline()[:-1]
				if self.getLogLevel() > 0:
					hostSentryLog.log('debug: moduleForeignDomain: login: looking for match with allowed host: %s' % allowData)

				if len(allowData) < 1:
					break
				elif allowData[0][0] == '#':
					pass
				elif re.search(allowData[0], loginHostname) != None:
					allow.close()
					return
			allow.close()
		except:
			hostSentryLog.log('adminalert: moduleForeignDomain: login: ERROR: Error during regex compare.: %s' % sys.exc_value[0])
			raise hostSentryError('adminalert: moduleForeignDomain: login: ERROR: Error during regex compare.: %s' % sys.exc_value[0])

		
		hostSentryLog.log('securityalert: Foreign domain login detected for user: ' + userObj.getUsername() + ' from: ' + loginHostname)
		self.setResult('securityalert: Foreign domain login detected for user: ' + userObj.getUsername() + ' from: ' + loginHostname)

	def logout(self, userObj, logoutStamp):
		if self.getLogLevel() > 0:
			hostSentryLog.log('debug: moduleForeignDomain: logout: processing user: ' + userObj.getUsername())		

	def setResult(self, result):
		self.__result = result

	def getResult(self):
		return self.__result


if __name__ == '__main__':

	test = moduleForeignDomain()
	test.setLogLevel(99)
	user = hostSentryUser.hostSentryUser()
	user.setUsername('hostSentrytest')
	user.setTotalLogins(1)

	config = hostSentryConfig.hostSentryConfig()
	config.configInit()
	dbFile = config.parseToken('DB_FILE')

	db = hostSentryDB.hostSentryDB(dbFile)
	db.store(user)
	db.close()

	test.login(user, '192.168.2.1@somewhere.com@tty0001@999999999@')
	test.logout(user, '192.168.2.1@somewhere.com@tty0001@999999999@98989898')

	db = hostSentryDB.hostSentryDB(dbFile)
	db.delete(user.getUsername())
	db.close()
