/*
 *  $Id: test_ini_dsns.c,v 1.4 1999/09/06 13:03:23 martin Exp martin $
 *
 *  $Log: test_ini_dsns.c,v $
 *  Revision 1.4  1999/09/06 13:03:23  martin
 *  Fixed main prototype.
 *
 *  Revision 1.3  1999/09/06 11:53:35  martin
 *  Add flags argument to enumerate_dsns() for internal bug ID 468 -
 *  	SQLDataSources does not differentiate between USER and SYSTEM DSNs.
 *
 *  Revision 1.2  1999/02/24 15:41:51  martin
 *  Minor cosmetic changes before beta release.
 *
 *  Revision 1.1  1999/01/29 17:15:20  martin
 *  Initial revision
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sql.h>
#include <sqlext.h>

#include <extra.h>

struct summary
{
    char                *dsn;
    int                 state;
#define DSN_STATE_OK 1
#define DSN_STATE_FAIL 2
    struct summary      *next;
};


static SQLRETURN do_errors(
    SQLSMALLINT type,
    SQLHANDLE handle);

/************************************************************************/
/*                                                                      */
/*  main                                                                */
/*  ====                                                                */
/*                                                                      */
/*  This program checks DSNs in the odbc.ini file to see if we can      */
/*  connect to it. It prints an ongoing status of each connection       */
/*  attempt and a summary at the end.                                   */
/*                                                                      */
/************************************************************************/
int main(int argc, char *argv[])
{
    void                *p;
    dsn_details_t       *dsn;                   /* enumerated dsn */
    SQLHANDLE           envh;                   /* environment */
    SQLHANDLE           dbch;                   /* connection */
    SQLRETURN           ret;                    /* SQL function return status */
    struct summary      *sum_start=NULL;        /* fisrt DSN summary */
    struct summary      *cur_sum=NULL;          /* current DSN summary */
    struct summary      *new_sum=NULL;          /* alloced DSN summary */
    
    setbuf(stdout, NULL);                       /* No buffering */
    
    p = enumerate_dsns(NULL, ENUMERATE_USER_DSNS);
    while(dsn = get_next_dsn(p))
    {
        char            in_connection_string[512];
        char            out_connection_string[1024];
        SQLSMALLINT     out_connection_length;
        
        if (!dsn->dsn) continue;
        
        if (!(new_sum = malloc(sizeof(*new_sum))))
        {
            fprintf(stderr, "Failed to allocate memory\n");
            exit(1);
        }
        if (!sum_start) sum_start = new_sum;
        new_sum->dsn = strdup(dsn->dsn);
        new_sum->state = DSN_STATE_FAIL;
        new_sum->next = NULL;
        
        if (cur_sum) cur_sum->next = new_sum;
        
        printf("Trying: %s: ", dsn->dsn);
        /*
         *  Allocate the necessary SQL handles.
         */
        ret = SQLAllocHandle(SQL_HANDLE_ENV, NULL, &envh);
        if (!SQL_SUCCEEDED(ret))
        {
            fprintf(stderr, "Failed to allocate environment handle\n");
            exit(1);
        }
        printf(".");
        (void)SQLSetEnvAttr(envh,
                            SQL_ATTR_ODBC_VERSION,
                            (void *)SQL_OV_ODBC3, 0);
        printf(".");
        ret = SQLAllocHandle(SQL_HANDLE_DBC, envh, &dbch);
        if (!SQL_SUCCEEDED(ret))
        {
            do_errors(SQL_HANDLE_ENV, envh);
            SQLFreeHandle(SQL_HANDLE_ENV, envh);
            continue;
        }
        printf(".");
        /*
         *  Set connection and login time outs.
         */
        (void)SQLSetConnectAttr(dbch, SQL_ATTR_LOGIN_TIMEOUT,
                                (SQLPOINTER)15, 0);
        printf(".");        
        (void)SQLSetConnectAttr(dbch, SQL_ATTR_CONNECTION_TIMEOUT,
                                (SQLPOINTER)30, 0);
        printf(".");
        /*
         *  Connect with this DSN.
         */
        sprintf(in_connection_string, "DSN=%s;", dsn->dsn);
        ret = SQLDriverConnect(dbch, NULL, in_connection_string, SQL_NTS,
                               out_connection_string,
                               sizeof(out_connection_string),
                               &out_connection_length, SQL_DRIVER_COMPLETE);
        if (!SQL_SUCCEEDED(ret))
        {
            new_sum->state = DSN_STATE_FAIL;
            printf("FAIL\n");
            fprintf(stderr, "Failed to connect to %s\n", dsn->dsn);
            do_errors(SQL_HANDLE_DBC, dbch);
        }
        else
        {
            new_sum->state = DSN_STATE_OK;
            printf("OK\n\n");
            SQLDisconnect(dbch);
        }
        /*
         *  Free up these handles and go on to the next DSN in the ini file.
         */
        SQLFreeHandle(SQL_HANDLE_DBC, dbch);
        SQLFreeHandle(SQL_HANDLE_ENV, envh);
        cur_sum = new_sum;
    }
    if (p) enumerate_dsns(p, 0);

    /*
     *  Print out to stdout a summary of which DSNs are OK and which are not.
     */
    printf("\n+-------------------------------------------------+\n");
    printf("|                  DSN Summary                    |\n");
    printf("+-------------------------------------------------+\n");
    cur_sum = sum_start;
    while(cur_sum)
    {
        printf("| %-40.40s| %-6s|\n",
               cur_sum->dsn,
               (cur_sum->state == DSN_STATE_OK) ? "OK" : "FAIL");
        new_sum = cur_sum;
        cur_sum = cur_sum->next;
        free(new_sum->dsn);
        free(new_sum);
    }
    printf("+-------------------------------------------------+\n");
    
    return 0;
}

/************************************************************************/
/*                                                                      */
/*  do_errors                                                           */
/*  =========                                                           */
/*                                                                      */
/*  do_errors prints to stderr all the errors on the specified type of  */
/*  ODBC handle.                                                        */
/*                                                                      */
/************************************************************************/
static SQLRETURN do_errors(
    SQLSMALLINT type,
    SQLHANDLE handle)
{
    SQLRETURN           ret;
    char                state[7];
    SQLCHAR             text[1024];
    SQLSMALLINT         len;
    int                 i=0;
    SQLINTEGER          native;
    char                cbuf[2048];

    do
    {
        ret = SQLGetDiagRec(type, handle, ++i, state,
                            &native, text, sizeof(text), &len);
        if (SQL_SUCCEEDED(ret))
        {
            sprintf(cbuf, "error: %s:%ld:%ld:%s\n",
                    state, i, native, text);
            fprintf(stderr, cbuf);
        }
    }
    while (SQL_SUCCEEDED(ret));
    return ret;
}

