/* mge-utalk.h

   Type Declarations and Constant Definitions for MGE UTALK Driver

   See docs/mge-utalk.txt for design information

   Version 0.50, 2002-04-30

   Copyright (C) 2002 Hans Ekkehard Plesser <hans.plesser@itf.nlh.no>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/* definitions of mgeups commands */

#include <shared-tables.h> 

/* --------------------------------------------------------------- */
/*                 Default Values for UPS Variables                */
/* --------------------------------------------------------------- */

#define DEFAULT_LOWBATT   30   /* low battery level, in %          */
#define DEFAULT_ONDELAY    1   /* delay between return of utility 
                                  power and powering up of load,
                                  in MINUTES                       */
#define DEFAULT_OFFDELAY 120   /* delay befor power off, in SECONDS*/ 

#define MIN_CONFIRM_TIME   3   /* shutdown must be confirmed in    */
#define MAX_CONFIRM_TIME  15   /* this interval                    */

/* --------------------------------------------------------------- */
/*                       Multiplier Tables                         */
/* --------------------------------------------------------------- */

/* First index : Table number, fetched with "Ai" command 
   Second index: unit, as in enum above                               
   NOTE: - to make the table index the same as the MGE table number, 
           dummy table row multiplier[0][] is inserted
         - unit MIN2SEC is used to convert values in minutes sent by
	   the UPS (WAKEDELAY) to seconds
	 - the final column corresponds to unit NONE
*/

/* units in multiplier table */
typedef enum eunits 
{VOLT=0, AMPERE, HERTZ, VOLTAMP, WATT, DEGCELS, MIN2SEC, NONE} units;

static const double multiplier[4][8] = {
/*   V     A     Hz   VA     W   C  MIN2SEC NONE */ 
  { 1   , 1   , 1  ,    1,    1, 1,    60,   1 },
  { 1   , 1   , 0.1, 1000, 1000, 1,    60,   1 },
  { 0.01, 0.01, 1  ,    1,    1, 1,    60,   1 },
  { 1   , 0.01, 0.1,    1,    1, 1,    60,   1 }
};

/* --------------------------------------------------------------- */
/*                       Explicit Booleans                         */
/* --------------------------------------------------------------- */

/* use explicit booleans */
#ifdef FALSE
#  undef FALSE
#endif /* FALSE */
#ifdef TRUE
#  undef TRUE
#endif /* TRUE */
typedef enum ebool { FALSE=0, TRUE } bool;


/* --------------------------------------------------------------- */
/*      Query Commands and their Mapping to INFO_ Variables        */
/* --------------------------------------------------------------- */

/* Structure defining how to query UPS for a variable and write
   information to INFO structure.
*/
typedef struct {
  int   type;                /* INFO_* element                        */
  int   flags;               /* INFO-element flags to set in addinfo  */
  int   length;              /* INFO-element length of strings        */  
  char  cmd[VALSIZE];        /* UPS command string to requets element */
  char  fmt[VALSIZE];        /* printf format string for INFO entry   */
  units unit;                /* unit of measurement, or NONE          */
  bool  ok;                  /* flag indicating if item is available  */
} mge_info_item;

/* Array containing all possible INFO items
   NOTE: - Array is terminated by element with type INFO_UNUSED.
	 - Essential INFO items (_MFR, _MODEL, _FIRMWARE, _STATUS) are
	   handled separately.
         - Array is NOT const, since "ok" can be changed.
*/

static mge_info_item mge_info[] = {
  { 
    INFO_UTILITY,              /* INFO_ element */
    0,			       /* Flag          */
    0,			       /* max length    */
    "Uv",		       /* UPS command   */
    "%05.1f",		       /* output format */
    VOLT, 		       /* unit          */
    TRUE		       /* ok?           */
  },
  { 
    INFO_BATTPCT,              /* INFO_ element */
    0,			       /* Flag          */
    0,			       /* max length    */
    "Bl",		       /* UPS command   */
    "%05.1f",		       /* output format */
    NONE,		       /* unit          */
    TRUE		       /* ok?           */
  },
  { 
    INFO_BATT_RUNTIME,         /* INFO_ element */
    0,			       /* Flag          */
    0,			       /* max length    */
    "Bn",		       /* UPS command   */
    "%05d",		       /* output format */
    NONE,		       /* unit          */
    TRUE		       /* ok?           */
  },
  { 
    INFO_UPSTEMP,              /* INFO_ element */
    0,			       /* Flag          */
    0,			       /* max length    */
    "St",		       /* UPS command   */
    "%05.1f",		       /* output format */
    DEGCELS,		       /* unit          */
    TRUE		       /* ok?           */
  },
  { 
    INFO_ACFREQ,               /* INFO_ element */
    0,			       /* Flag          */
    0,			       /* max length    */
    "Uf",		       /* UPS command   */
    "%05.2f",		       /* output format */
    HERTZ,		       /* unit          */
    TRUE		       /* ok?           */
  },
  { 
    INFO_LOADPCT,              /* INFO_ element */
    0,			       /* Flag          */
    0,			       /* max length    */
    "Ll",		       /* UPS command   */
    "%05.1f",		       /* output format */
    NONE,		       /* unit          */
    TRUE		       /* ok?           */
  },
  {       /* same as LOBOOSTXFER */
    INFO_LOWXFER,              /* INFO_ element */
    FLAG_RW | FLAG_STRING,     /* Flag          */
    5,			       /* max length    */
    "Ee ?",		       /* UPS command   */
    "%05.1f",		       /* output format */
    VOLT,		       /* unit          */
    TRUE		       /* ok?           */
  },
  { 
    INFO_LOBOOSTXFER,          /* INFO_ element */
    FLAG_RW | FLAG_STRING,     /* Flag          */
    5,			       /* max length    */
    "Ee ?",		       /* UPS command   */
    "%05.1f",		       /* output format */
    VOLT,		       /* unit          */
    TRUE		       /* ok?           */
  },
  { 
    INFO_HIBOOSTXFER,          /* INFO_ element */
    FLAG_RW | FLAG_STRING,     /* Flag          */
    5,			       /* max length    */
    "Eo ?",		       /* UPS command   */
    "%05.1f",		       /* output format */
    VOLT,		       /* unit          */
    TRUE		       /* ok?           */
  },
  {     
    INFO_LOTRIMXFER,           /* INFO_ element */
    FLAG_RW | FLAG_STRING,     /* Flag          */
    5,			       /* max length    */
    "Ea ?",		       /* UPS command   */
    "%05.1f",		       /* output format */
    VOLT,		       /* unit          */
    TRUE		       /* ok?           */
  },
  {     
    INFO_HITRIMXFER,           /* INFO_ element */
    FLAG_RW | FLAG_STRING,     /* Flag          */
    5,			       /* max length    */
    "Eu ?",		       /* UPS command   */
    "%05.1f",		       /* output format */
    VOLT,		       /* unit          */
    TRUE		       /* ok?           */
  },
  {         /* same as HITRIMXFER */
    INFO_HIGHXFER,             /* INFO_ element */
    FLAG_RW | FLAG_STRING,     /* Flag          */
    5,			       /* max length    */
    "Eu ?",		       /* UPS command   */
    "%05.1f",		       /* output format */
    VOLT,		       /* unit          */
    TRUE		       /* ok?           */
  },
  { 
    INFO_WAKEDELAY,            /* INFO_ element */
    FLAG_RW | FLAG_STRING,     /* Flag          */
    5,			       /* max length    */
    "Sm ?",		       /* UPS command   */
    "%03d",  		       /* output format */
    NONE,		       /* unit          */
    TRUE		       /* ok?           */
  },
  { 
    INFO_GRACEDELAY,           /* INFO_ element */
    FLAG_RW | FLAG_STRING,     /* Flag          */
    5,			       /* max length    */
    "Sn ?",		       /* UPS command   */
    "%03d",  		       /* output format */
    NONE,		       /* unit          */
    TRUE		       /* ok?           */
  },
  { 
    INFO_SLFTSTINT,            /* INFO_ element */
    FLAG_RW | FLAG_STRING,     /* Flag          */
    5,			       /* max length    */
    "Bp ?",		       /* UPS command   */
    "%03d",  		       /* output format */
    NONE,		       /* unit          */
    TRUE		       /* ok?           */
  },
  { 
    INFO_OUTVOLT,              /* INFO_ element */
    0,			       /* Flag          */
    0,			       /* max length    */
    "Lv",		       /* UPS command   */
    "%05.1f",		       /* output format */
    VOLT,		       /* unit          */
    TRUE		       /* ok?           */
  },
  { 
    INFO_BATTVOLT,             /* INFO_ element */
    0,			       /* Flag          */
    0,			       /* max length    */
    "Bv",		       /* UPS command   */
    "%05.1f",		       /* output format */
    VOLT,		       /* unit          */
    TRUE		       /* ok?           */
  },
  { 
    INFO_LOBATTPCT,            /* INFO_ element */
    FLAG_RW | FLAG_STRING,     /* Flag          */
    2,			       /* max length    */
    "Bl ?",		       /* UPS command   */
    "%02d",  		       /* output format */
    NONE,		       /* unit          */
    TRUE		       /* ok?           */
  },
  { 
    INFO_NOMBATVLT,            /* INFO_ element */
    0,			       /* Flag          */
    0,			       /* max length    */
    "Bv ?",		       /* UPS command   */
    "%05.1f",		       /* output format */
    VOLT,		       /* unit          */
    TRUE		       /* ok?           */
  },
  { 
    INFO_CURRENT,              /* INFO_ element */
    0,			       /* Flag          */
    0,			       /* max length    */
    "Lc",		       /* UPS command   */
    "%05.1f",		       /* output format */
    AMPERE,		       /* unit          */
    TRUE		       /* ok?           */
  },
  /* terminating element */
  { 
    INFO_UNUSED,               /* INFO_ element */
    0,			       /* Flag          */
    0,			       /* max length    */
    "\0",		       /* UPS command   */
    "\0",    		       /* output format */
    NONE,  		       /* unit          */
    FALSE		       /* ok?           */
  } 
};
