/* 
   bestuferrups.c - model specific routines for Best Power Micro-Ferrups

   This module is a 40% rewritten mangle of the bestfort module by
   Grant, which is a 75% rewritten mangle of the bestups module by
   Russell.   It has no test battery command since my ME3100 does this
   by itself. (same as Grant's driver in this respect)

   Copyright (C) 2002  Andreas Wrede  <andreas@planix.com>
   Copyright (C) 2000  John Stone  <johns@megapixel.com>
   Copyright (C) 2000  Grant Taylor <gtaylor@picante.com>
   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

*/

#include "main.h"

#define ENDCHAR		'\r'
#define IGNCHARS	"\012"

#include <stdio.h>
#include <string.h>
#include <unistd.h>

int debugging = 0;


/* Blob of UPS configuration data from the formatconfig string */
struct {
  int valid;			/* set to 1 when this is filled in */

  float  idealbvolts;		/* various interestin battery voltages */
  float  fullvolts;
  float  emptyvolts;
  int va;			/* capacity of UPS in Volt-Amps */
  int watts;			/* capacity of UPS in watts */
  int model;			/* enumerated model type */
} fc;


/* Forward decls */

/* Set up all the funky shared memory stuff used to communicate with upsd */
void  upsdrv_initinfo (void)
{
	/* now set up room for all future variables that are supported */
	addinfo (INFO_MFR, "", 0, 0);
	addinfo (INFO_MODEL, "", 0, 0);
	addinfo (INFO_UTILITY, "", 0, 0);
	addinfo (INFO_OUTVOLT, "", 0, 0);
	addinfo (INFO_ACFREQ, "", 0, 0);
	addinfo (INFO_LOADPCT, "", 0, 0);
	addinfo (INFO_BATTPCT, "", 0, 0);
	addinfo( INFO_BATTVOLT, "", 0, 0);
	addinfo (INFO_UPSTEMP, "", 0, 0);
	addinfo (INFO_STATUS, "", 0, 0);
	addinfo( INFO_RUNTIME, "", 0, 0);
  
	setinfo(INFO_MFR, "%s", "Best Power");
	setinfo(INFO_MODEL, "Micro Ferrups (ME) %d", fc.va);
  
	fprintf(stderr, "Best Power %s detected\n", getdata(INFO_MODEL));
	fprintf(stderr, "Battery voltages %5.1f nominal, %5.1f full, %5.1f empty\n", 
	 fc.idealbvolts,
	 fc.fullvolts,
	 fc.emptyvolts);
}


/* Debugging display from kermit:

----------------------------------------------------
time^M^M^JFeb 20, 22:13:32^M^J^M^J=>id^M^JUnit ID "ME3.1K12345"^M^J^M^J=>
----------------------------------------------------
*/

int execute(char *cmd, char *result, int resultsize) 
{
  int ret;
  char buf[256];
  
  upssend(cmd);
  upsrecv(buf, sizeof(buf), '\012', "");
  ret=upsrecv(result, resultsize, '\015', "\012");
  upsrecv(buf, sizeof(buf), '>', "");
  return ret;

}


void upsdrv_updateinfo(void)
{
  char fstring[512];

  if (! fc.valid) {
    fprintf(stderr, 
	    "upsupdate run before ups_ident() read ups config\n");
    assert(0);
  }

  if (execute("f\r", fstring, sizeof(fstring)) > 0) {
    int inverter=0, charger=0, vin=0, vout=0, btimeleft=0, linestat=0, 
      alstat=0, vaout=0;
    double ampsout=0.0, vbatt=0.0, battpercent=0.0, loadpercent=0.0,
      upstemp=0.0, acfreq=0.0;
    char tmp[16], statstr[128];

    /* Inverter status.  0=off 1=on */
    memcpy(tmp, fstring+16, 2);
    tmp[2] = '\0';
    inverter = atoi(tmp);

    /* Charger status.  0=off 1=on */
    memcpy(tmp, fstring+18, 2);
    tmp[2] = '\0';
    charger = atoi(tmp);
    
    /* Input Voltage. integer number */
    memcpy(tmp, fstring+24, 4);
    tmp[4] = '\0';
    vin = atoi(tmp);

    /* Output Voltage. integer number */
    memcpy(tmp, fstring+28, 4);
    tmp[4] = '\0';
    vout = atoi(tmp);

    /* Iout.  int times 10 */
    memcpy(tmp, fstring+36, 4);
    tmp[4] = '\0';
    ampsout = ((double)(atoi(tmp)) / 10.0);

    /* Battery voltage.  int times 10 */
    memcpy(tmp, fstring+50, 4);
    tmp[4] = '\0';
    vbatt = ((double)(atoi(tmp)) / 10.0);

    /* Volt-amps out.  int  */
    memcpy(tmp, fstring+40, 6);
    tmp[6] = '\0';
    vaout = atoi(tmp);

    /* Line status.  Bitmask */
    memcpy(tmp, fstring+72, 2);
    tmp[2] = '\0';
    linestat = atoi(tmp);

    /* Alarm status reg 1.  Bitmask */
    memcpy(tmp, fstring+20, 2);
    tmp[2] = '\0';
    alstat = atoi(tmp);

    /* Alarm status reg 2.  Bitmask */
    memcpy(tmp, fstring+22, 2);
    tmp[2] = '\0';
    alstat = alstat | (atoi(tmp) << 8);

    /* AC line frequency */
    memcpy(tmp, fstring+54, 4);
    tmp[4]= '\0';
    acfreq = ((double)(atoi(tmp)) / 100.0);

    /* Runtime remaining */
    memcpy(tmp, fstring+58, 4);
    tmp[4]= '\0';
    btimeleft = atoi(tmp);

    /* UPS Temperature */
    memcpy(tmp, fstring+62, 4);
    tmp[4]= '\0';
    upstemp = (double)(atoi(tmp));

    /* Percent Load */
    if (execute("d 16\r", fstring, sizeof(fstring)) > 0) {
      int l;
      sscanf(fstring, "16 FullLoad%% %d", &l);
      loadpercent = (double) l;
    }

    /* Compute battery percent left based on battery voltages. */
    battpercent = ((vbatt - fc.emptyvolts) 
		   / (fc.fullvolts - fc.emptyvolts) * 100.0);
    if (battpercent < 0.0) 
      battpercent = 0.0;
    else if (battpercent > 100.0)
      battpercent = 100.0;
    
    /* Compute status string */
    {
      int lowbatt, overload, replacebatt, boosting, trimming;

      lowbatt = alstat & (1<<1);
      overload = alstat & (1<<6);
      replacebatt = alstat & (1<<10);
      boosting = inverter && (linestat & (1<<2)) && (vin < 115);
      trimming = inverter && (linestat & (1<<2)) && (vin > 115);

      strcpy(tmp, inverter ? "OB" : "OL");
      if (lowbatt) strcat (tmp, " LB");
      if (trimming) strcat (tmp, " TRIM");
      if (boosting) strcat (tmp, " BOOST");
      if (replacebatt) strcat (tmp, " RB");
      if (overload) strcat (tmp, " OVER");

      strlcpy(statstr, tmp, sizeof(statstr));

    }

    if (debugging) {
      fprintf(stderr,
	      "Poll: inverter %d charger %d vin %d vout %d vaout %d btimeleft %d\n",
	      inverter, charger, vin, vout, vaout, btimeleft);
      fprintf(stderr,
	      "      ampsout %5.1f vbatt %5.1f batpcnt %5.1f loadpcnt %5.1f upstemp %5.1f acfreq %5.2f\n",
	      ampsout, vbatt, battpercent, loadpercent, upstemp, acfreq);
      fprintf(stderr,
	      "      STATUS '%s'\n", statstr);

    }

    /* Stuff information into info structures */

    setinfo(INFO_STATUS, "%s", statstr);
    setinfo(INFO_UTILITY, "%05.1f", (double)vin);
    setinfo(INFO_OUTVOLT, "%05.1f", (double)vout);
    setinfo(INFO_BATTPCT, "%02.1f", battpercent);
    setinfo(INFO_LOADPCT, "%02.1f", loadpercent);
    setinfo(INFO_BATTVOLT, "%02.1f", vbatt);
    setinfo(INFO_ACFREQ, "%05.2f", (double)acfreq);
    setinfo(INFO_UPSTEMP, "%05.1f", (double)upstemp);
    setinfo(INFO_RUNTIME, "%d", btimeleft);
    writeinfo();
  
  }

  return;
}


void ups_sync(void)
{
  char	buf[256];

  printf ("Syncing: ");
  fflush (stdout);

  /* A bit better sanity might be good here.  As is, we expect the
     human to observe the time being totally not a time. */

  if (execute("time\r", buf, sizeof(buf)) > 0) {
    fprintf(stderr, "UPS Time: %s\n", buf);
  } else {
    fprintf(stderr, "Error connecting to UPS.\n");
    exit(1);
  }
}

/* power down the attached load immediately */
void upsdrv_shutdown(void)
{
/* NB: hard-wired password */
  upssend("pw377\r");
  upssend("off 1 a\r");	/* power off in 1 second and restart when line power returns */
}

/* list flags and values that you want to receive via -x */
void upsdrv_makevartable(void)
{
        /* allow '-x xyzzy' */
        /* addvar(VAR_FLAG, "xyzzy", "Enable xyzzy mode"); */

        /* allow '-x foo=<some value>' */
        /* addvar(VAR_VALUE, "foo", "Override foo setting"); */
}

void upsdrv_help(void)
{
}


void upsdrv_banner(void)
{
        printf("Network UPS Tools - Best Ferrups Series ME 0.00 (%s)\n\n", UPS_VERSION);
}

static void sync_serial(void) {
	char buffer[10];

	upssend("\r");
  	upsrecv(buffer,sizeof(buffer), '\r', "\012");
	upsrecv(buffer, sizeof(buffer), ENDCHAR, IGNCHARS);

	while (upsrecv(buffer,sizeof(buffer), '>', "\012") <= 0) {
		upssend("\r");
	}
}

/* Begin code stolen from bestups.c */
static void setup_serial(void)
{  
	struct   termios  tio;
			     
	if (tcgetattr(upsfd, &tio) == -1)
		fatal("tcgetattr");
				     
	tio.c_iflag = IXON | IXOFF;
	tio.c_oflag = 0;
	tio.c_cflag = (CS8 | CREAD | HUPCL | CLOCAL);
	tio.c_lflag = 0;
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0;

#ifdef HAVE_CFSETISPEED
	cfsetispeed(&tio, B1200); /* baud change here */
	cfsetospeed(&tio, B1200);
#else
#error This system lacks cfsetispeed() and has no other means to set the speed
#endif

	if (tcsetattr(upsfd, TCSANOW, &tio) == -1)
		fatal("tcsetattr");
/* end code stolen from bestups.c */

	sync_serial();
}


void upsdrv_initups ()
{
  char	temp[256], fcstring[512];

  open_serial(device_path, B1200);
  setup_serial();
  ups_sync();

  /* Obtain Model */
  if (execute("id\r", fcstring, sizeof(fcstring)) < 0) {
    fprintf(stderr, "Failed execute in ups_ident()\n");
    exit(1);
  }
  
  /* response is a one-line packed string starting with $ */
  if (memcmp(fcstring, "Unit", 4)) {
    fprintf(stderr, "Bad response from formatconfig command in ups_ident()\n");
    fprintf(stderr, "id: %s\n", fcstring);
    exit(1);
  }

  if (debugging)
    fprintf(stderr, "id: %s\n", fcstring);
  
  /* chars 4:2  are a two-digit ascii hex enumerated model code */
  memcpy(temp, fcstring+9, 2);
  temp[2] = '\0';

  if (memcmp(temp, "ME", 2))  {
    fprintf(stderr, "Unknown model %s in ups_ident()\n", temp);
    exit(1);
  }
 
  /* hard code this to be an ME3100 for now */
  fc.va = 3100;
  fc.watts = 2200;

  /* determine shutdown battery voltage */
  if (execute("d 29\r", fcstring, sizeof(fcstring)) > 0) {
    sscanf(fcstring, "29 LowBat   %f", &fc.emptyvolts);
  }

  /* determine fully charged battery voltage */
  if (execute("d 31\r", fcstring, sizeof(fcstring)) > 0) {
    sscanf(fcstring, "31 HiBatt   %f", &fc.fullvolts);
  }
  fc.fullvolts=54.20;

  /* determine "ideal" voltage by a guess */
  fc.idealbvolts = ((fc.fullvolts - fc.emptyvolts) * 0.7) + fc.emptyvolts;

  fc.valid = 1;
  return;
}

/* tell main how many items you need */
int upsdrv_infomax(void)
{
	/* every addinfo() consumes an entry.  plan ahead, but don't go
	 * overboard, since this is an easy way to waste memory.
	 */

	return 128;
}

