/* upsrw - simple client for read/write variable access (formerly upsct2)

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "common.h"

#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>

#include "upsclient.h"

static int do_enum(UPSCONN *ups, const char *varname)
{
	char	out[SMALLBUF], temp[SMALLBUF];

	snprintf(out, sizeof(out), "ENUM %s\n", varname);

	if (upscli_sendline(ups, out, strlen(out)) < 0) {
		fprintf(stderr, "Enumerate %s failed: %s\n", 
			varname, upscli_strerror(ups));

		return -1;	/* failed */
	}

	if (upscli_readline(ups, temp, sizeof(temp)) < 0) {
		fprintf(stderr, "Enumerate %s failed: %s\n",
			varname, upscli_strerror(ups));

		return -1;
	}

	if (strncmp(temp, "ENUM", 4) != 0) {
		fprintf(stderr, "Bogus reply from server for %s\n", varname);
		return -1;
	}

	if (upscli_readline(ups, temp, sizeof(temp)) < 0) {
		fprintf(stderr, "Can't get next ENUM value: %s\n",
			upscli_strerror(ups));

		return -1;
	}

	while (strcmp(temp, "END") != 0) {
		printf("Option: %s\n", strstr(temp, "\""));

		if (upscli_readline(ups, temp, sizeof(temp)) < 0) {
			fprintf(stderr, "Can't get next ENUM value: %s\n",
				upscli_strerror(ups));
	
			return -1;
		}
	}

	return 0;
}

static void usage(char *prog)
{
	printf("Network UPS Tools upsrw %s\n\n", UPS_VERSION);
        printf("usage: %s [-h]\n", prog);
	printf("       %s [-s <variable>] [-u <username>] [-p <password>] <ups>\n\n", prog);
	printf("Demo program to set variables within UPS hardware.\n");
        printf("\n");
        printf("  -h            display this help text\n");
	printf("  -s <variable>	specify variable to be changed\n");
	printf("		use -s VAR=VALUE to avoid prompting for value\n");
        printf("  -u <username> set username for command authentication\n");
        printf("  -p <password> set password for command authentication\n");
        printf("\n");
        printf("  <ups>         UPS identifier - myups@localhost, etc.\n");
	printf("\n");
	printf("Call without -s to show all possible read/write variables.\n");

	exit(0);
}

static void clean_exit(UPSCONN *ups, char *upsname, char *hostname, int code)
{
	if (upsname)
		free(upsname);

	if (hostname)
		free(hostname);

	upscli_disconnect(ups);

	exit(code);
}

static void do_setvar(UPSCONN *ups, const char *varname, char *uin,
		const char *pass, char *upsname, char *hostname)
{
	char	newval[SMALLBUF], temp[SMALLBUF], *ptr, *user;

	if (!uin) {
		printf("Username: ");
		user = xmalloc(SMALLBUF);
		fgets(user, SMALLBUF, stdin);

		/* deal with that pesky newline */
		if (strlen(user) > 1)
			user[strlen(user) - 1] = '\0';
	} else {
		user = uin;
	}

	/* leaks - use -p when running in valgrind */
	if (!pass)
		pass = getpass("Password: " );

	/* Check if varname is in VAR=VALUE form */
	if ((ptr = strchr(varname, '=')) != NULL) {
		*ptr++ = 0;
		snprintf(newval, sizeof(newval), "%s", ptr);
	} else {
		printf("Enter new value for %s: ", varname);
		fflush(stdout);
		fgets(newval, sizeof(newval), stdin);
		newval[strlen(newval) - 1] = '\0';
	}

	snprintf(temp, sizeof(temp), "USERNAME %s\n", user);

	if (upscli_sendline(ups, temp, strlen(temp)) < 0) {
		fprintf(stderr, "Can't set username: %s\n", 
			upscli_strerror(ups));

		if (!uin)
			free(user);

		clean_exit(ups, upsname, hostname, 1);
	}

	if (upscli_readline(ups, temp, sizeof(temp)) < 0) {

		if (ups->upserror == UPSCLI_ERR_UNKCOMMAND) {
			fprintf(stderr, "Set username failed due to an "
				"unknown command.\n");

			fprintf(stderr, "You probably need to upgrade upsd.\n");

			if (!uin)
				free(user);
			clean_exit(ups, upsname, hostname, 1);
		}

		fprintf(stderr, "Set username failed: %s\n",
			upscli_strerror(ups));

		if (!uin)
			free(user);
		clean_exit(ups, upsname, hostname, 1);
	}

	snprintf(temp, sizeof(temp), "PASSWORD %s\n", pass);

	if (upscli_sendline(ups, temp, strlen(temp)) < 0) {
		fprintf(stderr, "Can't set password: %s\n", 
			upscli_strerror(ups));

		if (!uin)
			free(user);
		clean_exit(ups, upsname, hostname, 1);
	}

	if (upscli_readline(ups, temp, sizeof(temp)) < 0) {
		fprintf(stderr, "Set password failed: %s\n", 
			upscli_strerror(ups));

		if (!uin)
			free(user);
		clean_exit(ups, upsname, hostname, 1);
	}

	if (upsname)
		snprintf(temp, sizeof(temp), "SET %s@%s %s\n", 
			varname, upsname, newval);
	else
		snprintf(temp, sizeof(temp), "SET %s %s\n", varname, newval);

	if (upscli_sendline(ups, temp, strlen(temp)) < 0) {
		fprintf(stderr, "Can't set variable: %s\n", 
			upscli_strerror(ups));
		if (!uin)
			free(user);
		clean_exit(ups, upsname, hostname, 1);
	}

	if (upscli_readline(ups, temp, sizeof(temp)) < 0) {
		fprintf(stderr, "Set variable failed: %s\n", 
			upscli_strerror(ups));
		if (!uin)
			free(user);
		clean_exit(ups, upsname, hostname, 1);
	}

	if (!uin)
		free(user);
	clean_exit(ups, upsname, hostname, 0);
}	

static void print_rwlist(UPSCONN *ups, char *upsname, char *hostname)
{
	int	typelen;
	char	*v, *sp, *ptr, vars[SMALLBUF], out[SMALLBUF], temp[SMALLBUF],
		*type;

	printf("host: %s\n", hostname);

	if (upscli_getlist(ups, upsname, UPSCLI_LIST_RW, vars, 
		sizeof(vars)) < 0) {
		fprintf(stderr, "Unable to get variable list - %s\n",
		       upscli_strerror(ups));
		clean_exit(ups, upsname, hostname, 1);
	}

	if (strlen(vars) == 0) {
		fprintf(stderr, "Error: no read/write variables available\n");
		clean_exit(ups, upsname, hostname, 1);
	}

	v = vars;
	ptr = strchr(v, ' ');
	if (!ptr) {
		fprintf(stderr, "Broken string from server: %s\n", vars);
		clean_exit(ups, upsname, hostname, 1);
	}

	while (v != NULL) {
		ptr = strchr(v, ' ');
		if (ptr)
			*ptr++ = '\0';

		/* get description */
		snprintf(out, sizeof(out), "VARDESC %s\n", v);

		if (upscli_sendline(ups, out, strlen(out)) < 0) {
			fprintf(stderr, "Can't get description of %s: %s\n",
				v, upscli_strerror(ups));
			clean_exit(ups, upsname, hostname, 1);
		}

		if (upscli_readline(ups, temp, sizeof(temp)) < 0) {
			fprintf(stderr, "Can't get description of %s: %s\n",
				v, upscli_strerror(ups));
			clean_exit(ups, upsname, hostname, 1);
		}

		printf("[%s] %s ", v, strstr(temp, "\""));

		/* now get type */
		snprintf(out, sizeof(out),
			(upsname ? "VARTYPE %s@%s\n" : "VARTYPE %s\n"), v, upsname);

		if (upscli_sendline(ups, out, strlen(out)) < 0) {
			fprintf(stderr, "Can't get type of %s: %s\n",
				v, upscli_strerror(ups));
			clean_exit(ups, upsname, hostname, 1);
		}

		if (upscli_readline(ups, temp, sizeof(temp)) < 0) {
			fprintf(stderr, "Can't get type of %s: %s\n",
				v, upscli_strerror(ups));
			clean_exit(ups, upsname, hostname, 1);
		}

		/* TYPE (ENUM|STRING) <num> */

		sp = strchr(temp, ' ');
		if ((!sp) || (strncmp(temp, "TYPE ", 5) != 0)) {
			fprintf(stderr, "Unexpected response from server [%s]\n",
				temp);
			clean_exit(ups, upsname, hostname, 1);
		}

		*sp++ = '\0';

		type = sp;

		/* now make sure we can handle this vartype */

		sp = strchr(type, ' ');

		if (!sp) {
			fprintf(stderr, "Unexpected vartype from server [%s]\n",
				ptr);
			clean_exit(ups, upsname, hostname, 1);
		}
		
		*sp++ = '\0';

		typelen = strtol(sp, (char **) NULL, 10);

		printf("(%s:%d)\n", type, typelen);

		if (!strcmp(type, "ENUM"))
			if (do_enum(ups, v) != 0)
				clean_exit(ups, upsname, hostname, 1);

		if (!strcmp(type, "STRING")) {
			if (upscli_getvar(ups, upsname, v, temp, 
				sizeof(temp)) < 0) {

				fprintf(stderr, "Can't get value of %s: %s\n",
					temp, upscli_strerror(ups));

				clean_exit(ups, upsname, hostname, 1);
			}
				
			printf("Value: %s\n", temp);
		}
		
		v = ptr;
	}
}

int main(int argc, char **argv)
{
	int	i, port;
	char	*upsname, *hostname, *setvar, *prog;
	char	*password = NULL, *username = NULL;
	UPSCONN	ups;

	setvar = username = NULL;
	prog = argv[0];

	while ((i = getopt(argc, argv, "+s:p:u:")) != EOF) {
		switch (i) {
		case 's':
			setvar = optarg;
			break;
		case 'p':
			password = optarg;
			break;
		case 'u':
			username = optarg;
			break;
		default:
			usage(prog);
			break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc < 1)
		usage(prog);

	upscli_splitname(argv[0], &upsname, &hostname, &port);

	if (upscli_connect(&ups, hostname, port, 0) < 0) {
		fprintf(stderr, "Can't connect: %s\n", upscli_strerror(&ups));
		clean_exit(&ups, upsname, hostname, 1);
	}

	/* setting a variable? */
	if (setvar)
		do_setvar(&ups, setvar, username, password, upsname,
			hostname);

	/* if not, get the list of supported read/write variables */
	print_rwlist(&ups, upsname, hostname);

	clean_exit(&ups, upsname, hostname, 0);

	/* NOTREACHED */
	return 0;
}
