/*
 * Extension: Zmodem with FTP asynhronous background transfer (ZmFTP).
 * Copyright (c) 1995 CAD lab.
 *
 * v1.0 17/11/95 bob@turbo.nsk.su
 */
/*
 * Copyright (c) 1985, 1989, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char sccsid[] = "@(#)ftp.c	8.4 (Berkeley) 4/6/94";
#endif /* not lint */

#include <sys/param.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/file.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <arpa/inet.h>
#include <arpa/ftp.h>
#include <arpa/telnet.h>

#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <varargs.h>
#include <setjmp.h>
#include "ftp.h"

extern int h_errno;
extern int opt_v;
extern int opt_d;

struct	sockaddr_in hisctladdr;
struct	sockaddr_in data_addr;
int	data = -1;
int	abrtflag = 0;
struct	sockaddr_in myctladdr;
FILE	*cin, *cout;
char	reply_string[BUFSIZ];	/* last line of previous reply */

static int	sendport = 0;
static int	cpend;
static char	pasv[64];
static jmp_buf	recvabort;

void recvrequest __P((FILE *, char *));
FILE *dataconn __P((void));
void ptransfer __P((long, struct timeval *, struct timeval *));
void tvsub __P((struct timeval *, struct timeval *, struct timeval *));
void abort_remote __P((FILE *));
void lostpeer __P((void));

/*
 * Lookup "host" and try to connect into "port".
 * Return hostname connected to or NULL.
 */
char *
ftp_connect(host, port)
	char *host;
	int port;
{
	struct hostent *hp = 0;
	int s, len, tos;
	static char hostnamebuf[80];

	memset((char *)&hisctladdr, 0, sizeof (hisctladdr));
	hisctladdr.sin_addr.s_addr = inet_addr(host);
	if (hisctladdr.sin_addr.s_addr != -1) {
		hisctladdr.sin_family = AF_INET;
		(void) strncpy(hostnamebuf, host, sizeof(hostnamebuf));
	} else {
		hp = gethostbyname(host);
		if (hp == NULL) {
			fprintf(stderr, "ftp: %s: %s\n", host, hstrerror(h_errno));
			code = -1;
			return NULL;
		}
		hisctladdr.sin_family = hp->h_addrtype;
		memmove((caddr_t)&hisctladdr.sin_addr,
				hp->h_addr_list[0], hp->h_length);
		(void) strncpy(hostnamebuf, hp->h_name, sizeof(hostnamebuf));
	}
	hostname = hostnamebuf;
	s = socket(hisctladdr.sin_family, SOCK_STREAM, 0);
	if (s < 0) {
		fprintf(stderr, "ftp: socket: %s\n", strerror(errno));
		code = -1;
		return NULL;
	}
	hisctladdr.sin_port = htons(port);
	while (connect(s, (struct sockaddr *)&hisctladdr, sizeof (hisctladdr)) < 0) {
		if (hp && hp->h_addr_list[1]) {
			int oerrno = errno;
			char *ia;

			ia = inet_ntoa(hisctladdr.sin_addr);
			errno = oerrno;
			fprintf(stderr, "ftp: connect to address %s: %s\n",
				ia, strerror(errno));
			hp->h_addr_list++;
			memmove((caddr_t)&hisctladdr.sin_addr,
					hp->h_addr_list[0], hp->h_length);
			fprintf(stderr, "ftp: trying %s...\n",
				inet_ntoa(hisctladdr.sin_addr));
			(void) close(s);
			s = socket(hisctladdr.sin_family, SOCK_STREAM, 0);
			if (s < 0) {
				fprintf(stderr, "ftp: socket: %s\n",
					strerror(errno));
				code = -1;
				return NULL;
			}
			continue;
		}
		fprintf(stderr, "ftp: connect: %s\n", strerror(errno));
		code = -1;
		goto bad;
	}
	len = sizeof (myctladdr);
	if (getsockname(s, (struct sockaddr *)&myctladdr, &len) < 0) {
		fprintf(stderr, "ftp: getsockname: %s\n", strerror(errno));
		code = -1;
		goto bad;
	}
#ifdef IP_TOS
	tos = IPTOS_LOWDELAY;
	if (setsockopt(s, IPPROTO_IP, IP_TOS, (char *)&tos, sizeof(int)) < 0 &&
	    opt_d) {
		fprintf(stderr, "ftp: setsockopt: TOS LOWDELAY ignored\n");
	}
#endif
	if ((cin = fdopen(s, "r")) != NULL)
		cout = fdopen(s, "w");
	if (cin == NULL || cout == NULL) {
		fprintf(stderr, "ftp: fdopen: %s\n", strerror(errno));
		if (cin)
			(void) fclose(cin);
		if (cout)
			(void) fclose(cout);
		code = -1;
		goto bad;
	}
	fprintf(stderr, "ftp: connected to %s\n", hostname);
	if (getreply(0) > 2) { 	/* read startup message from server */
		if (cin)
			(void) fclose(cin);
		if (cout)
			(void) fclose(cout);
		code = -1;
		goto bad;
	}
#ifdef SO_OOBINLINE
	{
	int on = 1;

	if (setsockopt(s, SOL_SOCKET, SO_OOBINLINE, (char *)&on, sizeof(on))
	    < 0 && opt_d) {
			fprintf(stderr, "ftp: setsockopt: %s\n", strerror(errno));
		}
	}
#endif /* SO_OOBINLINE */

	return (hostname);
bad:
	(void) close(s);
	return NULL;
}

/*
 * Login into previously connected host under "username" and "password".
 * Return -1 on error and 0 for success.
 */
int
ftp_login(user, pass)
	char *user, *pass;
{
	int n;

	n = command("USER %s", user);
	if (n == CONTINUE) {
		n = command("PASS %s", pass);
	}
	if (n != COMPLETE) {
		fprintf(stderr, "ftp: login failed\n");
		return -1;
	}
	return 0;
}

/*
 * Get some pieces of file information from file "name" on remote host.
 * Return 0 and filled in struct stat or -1 on error.
 */
int
ftp_stat(name, sp)
	char *name;
	struct stat *sp;
{
	struct tm *t;

	if (command("TYPE I") != COMPLETE) return -1;

	if (command("SIZE %s", name) != COMPLETE) return -1;
	sscanf(reply_string, "%*s %d", &sp->st_size);

	if (command("MDTM %s", name) != COMPLETE) return -1;
	sp->st_mtime = time(NULL);
	t = gmtime(&sp->st_mtime);
	sscanf(reply_string, "%*s %04d%02d%02d%02d%02d%02d", &t->tm_year,
	       &t->tm_mon, &t->tm_mday, &t->tm_hour, &t->tm_min, &t->tm_sec);
	t->tm_year %= 100;
	t->tm_mon--;
	sp->st_mtime = mktime(t);

	return 0;
}

/*
 * Create temporary local file and receive remote file in background into it.
 * Return file stream to read from or NULL.
 */
FILE *
ftp_getbackground(remote, pos)
	char *remote;
	long pos;
{
	FILE *fp;
	char temp[sizeof(TMPFILE)];

	restart_point = pos;
	(void)mktemp(strcpy(temp, TMPFILE));

	if ((pid_ftp = fork()) == 0) {
		(void) signal(SIGQUIT, SIG_IGN);
		(void) signal(SIGHUP, SIG_IGN);
		if ((fp = fopen(temp, "w")) != NULL) {
			if (restart_point) {
				lseek(fileno(fp), restart_point, SEEK_SET);
			}
			if (opt_v) {
				if (restart_point)
					fprintf(stderr, "ftp: restarting \"%s\" from offset %ld\n", remote, restart_point);
				else	fprintf(stderr, "ftp: start receiving \"%s\"\n", remote);
			}
			recvrequest(fp, remote);
			fclose(fp);
		}
		exit(0);
	}
	if (pid_ftp < 0) {
		pid_ftp = 0;
		if (opt_v) {
			fprintf(stderr, "zmtx: can't fork(): %s\n", strerror(errno));
		}
		return NULL;
	}
	flag_ftp_done = 0;
	sleep(1);
	fp = fopen(temp, "r");
	unlink(temp);
	return fp;
}

void
cmdabort()
{
	abrtflag++;
}

/*VARARGS*/
int
command(va_alist)
va_dcl
{
	va_list ap;
	char *fmt;
	int r;
	sig_t oldintr;

	abrtflag = 0;
	if (cout == NULL) {
		fprintf(stderr, "ftp: No control connection for command\n");
		code = -1;
		return (0);
	}
	oldintr = signal(SIGINT, cmdabort);
	va_start(ap);
	fmt = va_arg(ap, char *);
	if (opt_d) {
		fprintf(stderr, "ftp: ");
		vfprintf(stderr, fmt, ap);
		fprintf(stderr, "\n");
		fflush(stderr);
	}
	vfprintf(cout, fmt, ap);
	va_end(ap);
	fprintf(cout, "\r\n");
	(void) fflush(cout);
	cpend = 1;
	r = getreply(!strcmp(fmt, "QUIT"));
	if (abrtflag && oldintr != SIG_IGN)
		(*oldintr)(SIGINT);
	(void) signal(SIGINT, oldintr);
	return (r);
}

int
getreply(expecteof)
	int expecteof;
{
	int c, n;
	int dig;
	int originalcode = 0, continuation = 0;
	sig_t oldintr;
	int pflag = 0;
	char *cp, *pt = pasv;

	oldintr = signal(SIGINT, cmdabort);
	for (;;) {
		dig = n = code = 0;
		cp = reply_string;
		while ((c = getc(cin)) != '\n') {
			if (c == IAC) {     /* handle telnet commands */
				switch (c = getc(cin)) {
				case WILL:
				case WONT:
					c = getc(cin);
					fprintf(cout, "%c%c%c", IAC, DONT, c);
					(void) fflush(cout);
					break;
				case DO:
				case DONT:
					c = getc(cin);
					fprintf(cout, "%c%c%c", IAC, WONT, c);
					(void) fflush(cout);
					break;
				default:
					break;
				}
				continue;
			}
			dig++;
			if (c == EOF) {
				if (expecteof) {
					(void) signal(SIGINT,oldintr);
					code = 221;
					return (0);
				}
				lostpeer();
				fprintf(stderr, "ftp: Service not available, remote server has closed connection\n");
				(void) fflush(stderr);
				code = 421;
				return (4);
			}
			if (c != '\r' && n == '5' && dig > 4)
				(void) putchar(c);
			if (dig < 4 && isdigit(c))
				code = code * 10 + (c - '0');
			if (!pflag && code == 227)
				pflag = 1;
			if (dig > 4 && pflag == 1 && isdigit(c))
				pflag = 2;
			if (pflag == 2) {
				if (c != '\r' && c != ')')
					*pt++ = c;
				else {
					*pt = '\0';
					pflag = 3;
				}
			}
			if (dig == 4 && c == '-') {
				if (continuation)
					code = 0;
				continuation++;
			}
			if (n == 0)
				n = c;
			if (cp < &reply_string[sizeof(reply_string) - 1])
				*cp++ = c;
		}
		if (n == '5') {
			(void) putchar(c);
			(void) fflush (stdout);
		}
		if (continuation && code != originalcode) {
			if (originalcode == 0)
				originalcode = code;
			continue;
		}
		*cp = '\0';
		if (n != '1')
			cpend = 0;
		(void) signal(SIGINT,oldintr);
		if (code == 421 || originalcode == 421)
			lostpeer();
		if (abrtflag && oldintr != cmdabort && oldintr != SIG_IGN)
			(*oldintr)(SIGINT);
		return (n - '0');
	}
}

int
empty(mask, sec)
	struct fd_set *mask;
	int sec;
{
	struct timeval t;

	t.tv_sec = (long) sec;
	t.tv_usec = 0;
	return (select(32, mask, (struct fd_set *) 0, (struct fd_set *) 0, &t));
}

void
abortrecv()
{

	abrtflag = 0;
	longjmp(recvabort, 1);
}

#define HASHBYTES 1024

void
recvrequest(fp, remote)
	FILE *fp;
	char *remote;
{
	FILE *din = 0;
	sig_t oldintr, oldintp;
	int c, d;
	char buf[8192];
	long bytes = 0, hashbytes = HASHBYTES;
	struct timeval start, stop;

	oldintr = NULL;
	oldintp = NULL;
	if (setjmp(recvabort)) {
		while (cpend) {
			(void) getreply(0);
		}
		if (data >= 0) {
			(void) close(data);
			data = -1;
		}
		if (oldintr)
			(void) signal(SIGINT, oldintr);
		code = -1;
		return;
	}
	oldintr = signal(SIGINT, abortrecv);
	if (initconn()) {
		(void) signal(SIGINT, oldintr);
		code = -1;
		return;
	}
	if (setjmp(recvabort)) goto abort;
	if (restart_point && command("REST %ld", (long) restart_point) != CONTINUE)
		return;
	if (command("RETR %s", remote) != PRELIM) {
		(void) signal(SIGINT, oldintr);
		return;
	}
	if ((din = dataconn()) == NULL) goto abort;
	(void) gettimeofday(&start, (struct timezone *)0);

	errno = d = 0;
	while ((c = read(fileno(din), buf, sizeof(buf))) > 0) {
		if ((d = write(fileno(fp), buf, c)) != c)
			break;
		bytes += c;
		if (opt_d) {
			while (bytes >= hashbytes) {
				fprintf(stderr, "#");
				fflush(stderr);
				hashbytes += HASHBYTES;
			}
		}
	}
	if (c < 0) {
		if (errno != EPIPE)
			fprintf(stderr, "ftp: netin: %s\n", strerror(errno));
		bytes = -1;
	}
	(void) signal(SIGINT, oldintr);
	if (oldintp)
		(void) signal(SIGPIPE, oldintp);
	(void) gettimeofday(&stop, (struct timezone *)0);
	(void) fclose(din);
	(void) getreply(0);
	if (bytes > 0) ptransfer(bytes, &start, &stop);
	return;
abort:

/* abort using RFC959 recommended IP,SYNC sequence  */

	(void) gettimeofday(&stop, (struct timezone *)0);
	if (oldintp)
		(void) signal(SIGPIPE, oldintr);
	(void) signal(SIGINT, SIG_IGN);
	if (!cpend) {
		code = -1;
		(void) signal(SIGINT, oldintr);
		return;
	}

	abort_remote(din);
	code = -1;
	if (data >= 0) {
		(void) close(data);
		data = -1;
	}
	if (din)
		(void) fclose(din);
	if (bytes > 0)
		ptransfer(bytes, &start, &stop);
	(void) signal(SIGINT, oldintr);
}

/*
 * Need to start a listen on the data channel before we send the command,
 * otherwise the server's connect may fail.
 */
int
initconn()
{
	char *p, *a;
	int result, len, tmpno = 0;
	int on = 1;

noport:
	data_addr = myctladdr;
	if (sendport)
		data_addr.sin_port = 0;	/* let system pick one */ 
	if (data != -1)
		(void) close(data);
	data = socket(AF_INET, SOCK_STREAM, 0);
	if (data < 0) {
		fprintf(stderr, "ftp: socket: %s\n", strerror(errno));
		if (tmpno)
			sendport = 1;
		return (1);
	}
	if (!sendport)
		if (setsockopt(data, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof (on)) < 0) {
			fprintf(stderr, "ftp: setsockopt (reuse address): %s\n",
				strerror(errno));
			goto bad;
		}
	if (bind(data, (struct sockaddr *)&data_addr, sizeof (data_addr)) < 0) {
		fprintf(stderr, "ftp: bind: %s\n", strerror(errno));
		goto bad;
	}
	len = sizeof (data_addr);
	if (getsockname(data, (struct sockaddr *)&data_addr, &len) < 0) {
		fprintf(stderr, "ftp: getsockname: %s\n", strerror(errno));
		goto bad;
	}
	if (listen(data, 1) < 0)
		fprintf(stderr, "ftp: listen: %s\n", strerror(errno));
	if (sendport) {
		a = (char *)&data_addr.sin_addr;
		p = (char *)&data_addr.sin_port;
#define	UC(b)	(((int)b)&0xff)
		result = command("PORT %d,%d,%d,%d,%d,%d",
		      UC(a[0]), UC(a[1]), UC(a[2]), UC(a[3]),
		      UC(p[0]), UC(p[1]));
		if (result == ERROR && sendport == -1) {
			sendport = 0;
			tmpno = 1;
			goto noport;
		}
		return (result != COMPLETE);
	}
	if (tmpno)
		sendport = 1;
#ifdef IP_TOS
	on = IPTOS_THROUGHPUT;
	if (setsockopt(data, IPPROTO_IP, IP_TOS, (char *)&on, sizeof(int)) < 0 &&
	    opt_d) {
		fprintf(stderr, "ftp: setsockopt: TOS THROUGHPUT ignored\n");
	}
#endif
	return (0);
bad:
	(void) close(data), data = -1;
	if (tmpno)
		sendport = 1;
	return (1);
}

FILE *
dataconn()
{
	struct sockaddr_in from;
	int s, fromlen = sizeof (from), tos;

	s = accept(data, (struct sockaddr *) &from, &fromlen);
	if (s < 0) {
		fprintf(stderr, "fpt: accept: %s\n", strerror(errno));
		(void) close(data), data = -1;
		return (NULL);
	}
	(void) close(data);
	data = s;
#ifdef IP_TOS
	tos = IPTOS_THROUGHPUT;
	if (setsockopt(s, IPPROTO_IP, IP_TOS, (char *)&tos, sizeof(int)) < 0 &&
	    opt_d) {
		fprintf(stderr, "ftp: setsockopt: TOS THROUGHPUT ignored\n");
	}
#endif
	return (fdopen(data, "r"));
}

void
ptransfer(bytes, t0, t1)
	long bytes;
	struct timeval *t0, *t1;
{
	struct timeval td;
	float s, bs;

	if (opt_v) {
		tvsub(&td, t1, t0);
		s = td.tv_sec + (td.tv_usec / 1000000.);
#define	nz(x)	((x) == 0 ? 1 : (x))
		bs = bytes / nz(s);
		fprintf(stderr, "ftp: %ld bytes received in %.2g seconds (%.2g Kbytes/s)\n",
		    bytes, s, bs / 1024.);
		fflush(stderr);
	}
}

void
tvsub(tdiff, t1, t0)
	struct timeval *tdiff, *t1, *t0;
{

	tdiff->tv_sec = t1->tv_sec - t0->tv_sec;
	tdiff->tv_usec = t1->tv_usec - t0->tv_usec;
	if (tdiff->tv_usec < 0)
		tdiff->tv_sec--, tdiff->tv_usec += 1000000;
}

void
abort_remote(din)
	FILE *din;
{
	char buf[BUFSIZ];
	int nfnd;
	struct fd_set mask;

	/*
	 * send IAC in urgent mode instead of DM because 4.3BSD places oob mark
	 * after urgent byte rather than before as is protocol now
	 */
	sprintf(buf, "%c%c%c", IAC, IP, IAC);
	if (send(fileno(cout), buf, 3, MSG_OOB) != 3)
		fprintf(stderr, "ftp: abort: %s\n", strerror(errno));
	fprintf(cout,"%cABOR\r\n", DM);
	(void) fflush(cout);
	FD_ZERO(&mask);
	FD_SET(fileno(cin), &mask);
	if (din) { 
		FD_SET(fileno(din), &mask);
	}
	if ((nfnd = empty(&mask, 10)) <= 0) {
		if (nfnd < 0) {
			fprintf(stderr, "ftp: abort: %s\n", strerror(errno));
		}
		lostpeer();
	}
	if (din && FD_ISSET(fileno(din), &mask)) {
		while (read(fileno(din), buf, BUFSIZ) > 0)
			/* LOOP */;
	}
	if (getreply(0) == ERROR && code == 552) {
		/* 552 needed for nic style abort */
		(void) getreply(0);
	}
	(void) getreply(0);
}

void
lostpeer()
{
	if (cout != NULL) {
		(void) shutdown(fileno(cout), 1+1);
		(void) fclose(cout);
		cout = NULL;
	}
	if (data >= 0) {
		(void) shutdown(data, 1+1);
		(void) close(data);
		data = -1;
	}
}
