/*
 *	Copyright (c) 1996 The CAD lab of the
 *	Siberian State Academy of Telecommunication
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <syslog.h>
#include <errno.h>
#include "netchat.h"


#ifdef	HAVE_MMAP	/* use mmap(); very fast! */

#include <sys/mman.h>

static struct loginfo *topminfo = NULL;	/* global mapped pointer */

/*
 * Save on-line shared loginfo; return 0 on success or -1 on error.
 */
int
saveinfo(flags)
	int flags;
{
	register i;
	static struct loginfo myinfo;
	pid_t mypid = getpid();

	if (topminfo == NULL) {
		struct stat st;
		/*
		 * Open loginfo file for reading and writing.
		 * Create it if doesn't exist.
		 */
		if ((i = open(LOGINFOFILE, O_RDWR|O_CREAT, 0644)) < 0) {
			syslog(LOG_ERR, "open: %s: %m", LOGINFOFILE);
			return -1;
		}
		if (fstat(i, &st) < 0) {
			syslog(LOG_ERR, "fstat: %s: %m", LOGINFOFILE);
			return -1;
		}
		if (sizeof(struct loginfo) * MAXBBSUSERS > st.st_size) {
			/*
			 * File too small.
			 * Expand it to full size of all allowed slots.
			 */
			if (ftruncate(i, sizeof(struct loginfo) * MAXBBSUSERS) < 0) {
				syslog(LOG_ERR, "ftruncate: %s: %m", LOGINFOFILE);
				return -1;
			}
		}
		/*
		 * Map it into memory.
		 */
		topminfo = (struct loginfo *)mmap((caddr_t)0,
				sizeof(struct loginfo) * MAXBBSUSERS,
			       (PROT_READ | PROT_WRITE), MAP_SHARED, i, (off_t)0);
		if (topminfo == (struct loginfo *)-1) {
			syslog(LOG_ERR, "mmap: %s: %m", LOGINFOFILE);
			return -1;
		}
		/*
		 * Prepare my static loginfo.
		 */
		memset(&myinfo, 0, sizeof(struct loginfo));
		strcpy(myinfo.tty, MYTTY);
		strcpy(myinfo.name, username);
		myinfo.ltime = time(NULL);
		myinfo.pid = mypid;
		myinfo.baud = 0;
		myinfo.port = myport;
	}
	/*
	 * Find my slot in memory.
	 */
	for (i = 0; i < MAXBBSUSERS; i++) {
		if (topminfo[i].pid == mypid &&
		    !strcmp(topminfo[i].tty, MYTTY)) break;
	}
	if (i == MAXBBSUSERS) {
		/*
		 * My slot not found. Try to find first empty slot.
		 */
		for (i = 0; i < MAXBBSUSERS; i++)
			if (!topminfo[i].pid || kill(topminfo[i].pid, 0))
				break;
		if (i == MAXBBSUSERS) {
			/*
			 * Empty slots not found. Unable to continue!
			 */
			syslog(LOG_ERR, "saveinfo: no more slots (max %d)", MAXBBSUSERS);
			return -1;
		}
	}
	/* 
	 * Insert loginfo into my own slot.
	 */
	myinfo.flags = flags;
	memcpy(&topminfo[i], &myinfo, sizeof(struct loginfo));
#ifdef	MS_ASYNC
	if (msync((caddr_t)topminfo, sizeof(struct loginfo) * MAXBBSUSERS, MS_ASYNC) < 0)
#else
	if (msync((caddr_t)topminfo, sizeof(struct loginfo) * MAXBBSUSERS) < 0)
#endif
	{
		syslog(LOG_ERR, "msync: %s: %m", LOGINFOFILE);
		return -1;
	}
	return 0;
}

/*
 * Purge myself from loginfo; return 0 on success or -1 on error.
 * Should be unmaped??
 */
int
purgeinfo()
{
	register i;
	pid_t mypid;

	if (topminfo == NULL) return 0;

	mypid = getpid();
	for (i = 0; i < MAXBBSUSERS; i++)
		if (topminfo[i].pid == mypid &&
		    !strcmp(topminfo[i].tty, MYTTY)) break;
	if (i == MAXBBSUSERS) {
		for (i = 0; i < MAXBBSUSERS; i++)
			if (!topminfo[i].pid || kill(topminfo[i].pid, 0))
				break;
		if (i == MAXBBSUSERS) {
			syslog(LOG_ERR, "purgeinfo: slot not found in \"%s\"",
			       LOGINFOFILE);
			return -1;
		}
	}
	memset(&topminfo[i], 0, sizeof(struct loginfo));
#ifdef	MS_ASYNC
	msync((caddr_t)topminfo, sizeof(struct loginfo) * MAXBBSUSERS, MS_ASYNC);
#else
	msync((caddr_t)topminfo, sizeof(struct loginfo) * MAXBBSUSERS);
#endif
	munmap((caddr_t)topminfo, sizeof(struct loginfo) * MAXBBSUSERS);
	topminfo = NULL;
	return 0;
}

static int seekinfo = -1;
static struct loginfo *topfinfo = NULL;

static struct loginfo *
getfinfo()
{
	static int nread = 0;
	static time_t mtime = 0;
	struct loginfo *info;

	if (seekinfo < 0) {
		struct stat st;

		if (stat(LOGINFOFILE, &st) < 0) return NULL;
		if (st.st_mtime != mtime) {
			int fd;
			if ((int)st.st_size % sizeof(struct loginfo)) return NULL;
			if ((fd = open(LOGINFOFILE, O_RDONLY)) < 0)
				return NULL;
			if ((topfinfo = realloc(topfinfo, (int)st.st_size)) == NULL) {
				close(fd);
				return NULL;
			}
			while ((nread = read(fd, topfinfo, (int)st.st_size)) < 0 && errno == EINTR);
			close(fd);
			if (nread != st.st_size) return NULL;
			nread = nread / sizeof(struct loginfo);
			mtime = st.st_mtime;
		}
		seekinfo = 0;
	}
	while (seekinfo < nread) {
		info = &topfinfo[seekinfo++];
		if (info->pid && info->port != myport && !kill(info->pid, 0))
			return info;
	}
	seekinfo = -1;
	return NULL;
}

/*
 * Get on-line shared loginfo; return it or NULL on EOF.
 */
struct loginfo *
getinfo()
{
	struct loginfo *info;

	if (topminfo == NULL) {
		/*
		 * Loginfo in mapped memory unavailable yet; get it from file.
		 */
		return getfinfo();
	}
	if (seekinfo < 0) {
		seekinfo = 0;
		if (topfinfo != NULL) free(topfinfo);
	}
	while (seekinfo < MAXBBSUSERS) {
		info = &topminfo[seekinfo++];
		if (info->pid && info->port != myport && !kill(info->pid, 0))
			return info;
	}
	seekinfo = -1;
	return NULL;
}

void
closeinfo()
{
	seekinfo = -1;
}

#else	/* Use the file operations instead of mmap/msync; very slow! */

int
saveinfo(flags)
	int flags;
{
	int fd, i, nread;
	struct stat st;
	struct loginfo myinfo;
	struct loginfo *topinfo;
	pit_t mypid = getpid();

	/* Prepare loginfo */
	memset(&myinfo, 0, sizeof(struct loginfo));
	strcpy(myinfo.tty, MYTTY);
	strcpy(myinfo.name, username);
	myinfo.ltime = time(NULL);
	myinfo.pid = mypid;
	myinfo.baud = 0;
	myinfo.port = myport;
	myinfo.flags = flags;

	if (stat(LOGINFOFILE, &st) < 0) {	/* Creat it */
		if ((fd = open(LOGINFOFILE, O_WRONLY|O_CREAT, 0664)) < 0) {
			syslog(LOG_ERR, "open: %s: %m", LOGINFOFILE);
			return -1;
		}
		while (write(fd, &myinfo, sizeof(struct loginfo)) < 0 && errno == EINTR);
		close(fd);
		return 0;
	}
	if ((int)st.st_size % sizeof(struct loginfo)) {
		if ((fd = open(LOGINFOFILE, O_WRONLY|O_TRUNC, 0664)) < 0) {
			syslog(LOG_ERR, "open: %s: %m", LOGINFOFILE);
			return -1;
		}
		while (write(fd, &myinfo, sizeof(struct loginfo)) < 0 && errno == EINTR);
		close(fd);
		syslog(LOG_WARNING, "saveinfo: %s: Wrong size -- truncated", LOGINFOFILE);
		return 0;
	}
	if ((topinfo = malloc((int)st.st_size)) == NULL) {
		syslog(LOG_ERR, "malloc: %m");
		return -1;
	}
	if ((fd = open(LOGINFOFILE, O_RDWR)) < 0) {
		free(topinfo);
		syslog(LOG_ERR, "open: %s: %m", LOGINFOFILE);
		return -1;
	}
	while ((nread = read(fd, topinfo, (int)st.st_size)) < 0 && errno == EINTR);
	if (nread != st.st_size) {
		close(fd);
		free(topinfo);
		syslog(LOG_ERR, "read: %s: %m", LOGINFOFILE);
		return -1;
	}
	nread = nread / sizeof(struct loginfo);
	for (i = 0; i < nread; i++)
		if (topinfo[i].pid == mypid &&
		    !strcmp(topinfo[i].tty, myinfo.tty)) break;
	if (i == nread) {
		for (i = 0; i < nread; i++)
			if (!topinfo[i].pid || kill(topinfo[i].pid, 0))
				break;
	}
	free(topinfo);
	lseek(fd, (off_t)(i * sizeof(struct loginfo)), SEEK_SET);
	while (write(fd, &myinfo, sizeof(struct loginfo)) < 0 && errno == EINTR);
	fsync(fd);
	close(fd);
	return 0;
}

int
purgeinfo()
{
	int fd, i, nread;
	struct stat st;
	struct loginfo myinfo;
	struct loginfo *topinfo;
	pid_t mypid;

	if (stat(LOGINFOFILE, &st) < 0) return 0;
	if ((topinfo = malloc((int)st.st_size)) == NULL) return -1;
	if ((fd = open(LOGINFOFILE, O_RDWR)) < 0) return -1;
	while ((nread = read(fd, topinfo, (int)st.st_size)) < 0 && errno == EINTR);
	if (nread != (int)st.st_size || (int)st.st_size % sizeof(struct loginfo)) {
		close(fd);
		free(topinfo);
		return -1;
	}
	nread = nread / sizeof(struct loginfo);
	mypid = getpid();
	for (i = 0; i < nread; i++) {
		if (topinfo[i].pid == mypid &&
		    !strcmp(topinfo[i].tty, MYTTY)) break;
	}
	free(topinfo);
	if (i == nread) {
		close(fd);
		return -1;
	}
	memset(&myinfo, 0, sizeof(struct loginfo));
	lseek(fd, (off_t)(i * sizeof(struct loginfo)), SEEK_SET);
	while (write(fd, &myinfo, sizeof(struct loginfo)) < 0 && errno == EINTR);
	fsync(fd);
	close(fd);
	return 0;
}

static int seekinfo = -1;
static struct loginfo *topinfo = NULL;

struct loginfo *
getinfo()
{
	static int nread = 0;
	static time_t mtime = 0;
	struct loginfo *info;

	if (seekinfo < 0) {
		struct stat st;

		if (stat(LOGINFOFILE, &st) < 0) return NULL;
		if (st.st_mtime != mtime) {
			int fd;
			if ((int)st.st_size % sizeof(struct loginfo)) return NULL;
			if ((fd = open(LOGINFOFILE, O_RDONLY)) < 0)
				return NULL;
			if ((topinfo = realloc(topinfo, (int)st.st_size)) == NULL) {
				close(fd);
				return NULL;
			}
			while ((nread = read(fd, topinfo, (int)st.st_size)) < 0 && errno == EINTR);
			close(fd);
			if (nread != st.st_size) return NULL;
			nread = nread / sizeof(struct loginfo);
			mtime = st.st_mtime;
		}
		seekinfo = 0;
	}
	while (seekinfo < nread) {
		info = &topinfo[seekinfo++];
		if (info->pid && info->port != myport && !kill(info->pid, 0))
			return info;
	}
	seekinfo = -1;
	return NULL;
}

void
closeinfo()
{
	seekinfo = -1;
}

#endif	/* HAVE_MMAP */

char *
flag2str(flag)
	int flag;
{
	static struct fst {
		int f;
		char *s;
	} fs[] = {
		{ INCONF,	"in conference"		},
		{ INTALK,	"in talk"		},
		{ INRECVF,	"receiving file"	},
		{ INSENDF,	"sending file"		},
		{ EXECEXT,	"running ext apps"	},
		{ MESGNO,	"have mesg no"		},
		{ 0,		"available for talk"	},
	};
	register struct fst *p;

	for (p = fs; p->f; p++) if (flag & p->f) break;
	return p->s;
}
