/*
 *	Copyright (c) 1996 by Vladimir Vorobyev <bob@turbo.nsk.su>
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <time.h>
#include <ctype.h>
#include "drive.h"
#include "msgapi.h"

#define	MP	"MsgPack v1.1"
#define	BUFSIZE	4096

int num_msgs, bflag, mflag, qflag;
time_t time_old;
char msgbase[1024];

main(argc, argv)
	int argc;
	char *argv[];
{
	int op, type;
	char *p;
	extern int optind, opterr;
	extern char *optarg;

	num_msgs = time_old = bflag = mflag = qflag = 0;
	opterr = 0;
	while ((op = getopt(argc, argv, "bd:qmn:?")) != EOF)
		switch (op) {
			case 'b':	/* make backup files */
				bflag++;
				break;
			case 'd':	/* days old to expire messages */
				time_old = (time_t)atoi(optarg);
				if (time_old < 1) usage();
				time_old = time(NULL) - time_old * 86400;
				break;
			case 'q':	/* quiet mode */
				qflag++;
				break;
			case 'm':	/* pack mailboxes */
				mflag++;
				break;
			case 'n':	/* max number of messages */
				num_msgs = atoi(optarg);
				if (num_msgs < 1) usage();
				break;
			case '?':
			default:
				usage();
		}
	if (argv[optind] == NULL) usage();

	for (op = optind; (p = argv[optind]) != NULL; optind++) {
		if (mflag) {
			if (!qflag) {
				printf("Scaning userdir in %s...\n", p);
				fflush(stdout);
			}
			if (!packuserbase(p)) op++;
		} else {
			if ((type = getmsgbase(p)) != -1) {
				if ((p = strrchr(msgbase, '/')) != NULL) p++;
				else p = msgbase;
				if (!msgpack(p, type)) op++;
			}
		}
	}
	exit(op != optind);
}

static int
acceptdir(dir)
	register struct dirent *dir;
{
	if (isupper(dir->d_name[0]) && strlen(dir->d_name) > 2) return 1;
	return 0;
}

int
packuserbase(usrdir)
	char *usrdir;
{
	register i, j;
	int nf, nl, type;
	struct dirent **fn, **ln;
	char buf[1024];

	if ((nf = scandir(usrdir, &fn, acceptdir, alphasort)) < 0) {
		perror(usrdir);
		return -1;
	}
	for (i = 0; i < nf; i++) {
		sprintf(buf, "%s/%s", usrdir, fn[i]->d_name);
		if ((nl = scandir(buf, &ln, acceptdir, alphasort)) < 0)
			continue;
		for (j = 0; j < nl; j++) {
			sprintf(buf, "%s/%s/%s/%s", usrdir, fn[i]->d_name, ln[j]->d_name, MAILBOX);
			if ((type = getmsgbase(buf)) != -1) {
				sprintf(buf, "%s %s", fn[i]->d_name, ln[j]->d_name);
				msgpack(buf, type);
			}
		}
		freedir(ln, nl);
	}
	freedir(fn, nf);
	return 0;
}

static int
findmsg(file)
	register struct dirent *file;

{
	register char *p;

	if (!isdigit(file->d_name[0])) return 0;
	if ((p = strchr(file->d_name, '.')) == NULL) return 0;
	if (strcmp(p, ".msg")) return 0;
	return 1;
}

int
getmsgbase(fname)
	char *fname;
{
	int type;
	char *p;
	struct stat st;

	if (*fname == '$') {
		fname++;
		type = MSGTYPE_SQUISH;
	} else	type = MSGTYPE_SDM;

	strcpy(msgbase, fname);

	if (stat(msgbase, &st) < 0) {
		strcat(msgbase, ".sqd");
		if (stat(msgbase, &st) < 0 ||
		    (st.st_mode & S_IFMT) != S_IFREG) {
			perror(fname);
			return -1;
		}
		type = MSGTYPE_SQUISH;
	} else if ((st.st_mode & S_IFMT) == S_IFREG) {
		if ((p = strrchr(msgbase, '.')) == NULL || strcmp(p, ".sqd")) {
			fprintf(stderr, "%s: Not a Squish MsgBase\n", fname);
			return -1;
		}
		type = MSGTYPE_SQUISH;
	} else if ((st.st_mode & S_IFMT) == S_IFDIR) {
		struct dirent **msgs;
		if ((type = scandir(msgbase, &msgs, findmsg, NULL)) <= 0) {
			if (type < 0) perror(fname);
			else fprintf(stderr, "%s: Not SDM or empty MsgBase\n", fname);
			return -1;
		}
		freedir(msgs, type);
		type = MSGTYPE_SDM;
	} else {
		fprintf(stderr, "%s: Not a Squish or SDM MsgBase\n", fname);
		return -1;
	}
	if (type == MSGTYPE_SQUISH && (p = strrchr(msgbase, '.')) != NULL)
		*p = '\0';
	return type;
}

time_t
MsgTime(dosdate)
	union stamp_combo *dosdate;
{
	struct tm tmmsg;
	DosDate_to_TmDate(dosdate, &tmmsg);
	return (mktime(&tmmsg));
}

int
msgpack(area, type)
	char *area;
	int type;
{
	XMSG msg;
	MSG *in_area, *out_area;
	MSGH *in_msg, *out_msg;
	char *what, *ctrl;
	int in_msgn, out_msgn, ctrllen, offs, got;
	struct _minf mi;
	char tmpname[1024], bakname[1024], buf[BUFSIZE];

	mi.req_version = 0;
	mi.def_zone = DEFMSGZONE;
	MsgOpenApi(&mi);

	if ((in_area = MsgOpenArea(msgbase, MSGAREA_NORMAL, type)) == NULL) {
		fprintf(stderr, "can't read \"%s\"\n", msgbase);
		MsgCloseApi();
		return -1;
	}
	MsgLock(in_area);

	strcpy(tmpname, msgbase);
	strcat(tmpname, "~tmp");
	if ((out_area = MsgOpenArea(tmpname, MSGAREA_CREATE, type)) == NULL) {
		fprintf(stderr, "can't creat \"%s\"\n", tmpname);
		MsgCloseArea(in_area);
		MsgCloseApi();
		return -1;
	}
	MsgLock(out_area);

	if (mflag) what = "Mbox";
	else what = "Area";
	for (in_msgn = out_msgn = 1; in_msgn <= MsgHighMsg(in_area); in_msgn++) {
		if (!qflag && (in_msgn % 5) == 0) {
			printf("%s: %-25.25s Msg: %d\r", what, area, in_msgn);
			fflush(stdout);
		}
		if (num_msgs && in_msgn <= (int)MsgGetNumMsg(in_area) - num_msgs)
			continue;

		if ((in_msg = MsgOpenMsg(in_area, MOPEN_READ, in_msgn)) == NULL)
			continue;
		ctrllen = MsgGetCtrlLen(in_msg);
		if ((ctrl = malloc(ctrllen)) == NULL) ctrllen = 0;
		if (MsgReadMsg(in_msg, &msg, 0, 0, NULL, ctrllen, (unsigned char *)ctrl)) {
			fprintf(stderr, "%s: %-25.25s Msg: %d - can't read (error %d)!\n",
				what, area, in_msgn, msgapierr);
			if (ctrl) free(ctrl);
			MsgCloseMsg(in_msg);
			continue;
		}
		if (time_old && MsgTime(&msg.date_written) < time_old) {
			if (ctrl) free(ctrl);
			MsgCloseMsg(in_msg);
			continue;
		}
		msg.attr |= MSGSCANNED;
		msg.replyto = 0;
		memset(msg.replies, '\0', sizeof(msg.replies));
		if ((out_msg = MsgOpenMsg(out_area, MOPEN_CREATE, 0)) == NULL) {
			fprintf(stderr, "%s: %-25.25s Msg: %d - can't write (error %d)!\n",
				what, area, in_msgn, msgapierr);
			if (ctrl) free(ctrl);
			MsgCloseMsg(in_msg);
			continue;
		}
		MsgWriteMsg(out_msg, FALSE, &msg, NULL, 0,
			    MsgGetTextLen(in_msg), ctrllen, (unsigned char *)ctrl);
		for (offs = 0; offs < MsgGetTextLen(in_msg); offs += got) {
			got = MsgReadMsg(in_msg, NULL, offs, BUFSIZE,
					 (unsigned char *)buf, 0, NULL);
			if (got < 1) break;
			MsgWriteMsg(out_msg, TRUE, NULL, (unsigned char *)buf,
				    got, MsgGetTextLen(in_msg), 0, NULL);
		}
		if (ctrl) free(ctrl);
		MsgCloseMsg(out_msg);
		MsgCloseMsg(in_msg);
		out_msgn++;
	}
	MsgCloseArea(out_area);
	MsgCloseArea(in_area);
	MsgCloseApi();

	if (type == MSGTYPE_SQUISH) {
		strcpy(tmpname, msgbase);
		strcat(tmpname, ".sqd");
		if (bflag) {
			strcpy(bakname, msgbase);
			strcat(bakname, ".sqd.bak");
			unlink(bakname);
			link(tmpname, bakname);
		}
		unlink(tmpname);
		strcpy(bakname, msgbase);
		strcat(bakname, "~tmp.sqd");
		link(bakname, tmpname);
		unlink(bakname);

		strcpy(tmpname, msgbase);
		strcat(tmpname, ".sqi");
		if (bflag) {
			strcpy(bakname, msgbase);
			strcat(bakname, ".sqi.bak");
			unlink(bakname);
			link(tmpname, bakname);
		}
		unlink(tmpname);
		strcpy(bakname, msgbase);
		strcat(bakname, "~tmp.sqi");
		link(bakname, tmpname);
		unlink(bakname);
	} else {
		if (bflag) {
			strcpy(bakname, msgbase);
			strcat(bakname, ".bak");
			deldirtree(bakname);
			sprintf(buf, "%s %s %s", SYS_MV, msgbase, bakname);
			system(buf);
		} else	deldirtree(msgbase);
		sprintf(buf, "%s %s %s", SYS_MV, tmpname, msgbase);
		system(buf);
	}
	if (qflag < 2)
		printf("%s: %-25.25s Msg: %d -> %d\n", what, area, in_msgn, out_msgn);
	return 0;
}

usage()
{
	fprintf(stderr, "%s for %s\n", MP, ID);
	fprintf(stderr, "usage: msgpack [-b] [-q] [-m] [-d days_old] [-n num_msgs] msgbases...\n");
	fprintf(stderr, "where:\n\
  -b      \tMake backup copy of message base\n\
  -q      \tQuiet mode, no progress metter (-qq = most quieter)\n\
  -m      \tScan for MailBoxes in userbase tree rooted in `msgbase'\n\
  days_old\tDays old to expire messages\n\
  num_msgs\tNumber of messages to keep in message base\n");
	fprintf(stderr, "\
example1: msgpack -n 500 /usr/bbs/msg/*  -- to pack MsgBases\n\
example2: msgpack -d 30 -m /usr/bbs/usr  -- to pack MailBoxes\n");
	exit(1);
}
