/*
 *	Copyright (c) 1995 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>
#include "bbsuser.h"
#include "finger.h"

extern int errno;

static pid_t bbsuid;		/* UID of bbs */
static char *orgdir;		/* home dir of bbs */
static char *bbsname;		/* name of bbs */
static char userhome[1024];	/* home dir of user */

static struct loginfo *getinfo(void);
static void closeinfo(void);
static char *getuserinfo(char *user, char *nam, char *def);
static int isuserexist(char *username);
static char *makeshowpath(char *path);
extern char *getnamebysign(char *sign);

/*
 * Initialize the bbs stuff.
 * Must be called before any other functions.
 * Return TRUE if bbs exist in system.
 */
int
initbbs()
{
	struct passwd *pwd;
	char *ptr;

	if ((pwd = getpwnam("bbs")) == NULL) return 0;
	bbsuid = pwd->pw_uid;
	orgdir = strdup(pwd->pw_dir);
	if ((ptr = index(pwd->pw_gecos, ',')) != NULL) *ptr = '\0';
	if (pwd->pw_gecos[0] != '\0') bbsname = strdup(pwd->pw_gecos);
	else bbsname = strdup("TNSDrive");
	initcallsign(orgdir);
	return 1;
}

/*
 * Prepare user information for finger and return them.
 * If bbs user not found on this line on-line then return NULL.
 */
PERSON *
enter_bbsuser(line)
	register char *line;
{
	register struct loginfo *info;
	register PERSON *pn;
	char *ptr;

	while ((info = getinfo()) != NULL)
		if (!strcmp(&info->tty[3], line)) break;

	/* No bbs user found on this line!? */
	if (info == NULL) return NULL;

	closeinfo();

	pn = palloc();
	entries++;
	if (phead == NULL) phead = ptail = pn;
	else {
		ptail->next = pn;
		ptail = pn;
	}
	pn->next = NULL;
	pn->hlink = NULL;

	pn->office = pn->officephone = pn->homephone = NULL;

	pn->uid = bbsuid;

	sprintf(tbuf, "bbs#%s", line);
	pn->name = strdup(tbuf);

	if (isuserexist(info->name)) pn->dir = strdup(userhome);
	else pn->dir = orgdir;

	pn->shell = bbsname;

	pn->realname = strdup(info->name);

	if ((ptr = getuserinfo(info->name, USERORG, NULL)) == NULL)
		ptr = getuserinfo(info->name, USERLOCATION, NULL);
	if (ptr != NULL) pn->office = strdup(ptr);

	if ((ptr = getuserinfo(info->name, PHONENUMBER, NULL)) != NULL) {
		char *p;
		if ((p = index(ptr, ',')) != NULL) {
			*p++ = '\0';
			pn->homephone = strdup(p);
		}
		pn->officephone = strdup(ptr);
	}

	pn->whead = NULL;
	return pn;
}

PERSON *
find_bbsuser(line)
	register char *line;
{
	register PERSON *pn;
	register char *ptr;

	for (pn = phead; pn; pn = pn->next)
		if ((ptr = index(pn->name, '#')) != NULL)
			if (!strcmp(++ptr, line)) return pn;
	return NULL;
}

/*
 * Return TRUE and filled in "userhome" if such user exist.
 * It is also return back to "user" full realname if callsign has been used.
 * This function will ignore case in user name.
 */
static int
isuserexist(user)
	char *user;
{
	struct stat st;
	register char *ptr;
	char tmp[256];

	if ((ptr = getnamebysign(user)) != NULL) {
		strncpy(user, ptr, MAXUSRNAMELEN);
		user[MAXUSRNAMELEN] = '\0';
	}
	strcpy(tmp, USRHOMEDIR);
	if (tmp[0] == '/') strcpy(userhome, tmp);
	else	sprintf(userhome, "%s/%s", orgdir, tmp);
	for (ptr = tmp; *user; ptr++, user++) *ptr = tolower(*user);
	*ptr = '\0';
	if (tmp[0] == '\0') return 0;
	for (ptr = tmp; (ptr = strtok(ptr, " .")) != NULL; ptr = NULL) {
		strcat(userhome, "/");
		*ptr = toupper(*ptr);
		strcat(userhome, ptr);
	}
	return (stat(userhome, &st) == 0 && (st.st_mode & S_IFMT) == S_IFDIR);
}

/*
 * Get string value from file. If variable "nam" not found in file "path"
 * or this variable is empty then return back default string "def".
 */
static char *
getstrpathconf(path, nam, def)
	char *path, *nam, *def;
{
	int len, flag = FALSE;
	char *p;
	FILE *fp;
	static char strval[512];

	len = strlen(nam);
	if ((fp = fopen(path, "r")) != NULL) {
		bzero(strval, sizeof(strval));
		while (fgets(strval, sizeof(strval), fp) != NULL) {
			if ((p = index(strval, '\n')) != NULL) *p = '\0';
			if (strval[0] != '\0' && strval[0] != '#' && strval[len] == '=' &&
			    !strncmp(strval, nam, len)) {
				flag = TRUE;
				break;
			}
		}
		fclose(fp);
	}
	if (flag && strval[len+1] != '\0') return &strval[len+1];
	return def;
}

/*
 * Get user's value of variable "nam" from "user".
 * If user "user" not found then return NULL.
 * If variable "nam" not found then return back default string "def".
 */
static int u_charge;

static char *
getuserinfo(user, nam, def)
	char *user, *nam, *def;
{
	struct stat st;
	static char u_name[MAXUSRNAMELEN+1];
	static char u_file[1024];

	if (strcmp(user, u_name)) {
		u_name[0] = '\0';
		u_charge = FALSE;
		if (!isuserexist(user)) return NULL;
		(void)sprintf(u_file, "%s/%s", userhome, CHARGEUSERCONF);
		if (stat(u_file, &st) < 0) {
			(void)sprintf(u_file, "%s/%s", userhome, USERCONF);
			if (stat(u_file, &st) < 0) return NULL;
		} else	u_charge = TRUE;
		(void)strncpy(u_name, user, MAXUSRNAMELEN);
		u_name[MAXUSRNAMELEN] = '\0';
	}
	return getstrpathconf(u_file, nam, def);
}

/*
 * Get loginfo record (on-line user's information).
 * Each call of this function will return the next record of loginfo file.
 * Return NULL if end of loginfo file was reached.
 * Use closeinfo() before if you want to scan loginfo from the first record.
 */
static int seekinfo = -1;
static struct loginfo *topinfo = NULL;

static struct loginfo *
getinfo()
{
	static int nread = 0;
	static time_t mtime = 0;
	struct loginfo *info;

	if (seekinfo < 0) {
		struct stat st;

		if (stat(LOGINFOFILE, &st) < 0) return NULL;
		if (st.st_mtime != mtime) {
			int fd;

			if ((int)st.st_size % sizeof(struct loginfo)) return NULL;
			if ((fd = open(LOGINFOFILE, O_RDONLY)) < 0) return NULL;
			if ((topinfo = realloc(topinfo, (int)st.st_size)) == NULL) {
				close(fd);
				return NULL;
			}
			while ((nread = read(fd, topinfo, (int)st.st_size)) < 0 && errno == EINTR);
			close(fd);
			if (nread != st.st_size) return NULL;
			nread = nread / sizeof(struct loginfo);
			mtime = st.st_mtime;
		}
		seekinfo = 0;
	}
	while (seekinfo < nread) {
		info = &topinfo[seekinfo++];
		if (info->pid && kill(info->pid, 0) == 0) return info;
	}
	seekinfo = -1;
	return NULL;
}

static void
closeinfo()
{
	seekinfo = -1;
}

/*
 * Output to stdout list of on-line users.
 * Return number of bbs users currently on-line.
 */
int
who_is_there()
{
	int lncnt;
	char *ptr;
	struct loginfo *info;
	char city[25];

	lncnt = 0;
	while ((info = getinfo()) != NULL) {
		if ((ptr = getuserinfo(info->name, USERLOCATION, "Unknown")) == NULL)
			continue;
		strncpy(city, ptr, sizeof(city)-1);
		city[sizeof(city)-1] = '\0';
		if (!lncnt++)
			printf("TTY    LoginTime        UserName                 City/State           BaudRate\n");
		printf("%-6.6s %.15s  %-24.24s %-20.20s %d\n", info->tty,
		       ctime(&info->ltime)+4, info->name, city, info->baud);
	}
	if (lncnt) putchar('\n');
	return lncnt;
}

/*
 * Output to stdout list of some usefull user's settings.
 * Return FALSE if "user" does not exist.
 */
int
who_is_who(user)
	char *user;
{
	int i;
	time_t lc_time;
	char *ptr, *ptr2, user_name[MAXUSRNAMELEN+1];

	strncpy(user_name, user, MAXUSRNAMELEN);
	user_name[MAXUSRNAMELEN] = '\0';
	if ((ptr = getuserinfo(user_name, USERPRIVLEVEL, "0")) == NULL)
		return FALSE;
	printf("User name: \t%s\n", user_name);
	printf("User access: \tlevel %d, ", atoi(ptr));
	ptr = getuserinfo(user_name, USERGROUP, USERDEFGROUP);
	printf("group %s, status %s", ptr, u_charge ? "Charge" : "Regular");
	i = -1;
	if ((ptr = getuserinfo(user_name, DAILYTIMELEFT, NULL)) != NULL) {
		i = atoi(ptr);
		if (!i) printf(",Infinity");
	}
	if (getuserinfo(user_name, PASSWORD, NULL) == NULL)
		printf(",Blocked");
	printf("\n");
	ptr = getuserinfo(user_name, DAILYDLLIMIT, NULL);
	if (i >= 0 || ptr != NULL) {
		printf("Daily limits: \t");
		if (i > 0)	printf("%dmin ", i);
		else if (!i)	printf("Free-Time ");
		if (ptr != NULL) {
			i = atoi(ptr);
			if (i)	printf("%dkb ", i);
			else	printf("Free-DL-size ");
		}
		printf("\n");
	}
	if ((ptr = getuserinfo(user_name, LASTCALLTIME, NULL)) != NULL) {
		lc_time = atoi(ptr);
		printf("Last call: \t%s", ctime(&lc_time));
	}
	if ((ptr = getuserinfo(user_name, CALLNUMBER, NULL)) != NULL)
		printf("Call number: \t%d\n", atoi(ptr));
	if ((ptr = getuserinfo(user_name, TERM, NULL)) != NULL)
		printf("Terminal: \t%s\n", ptr);
	if ((ptr = getuserinfo(user_name, LANGUAGE, NULL)) != NULL)
		printf("Language: \t%s\n", ptr);
	if ((ptr = getuserinfo(user_name, MENUPATH, NULL)) != NULL)
		printf("Menu path: \t%s\n", makeshowpath(ptr));
	if ((ptr = getuserinfo(user_name, USERDLSIZE, NULL)) != NULL)
		printf("Download: \t%dkb\n", atoi(ptr));
	if ((ptr = getuserinfo(user_name, USERULSIZE, NULL)) != NULL)
		printf("Upload:	\t%dkb\n", atoi(ptr));
	if ((ptr = getuserinfo(user_name, USERARTREAD, NULL)) != NULL)
		printf("ArtRead: \t%d articles\n", atoi(ptr));
	if ((ptr = getuserinfo(user_name, USERARTPOST, NULL)) != NULL)
		printf("ArtPost: \t%d articles\n", atoi(ptr));
	if ((ptr = getuserinfo(user_name, USERMSGREAD, NULL)) != NULL)
		printf("MsgRead: \t%d messages\n", atoi(ptr));
	if ((ptr = getuserinfo(user_name, USERMSGPOST, NULL)) != NULL)
		printf("MsgPost: \t%d messages\n", atoi(ptr));
	if ((ptr = getuserinfo(user_name, REALNAME, NULL)) != NULL)
		printf("Real name: \t%s\n", ptr);
	if ((ptr = getuserinfo(user_name, USERORG, NULL)) != NULL)
		printf("Organization: \t%s\n", ptr);
	if ((ptr = getuserinfo(user_name, USERLOCATION, NULL)) != NULL)
		printf("Location: \t%s\n", ptr);
	if ((ptr = getuserinfo(user_name, PHONENUMBER, NULL)) != NULL)
		printf("Phone number: \t%s\n", ptr);
	if ((ptr = getuserinfo(user_name, USERDEFFROM, NULL)) != NULL)
		printf("E-mail addr: \t%s\n", ptr);
	if ((ptr = getuserinfo(user_name, SIGNATURE, NULL)) != NULL)
		printf("Signature: \t%s\n", ptr);

	putchar('\n');
	return TRUE;
}

/*
 * Return the bbs-style string of pathname.
 */
static char *
makeshowpath(showpath)
	char *showpath;
{
	int i;
	register char *ptr, *ptr2;
	char buf[1024];
	static char makestrbuf[1024];

	makestrbuf[0] = 0;
	i = strlen(orgdir);
	if (strncmp(orgdir, showpath, i) != 0) i = 0;
	strcpy(buf, &showpath[i]);
	for (ptr = buf; *ptr != '\0'; ptr++) if (*ptr == '_') *ptr = ' ';
	ptr2 = buf;
	i = 0;
	while (ptr = strtok(ptr2, "/")) {
		if (i) strcat(makestrbuf, "-");
		strcat(makestrbuf, "<");
		if ((ptr2 = index(ptr, '.')) != NULL) i = ptr2-ptr;
		else i = strlen(ptr);
		strncat(makestrbuf, ptr, i);
		strcat(makestrbuf, ">");
		ptr2 = NULL;
	}
	return makestrbuf;
}
