/*
 *	Copyright (c) 1994 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	TNSDrive $Id$
 *
 *	$Log$
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <dirent.h>
#include <time.h>
#include <errno.h>
#include <unistd.h>

#include "drive.h"
#include "scandir.h"

#define	FI	"FullIdx v1.4"

#define	CONFIG		".fullidx.conf"
#define	DEFFL		"files.bbs 00_index.txt"

#define	NODESCONAREA	"*** No description on area ***"

char tmp_file[sizeof(TMPFILEMASK)];

char hostname[256];
char hidedir[256];
char fileslist[256];
char banner[MAXPATHLEN];
char areaconf[MAXPATHLEN];
char marknew[80];
char nodesconarea[256];
char nodesconfile[256];

char *flist[MAXCONFLINES];
time_t clknow;
struct tm tmnow;
struct stat st;
int rawlist;

int ndays = 0;
int marknewdays = -1;
unsigned long areabytes;
unsigned long allbytes = 0;
unsigned long allfiles = 0;

extern int errno;

char *progname;

void
err(int ret, ...)
{
	va_list ap;
	char *cp, errmsg[256];

	va_start(ap, ret);
	cp = va_arg(ap, char *);
	(void)vsnprintf(errmsg, sizeof(errmsg), cp, ap);
	va_end(ap);

	if (errno) cp = strerror(errno);
	else cp = "Unknown error";
	fprintf(stderr, "%s: %s: %s\n", progname, errmsg, cp);
	exit(ret);
}

main(argc, argv)
	int argc;
	char **argv;
{
	int op;
	char *conf = NULL;
	char buf[MAXPATHLEN];
	extern int optind, opterr;
	extern char *optarg;
	char *findorgdir();

	if ((conf = strrchr(argv[0], '/')) != NULL) progname = ++conf;
	else progname = argv[0];

	strcpy(tmp_file, TMPFILEMASK);
	mktemp(tmp_file);

	conf = NULL;
	opterr = 0;
	while ((op = getopt(argc, argv, "c:m:n:?")) != EOF)
		switch (op) {
			case 'c':
				conf = optarg;
				break;
			case 'm':
				marknewdays = atoi(optarg);
				break;
			case 'n':
				if (!(ndays = atoi(optarg))) usage();
				break;
			case '?':
			default:
				usage();
		}
	clknow = time(NULL);
	memcpy(&tmnow, localtime(&clknow), sizeof(struct tm));
	flist[0] = NULL;
	strcpy(fileslist, DEFFL);
	strcpy(nodesconfile, NODESCONFILE);
	strcpy(nodesconarea, NODESCONAREA);
	if (argv[optind] != NULL) {
		if (marknewdays < 0) marknewdays = 0;
		if (!arealist(argv[optind], stdout))
			fprintf(stdout, "No files here\n");
		exit(0);
	}
	if (conf == NULL) {
		if ((conf = findorgdir()) == NULL) err(1, "no BBS in this system");
		sprintf(buf, "%s/%s", conf, CONFIG);
		conf = buf;
	}
	parse_conf(conf);
	if (!areaconf[0]) err(1, "can't find AREACONF in \"%s\"", conf);
	if (stat(areaconf, &st) < 0 || !st.st_size || !st.st_mtime)
		err(1, "can't find or empty \"%s\"", areaconf);
	if (!hostname[0]) err(1, "can't find SYSTEMNAME in \"%s\"", conf);
	if (marknewdays < 0) {
		if (marknew[0]) marknewdays = atoi(marknew);
		else marknewdays = 0;
	}
	if (banner[0]) showfile(banner);
	if (ndays) fprintf(stdout,
			   "\nList of New Files for the last %d days on %s\n",
			   ndays, hostname);
	else fprintf(stdout, "\nList of All Files on %s\n", hostname);
	fprintf(stdout, "List created: %s", ctime(&clknow));
	if ((conf = strrchr(areaconf, '/')) != NULL)
		memcpy(buf, areaconf, conf - areaconf);
	else	buf[0] = '\0';
	if ((op = browseareas(areaconf, buf)) > 0)
		fprintf(stdout, "\n\n\tSummary %d KBytes in %d Files in %d Areas on-line\n", allbytes/1024, allfiles, op);
	exit(0);
}

char *
findorgdir()
{
	struct passwd *pwd;

	if ((pwd = getpwnam(BBS)) == NULL)
		if ((pwd = getpwnam("tns")) == NULL) return NULL;
	return pwd->pw_dir;
}

parse_conf(conf)
	char *conf;
{
	char *ptr, *line;

	if ((ptr = loadfile(conf)) == NULL)
		err(1, "can't find \"%s\"", conf);

	hostname[0] = '\0';
	hidedir[0] = '\0';
	fileslist[0] = '\0';
	banner[0] = '\0';
	areaconf[0] = '\0';
	marknew[0] = '\0';

	for (; (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = 0;
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#') continue;
		if (!strncmp(ptr, "SYSTEMNAME=", 11))
			strcpy(hostname, &ptr[11]);
		else if (!strncmp(ptr, "HIDEDIR=", 8))
			strcpy(hidedir, &ptr[8]);
		else if (!strncmp(ptr, "FILESLIST=", 10))
			strcpy(fileslist, &ptr[10]);
		else if (!strncmp(ptr, "BANNER=", 7))
			strcpy(banner, &ptr[7]);
		else if (!strncmp(ptr, "AREACONF=", 9))
			strcpy(areaconf, &ptr[9]);
		else if (!strncmp(ptr, "MARKNEW=", 8))
			strcpy(marknew, &ptr[8]);
		else if (!strncmp(ptr, "NODESCONFILE=", 13))
			strcpy(nodesconfile, &ptr[13]);
		else if (!strncmp(ptr, "NODESCONAREA=", 13))
			strcpy(nodesconarea, &ptr[13]);
	}
	return;
}

showfile(fn)
	char *fn;
{
	FILE *fp;
	char buf[256];

	if ((fp = fopen(fn, "r")) == NULL) err(1, "can't open \"%s\"", fn);
	while (fgets(buf, sizeof(buf), fp) != NULL) fputs(buf, stdout);
	fclose(fp);
	return;
}

browseareas(conf, path)
	char *conf, *path;
{
	int i, n, areafiles, allareas, len;
	char *av[MAXCONFLINES], **ap = av;
	char *av1[MAXCONFLINES], **ap1 = av1;
	int priv[MAXCONFLINES];
	char *ptr, *ptr2, *line, buf[MAXPATHLEN];

	if ((ptr = loadfile(conf)) == NULL)
		err(1, "can't find \"%s\"", conf);

	for (n = 0; n < MAXCONFLINES &&
	     (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = 0;
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#' || *ptr == '.') continue;
		*ap = ptr;
		while ((u_char)*ptr > 0x20) ptr++;
		if (*ptr) for (*ptr++ = 0; (u_char)*ptr <= 0x20 && *ptr != 0; ptr++);
		if (*ptr) *ap1 = ptr;
		else *ap1 = nodesconarea;
		priv[n] = 0;
		if ((ptr2 = strrchr(*ap, '.')) != NULL && isdigitstr(&ptr2[1])) {
			priv[n] = atoi(&ptr2[1]);
			*ptr2 = '\0';
		}
		if (stat(*ap, &st) < 0 || (st.st_mode & S_IFMT) != S_IFDIR ||
		    !(st.st_mode & S_IRUSR) || !(st.st_mode & S_IXUSR))
			continue;
		ap++;
		ap1++;
		n++;
	}
	allareas = 0;
	len = strlen(hidedir);
	for (i = 0; i < n; i++) {
		if (*av[i] == '/') strcpy(buf, av[i]);
		else sprintf(buf, "%s/%s", path, av[i]);
		if ((areafiles = arealist(buf, NULL)) == 0) continue;
		if (len && !strncmp(buf, hidedir, len))
			ptr = &buf[len];
		else	ptr = buf;
		fprintf(stdout, "\n\n\tFile Area Name: %s\n", av1[i]);
		fprintf(stdout, "\tFile Area Path: %s\n", *ptr ? ptr : "*hidden*");
		fprintf(stdout, "\t%ld bytes in %d files", areabytes, areafiles);
		if (priv[i]) fprintf(stdout, "  (privlevel %d)", priv[i]);
		fprintf(stdout, "\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n");
		showfile(tmp_file);
		allbytes += areabytes;
		allfiles += areafiles;
		allareas++;
	}
	unlink(tmp_file);
	return allareas;
}

int
arealist(path, fddef)
	char *path;
	FILE *fddef;
{
	int fd, lncnt, tmsize;
	FILE *fdout;
	char *av[MAXSTACKSIZE], **ap = av;
	char ch, flen[8], fdate[8], outline[1024], tmpbuf[1024];
	char *ptr, *ptr2, *line;
	struct tm *tmfile;
	struct dir_ent *flst;
	static char *month[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
	static char *alloctop = NULL;

	if (alloctop != NULL) {
	    free(alloctop);
	    alloctop = NULL;
	}
	
	areabytes = 0;
	if ((fd = openlist(path)) < 0) {
		if ((lncnt = scan_dir(path, &flst, files_only, 0)) > 0) {
			register i;
			if (lncnt > MAXSTACKSIZE) lncnt = MAXSTACKSIZE;
			for (i = fd = 0; i < lncnt; i++)
				fd += flst[i].len + 1;
			if ((alloctop = ptr = malloc(fd + 2)) == NULL)
				err(1, "can't allocate %d bytes of memory", fd+1);
			for (i = 0, line = ptr; i < lncnt; i++) {
				fd = flst[i].len;
				memcpy(line, flst[i].name, fd);
				line[fd] = '\n';
				line += fd + 1;
			}
			*line = '\0';
			free_dir(&flst, lncnt);
		} else	{
			if (lncnt == 0) return 0;
			err(1, "can't scandir \"%s\"", path);
		}
	} else {
		if (fstat(fd, &st) < 0)
			err(1, "can't fstat \"%s\"", flist[rawlist]);
		if (!st.st_size) return 0;
		if ((alloctop = ptr = malloc((int)st.st_size + 2)) == NULL)
			err(1, "can't allocate %d bytes of memory", (int)st.st_size+1);
		while ((lncnt = read(fd, ptr, (int)st.st_size)) < 0 && errno == EINTR);
		close(fd);
		if (lncnt != st.st_size) err(1, "error reading \"%s\"", tmpbuf);
		if (ptr[lncnt-1] != '\n') ptr[lncnt++] = '\n';
		ptr[lncnt] = 0;
	}
	if (fddef != NULL) fdout = fddef;
	else if ((fdout = fopen(tmp_file, "w")) == NULL)
		err(1, "can't write \"%s\"", tmp_file);
	for (lncnt = 0; ptr != NULL && lncnt < MAXSTACKSIZE &&
	     (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		if (line > ptr && *(line-1) == '\r') *(line-1) = 0;
		*line++ = 0;
		switch(istruefile(path, ptr)) {
			case 0: if (!ndays) fprintf(fdout, "%s\n", ptr);
			case -1: continue;
		}
		(void)strncpy(outline, ptr, sizeof(outline));
		outline[sizeof(outline)-1] = '\0';
		*ap = ptr2 = ptr;
		ptr = outline;
		while ((u_char)*ptr > 0x20) ptr++, ptr2++;
		*ptr2 = '\0';
		if (*ptr) for (*ptr++ = 0; (u_char)*ptr <= 0x20 && *ptr != 0; ptr++);
		if (!*ptr) ptr = nodesconfile;
		ch = ' ';
		for (fd = 0; fd < 2; fd++) {
			sprintf(tmpbuf, "%s/%s", path, *ap);
			if (stat(tmpbuf, &st) == 0) break;
			for (ptr2 = *ap; *ptr2 && *ptr2 != '-' && *ptr2 != '_'; ptr2++);
			if (*ptr2 == '-') *ptr2 = '_';
			else if (*ptr2 == '_') *ptr2 = '-';
			else fd = 2;
		}
		if (fd > 1) {
			if (ndays) continue;
			strcpy(flen, "off");
			strcpy(fdate, "line");
		} else {
			if ((st.st_mode & S_IFMT) != S_IFREG) continue;
			if (ndays && st.st_mtime < clknow - ndays * 86400)
				continue;
			if((int)st.st_size > 9999999) {
			    tmsize = (int)st.st_size / 1024;
			    if(((int)st.st_size % 1024) > 512) tmsize++;
			    sprintf(flen, "%6dK", tmsize);
			} else 
			    sprintf(flen, "%7d", (int)st.st_size);
			areabytes += (int)st.st_size;
			tmfile = localtime(&st.st_mtime);
			if (tmfile->tm_year < tmnow.tm_year)
				sprintf(fdate, "%s-%02d", month[tmfile->tm_mon], 
				(tmfile->tm_year > 99) ? tmfile->tm_year-100 : tmfile->tm_year);
			else	sprintf(fdate, "%s %-2d", month[tmfile->tm_mon], tmfile->tm_mday);
			if (st.st_mtime >= clknow - marknewdays * 86400)
				ch = '*';
		}
		if (strlen(outline) > 14) {
			fprintf(fdout, "%-.79s\n", outline);
			strcpy(outline, " ");
		}
		if (rawlist) sprintf(tmpbuf, "%-14.14s", outline);
		else sprintf(tmpbuf, "%-14.14s%7.7s %-6.6s%c ", outline, flen, fdate, ch);
		do {
			if ((fd = strlen(ptr)) > 49 && !rawlist) {
				for (ptr2 = &ptr[48]; ptr2 != ptr &&
				     *ptr2 != '/' && *ptr2 != '-' &&
				     *ptr2 != '.' && *ptr2 != ',' &&
				     (u_char)*ptr2 > 0x20; ptr2--);
				if (ptr2 == ptr) ptr2 = &ptr[48];
				fd = ++ptr2 - ptr;
			} else ptr2 = NULL;
			fprintf(fdout, "%s%.*s\n", tmpbuf, fd, ptr);
			ptr = ptr2;
			strcpy(tmpbuf, "                              ");
		} while (ptr != NULL);
		ap++;
		lncnt++;
	}
	if (fddef == NULL) fclose(fdout);
	return lncnt;
}

int
istruefile(fareadir, filename)
	char *fareadir, *filename;
{
	register char *p = filename;
	struct stat st;
	char fpath[1024];

	if (*p == '#' || *p == '.' || *p == ';') return -1;
	if ((u_char)*p <= 0x20 || *p == '-') return 0;
	while ((u_char)*p > 0x20 && *p != '.') p++;
	if (p[0] == '.') return (p[1] > 0x20 && p[1] != '.');
	sprintf(fpath, "%s/%.*s", fareadir, p - filename, filename);
	if (stat(fpath, &st) < 0) return 0;
	if ((st.st_mode & S_IFMT) == S_IFREG) return 1;
	return -1;
}

/*
int
istruefile(p)
	register char *p;
{
	if (*p == '#' || *p == '.' || *p == ';') return -1;
	while ((u_char)*p > 0x20 && *p != '.') p++;
	if (p[0] == '.' && p[1] > 0x20 && p[1] != '.') return 1;
	return 0;
}
*/

int
openlist(path)
	char *path;
{
	register fd, i = 0;
	char *ptr, buf[1024];

	if (flist[0] == NULL) {
		ptr = fileslist;
		while ((ptr = strtok(ptr, " \t,;")) != NULL && i < MAXCONFLINES) {
			flist[i++] = ptr;
			ptr = NULL;
		}
		flist[i] = NULL;
		i = 0;
	}
	while (flist[i] != NULL) {
		sprintf(buf, "%s/%s", path, flist[i]);
		if ((fd = open(buf, O_RDONLY)) >= 0) break;
		i++;
	}
	if (flist[i] == NULL) return -1;
	rawlist = i;
	return fd;
}

usage()
{
	fprintf(stderr, "%s\n", FI);
	fprintf(stderr, "usage: fullidx [-c config] [-m marknew] [-n newdays] [directory]\n");
	fprintf(stderr, "where:\n\
  config\tConfig file, default \"%s\" in BBS home directory\n\
  marknew\tMark files as new for the last Number of days (by asterisk)\n\
  newdays\tList of files for the last Number of days, default all\n\
  directory\tList of files from given directory\n", CONFIG);
	exit(1);
}
