/*
 *	Copyright (c) 1997 Rinet Corp., Novosibirsk, Russia
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#ifdef	__STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include <syslog.h>	/* just for error logging levels */

#include "loginfo.h"

extern int errno;

/* error reporting function by default */
void
#ifdef	__STDC__
report(int level, const char *fmt, ...)
#else
report(level, fmt, va_alist)
	int level;
	const char *fmt;
	va_dcl
#endif
{
	va_list ap;

#ifdef	__STDC__
	va_start(ap, fmt);
#else
	va_start(ap);
#endif
	vfprintf(stderr, fmt, ap);
	va_end(ap);
}

void (*error)(int level, const char *fmt, ...) = report;

static pid_t selfpid = 0;
static int seekinfo = -1;
static struct loginfo *topfinfo = NULL;

static struct loginfo *
getfinfo(myself)
	int myself;
{
	static int nread = 0;
	static time_t mtime = 0;
	struct loginfo *info;

	if (seekinfo < 0) {
		struct stat st;

		if (stat(LOGINFO_FILE, &st) < 0) return NULL;
		if (st.st_mtime != mtime) {
			int fd;

			if ((int)st.st_size % sizeof(struct loginfo))
				return NULL;
			if ((fd = open(LOGINFO_FILE, O_RDONLY)) < 0)
				return NULL;
			if ((topfinfo = realloc(topfinfo, (int)st.st_size)) == NULL) {
				close(fd);
				return NULL;
			}
			while ((nread = read(fd, topfinfo, (int)st.st_size)) < 0 && errno == EINTR);
			close(fd);
			if (nread != st.st_size) return NULL;
			nread /= sizeof(struct loginfo);
			mtime = st.st_mtime;
		}
		seekinfo = 0;
	}
	while (seekinfo < nread) {
		info = &topfinfo[seekinfo++];
		if (!info->pid) continue;
		if (myself) {
			if (info->pid == selfpid || !kill(info->pid, 0))
				return info;
		} else {
			if (info->pid != selfpid && !kill(info->pid, 0))
				return info;
		}
	}
	seekinfo = -1;

	return NULL;
}

void
closeinfo()
{
	seekinfo = -1;
}

time_t
infotime()
{
	struct stat st;

	if (stat(LOGINFO_FILE, &st) < 0) return 0;

	return st.st_mtime;
}

#ifdef	HAVE_MMAP	/* use mmap(); very fast! */

#include <sys/mman.h>

static struct loginfo *topminfo = NULL;	/* global mapped pointer */

/*
 * Save on-line shared loginfo; return 0 on success or -1 on error.
 */
int
saveinfo(info)
	struct loginfo *info;
{
	register i;

	if (topminfo == NULL) {
		struct stat st;
		/*
		 * Open loginfo file for reading and writing.
		 * Create it if doesn't exist.
		 */
		if ((i = open(LOGINFO_FILE, O_RDWR|O_CREAT, 0660)) < 0) {
			(*error)(LOG_ERR, "saveinfo: open: %s: %s",
				 LOGINFO_FILE, strerror(errno));
			return -1;
		}
		(void)fchmod(i, 0660);
		(void)fchown(i, getuid(), getgid());
		if (fstat(i, &st) < 0) {
			(*error)(LOG_ERR, "saveinfo: fstat: %s: %s",
				 LOGINFO_FILE, strerror(errno));
			return -1;
		}
		if (sizeof(struct loginfo) * LOGINFO_MAXENT > st.st_size) {
			/*
			 * File too small.
			 * Expand it to full size of all allowed slots.
			 */
			if (ftruncate(i, sizeof(struct loginfo) * LOGINFO_MAXENT) < 0) {
				(*error)(LOG_ERR, "saveinfo: ftruncate: %s: %s",
					 LOGINFO_FILE, strerror(errno));
				return -1;
			}
		}
		/*
		 * Map it into memory.
		 */
		topminfo = (struct loginfo *)mmap((caddr_t)0,
			sizeof(struct loginfo) * LOGINFO_MAXENT,
			(PROT_READ | PROT_WRITE), MAP_SHARED, i, (off_t)0);
		if (topminfo == (struct loginfo *)-1) {
			(*error)(LOG_ERR, "saveinfo: mmap: %s: %s",
				 LOGINFO_FILE, strerror(errno));
			return -1;
		}
	}
	/*
	 * Find this info slot in memory.
	 */
	for (i = 0; i < LOGINFO_MAXENT; i++) {
		if (topminfo[i].pid == info->pid &&
		    !strcmp(topminfo[i].tty, info->tty))
			break;
	}
	if (i == LOGINFO_MAXENT) {
		/*
		 * Appropriate slot not found; try to find first empty slot.
		 */
		for (i = 0; i < LOGINFO_MAXENT; i++) {
			if (!topminfo[i].pid || kill(topminfo[i].pid, 0))
				break;
		}
		if (i == LOGINFO_MAXENT) {
			/*
			 * Empty slot not found -- unable to continue!
			 */
			(*error)(LOG_ERR, "saveinfo: %s: No more slots (max %d)",
				 LOGINFO_FILE, LOGINFO_MAXENT);
			return -1;
		}
	}
	/* 
	 * Insert info slot.
	 */
	memcpy(&topminfo[i], info, sizeof(struct loginfo));
#ifdef	MS_ASYNC
	if (msync((caddr_t)topminfo, sizeof(struct loginfo) * LOGINFO_MAXENT, MS_ASYNC) < 0)
#else
	if (msync((caddr_t)topminfo, sizeof(struct loginfo) * LOGINFO_MAXENT) < 0)
#endif
		(*error)(LOG_WARNING, "saveinfo: msync: %s: %s",
			 LOGINFO_FILE, strerror(errno));

	return 0;
}

/*
 * Delete info slot from loginfo; return 0 on success or -1 on error.
 */
void
purgeinfo(info)
	struct loginfo *info;
{
	register i;

	if (topminfo == NULL) return;

	for (i = 0; i < LOGINFO_MAXENT; i++) {
		if (topminfo[i].pid == info->pid &&
		    !strcmp(topminfo[i].tty, info->tty))
			break;
	}
	if (i == LOGINFO_MAXENT) {
		for (i = 0; i < LOGINFO_MAXENT; i++) {
			if (!topminfo[i].pid || kill(topminfo[i].pid, 0))
				break;
		}
		if (i == LOGINFO_MAXENT) {
			(*error)(LOG_WARNING, "purgeinfo: %s: Slot not found",
				 LOGINFO_FILE);
			return;
		}
	}
	memset(&topminfo[i], 0, sizeof(struct loginfo));
#ifdef	MS_ASYNC
	msync((caddr_t)topminfo, sizeof(struct loginfo) * LOGINFO_MAXENT, MS_ASYNC);
#else
	msync((caddr_t)topminfo, sizeof(struct loginfo) * LOGINFO_MAXENT);
#endif
	munmap((caddr_t)topminfo, sizeof(struct loginfo) * LOGINFO_MAXENT);
	topminfo = NULL;
}

/*
 * Get on-line shared loginfo; return it or NULL on EOF.
 */
struct loginfo *
getinfo(myself)
	int myself;
{
	struct loginfo *info;

	/* first time initialization */
	if (!selfpid) selfpid = getpid();

	if (topminfo == NULL) {
		/*
		 * Loginfo in mapped memory unavailable yet; get it from file.
		 */
		return getfinfo(myself);
	}
	if (seekinfo < 0) {
		seekinfo = 0;
		if (topfinfo != NULL) {
			free(topfinfo);
			topfinfo = NULL;
		}
	}
	while (seekinfo < LOGINFO_MAXENT) {
		info = &topminfo[seekinfo++];
		if (!info->pid) continue;
		if (myself) {
			if (info->pid == selfpid || !kill(info->pid, 0))
				return info;
		} else {
			if (info->pid != selfpid && !kill(info->pid, 0))
				return info;
		}
	}
	seekinfo = -1;

	return NULL;
}

#else	/* Use the file operations instead of mmap; very slow! */

int
saveinfo(info)
	struct loginfo *info;
{
	int fd, i, nread;
	struct stat st;
	struct loginfo *topinfo;

	if (stat(LOGINFO_FILE, &st) < 0) {	/* Creat it */
		if ((fd = open(LOGINFO_FILE, O_WRONLY|O_CREAT, 0660)) < 0) {
			(*error)(LOG_ERR, "saveinfo: creat: %s: %s",
				 LOGINFO_FILE, strerror(errno));
			return -1;
		}
		(void)fchmod(fd, 0660);
		(void)fchown(fd, getuid(), getgid());
		while (write(fd, info, sizeof(struct loginfo)) < 0 && errno == EINTR);
		close(fd);
		return 0;
	}
	if ((int)st.st_size % sizeof(struct loginfo)) {
		/* bad file; rewrite completely */
		if ((fd = open(LOGINFO_FILE, O_WRONLY|O_TRUNC, 0660)) < 0) {
			(*error)(LOG_ERR, "saveinfo: truncate: %s: %s",
				 LOGINFO_FILE, strerror(errno));
			return -1;
		}
		while (write(fd, info, sizeof(struct loginfo)) < 0 && errno == EINTR);
		close(fd);
		(*error)(LOG_WARNING, "saveinfo: %s: Wrong size %d -- truncated",
			 LOGINFO_FILE, (int)st.st_size);
		return 0;
	}
	if ((topinfo = malloc((int)st.st_size)) == NULL) {
		(*error)(LOG_ERR, "saveinfo: malloc: %s: %s",
			 LOGINFO_FILE, strerror(errno));
		return -1;
	}
	if ((fd = open(LOGINFO_FILE, O_RDWR)) < 0) {
		(*error)(LOG_ERR, "saveinfo: open: %s: %s",
			 LOGINFO_FILE, strerror(errno));
		return -1;
	}
	while ((nread = read(fd, topinfo, (int)st.st_size)) < 0 && errno == EINTR);
	if (nread != st.st_size) {
		close(fd);
		free(topinfo);
		(*error)(LOG_ERR, "saveinfo: read: %s: Short read", LOGINFO_FILE);
		return -1;
	}
	nread /= sizeof(struct loginfo);
	for (i = 0; i < nread; i++) {
		if (topinfo[i].pid == info->pid &&
		    !strcmp(topinfo[i].tty, info->tty)) break;
	}
	if (i == nread) {
		for (i = 0; i < nread; i++) {
			if (!topinfo[i].pid || kill(topinfo[i].pid, 0))
				break;
		}
	}
	free(topinfo);
	lseek(fd, (off_t)(i * sizeof(struct loginfo)), SEEK_SET);
	while (write(fd, info, sizeof(struct loginfo)) < 0 && errno == EINTR);
	fsync(fd);
	close(fd);

	return 0;
}

void
purgeinfo(info)
	struct loginfo *info;
{
	int fd, i, nread;
	struct stat st;
	struct loginfo *topinfo, zero;

	if (stat(LOGINFO_FILE, &st) < 0) return;
	if ((topinfo = malloc((int)st.st_size)) == NULL) return;
	if ((fd = open(LOGINFO_FILE, O_RDWR)) < 0) return;
	while ((nread = read(fd, topinfo, (int)st.st_size)) < 0 && errno == EINTR);
	if (nread != (int)st.st_size || (int)st.st_size % sizeof(struct loginfo)) {
		close(fd);
		free(topinfo);
		return;
	}
	nread = nread / sizeof(struct loginfo);
	for (i = 0; i < nread; i++) {
		if (topinfo[i].pid == info->pid &&
		    !strcmp(topinfo[i].tty, info->tty)) break;
	}
	free(topinfo);
	if (i == nread) {
		close(fd);
		return;
	}
	memset(&zero, 0, sizeof(struct loginfo));
	lseek(fd, (off_t)(i * sizeof(struct loginfo)), SEEK_SET);
	while (write(fd, &zero, sizeof(struct loginfo)) < 0 && errno == EINTR);
	fsync(fd);
	close(fd);
}

struct loginfo *
getinfo(myself)
	int myself;
{
	/* first time initialization */
	if (!selfpid) selfpid = getpid();

	return getfinfo(myself);
}

#endif	/* HAVE_MMAP */
