/*
 *	Copyright (c) 1997 Rinet Corp., Novosibirsk, Russia
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <pwd.h>
#include <string.h>
#include <ctype.h>

#include "drive.h"
#include "getconf.h"
#include "callsign.h"
#include "variables.h"

/*
 * The bbs originate directory (from struct passwd -> pw_dir).
 */
char *bbsdir = NULL;

int
initgetconf(orgdir)
	char *orgdir;
{
	if (orgdir != NULL)
		bbsdir = strdup(orgdir);
	else	bbsdir = getbbsdir();

	return (bbsdir != NULL ? 0 : -1);
}

/*
 * Return bbs home directory, or NULL if no bbs found.
 */
char *
getbbsdir()
{
	struct passwd *pw;

	if ((pw = getpwnam("bbs")) == NULL &&
	    (pw = getpwnam("tns")) == NULL)
		return NULL;

	if ((bbsdir = strdup(pw->pw_dir)) == NULL)
		return NULL;

	return bbsdir;
}

/*
 * Return user home directory, or NULL if no such user.
 */
char *
getuserdir(user)
	char *user;
{
	int n;
	register char *p;
	char name[100];
	struct stat st;
	static char home[1024];

	/* prepare users dir */
	if (USRHOMEDIR[0] != '/') {

		if (bbsdir == NULL && getbbsdir() == NULL)
			return NULL;

		(void)strcpy(home, bbsdir);
		(void)strcat(home, "/");
	} else *home = '\0';
	(void)strcat(home, USRHOMEDIR);

	/* make lower case user name; skip leading junk */
	for (n = 1, p = name; *user && p < &name[sizeof(name)-1]; user++) {
		if (n) {
			if (isspace(*user) || *user == '/' || *user == '.')
				continue;
			n = 0;
		}
		*p++ = tolower(*user);
	}
	*p = '\0';
	if (*name == '\0') return NULL;

	/* make sure it's realname not callsign */
	if ((p = getnamebysign(name)) != NULL) {
		(void)strncpy(name, p, sizeof(name));
		name[sizeof(name)-1] = '\0';
	}

	for (n = 0, p = name; (p = strtok(p, " .")) != NULL; n++, p = NULL) {
		(void)strcat(home, "/");
		*p = toupper(*p);
		(void)strcat(home, p);
	}

	if (n == 2 && stat(home, &st) == 0 && (st.st_mode & S_IFMT) == S_IFDIR)
		return home;

	return NULL;
}

/*
 * Return path to user config file, or NULL if no such user.
 */
char *
getuserfile(user)
	char *user;
{
	char *file;
	struct stat st;
	static char path[1024];

	if ((file = getuserdir(user)) == NULL)
		return NULL;

	file = &path[strlen(strcpy(path, file))];
	*file++ = '/';

	(void)strcpy(file, CHARGEUSERCONF);
	if (stat(path, &st) == 0 && st.st_size > sizeof(PASSWORD)-1 &&
	    (st.st_mode & S_IFMT) == S_IFREG) {
		if (atoi(getfileconf(path, USERTIMELEFT, "0")) > 1)
			return path;
		/* have not enough time left */
	}
	(void)strcpy(file, USERCONF);
	if (stat(path, &st) == 0 && st.st_size > sizeof(PASSWORD)-1 &&
	    (st.st_mode & S_IFMT) == S_IFREG)
		return path;

	return NULL;
}

/*
 * Determine user name from his config file path, or NULL.
 */
char *
getusername(path)
	char *path;
{
	int n;
	register char *p;
	struct stat st;
	char home[512];
	static char name[512];

	if (stat(path, &st) < 0 || (st.st_mode & S_IFMT) != S_IFREG)
		return NULL;

	/* prepare users dir */
	if (USRHOMEDIR[0] != '/') {

		if (bbsdir == NULL && getbbsdir() == NULL)
			return NULL;

		(void)strcpy(home, bbsdir);
		(void)strcat(home, "/");
	} else *home = '\0';
	n = strlen(strcat(home, USRHOMEDIR));

	if (strncmp(path, home, n)) /* common user dirs not matched */
		return NULL;

	p = &path[n];
	if (*p != '/') return NULL;

	*name = '\0';
	p = strcpy(home, ++p);
	for (n = 0; (p = strtok(p, "/")) != NULL && n < 2; n++, p = NULL) {
		if (n) (void)strcat(name, " ");
		(void)strcat(name, p);
	}

	return (p != NULL ? name : NULL);
}

/*
 * Get string value from config file.
 */
char *
getfileconf(path, nam, def)
	char *path, *nam, *def;
{
	int len, flag = FALSE;
	char *p;
	FILE *fp;
	static char strval[100];

	len = strlen(nam);
	if ((fp = fopen(path, "r")) != NULL) {
		memset(strval, 0, sizeof(strval));
		while (fgets(strval, sizeof(strval), fp) != NULL) {
			strval[sizeof(strval)-1] = '\0';
			if ((p = strchr(strval, '\n')) != NULL) *p = '\0';
			if (strval[0] && strval[0] != '#' && strval[len] == '=' &&
			    !strncmp(strval, nam, len)) {
				flag = TRUE;
				break;
			}
		}
		fclose(fp);
	}
	if (flag && strval[len+1] != '\0') return &strval[len+1];
	return def;
}

/*
 * Get string value from system config.
 */
char *
getsysconf(var, def)
	char *var, *def;
{
	char *p;

	if ((p = getfileconf(MAINSYSTEMCONF, var, NULL)) == NULL) {
		char path[1024];

		if (bbsdir == NULL && getbbsdir() == NULL)
			return NULL;

		(void)strcpy(path, bbsdir);
		(void)strcat(path, "/");
		(void)strcat(path, SYSTEMCONF);
		p = getfileconf(path, var, def);
	}
	return p;
}

/*
 * Return path string in form suitable for users view.
 */
char *
getshowpath(path)
	char *path;
{
	int i;
	register char *ptr, *ptr2;
	char buf[1024];
	static char makestrbuf[1024];

	makestrbuf[0] = 0;
	if (bbsdir == NULL && getbbsdir() == NULL) i = 0;
	else {
		i = strlen(bbsdir);
		if (strncmp(bbsdir, path, i) != 0) i = 0;
	}
	(void)strcpy(buf, &path[i]);
	for (ptr = buf; *ptr != '\0'; ptr++) if (*ptr == '_') *ptr = ' ';
	ptr2 = buf;
	while ((ptr = strtok(ptr2, "/")) != NULL) {
		if ((ptr2 = strchr(ptr, '.')) != NULL) i = ptr2-ptr;
		else i = strlen(ptr);
		if (i) {
			strcat(makestrbuf, "<");
			strncat(makestrbuf, ptr, i);
			strcat(makestrbuf, ">-");
		}
		ptr2 = NULL;
	}
	i = strlen(makestrbuf);
	if (i && makestrbuf[i-1] == '-') makestrbuf[--i] = '\0';
	return makestrbuf;
}

/*
 * Print detailed user information.
 * Return number of printed lines, or -1 if no such user.
 */
int
getuserinfo(print, user)
	int (*print)(const char *, ...);
	char *user;
{
	int i, lines = 0;
	register char *ptr;
	char *file, gfile[1024], buf[1024];

	if ((file = getuserfile(user)) == NULL)
		return -1;

	if ((ptr = getusername(file)) == NULL) return 0;
	(*print)("%16s: %s", "User name", ptr);
	if ((ptr = getsignbyname(ptr)) != NULL)
		(*print)(" (%s)", ptr);
	(*print)("\n");
	lines++;

	ptr = getfileconf(file, USERPRIVLEVEL, "0");
	(*print)("%16s: level %d, ", "User access", atoi(ptr));

	ptr = getfileconf(file, USERGROUP, USERDEFGROUP);
	(*print)("group %s, ", ptr);

	if ((ptr = strrchr(file, '/')) != NULL &&
	    !strcmp(++ptr, CHARGEUSERCONF)) {
		i = ptr - file;
		(void)strncpy(gfile, file, i);
		(void)strcpy(&gfile[i], TIMEGETCONF);
		ptr = "Charge";
	} else {
		*gfile = '\0';
		ptr = "Regular";
	}
	(*print)("status %s", ptr);

	i = -1;
	if ((ptr = getfileconf(file, DAILYTIMELEFT, NULL)) != NULL) {
		i = atoi(ptr);
		if (!i) (*print)(",Infinity");
	}
	if (getfileconf(file, PASSWORD, NULL) == NULL)
		(*print)(",Blocked");
	(*print)("\n");
	lines++;

	ptr = getfileconf(file, DAILYDLLIMIT, NULL);
	if (i >= 0 || ptr != NULL) {
		(*print)("%16s: ", "Daily limits");
		if (i > 0)	(*print)("%dmin ", i);
		else if (!i)	(*print)("Free-Time ");
		if (ptr != NULL) {
			i = atoi(ptr);
			if (i)	(*print)("%dkb ", i);
			else	(*print)("Free-DL-size ");
		}
		(*print)("\n");
		lines++;
	}

	if (*gfile != '\0' && (i = readlink(gfile, buf, sizeof(buf))) >= 0) {
		buf[i] = '\0';
		(*print)("%16s: %s", "Group access", *buf ? buf : "none");
		if ((ptr = getfileconf(gfile, TIMEGET, NULL)) != NULL)
			(*print)(", timeget %d", atoi(ptr));
		if ((ptr = getfileconf(gfile, USERTIMELEFT, NULL)) != NULL)
			(*print)(", timeleft %d", atoi(ptr));
		(*print)("\n");
		lines++;
	}

	if ((ptr = getfileconf(file, LASTCALLTIME, NULL)) != NULL) {
		i = atoi(ptr);
		(*print)("%16s: %s", "Last call", ctime((time_t *)&i));
		lines++;
	}
	if ((ptr = getfileconf(file, CALLNUMBER, NULL)) != NULL) {
		(*print)("%16s: %d\n", "Call number", atoi(ptr));
		lines++;
	}
	if ((ptr = getfileconf(file, TERM, NULL)) != NULL) {
		(*print)("%16s: %s\n", "Terminal", ptr);
		lines++;
	}
	if ((ptr = getfileconf(file, LANGUAGE, NULL)) != NULL) {
		(*print)("%16s: %s\n", "Language", ptr);
		lines++;
	}
	if ((ptr = getfileconf(file, MENUPATH, NULL)) != NULL) {
		(*print)("%16s: %s\n", "Menu path", getshowpath(ptr));
		lines++;
	}
	if ((ptr = getfileconf(file, USERDLSIZE, NULL)) != NULL) {
		(*print)("%16s: %dkb\n", "Download", atoi(ptr));
		lines++;
	}
	if ((ptr = getfileconf(file, USERULSIZE, NULL)) != NULL) {
		(*print)("%16s: %dkb\n", "Upload", atoi(ptr));
		lines++;
	}
	if ((ptr = getfileconf(file, USERARTREAD, NULL)) != NULL) {
		(*print)("%16s: %d articles\n", "ArtRead", atoi(ptr));
		lines++;
	}
	if ((ptr = getfileconf(file, USERARTPOST, NULL)) != NULL) {
		(*print)("%16s: %d articles\n", "ArtPost", atoi(ptr));
		lines++;
	}
	if ((ptr = getfileconf(file, USERMSGREAD, NULL)) != NULL) {
		(*print)("%16s: %d messages\n", "MsgRead", atoi(ptr));
		lines++;
	}
	if ((ptr = getfileconf(file, USERMSGPOST, NULL)) != NULL) {
		(*print)("%16s: %d messages\n", "MsgPost", atoi(ptr));
		lines++;
	}
	if ((ptr = getfileconf(file, REALNAME, NULL)) != NULL) {
		(*print)("%16s: %s\n", "Real name", ptr);
		lines++;
	}
	if ((ptr = getfileconf(file, USERORG, NULL)) != NULL) {
		(*print)("%16s: %s\n", "Organization", ptr);
		lines++;
	}
	if ((ptr = getfileconf(file, USERLOCATION, NULL)) != NULL) {
		(*print)("%16s: %s\n", "Location", ptr);
		lines++;
	}
	if ((ptr = getfileconf(file, PHONE, NULL)) != NULL) {
		(*print)("%16s: %s\n", "Phone number", ptr);
		lines++;
	}
	if ((ptr = getfileconf(file, USERDEFFROM, NULL)) != NULL) {
		(*print)("%16s: %s\n", "E-mail addr", ptr);
		lines++;
	}
	if ((ptr = getfileconf(file, SIGNATURE, NULL)) != NULL) {
		(*print)("%16s: %s\n", "Signature", ptr);
		lines++;
	}

	return lines;
}
