/*
 *	Copyright (c) 1997 The CAD lab of the
 *	Siberian State Academy of Telecommunication
 *
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

/* mail folder functions */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <unistd.h>
#include <fcntl.h>
#include "drive.h"
#include "sysmsg.h"
#include "usenet.h"
#include "group.h"
#include "ripwraps.h"
#include "variables.h"

#define	BIG_BUF_SIZE	51200	/* 50kb */
#define	STAT_READ	1
#define	STAT_OLD	2
#define	STAT_DEL	4
#define	STAT_STATUS	(STAT_READ | STAT_OLD | STAT_DEL)
#define	STAT_MARK	8

int MsgReadLimit = 0, MsgPostLimit = 0;
int UserMsgRead = 0, UserMsgPost = 0;

int mailbox;		/* flag: mailbox or message folder */
char folder_path[1024];	/* path of current message folder */
FILE *folder_fp = NULL;	/* fp of opened message folder */
long folder_eof;	/* file size of opened message folder */
char *bigbuf;

struct folder_ent {
	char *path;	/* path of message folder */
	char *desc;	/* description of message folder */
	int nmsg;	/* number of messages in folder */
	time_t mtime;	/* modification time of message folder */
	time_t lread;	/* lastread of message folder */
};

struct msg_ent {
	int ostatus;
	int status;	/* message status */
	long head;	/* offset of header */
	long body;	/* offset of body */
	long eof;	/* offset of end */
	int lines;	/* lines in message body */
	time_t date;	/* date/time */
	char *from;	/* from address */
	char *name;	/* from name */
	char *org;	/* organization */
	char *to;	/* to: field as is */
	char *subj;	/* subject */
};

static struct msg_ent *msg;	/* pointer: message array of scaned folder */
static int nmsg;		/* number of messages in scaned folder */
static char *pending_folder_path;
static int pending_mailbox;

extern int ch_instead;

initmailarea()
{
	char *p;

	mailbox = TRUE;		/* default message folder to mailbox */
	folder_fp = NULL;
	msg = NULL;
	nmsg = 0;
	bigbuf = NULL;
	pending_folder_path = NULL;
	sprintf(folder_path, "%s/%s", homedir, MAILBOX);

	if ((p = getuserconf(MSGREADLIMIT, NULL)) != NULL)
		MsgReadLimit = atoi(p);
	if ((p = getuserconf(MSGPOSTLIMIT, NULL)) != NULL)
		MsgPostLimit = atoi(p);
	if ((p = getuserconf(USERMSGREAD, NULL)) != NULL)
		UserMsgRead = atoi(p);
	if ((p = getuserconf(USERMSGPOST, NULL)) != NULL)
		UserMsgPost = atoi(p);
	if ((p = getuserconf(MSGAREAPATH, NULL)) != NULL) {
		if (*p == '$') p++;
 		if (strcmp(folder_path, p) && access(p, R_OK) == 0) {
			strcpy(folder_path, p);
			mailbox = FALSE;
		}
	}
	if (!isgrpright(GSID_MESSAGE, mailbox ? HOMEDIR : folder_path, GRF_S))
		folder_path[0] = '\0';
}

static void
set_pending_folder()
{
	if (pending_folder_path != NULL) {
		strcpy(folder_path, pending_folder_path);
		mailbox = pending_mailbox;
		free(pending_folder_path);
		pending_folder_path = NULL;
	}
}

int
makefolderpath(ptr, buf)
	char *ptr, *buf;
{
	int type;

	if (!strncasecmp(ptr, "mbox:", 5)) {
		ptr += 5;
		type = TRUE;
	} else	type = FALSE;
	if (!strcmp(ptr, HOMEDIR)) {
		sprintf(buf, "%s/%s", homedir, MAILBOX);
		type = TRUE;
	} else if (*ptr != '/') {
		if (*ptr == '~') {
			ptr++;
			if (*ptr == '/') ptr++;
			sprintf(buf, "%s/%s", homedir, ptr);
		} else	sprintf(buf, "%s/%s", dirpath, ptr);
	} else (void)strcpy(buf, ptr);

	return type;
}

void
change_folder(folder)
	char *folder;
{
	int type;
	char buf[1024];

	type = makefolderpath(folder, buf);
	if (folder_fp != NULL) {
		if (pending_folder_path != NULL) free(pending_folder_path);
		pending_folder_path = strdup(buf);
		pending_mailbox = type;
	} else {
		strcpy(folder_path, buf);
		mailbox = type;
	}
	LOGIT(LOG_INFO, "Change MsgArea to \"%s\"", buf);
	putuserconf(MSGAREAPATH, buf);
}

/* Select message folder and return it */
char *
select_folder()
{
	int i, n, j, rval, lncnt, fidx[MAXSCRLINES*2];
	char ch, *ptr, *line, buf[1024];
	struct stat st;
	struct folder_ent folder[MAXCONFLINES];
	extern char working[];

	if (nchr > 5) nchr = 0;
	sprintf(buf, "%s/%s", dirpath, MSGAREACONF);
	if ((ptr = loadfile(buf)) == NULL) return NULL;
	sprintf(buf, "%s/%s", homedir, MAILBOX);

	pleasewait(1);
	for (n = j = 0; n < MAXCONFLINES &&
	     (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = '\0';
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#') continue;
		if (*ptr == '$') ptr++;
		folder[n].path = ptr;
		while ((u_char)*ptr > 0x20) ptr++;
		if (*ptr) for (*ptr++ = 0; *ptr && (u_char)*ptr <= 0x20; ptr++);
		if (!*ptr) continue;
		folder[n].desc = ptr;
		if ((ptr = strrchr(folder[n].path, '.')) != NULL) {
			if (isdigitstr(&ptr[1])) {
				if (atoi(&ptr[1]) > privlevel) continue;
				else *ptr = '\0';
			}
		}
		ptr = folder[n].path;
		if (!isgrpright(GSID_MESSAGE, ptr, GRF_S)) continue;
		rval = makefolderpath(ptr, buf);
		if (stat(buf, &st) == 0 && (st.st_mode & S_IFMT) == S_IFREG) {
			folder[n].nmsg = getmsgnum(buf);
			folder[n].mtime = st.st_mtime;
			if (!rval) {
				folder[n].mtime -= 60;	/* workaround */
				if ((ptr = getrcent(MSG_RC, buf)) != NULL)
					folder[n].lread = atoi(ptr) + 2;
				else	folder[n].lread = lastcalltime;
			} else	folder[n].lread = st.st_atime + 2;
		} else {	/* just create it */
			if ((i = open(buf, O_WRONLY|O_CREAT, 0660)) < 0)
				LOGIT(LOG_ERR, "can't create \"%s\": %m", buf);
			else close(i);
			folder[n].nmsg = 0;
			folder[n].mtime = 0;
			folder[n].lread = 0;
		}
		if (!(termflags & RIPTERM) && !(n % 5))
			putstr("\b%c", working[++j & 3]);
		n++;
	}
	if (!n) {
		warning(MSG_NOAREASAVAILABLE, 1);
		return NULL;
	}
	if (nchr) {
		if ((i = prompt_num(0, 0, n)) > 0) return folder[i-1].path;
		return NULL;
	}
	lncnt = 1;
	for (i = 0; i < n; i++) {
		if (lncnt == 1) {
			lncnt += 2;
			j = 4;
			memset(fidx, 0, sizeof(fidx));
			fidx[0] = -1;
			ptr = makeshowpath(dirpath);
			rval = strlen(ptr) - 40;
			if (rval > 0) ptr += rval;
			sprintf(buf, "%s %s", sysmsg(MSG_AFILEAREAHEAD), ptr);
			if (termflags & (ANSITERM | RIPTERM))
				putstr("\033[H\033[J\033[1;37;44m %-72.72s %5d\033[K\033[40m\n\n", buf, n);
			else {
				clearline();
				putstr("\n %s (%d)\n\n", buf, n);
			}
		}
		fidx[j++] = i+1;
		if (folder[i].mtime > folder[i].lread) ch = '*';
		else ch = ' ';
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\033[1;33m%4d \033[32m%4d\033[31m%c \033[0;36m%-28.28s",
			       i+1, folder[i].nmsg, ch, folder[i].desc);
		else	putstr("%4d) [%4d]%c %-25.25s",
			       i+1, folder[i].nmsg, ch, folder[i].desc);
		if (!(i & 1)) {
			putchr(' ');
			continue;
		}
		putchr('\n');
more_again:
		rval = morenum(MSG_SELECTFAREA, &lncnt, n);
		if (rval == -1) return NULL;
		if (rval > 0) {
			if (rval > n) {
				rval = fidx[rval-n-1];
				if (rval < 0) return NULL;
				if (!rval) {
					lncnt = scrlines;
					goto more_again;
				}
			}
			return folder[rval-1].path;
		}
		if (rval == -2)	i -= (scrlines - 2) * 4;
		else if (rval == -3) i = n - ((scrlines - 2) * 2) - 1;
		else if (rval == -4) i = -1;
		if (i < -1) i = -1;
	}
	if (n & 1) putchr('\n');
	lncnt = scrlines;
	goto more_again;
}

void
havenewmail()
{
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[1;36m");
	putstr("%s\n", sysmsg(MSG_NEWMAILARRIVED));
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[0;37m");
}

int
fast_copy_fp(sfp, tfp, len)
	FILE *sfp, *tfp;
	int len;
{
	int n;

	if (len < 1) return 0;
	if (bigbuf == NULL && (bigbuf = malloc(BIG_BUF_SIZE)) == NULL)
		return -1;
	while (len > 0) {
		if (len > BIG_BUF_SIZE) n = BIG_BUF_SIZE;
		else n = len;
		if (fread(bigbuf, sizeof(char), n, sfp) != n) break;
		if (fwrite(bigbuf, sizeof(char), n, tfp) != n) break;
		len = len - n;
	}
	return (len ? -1 : 0);
}

char *
fast_gets_fp(fp)
	FILE *fp;
{
	int n;
	char *p, *s;
	static char *buf = NULL;

	if (bigbuf == NULL && (bigbuf = malloc(BIG_BUF_SIZE)) == NULL)
		return NULL;
	if (buf == NULL) {	/* first read */
		n = fread(bigbuf, sizeof(char), BIG_BUF_SIZE-1, fp);
		bigbuf[n] = '\0';
		buf = bigbuf;
	}
	s = NULL;
	if ((p = strchr(buf, '\n')) != NULL) {
		*p++ = '\0';
		s = buf;
	} else if (buf > bigbuf) {	/* next read */
		n = strlen(buf);
		if (n) memmove(bigbuf, buf, n);
		n = fread(&bigbuf[n], sizeof(char), (BIG_BUF_SIZE-1)-n, fp);
		bigbuf[n] = '\0';
		buf = bigbuf;
		if ((p = strchr(buf, '\n')) != NULL) {
			*p++ = '\0';
			s = buf;
		}
	}
	buf = p;
	return s;
}

int
getmsgnum(folder)
	char *folder;
{
	FILE *fp;
	int flag, msgs;
	char *buf;

	if ((fp = sfopen(folder, "r")) == NULL) return 0;
	flag = 1;
	msgs = 0;
	while ((buf = fast_gets_fp(fp)) != NULL) {
		if (flag && !strncmp(buf, "From ", 5)) msgs++;
		flag = (*buf == '\0');
	}
	sfclose(fp);
	if (bigbuf != NULL) {
		free(bigbuf);
		bigbuf = NULL;
	}
	return msgs;
}

void
free_msg()
{
	register i;
	if (msg != NULL) {
		for (i = 0; i < nmsg; i++) {
			free(msg[i].from);
			free(msg[i].name);
			free(msg[i].org);
			free(msg[i].to);
			free(msg[i].subj);
		}
		free(msg);
		msg = NULL;
	}
	nmsg = 0;
}

void
getparam(addr, target, source)
	int addr;
	char *target, *source;
{
	char *p;
	while (*source && *source == ' ') source++;
	if (addr && (p = strchr(source, ',')) != NULL) *p = '\0';
	(void) strncpy(target, source, MAX_PARAM_LEN);
	target[MAX_PARAM_LEN-1] = '\0';
}

int
open_folder()
{
	sigfunc oldintr;
	int lines, status, flag;
	long offs;
	struct stat st;
	time_t lastread;
	static time_t mtime;
	char *p, buf[1024];

	char from[MAX_PARAM_LEN];	/* From joe@foo.bar Date... */
	char addr[MAX_PARAM_LEN];	/* From:	*/
	char to[MAX_PARAM_LEN];		/* To:		*/
	char subj[MAX_PARAM_LEN];	/* Subject:	*/
	char date[MAX_PARAM_LEN];	/* Date:	*/
	char org[MAX_PARAM_LEN];	/* Organization:*/

	set_pending_folder();
	if (folder_path[0] == '\0') {
		warning(MSG_SELECTAREAFIRST, 1);
		return 0;
	}
	if (!isgrpright(GSID_MESSAGE, mailbox ? HOMEDIR : folder_path, GRF_R)) {
		nopermission(GRF_R);
		return 0;
	}
	if (stat(folder_path, &st) < 0 || st.st_size < 2 ||
	    (st.st_mode & S_IFMT) != S_IFREG)
		return 0;
	if (msg != NULL && nmsg != 0 &&
	    st.st_mtime == mtime && folder_eof == (long)st.st_size) {
		if (folder_fp == NULL &&
		    (folder_fp = fopen(folder_path, "r")) == NULL) {
			free_msg();
			return 0;
		}
		return nmsg;
	}
	mtime = st.st_mtime;
	folder_eof = (long)st.st_size;
	free_msg();
	if (folder_fp != NULL) fclose(folder_fp);
	if ((folder_fp = fopen(folder_path, "r")) == NULL) return 0;

	oldintr = signal(SIGINT, SIG_IGN);

	if (mailbox == FALSE) {
		if ((p = getrcent(MSG_RC, folder_path)) != NULL)
			lastread = atoi(p) + 2;
		else	lastread = lastcalltime;
	}
	*from = '\0';
	flag = 1;
	while (fgets(buf, sizeof(buf), folder_fp) != NULL) {
		buf[sizeof(buf)-1] = '\0';
		for (p = buf; *p && *p != '\n'; p++)
			if ((unsigned char)*p < 0x20) *p = ' ';
		if (*p == '\0') {	/* too long line */
			flag = 0;
			continue;
		}
		*p++ = '\0';
		if (flag && !strncmp(buf, "From ", 5)) {
			offs = ftell(folder_fp) - (p - buf);
			getparam(0, from, &buf[5]);
			*addr = *org = *date = *subj = *to = '\0';
			if (nmsg) {	/* body lines && end of message */
				if (!msg[nmsg-1].eof) {
					msg[nmsg-1].eof = offs;
					msg[nmsg-1].lines = lines - msg[nmsg-1].lines;
				}
			}
			lines = 1;
			flag = status = 0;
			continue;
		}
		flag = (*buf == '\0');
		lines++;
		if (*from == '\0') continue;
		if (*buf == '\0') {	/* message body follow if any */
			char name[256];
			if ((p = strchr(from, ' ')) != NULL) *p++ = '\0';
			else p = "";
			parsefrom(*addr ? addr : from, buf, name);
			if (mailbox == FALSE || isgrpright(GSID_EMAIL, buf, GRF_S)) {
				msg = realloc(msg, (nmsg+1) * sizeof(struct msg_ent));
				if (msg == NULL) {
					nmsg = 0;
					break;
				}
				msg[nmsg].ostatus = status;
				msg[nmsg].status = status;
				msg[nmsg].head = offs;
				msg[nmsg].body = ftell(folder_fp);
				msg[nmsg].eof = 0; /* assign later */
				msg[nmsg].lines = lines; /* header lines */
				msg[nmsg].date = parsedate(*date ? date : p, NULL);
				msg[nmsg].from = strdup(buf);
				msg[nmsg].name = strdup(name);
				msg[nmsg].org = strdup(*org ? org : "Unknown");
				msg[nmsg].to = strdup(*to ? to : "All");
				msg[nmsg].subj = strdup(*subj ? subj : "Unidentified subject!");
				if (mailbox == FALSE) {
					if (msg[nmsg].date > lastread)
						lastread = msg[nmsg].date;
					else	msg[nmsg].status |= STAT_READ;
				}
				nmsg++;
			}
			*from = '\0';
		} else if (!strncmp(buf, "From: ", 6))
			getparam(1, addr, &buf[6]);
		else if (!strncmp(buf, "To: ", 4))
			getparam(1, to, &buf[4]);
		else if (!strncmp(buf, "Subject: ", 9))
			getparam(0, subj, &buf[9]);
		else if (!strncmp(buf, "Date: ", 6))
			getparam(0, date, &buf[6]);
		else if (!strncmp(buf, "Organization: ", 14))
			getparam(0, org, &buf[14]);
		else if (!strncmp(buf, "Apparently-To: ", 15))
			getparam(1, to, &buf[15]);
		else if (mailbox == TRUE && !strncmp(buf, "Status: ", 8)) {
			for (p = &buf[8]; *p; p++) switch (*p) {
				case 'R': status |= STAT_READ; break;
				case 'O': status |= STAT_OLD; break;
				case 'D': status |= STAT_DEL; break;
				case 'K': status |= STAT_DEL; break;
			}
		}
	}
	if (nmsg) {
		if (!msg[nmsg-1].eof) {
			msg[nmsg-1].eof = folder_eof;
			msg[nmsg-1].lines = lines - msg[nmsg-1].lines;
		}
	} else {
		fclose(folder_fp);
		folder_fp = NULL;
	}
	signal(SIGINT, oldintr);
	return nmsg;
}

void
close_folder()
{
	if (folder_fp != NULL) {
		fclose(folder_fp);
		folder_fp = NULL;
	}
}

/* return -1 on error, 0 - no new message, 1 - have new messages */
int
update_folder()
{
	sigfunc oldintr;
	FILE *fp;
	int i, flag;
	struct stat st;
	char buf[1024];
	
	if (folder_fp == NULL) return -1;

	/*
	 * Currently only mailbox can be updated.
	 * Just close folder and save it lastread.
	 */
	if (mailbox == FALSE) {
		time_t lastread = 0;
		fclose(folder_fp);
		folder_fp = NULL;
		for (i = 0; i < nmsg; i++) {
			if ((msg[i].status & STAT_READ) &&
			    msg[i].date > lastread) lastread = msg[i].date;
		}
		if (lastread) putrcent(MSG_RC, folder_path, itoa(lastread));
		set_pending_folder();
		return 0;
	}
	for (i = 0; i < nmsg; i++) {
		if ((msg[i].status & STAT_STATUS) != (msg[i].ostatus & STAT_STATUS))
			break;
	}
	if (i == nmsg) {	/* not need to update mbox */
		fclose(folder_fp);
		folder_fp = NULL;
		flag = (stat(folder_path, &st) == 0 &&
			(long)st.st_size != folder_eof);
		set_pending_folder();
		return flag;
	}
	sprintf(buf, "%s.%d", folder_path, (int)getpid());
	if ((fp = fopen(buf, "w")) == NULL) {
		LOGIT(LOG_ERR, "can't write \"%s\": %m", buf);
		return -1;
	}
	oldintr = signal(SIGINT, SIG_IGN);
	for (i = 0; i < nmsg; i++) {
		/* skip deleted message */
		if (msg[i].status & STAT_DEL) continue;

		/* save message header */
		fseek(folder_fp, msg[i].head, SEEK_SET);
		flag = -1;
		while (ftell(folder_fp) < msg[i].eof &&
		       fgets(buf, sizeof(buf), folder_fp) != NULL) {
			buf[sizeof(buf)-1] = '\0';
			if (flag) {
				if (buf[0] == '\n') {
					if (flag == -1)
						fprintf(fp, "Status: %sO\n",
							(msg[i].status & STAT_READ) ? "R" : "");
					flag = 0;
				} else if (!strncmp(buf, "Status: ", 8)) {
					fprintf(fp, "Status: %sO\n",
						(msg[i].status & STAT_READ) ? "R" : "");
					flag = 1;
					continue;
				}
			}
			fputs(buf, fp);
			if (!flag) /* quick save message body if any */
				fast_copy_fp(folder_fp, fp, msg[i].eof - ftell(folder_fp));
		}
	}
	flag = 0;
	/* check for new messages */
	if (stat(folder_path, &st) == 0 && (long)st.st_size > folder_eof) {
		/* append new messages */
		close_folder();
		if ((folder_fp = fopen(folder_path, "r")) != NULL) {
			fseek(folder_fp, folder_eof, SEEK_SET);
			fast_copy_fp(folder_fp, fp, (long)st.st_size - folder_eof);
			flag = 1;
		} else flag = -1;
	}
	if (folder_fp != NULL) {
		fclose(folder_fp);
		folder_fp = NULL;
	}
	fclose(fp);
	/* rename temp file to mbox */
	sprintf(buf, "%s.%d", folder_path, (int)getpid());
	if (rename(buf, folder_path) < 0) {
		LOGIT(LOG_ERR, "can't rename \"%s\": %m", buf);
		flag = -1;
	} else	unlink(buf);
	chmod(folder_path, 0660);
	if (bigbuf != NULL) {
		free(bigbuf);
		bigbuf = NULL;
	}
	signal(SIGINT, oldintr);
	set_pending_folder();
	return flag;
}

char *
search_msgbody(mode, msgnum, substr)
	int mode;
	int msgnum;
	char *substr;
{
	long saveoffs;
	char *p, buf[1024];

	saveoffs = ftell(folder_fp);
	fseek(folder_fp, mode ? msg[msgnum].head : msg[msgnum].body, SEEK_SET);
	p = NULL;
	while (ftell(folder_fp) < msg[msgnum].eof &&
	       fgets(buf, sizeof(buf), folder_fp) != NULL) {
		buf[sizeof(buf)-1] = '\0';
		if ((p = __strcasestr(buf, substr)) != NULL) break;
	}
	fseek(folder_fp, saveoffs, SEEK_SET);
	return p;
}

#define	_SBA	1
#define	_SBS	2
#define	_SBB	3
#define	_SBG	4
#define	_SBT	5

int
search_msg(msgnum, where, forward)
	int msgnum, where, forward;
{
	int i;
	char *p, *s_buf;
	extern char author_search_string[], subject_search_string[], body_search_string[];

	switch (where) {
		case _SBA:
			if (forward)	i = MSG_AUTHORSEARCHF;
			else		i = MSG_AUTHORSEARCHB;
			s_buf = author_search_string;
			break;
		case _SBS:
			if (forward)	i = MSG_SUBJSEARCHF;
			else		i = MSG_SUBJSEARCHB;
			s_buf = subject_search_string;
			break;
		case _SBB:
		case _SBG:
			i = MSG_BODYSEARCH;
			s_buf = body_search_string;
			break;
		case _SBT:
			s_buf = (char *)fromfield(FALSE);
			break;
		default: return -1;
	}
	if (where != _SBT) {
		if ((p = prompt_str(i, s_buf, 20)) == NULL) return -1;
		strcpy(s_buf, p);
	}

	i = msgnum;
	do {
		if (forward) i++;
		else i--;
		if (i >= nmsg) i = 0;
		if (i < 0) i = nmsg - 1;
		switch (where) {
			case _SBA:
				if (__strcasestr(msg[i].from, s_buf) != NULL ||
				    __strcasestr(msg[i].name, s_buf) != NULL ||
				    __strcasestr(msg[i].to, s_buf) != NULL)
					return i;
				break;
			case _SBS:
				if (__strcasestr(msg[i].subj, s_buf) != NULL)
					return i;
				break;
			case _SBB:
			case _SBG:
				if (search_msgbody(where != _SBB, i, s_buf) != NULL)
					return i;
				break;
			case _SBT:
				if (__strcasestr(msg[i].to, s_buf) != NULL)
					return i;
				break;
			default: return -1;
		}
	} while (i != msgnum);
	warning(MSG_NOTFOUND, -1);
	return -2;	/* not redraw page */
}

int
open_msg(msgnum)
	int msgnum;
{
	note_page = -1;
	if (folder_fp == NULL || msgnum < 0 || msgnum >= nmsg) return -1;
	if (mailbox == TRUE) {
		if (!isgrpright(GSID_EMAIL, msg[msgnum].from, GRF_R)) {
			nopermission(GRF_R);
			return -1;
		}
		if (MsgReadLimit && UserMsgRead >= MsgReadLimit) {
			warning(MSG_MSGREADLIMIT, 1);
			return -1;
		}
	}
	if ((msg[msgnum].status & STAT_READ) == 0) {
		msg[msgnum].status |= STAT_READ; /* mark message as read */
		LOGIT(LOG_INFO, "ReadMsg #%d in \"%s\"", msgnum, folder_path);
		if (mailbox == TRUE) UserMsgRead++;
	}
	rotate = 0; /* normal mode, not rot13 */
	fseek(folder_fp, msg[msgnum].body, SEEK_SET);
	note_page = 0;
	note_mark[0] = ftell(folder_fp);
	note_end = FALSE;
	return 0;
}

int
show_msg(msgnum)
	int msgnum;
{
	char ch, *p;
	int n, i;
	void MsgToFile();

restart:
	if (msgnum != this_resp) {	   /* remember current & previous */
		last_resp = this_resp;	   /* message for - command */
		this_resp = msgnum;
	}
	if (open_msg(msgnum) < 0) {
		warning(MSG_MESSAGEUNAVAIL, 1);
		return msgnum;
	}
	show_msg_page(msgnum);
	showbuttons(0);

	while (1) {
		if (ch_instead) {
			ch = ch_instead;
			ch_instead = 0;
		} else	ch = getchr(1);

		if (ch >= '0' && ch <= '9') {
			if ((n = prompt_num(MSG_ENTERRESPNUM, ch - '0', nmsg)) > 0) {
				msgnum = n - 1;
				goto restart;
			}
		} else switch (ch) {
			case 'a':	/* author search forward */
			case 'A':	/* author search backward */
				i = (ch == 'a');
				if ((n = search_msg(msgnum, _SBA, i)) >= 0) {
					msgnum = n;
					goto restart;
				}
				if (n == -1) redraw_msg_page(msgnum);
				break;
			case 'y':	/* to this user search forward */
				if ((n = search_msg(msgnum, _SBT, 1)) >= 0) {
					msgnum = n;
					goto restart;
				}
				if (n == -1) redraw_msg_page(msgnum);
				break;
			case 's':	/* subject search forward */
			case 'S':	/* subject search backward */
				i = (ch == 's');
				if ((n = search_msg(msgnum, _SBS, i)) >= 0) {
					msgnum = n;
					goto restart;
				}
				if (n == -1) redraw_msg_page(msgnum);
				break;
			case 'B':	/* body search */
			case 'G':	/* global search */
				i = (ch == 'B' ? _SBB : _SBG);
				if ((n = search_msg(msgnum, i, 1)) >= 0) {
					msgnum = n;
					goto restart;
				}
				if (n == -1) redraw_msg_page(msgnum);
				break;
			case ctrl('X'):
			case '%':	/* toggle rot-13 mode */
				if (rotate)	rotate = 0;
				else		rotate = 13;
				redraw_msg_page(msgnum);
				break;
			case 'P':	/* previous unread message */
				for (n = msgnum-1; n >= 0; n--) {
					if ((msg[n].status & STAT_READ) == 0) {
						msgnum = n;
						goto restart;
					}
				}
				warning(MSG_NOPREVUNREADMSG, -1);
				break;
			case 'q':	/* return */
			case 'Q':
				return msgnum;
			case ctrl('R'):	  /* redraw beginning of message */
				if (note_page < 0)
					warning(MSG_MESSAGEUNAVAIL, 1);
				else {
					note_page = 0;
					note_end = FALSE;
					fseek(folder_fp, note_mark[0], SEEK_SET);
					show_msg_page(msgnum);
				}
				break;
			case ctrl('L'):
			case ctrl('W'):
				redraw_msg_page(msgnum);
				break;
			case '\b':
			case 'b':	/* back a page */
			case 'U':
				if (note_page <= 1) {
					if (msgnum - 1 < 0) return msgnum;
					msgnum--;
					goto restart;
				}
				note_page -= 2;
				note_end = FALSE;
				fseek(folder_fp, note_mark[note_page], SEEK_SET);
				show_msg_page(msgnum);
				break;
			case '-':	/* show last viewed message */
				if (last_resp < 0 || last_resp >= nmsg) {
					warning(MSG_NOLASTMESSAGE, -1);
					break;
				}
				msgnum = last_resp;
				goto restart;
			case 'p':	/* previous message */
			case '<':
				if (msgnum - 1 < 0) return msgnum;
				msgnum--;
				goto restart;
			case 'n':	/* next message */
			case '>':
				if (msgnum + 1 >= nmsg) return msgnum;
				msgnum++;
				goto restart;
			case ' ': 	/* next page or message */
			case 'D':
				if (note_page < 0 || note_end) {
					if (msgnum + 1 >= nmsg)
						return msgnum;
					msgnum++;
					goto restart;
				}
				show_msg_page(msgnum);
				break;
			case '\t': 	/* next page or unread message */
				if (note_page < 0 || note_end) {
					for (n = msgnum+1; n < nmsg; n++) {
						if ((msg[n].status & STAT_READ) == 0) {
							msgnum = n;
							goto restart;
						}
					}
					return msgnum;
				}
				show_msg_page(msgnum);
				break;
			case '\r':
			case '\n':
			case 'N':	/* next unread message */
				for (n = msgnum+1; n < nmsg; n++) {
					if ((msg[n].status & STAT_READ) == 0) {
						msgnum = n;
						goto restart;
					}
				}
				if (ch != 'N') return msgnum;
				warning(MSG_NONEXTUNREADMSG, -1);
				break;
			case 'H':	/* show message header */
				note_page = 0;
				note_end = FALSE;
				fseek(folder_fp, msg[msgnum].head, SEEK_SET);
				show_msg_page(msgnum);
				break;
			case 'k':	/* mark/unmark message for delete */
			case 'd':
				if (mailbox == TRUE &&
				    isgrpright(GSID_MESSAGE, HOMEDIR, GRF_D) &&
				    isgrpright(GSID_EMAIL, msg[msgnum].from, GRF_D)) {
					if (msg[msgnum].status & STAT_DEL) {
						msg[msgnum].status &= ~STAT_DEL;
						LOGIT(LOG_NOTICE, "UnDeleteMsg #%d in \"%s\"",
						    msgnum+1, folder_path);
					} else {
						msg[msgnum].status |= STAT_DEL;
						LOGIT(LOG_NOTICE, "DeleteMsg #%d in \"%s\"",
						    msgnum+1, folder_path);
					}
					goto restart;
				}
				nopermission(GRF_D);
				redraw_msg_page(msgnum);
				break;
			case 'e':	/* enter new mail */
				if (isgrpright(GSID_MESSAGE, mailbox ?
					       HOMEDIR : folder_path, GRF_W)) {
					if (mailbox)
						enter_mail(NULL, NULL, NULL, 0);
					else	enter_folder(folder_path, NULL, NULL, NULL, 0);
				} else	nopermission(GRF_W);
				redraw_msg_page(msgnum);
				break;
			case 'f':	/* reply to folder */
			case 'F':
				if (mailbox == FALSE) {
					if (isgrpright(GSID_MESSAGE, folder_path, GRF_W))
						reply_letter(folder_path, msgnum, ch == 'F');
					else	nopermission(GRF_W);
					redraw_msg_page(msgnum);
					break;
				}
			case 'r':	/* reply by mail */
			case 'R':
				i = (ch == 'R' || ch == 'F');
				if (isgrpright(GSID_MESSAGE, HOMEDIR, GRF_W))
					reply_letter(NULL, msgnum, i);
				else	nopermission(GRF_W);
				redraw_msg_page(msgnum);
				break;
			case 'm':	/* forward by mail */
				if (isgrpright(GSID_MESSAGE, HOMEDIR, GRF_W))
					forward_letter(NULL, msgnum);
				else	nopermission(GRF_W);
				redraw_msg_page(msgnum);
				break;
			case 'M':	/* forward to folder */
				if ((p = select_folder()) != NULL) {
					char buf[1024];
					i = makefolderpath(p, buf);
					if (isgrpright(GSID_MESSAGE, i ? HOMEDIR : buf, GRF_W))
						forward_letter(i ? NULL : buf, msgnum);
					else	nopermission(GRF_W);
				}
				redraw_msg_page(msgnum);
				break;
			case 'x':	/* extract/decode files */
			case 'X':
				if ((n = extract_mail(ch != 'x', msgnum)) < 0)
					warning(MSG_NOTHINGTOEXTRACT, -1);
				else {
					clearline();
					putstr(sysmsg(MSG_FILESEXTRACTED), n);
				}
				break;
			case 'w':	/* write mail to file */
				n = save_mail(msgnum);
				clearline();
				putstr(sysmsg(MSG_SAVEMSGCOMPLETE), n);
				break;
			case 'W':	/* write mbox to file */
				MsgToFile();
				goto restart;
			case 'h':
			case '?':
				if (termflags & (ANSITERM | RIPTERM))
					putstr("\033[1;44m\033[H\033[J");
				psysmsg(MSG_MAILREADERHELP);
				anykey();
				if (termflags & (ANSITERM | RIPTERM))
					putstr("\033[40m\033[J");
				redraw_msg_page(msgnum);
				break;
			default:
				putchr('\007');
				warning(MSG_BADCOMMAND, -1);
		}
	}
	/* unreached */
	return msgnum;
}

redraw_msg_page(msgnum)
	int msgnum;
{
	if (note_page < 0) warning(MSG_MESSAGEUNAVAIL, 1);
	else if (note_page > 0) {
		note_page--;
		fseek(folder_fp, note_mark[note_page], SEEK_SET);
		show_msg_page(msgnum);
	}
}

show_msg_page(msgnum)
	int msgnum;
{
	char *p, buf[1024], buf2[1024+50];
	int ctrl_L;		/* form feed character detected */

	note_line = 1;
	if (termflags & (ANSITERM | RIPTERM)) {
		putstr("\033[H\033[J\033[0;36m");
		if (note_page == 0)	show_first_msgheader(msgnum);
		else			show_cont_msgheader(msgnum);
	} else	{
		clearline();
		if (note_page == 0) {
			putstr("\n%s\n", sysmsg(MSG_DELIMITER));
			show_first_msgheader(msgnum);
		}
	}
	ctrl_L = FALSE;
	while (note_line < scrlines) {
		if (ftell(folder_fp) >= msg[msgnum].eof ||
		    fgets(buf, sizeof(buf), folder_fp) == NULL) {
			note_end = TRUE;
			break;
		}
		buf[sizeof(buf)-1] = '\0';
		ctrl_L = unrot(buf, buf2);
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\033[%dm", isquotedstr(buf2) ? 0 : 1);
		p = buf2;
		note_line++;
		while ((p = putstrmax(p, 79)) != NULL) {
			if (note_line >= scrlines) {
				fseek(folder_fp, -(strlen(p)+1), SEEK_CUR);
				break;
			}
			note_line++;
		}
		if (ctrl_L) break;
	}
	note_mark[++note_page] = ftell(folder_fp);

	if (termflags & (ANSITERM | RIPTERM)) {
		while (note_line++ < scrlines) putchr('\n');
		putstr("\033[0;7m");
	}
	if (note_end) {
		if (msgnum >= nmsg-1) putstr("- last -\r");
		else putstr("- next -\r");
	} else {
		int bytes = msg[msgnum].eof - msg[msgnum].head;
		if (bytes > 0) {
			ctrl_L = (note_mark[note_page] - msg[msgnum].head) * 100 / bytes;
			putstr("- more - (%d%%)\r", ctrl_L);
		} else	putstr("- more -\r");
	}
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[0m");
}

show_first_msgheader(msgnum)
	int msgnum;
{
	char buf[80], *p;

	if (msg[msgnum].status & STAT_DEL) {
		p = "*Deleted*";
		if (termflags & (ANSITERM | RIPTERM))
			putstr("Msg %4d of %-4d  \033[1;5;31m%-28.28s\033[0;36m  Lines %-5d %19.19s\n",
			       msgnum+1, nmsg, p, msg[msgnum].lines, ctime(&msg[msgnum].date));
		else	putstr("Msg %4d of %-4d  %-28.28s  Lines %-5d %19.19s\n",
			       msgnum+1, nmsg, p, msg[msgnum].lines, ctime(&msg[msgnum].date));
	} else {
		int len;
		p = makeshowpath(folder_path);
		len = strlen(p) - 28;
		if (len > 0) p += len;
		if (termflags & (ANSITERM | RIPTERM))
			putstr("Msg %4d of %-4d  \033[1;34m%-28.28s\033[0;36m  Lines %-5d %19.19s\n",
			       msgnum+1, nmsg, p, msg[msgnum].lines, ctime(&msg[msgnum].date));
		else	putstr("Msg %4d of %-4d  %-28.28s  Lines %-5d %19.19s\n",
			       msgnum+1, nmsg, p, msg[msgnum].lines, ctime(&msg[msgnum].date));
	}
	p = msg[msgnum].name;
	if (*p == '\0') p = msg[msgnum].from;
	snprintf(buf, sizeof(buf)-1, "%s at %s", p, msg[msgnum].org);
	if (termflags & (ANSITERM | RIPTERM))
		putstr("\033[1;33;44m\033[K%-30.30s \033[32m%48.48s\n\033[37m", msg[msgnum].from, buf);
	else	putstr("%-30.30s %48.48s\n", msg[msgnum].from, buf);
	note_line += 2;
	if (mailbox == FALSE) {
		char addr[256], name[256];
		parsefrom(msg[msgnum].to, addr, name);
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\033[1;33;44m\033[K%-30.30s \033[32m%48.48s\n\033[37m", addr, name);
		else	putstr("%-30.30s %48.48s\n", addr, name);
		note_line++;
	}
	p = msg[msgnum].subj;
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[44m\033[K");
	while ((p = putstrmax(p, 79)) != NULL) {
		if (termflags & (ANSITERM | RIPTERM)) putstr("\033[44m\033[K");
		note_line++;
	}
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[40m");
	else {
		putstr("%s\n", sysmsg(MSG_DELIMITER));
		note_line++;
	}
}

show_cont_msgheader(msgnum)
	int msgnum;
{
	char buf[80];

	snprintf(buf, sizeof(buf)-1, "Msg %d of %d (page %d):  %s",
		 msgnum+1, nmsg, note_page+1, msg[msgnum].subj);
	buf[79] = '\0';
	putstr("%s\n\n", buf);
	note_line++;
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[37m");
}

void
show_list_header()
{
	int len;
	char *p, buf[256];

	p = makeshowpath(folder_path);
	len = strlen(p) - 40;
	if (len > 0) p += len;
	sprintf(buf, "%s %s", sysmsg(MSG_MSGAREAHEAD), p);
	if (termflags & (ANSITERM | RIPTERM))
		putstr("\033[H\033[J\033[1;37;44m %-72.72s %5d\033[K\033[40m\n\n", buf, nmsg);
	else {
		clearline();
		putstr("\n %s (%d)\n\n", buf, nmsg);
	}
	if (mailbox == TRUE) {
		putstr("Msg   %-12.12s  %-20.20s", "Date", sysmsg(MSG_MSGFROM));
		putstr("Lines %s\n", sysmsg(MSG_MSGSUBJECT));
	} else {
		putstr("Msg   %-18.18s ", sysmsg(MSG_MSGFROM));
		putstr("%-18.18s ", sysmsg(MSG_MSGTO));
		putstr("%s\n", sysmsg(MSG_MSGSUBJECT));
	}
	putstr("%s\n", sysmsg(MSG_DELIMITER));
}

void
show_msg_line(i)
	int i;
{
	char ch, *p;

	if (mailbox == TRUE) {
		if (msg[i].status & STAT_DEL) ch = 'D';
		else if (msg[i].status & STAT_READ) ch = ' ';
		else if (msg[i].status & STAT_OLD) ch = 'U';
		else ch = 'N';
		p = ctime(&msg[i].date) + 4;
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\033[1;33m%-4d\033[31m%c \033[37m%-12.12s  \033[32m%-20.20s \033[37m%-5d\033[0;36m%-33.33s\n",
			       i+1, ch, p, msg[i].from, msg[i].lines, msg[i].subj);
		else	putstr("%-4d%c %-12.12s  %-20.20s %-5d%-33.33s\n",
			       i+1, ch, p, msg[i].from, msg[i].lines, msg[i].subj);
	} else {
		char *p2, addr[256], name[256];

		p = msg[i].name;
		if (*p == '\0') p = msg[i].from;
		parsefrom(msg[i].to, addr, name);
		if (*name != '\0') p2 = name;
		else p2 = addr;
		if (msg[i].status & STAT_READ) ch = ' ';
		else ch = '*';
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\033[1;33m%-4d\033[31m%c \033[32m%-18.18s %-18.18s \033[0;36m%-35.35s\n",
			       i+1, ch, p, p2, msg[i].subj);
		else	putstr("%-4d%c %-18.18s %-18.18s %-35.35s\n",
			       i+1, ch, p, p2, msg[i].subj);
	}
}

int
reply_letter(folder, msgnum, copy_text)
	char *folder;
	int msgnum, copy_text;
{
	int lines;
	char *p, buf[1024], to[MAX_PARAM_LEN], id[MAX_PARAM_LEN];

	if (folder_fp == NULL) return FALSE;
	unlink(editorworkfile);
	*to = *id = '\0';

	/* Review original header for Reply-To and Message-Id fields */
	fseek(folder_fp, msg[msgnum].head, SEEK_SET);
	while (ftell(folder_fp) < msg[msgnum].body &&
	       fgets(buf, sizeof(buf)-1, folder_fp) != NULL) {
		buf[sizeof(buf)-1] = '\0';
		for (p = buf; *p && *p != '\n'; p++)
			if ((unsigned char)*p < 0x20) *p = ' ';
		if (!*p) continue;	/* too long line */
		*p = '\0';
		if (!strncasecmp(buf, "Reply-To: ", 10))
			getparam(1, to, &buf[10]);
		else if (!strncasecmp(buf, "Message-Id: ", 12))
			getparam(0, id, &buf[12]);
	}
	lines = 0;
	if (copy_text) {	/* Insert original text */
		FILE *fp;
		char name[80];

		if ((fp = sfopen(editorworkfile, "w")) == NULL)
			return FALSE;
		p = msg[msgnum].name;
		if (*p != '\0')
			snprintf(name, sizeof(name), "%s <%s>", p, msg[msgnum].from);
		else	snprintf(name, sizeof(name), "%s", msg[msgnum].from);
		p = ctime(&msg[msgnum].date);
		p[19] = '\0';
		sprintf(buf, sysmsg(MSG_MSGREPLYHEADER), p, name);
		fprintf(fp, "%s\n", buf);
		lines = 2;
		fseek(folder_fp, msg[msgnum].body, SEEK_SET);
		while (ftell(folder_fp) < msg[msgnum].eof &&
		       fgets(buf, sizeof(buf)-1, folder_fp) != NULL) {
			buf[sizeof(buf)-1] = '\0';
			for (p = buf; *p && *p != '\n'; p++)
				if ((unsigned char)*p < 0x20) *p = ' ';
			if (!*p) continue;	/* too long line */
			*p = '\0';
			fprintf(fp, "> %s\n", buf);
			lines++;
		}
		sfclose(fp);
	}
	sprintf(buf, "Re: %s", eat_re(msg[msgnum].subj));
	if (folder == NULL) {
		p = msg[msgnum].from;
		lines = enter_mail(*to ? to : p, buf, *id ? id : NULL, lines);
	} else {
		p = msg[msgnum].name;
		if (*p == '\0') {
			strcpy(to, msg[msgnum].from);
			if ((p = strchr(to, '@')) != NULL) *p = '\0';
			p = to;
		}
		lines = enter_folder(folder, p, buf, *id ? id : NULL, lines);
	}
	return lines;
}

int
forward_letter(folder, msgnum)
	char *folder;
	int msgnum;
{
	int rval;
	FILE *fp;
	char *p, buf[1024];

	if (folder_fp == NULL ||
	    (fp = sfopen(editorworkfile, "w")) == NULL)
		return FALSE;
	fprintf(fp, ">--- Begin of forwarded message ---<\n>");
	fseek(folder_fp, msg[msgnum].head, SEEK_SET);
	fast_copy_fp(folder_fp, fp, msg[msgnum].eof - msg[msgnum].head);
	fprintf(fp, ">--- End of forwarded message ---<\n");
	sfclose(fp);
	if (bigbuf != NULL) {
		free(bigbuf);
		bigbuf = NULL;
	}
	sprintf(buf, "(fwd) %s", msg[msgnum].subj);
	if (folder == NULL) rval = enter_mail(NULL, buf, NULL, 0);
	else rval = enter_folder(folder, NULL, buf, NULL, 0);
	return rval;
}

int
save_mail(msgnum)
	int msgnum;
{
	FILE *fp;
	long saveoffs;
	char buf[1024], *p;

	if (folder_fp == NULL) return 0;

	do {
		p = prompt_str(MSG_SAVEMSGFILE,
			       getuserconf("SAVEMSGFILE", NULL), MAXFILENAMELEN);
		if (p == NULL) return 0;
	} while (!legalfname(p));

	sprintf(buf, "%s/%s", homedir, p);
	if ((fp = sfopen(buf, "a+")) == NULL) return 0;
	putuserconf("SAVEMSGFILE", p);
	saveoffs = ftell(folder_fp);
	fseek(folder_fp, msg[msgnum].head, SEEK_SET);
	fast_copy_fp(folder_fp, fp, msg[msgnum].eof - msg[msgnum].head);
	fseek(folder_fp, saveoffs, SEEK_SET);
	sfclose(fp);
	LOGIT(LOG_INFO, "WriteMail to \"%s\"", buf);
	if (bigbuf != NULL) {
		free(bigbuf);
		bigbuf = NULL;
	}
	return 1;
}

int
extract_mail(method, msgnum)
	int method, msgnum;
{
	int found, saved;
	char ct[256];
	long saveoffs;

	if (folder_fp == NULL) return 0;
	saveoffs = ftell(folder_fp);
	found = saved = 0;

	if (method == 0) {	/* check for uuencoded files */
		fseek(folder_fp, msg[msgnum].body, SEEK_SET);
		while (get_begin(folder_fp, msg[msgnum].eof, ct)) {
			found++;
			saved += extract_file(folder_fp, uudecode, ct);
		}
	} else {	/* check for MIME files in base64 */
		char *p, ce[256];

		fseek(folder_fp, msg[msgnum].head, SEEK_SET);
		while (get_content(folder_fp, msg[msgnum].eof, ct, ce)) {
			if (!strcasecmp(ce, "base64")) {
				found++;
				if ((p = __strcasestr(ct, "name=")) != NULL)
					p += 5;
				saved += extract_file(folder_fp, frombase64, p);
			}
		}
	}
	fseek(folder_fp, saveoffs, SEEK_SET);
	return found ? saved : -1;
}


/* --------------------------------------- */
/* --- User Interface functions follow --- */
/* --------------------------------------- */

/* Check mailbox for new mail */
void
MailCheck()
{
	struct stat st;
	char mbox[1024];

	sprintf(mbox, "%s/%s", homedir, MAILBOX);
	if (stat(mbox, &st) == 0 && (st.st_mode & S_IFMT) == S_IFREG &&
	    st.st_mtime > st.st_atime + 2) havenewmail();
}

/* Change message folder */
void
MsgAreaChange()
{
	char *p;

	close_folder();
	if ((p = select_folder()) != NULL) change_folder(p);
}

/* Show list of messages in current folder */
void
MsgList()
{
	register i;
	int lncnt, startmsg, rval, tmp, fidx[MAXSCRLINES];

	if (!open_folder()) {
		warning(MSG_NOMESSAGESAVAIL, 1);
		return;
	}
	startmsg = 1;
	if (globalflags & NEWFILES) {
		if (setnewfilestime() < 0) {
			close_folder();
			return;
		}
	} else if (nchr) {
		startmsg = atoi(getstr(0, 1, ECHODISABLE));
		if (startmsg < 1) startmsg = 1;
	} else {
		for (i = 0; i < nmsg && (msg[i].status & STAT_READ); i++);
		if (i < 0 || i >= nmsg) i = nmsg-1;
		if (i - (scrlines/2 - 2) >= 0) i = i - (scrlines/2 - 2);
		else i = 0;
		startmsg = i+1;
	}
	lncnt = 1;
	for (i = startmsg; i <= nmsg; i++) {
		if ((globalflags & NEWFILES) && msg[i-1].date < newfilestime)
			continue;
		startmsg = 0;
		if (lncnt == 1) {	/* just header of list */
			lncnt += 4;
			memset(fidx, 0, sizeof(fidx));
			fidx[0] = -1;
			tmp = i;
			show_list_header();
		}
		fidx[lncnt-1] = i;
		show_msg_line(i-1);
more_again:
		rval = morenum(MSG_SELECTMESSAGE, &lncnt, nmsg);
		if (rval == -1) goto return_list;
		if (rval > 0) {
			if (rval > nmsg) {
				rval = fidx[(rval-nmsg-1)/2];
				if (rval < 0) goto return_list;
				if (!rval) {
					lncnt = scrlines;
					goto more_again;
				}
			}
			savescr();
			if ((rval = show_msg(rval-1)) < 0)
				goto return_list;
			if ((termflags & RIPTERM) &&
			    rval >= tmp && rval < tmp + (scrlines - 4)) {
				restorescr();
				lncnt = scrlines;
				goto more_again;
			}
			i = rval - (scrlines - 4) / 2;
		} else if (rval == -2) i -= (scrlines - 4) * 2;
		else if (rval == -3) i = nmsg - (scrlines - 4);
		else if (rval == -4) i = 0;
		if (i < 0) i = 0;
	}
	if (!startmsg) {
		lncnt = scrlines;
		goto more_again;
	}
return_list:
	if (update_folder() == 1) havenewmail();
}

/* Show list of new messages in current folder */
void
MsgListNew()
{
	globalflags |= NEWFILES;
	MsgList();
	globalflags &= ~NEWFILES;
}

/* Read messages in current folder */
void
MsgReader()
{
	int n;

	if (!open_folder()) {
		warning(MSG_NOMESSAGESAVAIL, 1);
		return;
	}
	if (!ch_instead && nchr)
		n = atoi(getstr(0, 1, ECHODISABLE)) - 1;
	else for (n = 0; n < nmsg && (msg[n].status & STAT_READ); n++);
	if (n < 0 || n >= nmsg) n = nmsg-1;
	show_msg(n);
	if (update_folder() == 1) havenewmail();
}

/* Search for string in current folder */
void
MsgSearch()
{
	ch_instead = 'G';
	MsgReader();
	ch_instead = 0;
}

/* Search to this user in current folder */
void
MsgForYou()
{
	ch_instead = 'y';
	MsgReader();
	ch_instead = 0;
}

/* Write/Move new/unread messages to file */
void
MsgToFile()
{
	int i, cnt, mode, mark;
	time_t date;
	struct tm *tm;
	FILE *fp;
	char *p, buf[1024], name[MAXFILENAMELEN+1];

	mode = (folder_fp == NULL);
	if (mode) { /* MODE: remove after save if possible else mark as read */
		if (!open_folder()) {
			warning(MSG_NOMESSAGESAVAIL, 1);
			return;
		}
	} else {    /* MODE: mark as read after save */
		if (setnewfilestime() < 0) return;
	}
	for (i = 0, cnt = 0; i < nmsg; i++) {
		msg[i].status &= ~STAT_MARK;
		if (mode) { /* MODE: only unreaded (new) messages */
			if (msg[i].status & STAT_READ) continue;
		} else {    /* MODE: only new specific time messages */
			if ((msg[i].status & STAT_DEL) || msg[i].date < newfilestime)
				continue;
		}
		if (mailbox == FALSE ||
		    isgrpright(GSID_EMAIL, msg[i].from, GRF_R)) {
			msg[i].status |= STAT_MARK;
			cnt++;
		}
	}
	if (!cnt) {
		if (mode) close_folder();
		warning(MSG_NONEWMESSAGES, 1);
		return;
	}
	p = fixfname(folder_path, FALSE);
	if (mode) { /* MODE: current date in file name for save */
		date = time(NULL);
		if (mailbox == TRUE) p = "mbox";
	} else {    /* MODE: specific time in file name for save */
		date = newfilestime;
		if (mailbox == TRUE) p = "mail";
	}
	tm = localtime(&date);
	snprintf(name, sizeof(name), "%.4s%02d%02d.msg", p, tm->tm_mon+1, tm->tm_mday);
	do {
		p = prompt_str(MSG_SAVEMSGFILE, name, MAXFILENAMELEN);
		if (p == NULL) {
			if (mode) close_folder();
			return;
		}
	} while (!legalfname(p));
	sprintf(buf, "%s/%s", homedir, p);
	if ((fp = sfopen(buf, "w")) == NULL) {
		if (mode) close_folder();
		warning(MSG_SAVEMSGFAILED, 1);
		return;
	}
	mark = STAT_READ;
	if (mailbox == TRUE && mode &&
	    isgrpright(GSID_MESSAGE, HOMEDIR, GRF_D)) mark |= STAT_DEL;
	for (i = 0; i < nmsg; i++) {
		if (msg[i].status & STAT_MARK) {
			msg[i].status &= ~STAT_MARK;
			fseek(folder_fp, msg[i].head, SEEK_SET);
			fast_copy_fp(folder_fp, fp, msg[i].eof - msg[i].head);
			msg[i].status |= mark;
		}
	}
	sfclose(fp);
	if (bigbuf != NULL) {
		free(bigbuf);
		bigbuf = NULL;
	}
	LOGIT(LOG_INFO, "WriteMail (%d) to \"%s\"", cnt, buf);
	clearline();
	putstr(sysmsg(MSG_SAVEMSGCOMPLETE), cnt);
	putchr('\n');
	if (mode && update_folder() == 1) havenewmail();
}

/* Delete messages from mailbox */
void
MsgKill()
{
	int i, mn, delperm;
	char *ptr, *ptr2;

	if (mailbox == FALSE || !isgrpright(GSID_MESSAGE, HOMEDIR, GRF_D)) {
		nopermission(GRF_D);
		return;
	}
	if ((ptr = prompt_str(MSG_ENTERNUMBERS, NULL, 0)) == NULL) return;
	putchr('\n');

	if (!open_folder()) {
		warning(MSG_NOMESSAGESAVAIL, 1);
		return;
	}
	for (i = 0; i < nmsg; i++) msg[i].status &= ~STAT_MARK;

	for (delperm = 0; (ptr = strtok(ptr, ",; \t")) != NULL; ptr = NULL) {
		if (strlen(ptr) > 10) continue;
		if (*ptr == '*' || !strcasecmp(ptr, "All")) {
			for (i = 0; i < nmsg; i++)
				msg[i].status |= STAT_MARK;
			delperm++;
			break;
		}
		if (isdigitstr(ptr)) {
			mn = atoi(ptr);
			if (mn > 0 && mn <= nmsg) {
				msg[mn-1].status |= STAT_MARK;
				delperm++;
			}
			continue;
		}
		if ((ptr2 = strchr(ptr, '-')) == NULL) continue;
		if (ptr2 == ptr) {
			mn = atoi(++ptr2);
			if (mn > 0 && mn <= nmsg) {
				for (i = 0; i < mn; i++)
					msg[i].status |= STAT_MARK;
				delperm++;
			}
			continue;	
		}
		*ptr2++ = '\0';
		mn = atoi(ptr);
		if (mn < 1 || mn > nmsg) continue;
		mn--;
		if (*ptr2 != '\0') {
			delperm = atoi(ptr2);
			if (delperm < 1 || delperm > nmsg) continue;
		} else	delperm = nmsg;
		if (delperm > mn) {
			for (i = mn; i < delperm; i++)
				msg[i].status |= STAT_MARK;
		} else	delperm = 0;
	}
	if (delperm) {
		for (i = 0; i < nmsg; i++) {
			if ((msg[i].status & STAT_MARK) == 0) continue;
			show_msg_line(i);
			if (privlevel >= MAXPRIVLEVEL ||
			    isgrpright(GSID_EMAIL, msg[i].from, GRF_D)) {
				msg[i].status |= STAT_DEL;
				delperm = MSG_MSGKILLED;
				LOGIT(LOG_NOTICE, "DeleteMsg #%d in \"%s\"", i+1, folder_path);
			} else	delperm = MSG_MSGYOUCANTKILL;
			if (termflags & (ANSITERM | RIPTERM))
				putstr("\033[1;37mMsg %d/%d - \033[31m%s\n",
				       i+1, nmsg, sysmsg(delperm));
			else	putstr("Msg %d/%d - %s\n",
				       i+1, nmsg, sysmsg(delperm));
		}
	}
	update_folder();
}

/* Enter new message in current folder */
void
MsgEnter()
{
	if (folder_path[0] == '\0') {
		warning(MSG_SELECTAREAFIRST, 1);
		return;
	}
	if (isgrpright(GSID_MESSAGE, mailbox ? HOMEDIR : folder_path, GRF_W)) {
		if (mailbox)
			enter_mail(NULL, NULL, NULL, 0);
		else	enter_folder(folder_path, NULL, NULL, NULL, 0);
	} else	nopermission(GRF_W);
}
