/*
 *  $Id: vscan-fsav_core.c,v 1.1.2.8 2004/04/03 16:27:13 reniar Exp $
 *
 * virusscanning module for samba. provides helper methods to call fsecure 
 *
 * Copyright (C) Monex AG  Oliver Jehle, 2003
 * Copyright (C) Rainer Link, 2004
 *               OpenAntiVirus.org <rainer@openantivirus.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *  
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "vscan-global.h"
#include "vscan-fsav.h"

extern BOOL verbose_file_logging;
extern BOOL send_warning_message;

static const char module_id[] = VSCAN_MODULE_STR " " SAMBA_VSCAN_VERSION_STR;

/**
 * create handle
 * 
 * all functions use a handle
 *
 * returns NULL on failure
 */


fsav_handle *fsav_create_handle()
{
	fsav_handle *h;

	DEBUG(5, ("samba-vscan (%s) create handle\n", module_id));
	h = (fsav_handle *) malloc(sizeof(fsav_handle));
	if (!h)
		return h;

	h->server =
	    (struct sockaddr_un *) malloc(sizeof(struct sockaddr_un));
	if (!h->server) {
		fsav_free_handle(h);
		return NULL;
	}

	/* allocate buffer for result */
	h->buffer = (char *) malloc((BUFFERSIZE + 1));
	if (!h->buffer) {
		fsav_free_handle(h);
		return NULL;
	}

	/* allocate buffer for receive */
	h->recv_buffer = (char *) malloc(RCVSIZE + 1);
	if (!h->recv_buffer) {
		fsav_free_handle(h);
		return NULL;
	}

	/* allocate buffer for virusname */
        h->virusname = (char *) malloc((BUFFERSIZE + 1));
        if (!h->virusname) {
                fsav_free_handle(h);
                return NULL;
        }


	h->uid = -1;
	DEBUG(5, ("samba-vscan (%s) create handle success\n", module_id));
	return h;
}


/**
 * free a handle
 * 
 * release all storage areas and close sockets 
 */

void fsav_free_handle(fsav_handle * h)
{
	DEBUG(5, ("samba-vscan (%s) free handle\n", module_id));
	if (!h)
		return;
	/* close socket */
	if (h->sockd)
		close(h->sockd);

	/* release all used storage */
	if (h->server)
		free(h->server);

	if (h->buffer)
		free(h->buffer);

	if (h->recv_buffer)
		free(h->recv_buffer);

	free(h);
}

/**
 * clean handle
 * 
 * clean all buffers and fields
 */

void fsav_clean_handle(fsav_handle * h)
{
	if (!h)
		return;

	h->infected = 0;
	h->fail = 0;
	h->configured = 0;

	if (h->buffer)
		memset(h->buffer, 0, BUFFERSIZE);
	if (h->recv_buffer)
		memset(h->recv_buffer, 0, RCVSIZE);
}

/**
 * kill a running fsav daemon 
 * 
 * try to kill a running fsavd, looks like you have to delete the socket
 */

void fsav_kill(fsav_handle * h)
{
	if (!h)
		return;
	fsav_socket_name_create(h);
	DEBUG(5,
	      ("samba-vscan (%s) kill fsavd %s\n ", module_id,
	       h->server->sun_path));
	unlink(h->server->sun_path);
}

/**
 * starting a new fsav daemon
 *
 */

void fsav_start(fsav_handle * h)
{
	pid_t pid;
	int status;

	if (!h)
		return;

	fsav_socket_name_create(h);
	fsav_kill(h);

	DEBUG(5,
	      ("samba-vscan (%s) master start fsavd %s\n ", module_id,
	       h->server->sun_path));
	pid = fork();
	if (pid < 0) {
		DEBUG(5,
		      ("samba-vscan (%s) master cannot fork for start fsavd %s\n ",
		       module_id, h->server->sun_path));
		return;
	}
	if (pid > 0) {
		waitpid(pid, &status, 0);
		DEBUG(5,
		      ("samba-vscan (%s) master wait finished %i\n ",
		       module_id, pid));
	} else {
		DEBUG(5,
		      ("samba-vscan (%s) start slave start %i\n ",
		       module_id, pid));
		fsav_daemonize(h);
		exit(0);
	}
}

/**
 * daemonize the fsav daemon
 *
 * to ensure, that fsav is not making problems sending signals and other things,
 * run it in a separate session, try to give up all filedescriptors hold
 *
 */

void fsav_daemonize(fsav_handle * h)
{
	pid_t pid;
	int status;
	int rc;

	char socket[256];
	char config[256];
	char dbdir[256];
	char binary[256];
	char command[256];
	int maxfd = 0;

	DEBUG(5,
	      ("samba-vscan (%s) slave start fsav %s\n", module_id,
	       h->server->sun_path));
	snprintf(socket, 255, "--socketname=%s", h->server->sun_path);
	snprintf(config, 255, "--configfile=%s", h->config_file);
	snprintf(dbdir, 255, "--databasedirectory=%s", h->db_dir);
	snprintf(binary, 255, "%s", h->binary);

	signal(SIGALRM, SIG_IGN);
	signal(SIGCHLD, SIG_IGN);

	pid = fork();
	if (pid < 0) {
		DEBUG(5,
		      ("samba-vscan (%s) slave cannot fork %s (%i)\n ",
		       module_id, strerror(errno), errno));
		exit(1);
	}
	if (pid > 0) {
		DEBUG(5,
		      ("samba-vscan (%s) slave wait %i\n ", module_id,
		       pid));
		waitpid(pid, NULL, 0);
		DEBUG(5,
		      ("samba-vscan (%s) slave wait finished %i\n ",
		       module_id, pid));
		return;
	} else {
		DEBUG(5,
		      ("samba-vscan (%s) slave(%i) starting %s %s %s %s \n ",
		       module_id, pid, binary, config, socket, dbdir));

		/* 
		 * close all filedescriptors before calling fsav
		 */

		maxfd = open("/dev/null", O_RDONLY);
		for (; maxfd > 2; maxfd--)
			close(maxfd);

		fsav_free_handle(h);


		/*
		 * switch user id of the fsavd process to the effective user id, else we cant connect
		 */

                rc=setreuid(geteuid(), geteuid());
                if (rc != 0) {
			DEBUG(5,
                        	("samba-vscan (%s): slave cannot setreuid %s (%i)\n",
                               	module_id, strerror(errno), errno));
                        exit(0);
                }
                rc=setsid();
                if (rc != 0) {
                        DEBUG(5,
				("samba-vscan (%s): slave cannot setsid %s (%i)\n",
                                 module_id, strerror(errno), errno));
                        exit(0);
                }


		rc = execlp(binary, binary, config, socket, dbdir,
			    "--standalone", (char *) 0);

		if (rc != 0) {
                        DEBUG(5,
				("samba-vscan (%s): slave cannot execlp %s %s (%i)\n",
                                 module_id, binary, strerror(errno), errno));
		}
		exit(0);
	}
}

void vscan_fsav_log_virus(char *infected_file, char *result, char* client_ip)
{
        char *str;
        size_t len;

	/* format is: "INFECTED\t/path/file\t<virusname> <Engine>\n" */

        /* some sanity checks ... */
        len = strlen(result);
	/*				 \t                         \t   \n (see later) */
        if ( len < (strlen("INFECTED") + 1 + strlen(infected_file) + 1 + 1)  ) {
                /* hum, sth went wrong */
                vscan_syslog_alert("ALERT - Scan result: '%s' infected with virus 'UNKNOWN', client: '%s'", infected_file, 
client_ip);
                if ( send_warning_message )
                        vscan_send_warning_message(infected_file, "UNKNOWN", client_ip);

        } else {
                str = result;
                str+= strlen("INFECTED") + 1 + strlen(infected_file) + 1;
		if ( str[strlen(str) - 1] == '\n' )
			str[strlen(str) - 1] = '\0';

                vscan_syslog_alert("ALERT - Scan result: '%s' infected with virus '%s', client: '%s'", infected_file, str, 
client_ip);
                if ( send_warning_message )
                        vscan_send_warning_message(infected_file, str, client_ip);

        }
        
}


/**
 * scan a file 
 *
 *  Returns -2 on a minor error, -1 on error, 0 if no virus was found,
 *  1 if a virus was found
 */

int fsav_scan(fsav_handle * h, char *file, char* client_ip)
{
	if (!h)
		return -1;

	DEBUG(5, ("samba-vscan (%s) scan %s\n ", module_id, file));
        if ( verbose_file_logging )
                vscan_syslog("INFO: Scanning file : '%s'", file);

	fsav_clean_handle(h);

	snprintf(h->buffer, BUFFERSIZE, "SCAN\t%s\n", file);
	if (fsav_process(h)) {
		vscan_syslog("ERROR: unknown error occured");
		return -1;
	}

	if (h->infected) {
		/* virus found */
		h->rc = 1;
		vscan_fsav_log_virus(file, h->virusname, client_ip);
	} else {
		if (h->fail) {
			/* error */
			h->rc = -1;
			if ( verbose_file_logging )
				vscan_syslog("ERROR: file %s not found, not readable or an error occured", file);

		} else {
			/* no virus found, everything OK */
			h->rc = 0;
			if ( verbose_file_logging )
	                        vscan_syslog("INFO: file %s is clean", file);
		}
	}
	DEBUG(5,
	      ("samba-vscan (%s) scan %s rc=(%i) buffer=%s\n", module_id,
	       file, h->rc, h->buffer));

	return h->rc;

}

/**
 * create socket name 
 *
 */


void fsav_socket_name_create(fsav_handle * h)
{
	if (h->userinstance) {
		snprintf(h->server->sun_path, sizeof(h->server->sun_path)
			 , "%s-%s", h->socket, h->user);
	} else {
		snprintf(h->server->sun_path, sizeof(h->server->sun_path)
			 , "%s", h->socket);
	}
}

/**
 * create socket 
 * 
 */

void fsav_socket_create(fsav_handle * h)
{
	if (!h)
		return;

	/*
	 * set the correct socketname
	 */
	fsav_socket_name_create(h);

	DEBUG(5,
	      ("samba-vscan (%s) socket_create  %s \n", module_id,
	       h->server->sun_path));

	h->server->sun_family = AF_UNIX;
	h->sockd = socket(AF_UNIX, SOCK_STREAM, 0);

	if (h->sockd < 0) {
		DEBUG(5,
		    ("samba-vscan (%s): socket_create cannot connect %s\n",
		     module_id, h->server->sun_path));
		h->rc = 2;
	}
	DEBUG(5,
	      ("samba-vscan (%s) socket_create done %s \n", module_id,
	       h->server->sun_path));
	return;
}

/**
 * connect handle
 *
 * connect to a fsav instance, if not successfull and in userinstance mode, try
 * start a new instance
 *
 * returns 0 on success, 1 or 2 as error
 */

int fsav_connect_handle(fsav_handle * h)
{
	int rc;

	DEBUG(5, ("samba-vscan (%s) connect handle check\n", module_id));
	if (!h)
		return 1;

	fsav_socket_create(h);

	if (h->sockd < 0) {
		DEBUG(5,
		      ("samba-vscan (%s) socket_create not successfull\n",
		       module_id));
		h->rc = 2;
		return h->rc;
	}
        if(fsav_switch_uid(h)) {
		DEBUG(5,
                     ("samba-vscan (%s) switch user  not successfull\n",
                      module_id));
                h->rc = 2;
                return h->rc;
        }

	DEBUG(5, ("samba-vscan (%s) connect try connect \n", module_id));
	rc = connect(h->sockd, (struct sockaddr *) h->server,
		     sizeof(struct sockaddr_un));


        if(fsav_switch_uid(h)) {
                DEBUG(5,
                      ("samba-vscan (%s) switch user back  not successfull\n",
                       module_id));
                h->rc = 2;
                return h->rc;
        }

	if (rc != 0 && h->userinstance) {
		DEBUG(5,
		      ("samba-vscan (%s) connect try restart and  connect \n",
		       module_id));
		fsav_start(h);

                if(fsav_switch_uid(h)) {
                        DEBUG(5,
                                ("samba-vscan (%s) switch user  not successfull\n",
                                module_id));
                        h->rc = 2;
                        return h->rc;
                }
		rc = connect(h->sockd, (struct sockaddr *) h->server,
			     sizeof(struct sockaddr_un));

                if(fsav_switch_uid(h)) {
                        DEBUG(5,
                                ("samba-vscan (%s) switch user  not successfull\n",
                                module_id));
                        h->rc = 2;
                        return h->rc;
                }
	}
	DEBUG(5,
	      ("samba-vscan (%s) connect done rc=%i \n", module_id, rc));


	if (rc != 0) {
		DEBUG(5,
		      ("samba-vscan (%s) connect returns %s(%i) \n",
		       module_id, strerror(errno), errno));
		return rc;
	}
	rc = fsav_configure(h, "ARCHIVE", h->archive) != 0;
	if (rc != 0) {
		DEBUG(5,
		      ("samba-vscan (%s) connect configure archive (%i) \n",
		       module_id, rc));
		return rc;
	}
	rc = fsav_configure(h, "TIMEOUT", h->timeout);
	if (rc != 0) {
		DEBUG(5,
		      ("samba-vscan (%s) connect configure timeout (%i) \n",
		       module_id, rc));
		return rc;
	}
	rc = fsav_configure(h, "MAXARCH", h->maxnested);
	if (rc != 0) {
		DEBUG(5,
		      ("samba-vscan (%s) connect configure maxarch (%i) \n",
		       module_id, rc));
		return rc;
	}
	rc = fsav_configure(h, "MIME", h->mime);
	if (rc != 0) {
		DEBUG(5,
		      ("samba-vscan (%s) connect configure mime (%i) \n",
		       module_id, rc));
		return rc;
	}
	return rc;
}

/**
 * configure
 * 
 * send configure commands to the fsav instance
 */

int fsav_configure(fsav_handle * h, char *option, int value)
{
	DEBUG(5, ("samba-vscan (%s) configure \n ", module_id));
	if (!h)
		return 1;
	fsav_clean_handle(h);

	snprintf(h->buffer, BUFFERSIZE, "CONFIGURE\t%s\t%i\n", option,
		 value);

	if (fsav_process(h))
		return 2;

	if (h->configured)
		h->rc = 0;
	else
		h->rc = 1;

	DEBUG(5,
	      ("samba-vscan (%s) configure return %i\n ", module_id,
	       h->rc));
	return h->rc;
}

/**
 * switch the uid , depend on the uid set in the handle
 *
 */
int fsav_switch_uid(fsav_handle * h) {
	int rc;
	if (h->connect_uid == -1)
		return 0;

	DEBUG(5,("samba-vscan (%s) switching user uid (%i) euid (%i)\n ",
	module_id, getuid(),geteuid()));

	if (h->uid == -1) {
		h->uid = geteuid();
		rc=seteuid(h->connect_uid);
		DEBUG(5,("samba-vscan (%s) switching user from (%i) to (%i)\n ",
                        module_id, h->connect_uid,h->uid));

	if (rc != 0) {
		DEBUG(5,
			 ("samba-vscan (%s): cannot switch user rc = %i / %s\n",
			  module_id,rc,strerror(errno)));
		return 1;
	}

	} else {
		DEBUG(5,("samba-vscan (%s) switching user back (%i) to (%i)\n ",
                        module_id, h->connect_uid,h->uid));
		rc=seteuid(h->uid);
		if (rc != 0) {
			DEBUG(5,
				("samba-vscan (%s): cannot switch user rc = %i / %s\n",
				module_id, rc,strerror(errno)));
			return 1;
		}
		h->uid = -1;
	}
	return 0;
}



/**
 * process
 * 
 * send a command to fsav and handle the result 
 * Returns 0 on success, 1 on error
 */

int fsav_process(fsav_handle * h)
{
	char *end;
	char *t;
	char *ptrptr;
	int bl;

	DEBUG(5,
	      ("samba-vscan (%s) process write %s\n ", module_id,
	       h->buffer));
	h->rc = 0;
        if(fsav_switch_uid(h)) {
                DEBUG(5,
                      ("samba-vscan (%s) switch user  not successfull\n",
                       module_id));
                h->rc = 2;
                return h->rc;
        }

	if (write(h->sockd, h->buffer, strlen(h->buffer)) <= 0) {
		h->rc = 1;
                fsav_switch_uid(h);
                if(fsav_switch_uid(h)) {
                        DEBUG(5,
                      ("samba-vscan (%s) switch user back  not successfull\n",
                       module_id));
                }
		return h->rc;
	}
	fsav_clean_handle(h);
	DEBUG(5,
	      ("samba-vscan (%s) process read %s\n ", module_id,
	       h->buffer));
	memset(h->buffer, 0, BUFFERSIZE);
	t = h->buffer;
	end = h->buffer;
	while (read(h->sockd, h->recv_buffer, RCVSIZE - 1)) {
		bl = strlen(h->buffer);

		/*
		 * check if buffer ends with end of line, all possible values for us ends with a newline
		 */

		for (t = h->buffer + bl;
		     t > end && strncmp(t, "\n", 1) != 0; t--);

		/*
		 * trim buffer ... only last line is needed
		 */

		if (t > end && strncmp(t, "\n", 1) == 0) {
			snprintf(h->buffer, BUFFERSIZE, "%s", t + 1);
			t = h->buffer;
			end = t;
		} else {
			end = h->buffer + bl - 1;
		}

		/*
		 * add receive buffer to global buffer
		 */

		pstrcat(h->buffer, h->recv_buffer);

		/*
		 * search for tokens
		 */

		if (strstr(h->buffer, "INFECTED")) {
			pstrcpy(h->virusname, h->buffer);
			h->infected = 1;
		}

		if (strstr(h->buffer, "FAILURE"))
			h->fail = 1;

		if (strstr(h->buffer, "Server configured"))
			h->configured = 1;

		/*
		 * fsav terminates with a point.
		 */

		if (strstr(h->buffer, ".\n"))
			break;

		memset(h->recv_buffer, 0, RCVSIZE);

	}

        fsav_switch_uid(h);
        if(fsav_switch_uid(h)) {
		DEBUG(5,
			("samba-vscan (%s) switch user back  not successfull\n",
			module_id));
	}

	DEBUG(5,
	      ("samba-vscan (%s) process read end infected: %i fail: %i configured: %i  buffer: %s \n ",
	       module_id, h->infected, h->fail, h->configured, h->buffer));
	end = (char *) index(h->buffer, '.');
	if (!end) {
		h->rc = 1;
		DEBUG(5,
		      ("samba-vscan (%s) process point not found return %i\n ",
		       module_id, h->rc));
		return h->rc;
	}
	h->rc = 0;
	DEBUG(5,
	      ("samba-vscan (%s) process return %i\n ", module_id, h->rc));
	return h->rc;
}
