/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: st_sql_postgres.c,v 1.48.4.6 2005/02/18 18:03:28 anton Exp $ */

#ifdef USE_POSTGRES

#include "netams.h"

extern char *tr_func[ST_CONN_TYPES_NUM];

//////////////////////////////////////////////////////////////////////////
void *pg_stOpenSql(Service *s, st_conn_type type){

	ServiceStorage_cfg *cfg=(ServiceStorage_cfg *)s->cfg;
	PGresult  *res;
	PGconn *conn=NULL;

	//check for reconnect
	if(conn) {
		if(PQstatus(conn) != CONNECTION_BAD)
			return (void*)conn;
		else
			pg_stCloseSql(conn);
	}

	char *pg_host, *pg_user, *pg_pass, *pg_dbname;
	char buffer[1024]; 
        
	if (!cfg->hostname) pg_host="localhost"; else pg_host=cfg->hostname;
        if (!cfg->username) pg_user="root"; else pg_user=cfg->username;
        if (!cfg->password) pg_pass=""; else pg_pass=cfg->password;
        if (!cfg->dbname) pg_dbname="netams"; else pg_dbname=cfg->dbname;
	
	conn = PQsetdbLogin(pg_host,NULL,NULL,NULL,pg_dbname,pg_user,pg_pass);
	if (PQstatus(conn) == CONNECTION_BAD) {
		PQfinish(conn);
		conn = PQsetdbLogin(pg_host,NULL,NULL,NULL,"template1",pg_user,pg_pass);
		if (PQstatus(conn) == CONNECTION_BAD) {	
             		aLog(D_WARN, "postgres connect failed: %s, continuing closed\n",  PQerrorMessage(conn));
			PQfinish(conn);
			return NULL;
		}
		res = PQexec(conn,"SET autocommit TO 'on'"); 
		PQclear(res);

		sprintf(buffer,"SELECT * FROM pg_database WHERE datname='%s'",pg_dbname);
		res = PQexec(conn, buffer);
        	if (PQresultStatus(res) != PGRES_TUPLES_OK) {
			aLog(D_WARN, "postgres list dbs failed: %s\n", PQerrorMessage(conn));
			PQfinish(conn);
			return NULL;
		}
		if(!PQntuples(res)) { //hmmm, there are no netams database there... creating it...
			PQclear(res);
			snprintf(buffer, 254, "CREATE DATABASE %s TEMPLATE template0", pg_dbname);
			res = PQexec(conn, buffer);
			if (PQresultStatus(res) != PGRES_COMMAND_OK) {
				aLog(D_WARN, "postgres create db failed: %s\n",PQerrorMessage(conn));
				PQfinish(conn);
				return NULL;
			} else 
				aLog(D_INFO, "postgres database [%s] created\n", pg_dbname);

			PQclear(res);
			PQfinish(conn);
			
			conn = PQsetdbLogin(pg_host,NULL,NULL,NULL,pg_dbname,pg_user,pg_pass);
			if (PQstatus(conn) == CONNECTION_BAD) {
                        	aLog(D_WARN, "postgres connect failed: %s, continuing closed\n",  PQerrorMessage(conn));
                        	PQfinish(conn);
				return NULL;
                	}
			
			char *query;
			//create function && load language ignore errors
			query="CREATE FUNCTION plpgsql_call_handler() RETURNS language_handler 	\
				AS '$libdir/plpgsql', 'plpgsql_call_handler' LANGUAGE c ";
			res=PQexec(conn, query); PQclear(res);

			//create language
			query="CREATE TRUSTED PROCEDURAL LANGUAGE plpgsql HANDLER plpgsql_call_handler";
			res=PQexec(conn, query); PQclear(res);
		}
	}
	
//	FILE *debug = fopen("/tmp/trace.out","a");
//	PQtrace(conn, debug);

	res=PQexec(conn,"BEGIN");PQclear(res);
	
	char *table=st_table_name[type];
	sprintf(buffer, "SELECT * FROM pg_tables WHERE tablename='%s'", table);
	res = PQexec(conn, buffer);
	if (PQresultStatus(res) != PGRES_TUPLES_OK) {
		aLog(D_WARN, "postgres list table failed: %s\n", PQerrorMessage(conn));
		PQfinish(conn);
		return NULL;
	}
	int num=PQntuples(res);
	PQclear(res);

	if(!num) {
		char *query=NULL;
		switch(type) {
			case ST_CONN_RAW:
				query="CREATE TABLE raw (unit_oid INT NOT NULL, policy_oid INT NOT NULL, t_from INT , t_to INT , bytes_in BIGINT , bytes_out BIGINT );\
					CREATE INDEX raw_unit_oid_idx ON raw(unit_oid);\
					CREATE INDEX raw_policy_oid_idx ON raw(policy_oid);\
					CREATE INDEX raw_t_from_idx ON raw(t_from);\
					CREATE INDEX raw_t_to_idx ON raw(t_to)\
				";
				break;
			case ST_CONN_SUMMARY:
				query="CREATE TABLE summary (prefix CHAR(1) NOT NULL, unit_oid INT NOT NULL, policy_oid INT NOT NULL, t_from INT NOT NULL, bytes_in BIGINT, bytes_out BIGINT, PRIMARY KEY(prefix, unit_oid, policy_oid, t_from));\
                        		CREATE INDEX summary_unit_oid_idx ON summary(unit_oid);\
                        		CREATE INDEX summary_policy_oid_idx ON summary(policy_oid);\
                        		CREATE INDEX summary_t_from_idx ON summary(t_from)\
				";
				break;
			case ST_CONN_MONITOR:
				query="CREATE TABLE monitor (time INT NOT NULL, flowt INT , unit_oid INT , proto SMALLINT , src BIGINT NOT NULL, srcport INT , dst BIGINT NOT NULL, dstport INT , dpkts BIGINT , len BIGINT );\
					CREATE INDEX monitor_time_idx ON monitor(time);\
					CREATE INDEX monitor_unit_oid_idx ON monitor(unit_oid)\
				";
				break;
			case ST_CONN_LOGIN:
				query="CREATE TABLE login (unit_oid INT PRIMARY KEY NOT NULL, password VARCHAR(32), inact INT, abs INT , last_changed INT , last_opened_time INT , ip BIGINT NULL, mac VARCHAR(18), flags INT NULL)\
				";
				break;
			case ST_CONN_QUOTA:
				query="CREATE TABLE quota (unit_oid INT NOT NULL,policy_oid INT NOT NULL, syspolicy_oid INT , soft_treshold INT , flags INT , last_blocked_time INT , notify_soft_oid INT , notify_hard_oid INT , notify_return_oid INT , h_in BIGINT NULL, h_out BIGINT NULL, h_sum BIGINT NULL, d_in BIGINT NULL, d_out BIGINT NULL, d_sum BIGINT NULL, w_in BIGINT NULL, w_out BIGINT NULL, w_sum BIGINT NULL, m_in BIGINT NULL, m_out BIGINT NULL, m_sum BIGINT NULL, block_policy INT, block_policy_flags INT, PRIMARY KEY (unit_oid,policy_oid));\
                                CREATE INDEX quota_unit_oid_idx ON quota(unit_oid);\
                                CREATE INDEX quota_policy_oid_idx ON quota(policy_oid)\
				";
				break;
			case ST_CONN_OIDS:
				query="CREATE TABLE oids (object_oid INT PRIMARY KEY NOT NULL, name VARCHAR(32))\
				";
				break;
			case ST_CONN_EVENTS:
				query="CREATE TABLE events (time INT , type VARCHAR(8), unit_oid INT, user_oid INT, account_oid INT, param VARCHAR(254))\
				";
				break;
			case ST_CONN_BILLING:
				query="CREATE TABLE billing (account_oid INT PRIMARY KEY NOT NULL, name VARCHAR(32), description VARCHAR(254), balance DOUBLE PRECISION, units TEXT, plan INT, plan_ch INT, nextplan INT, nextplan_ch INT, blocked INT, created INT, changed INT, last_fee_ch INT, email VARCHAR(32), passwd VARCHAR(32), status INT NULL)\
				";
				break;
			case ST_CONN_BDATA:
				query="CREATE TABLE bdata (prefix CHAR(1) NOT NULL, account_oid INT NOT NULL, subplan_oid INT NOT NULL, t_from INT NOT NULL, bytes_in BIGINT NULL, bytes_out BIGINT NULL, pay_in DOUBLE PRECISION NULL, pay_out DOUBLE PRECISION NULL, PRIMARY KEY(prefix, account_oid, subplan_oid, t_from));\
					CREATE INDEX bdata_account__oid_idx ON bdata(account_oid);\
					CREATE INDEX bdata_subplan_oid_idx ON bdata(subplan_oid);\
					CREATE INDEX bdata_t_from_idx ON bdata(t_from)\
				";
				break;
			default:
				break;
		}
		res = PQexec(conn, query);
		if (PQresultStatus(res) != PGRES_COMMAND_OK) {
			aLog(D_WARN, "postgres create table '%s' failed: %s\n", table, PQerrorMessage(conn));
			PQfinish(conn);
			return NULL;
		} else 
			aLog(D_INFO, "postgres table '%s' created\n", table);
		PQclear(res);
		
		//create triggers
		if(tr_func[type]) {
			res = PQexec(conn, tr_func[type]);
			if (PQresultStatus(res) != PGRES_COMMAND_OK) {
				aLog(D_WARN, "postgres create function 'netams_tr_%s' failed: %s\n",
						table, PQerrorMessage(conn));
			} else 
				aLog(D_INFO, "postgres function 'netams_tr_%s' created\n", table);
			PQclear(res);

			sprintf(buffer, "CREATE TRIGGER netams_tr_%s		\
						BEFORE INSERT ON \"%s\"		\
						FOR EACH ROW			\
						EXECUTE PROCEDURE netams_tr_%s()\
			", table, table, table);

			res = PQexec(conn, buffer);
			if (PQresultStatus(res) != PGRES_COMMAND_OK) {
				aLog(D_WARN, "postgres create trigger 'netams_tr_%s' failed: %s\n", 
						table, PQerrorMessage(conn));
			} else 
				aLog(D_INFO, "postgres trigger 'netams_tr_%s' created\n", table);
			PQclear(res);
		}
	}

	res=PQexec(conn,"COMMIT");PQclear(res);
	res=PQexec(conn,"BEGIN");PQclear(res);
	
	return (void*)conn;
}

//////////////////////////////////////////////////////////////////////////
void pg_stCloseSql(void *fd){
	PGconn *conn=(PGconn*)fd;
	PGresult *res;

	if(PQstatus((PGconn*)conn) == CONNECTION_OK) {
		res=PQexec((PGconn*)conn,"END");
		PQclear(res);
	}
	PQfinish((PGconn*)conn);
}

//////////////////////////////////////////////////////////////////////////
unsigned pg_stSaveSql(void *fd, char *filename, st_conn_type type){
	
	char buf[255];
	PGconn *conn=(PGconn*)fd;
	PGresult *res;
	int i, j=0;
	// this is not working. see also
	// http://groups.google.com/groups?q=PGRES_COPY_IN+PQputCopyData&hl=ru&lr=&selm=272e4be7.0404300512.49592ecb%40posting.google.com&rnum=1
	
	sprintf(buf, "COPY %s FROM STDIN DELIMITER AS ',';", st_table_name[type]);
	res = PQexec(conn, buf );
	if (PQresultStatus (res) != PGRES_COPY_IN) {
		aLog(D_WARN, "SQL Load data: postgres: COPY failed: %s\n", PQerrorMessage(conn)); 
		PQclear(res);
		return 0;
	}
	PQclear(res);
			
	FILE *f = fopen(filename, "r");
	if (!f) { 
		aLog(D_WARN, "SQL Load data: postgres: file %s: %s \n", filename, strerror(errno));
		goto COPY_END;
	}

	while (!feof(f)){
		fgets(buf, 255, f);
		i=PQputCopyData(conn, buf, strlen(buf));
		j++;
		if (i==-1) aLog(D_WARN, "SQL/pg copy data: (size=%d) %s\n", strlen(buf), PQerrorMessage(conn) );
	}
	fclose(f);
COPY_END:
	i=PQputCopyEnd(conn, NULL);
	if (i==-1) aLog(D_WARN, "SQL/pg copy end: %s\n", PQerrorMessage(conn) );
	res=PQgetResult(conn);
	aLog(D_INFO, "%s\n", PQerrorMessage(conn));
	PQclear(res);
	
	return j;	
}
//////////////////////////////////////////////////////////////////////////
u_char pg_stLoadSql(void *fd, Message *message){
	Message_Read *msg=(Message_Read *)message;
	char query[255];
	PGresult  *res;
	PGconn *conn=(PGconn*)fd;
	pstat *cps=NULL;
	
	switch (msg->prefix){
		case 'M': cps=&(msg->pdata->m); break;
		case 'W': cps=&(msg->pdata->w); break;
		case 'D': cps=&(msg->pdata->d); break;
		case 'H': cps=&(msg->pdata->h); break;
	}

	snprintf(query, 254, "SELECT bytes_in, bytes_out FROM summary WHERE prefix='%c' AND unit_oid=%u AND policy_oid=%u AND t_from=%lu", msg->prefix, msg->netunit, msg->ap, (unsigned long)cps->from);
	
	res=PQexec(conn,query);
	if(PQresultStatus(res) == PGRES_TUPLES_OK) {
		if(PQntuples(res)) {		
			unsigned long long in;
			unsigned long long out;
			sscanf(PQgetvalue(res,0,0), "%llu", &in); 
			sscanf(PQgetvalue(res,0,1), "%llu", &out);
			//update counters
			cps->in+=in;
			cps->out+=out;
		}
	}
	PQclear(res);
	aDebug(DEBUG_STORAGE, "SQL<-HDD/%c query %u bytes, fr=%lu in=%llu out=%llu\n", msg->prefix, strlen(query), cps->from, cps->in, cps->out);
	return 1;
}

//////////////////////////////////////////////////////////////////////////
u_char pg_stLgObtainDbData(void *fd){

	char buffer[]="SELECT * from login";

	PGresult  *res;
	PGconn *conn=(PGconn*)fd;

	res=PQexec(conn,buffer);
	if(PQresultStatus(res) != PGRES_TUPLES_OK) {
                aLog(D_WARN, "postgres select login failed: %s\n", PQerrorMessage(conn));
                return 0;
        }

	if(PQntuples(res)) {
		
		sLoginData *logindata;
		unsigned tmp;
		NetUnit *u=NULL;
		oid id;
		
		for(int row=0;row < PQntuples(res); row++) {
			sscanf(PQgetvalue(res,row,0), "%u",&id);
		
			if(id && !(u=Units.getUnitById(id))) continue;
			if(u->logindata) {
				logindata=u->logindata;
				aFree(logindata->password);
			} else 
				logindata = (sLoginData*)aMalloc(sizeof(sLoginData));		

			logindata->password=set_string(PQgetvalue(res,0,1));
			sscanf(PQgetvalue(res,row,2), "%lu", (unsigned long*)&logindata->inact);
			sscanf(PQgetvalue(res,row,3), "%lu", (unsigned long*)&logindata->abs);
			sscanf(PQgetvalue(res,row,5), "%lu", (unsigned long*)&logindata->opened);
			sscanf(PQgetvalue(res,row,6), "%u", &logindata->ip_from.s_addr);
			memcpy(&logindata->mac_from, PQgetvalue(res,row,7), sizeof (struct ether_addr));
			sscanf(PQgetvalue(res,row,8), "%u", &tmp); logindata->flags=tmp;
			u->logindata=logindata;

			aDebug(DEBUG_LOGIN, "Unit: %06X inact %lu abs %lu flags:%u\n", u->id, u->logindata->inact, u->logindata->abs, u->logindata->flags);
		}
    	} else 
		aDebug(DEBUG_LOGIN, "No login entries were not found or access problem\n");
	
	PQclear(res);
	return 1;
}
//////////////////////////////////////////////////////////////////////////
u_char pg_stQuObtainDbData(void *fd){
	char buffer[]="SELECT * from quota";
	PGresult  *res;
	PGconn *conn=(PGconn*)fd;
       
	res=PQexec(conn,buffer);
        if(PQresultStatus(res) != PGRES_TUPLES_OK) {
                aLog(D_WARN, "postgres select quota failed: %s\n", PQerrorMessage(conn));
                return 0;
        }

       if(PQntuples(res)) {

		sQuotaData *q;
		unsigned tmp;
		NetUnit *u=NULL;
		oid id;

		for(int row=0;row < PQntuples(res); row++) {
			sscanf(PQgetvalue(res,row,0), "%u", &id);

			if(id && !(u=Units.getUnitById(id))) continue;
			if(u->quotadata) {
				q=u->quotadata;
			} else
				q = (sQuotaData*)aMalloc(sizeof(sQuotaData));

			sscanf(PQgetvalue(res,row,1), "%u", &id); q->policy=PolicyL.getPolicyById(id);
            		sscanf(PQgetvalue(res,row,3), "%u", &tmp); q->soft_treshold=tmp;
            		sscanf(PQgetvalue(res,row,4), "%u", &tmp); q->flags=tmp; 
            		sscanf(PQgetvalue(res,row,5), "%lu", (unsigned long*)&q->blocked_time);

            		sscanf(PQgetvalue(res,row,6), "%u", &q->nso);
            		sscanf(PQgetvalue(res,row,7), "%u", &q->nho);
            		sscanf(PQgetvalue(res,row,8), "%u", &q->nro);

            		sscanf(PQgetvalue(res,row,9), "%llu", &q->h.in);
            		sscanf(PQgetvalue(res,row,10), "%llu", &q->h.out);
            		sscanf(PQgetvalue(res,row,11), "%llu", &q->h.sum);

            		sscanf(PQgetvalue(res,row,12), "%llu", &q->d.in);
            		sscanf(PQgetvalue(res,row,13), "%llu", &q->d.out);
            		sscanf(PQgetvalue(res,row,14), "%llu", &q->d.sum);

            		sscanf(PQgetvalue(res,row,15), "%llu", &q->w.in);
            		sscanf(PQgetvalue(res,row,16), "%llu", &q->w.out);
            		sscanf(PQgetvalue(res,row,17), "%llu", &q->w.sum);

            		sscanf(PQgetvalue(res,row,18), "%llu", &q->m.in);
            		sscanf(PQgetvalue(res,row,19), "%llu", &q->m.out);
            		sscanf(PQgetvalue(res,row,20), "%llu", &q->m.sum);

					sscanf(PQgetvalue(res,row,21), "%u", &id); if (id) q->fw_block_policy=PolicyL.getPolicyById(id); else q->fw_block_policy=NULL;
					sscanf(PQgetvalue(res,row,22), "%u", &tmp); q->fw_block_policy_flags=tmp;
            		
            		aDebug(DEBUG_QUOTA, "Unit: %06X soft %d flags %d\n", u->id, q->soft_treshold, q->flags);
            u->quotadata=q;
		}
	} else 
		aDebug(DEBUG_QUOTA, "No quota entry were not found or access problem\n");
	PQclear(res);
return 1;
}
//////////////////////////////////////////////////////////////////////////
#ifdef HAVE_BILLING

u_char pg_stBiObtainDbData(void *fd){
	char *buffer;
	PGresult  *res;
	PGconn *conn=(PGconn*)fd;
	
	aDebug(DEBUG_BILLING, "SQL->HDD/billing data request\n");
	buffer="SELECT * FROM billing ORDER BY CREATED";
	
	res=PQexec(conn,buffer);
	if (PQresultStatus(res) != PGRES_TUPLES_OK) { 
		aLog(D_WARN, "pgsql select billing failed: %s\n", PQerrorMessage(conn)); 
		return 0; 
	}
	
   if(PQntuples(res)) {

		Account *ac;
		NetUnit *u;
		unsigned t;
	
		for(int row=0;row < PQntuples(res); row++) {
			sscanf(PQgetvalue(res,row,0), "%u", &t);
			newOid(t);
			
			sscanf(PQgetvalue(res,row,10), "%u", &t); //created ?
			if(!t) continue; //this account deleted

			sscanf(PQgetvalue(res,row,0), "%u", &t);
			ac=new Account();
			ac->id=newOid(t);

			if (PQgetvalue(res,row,1)) ac->name=set_string(PQgetvalue(res,row,1));
			if (PQgetvalue(res,row,2)) ac->description=set_string(PQgetvalue(res,row,2));
			
			sscanf(PQgetvalue(res,row,3), "%lf", &ac->balance);
			
			sscanf(PQgetvalue(res,row,5), "%u", &t); ac->plan=bPlans->Check(t);
			sscanf(PQgetvalue(res,row,6), "%lu", (unsigned long*)&ac->plan_ch);
			sscanf(PQgetvalue(res,row,7), "%u", &t); ac->nextplan=bPlans->Check(t);
			sscanf(PQgetvalue(res,row,8), "%lu", (unsigned long*)&ac->nextplan_ch);
			sscanf(PQgetvalue(res,row,9), "%lu", (unsigned long*)&ac->blocked);
			sscanf(PQgetvalue(res,row,10), "%lu", (unsigned long*)&ac->created);
			sscanf(PQgetvalue(res,row,11), "%lu", (unsigned long*)&ac->changed);
			sscanf(PQgetvalue(res,row,12), "%lu", (unsigned long*)&ac->last_fee_ch);
							  
			if (PQgetvalue(res,row,13)) ac->email=set_string(PQgetvalue(res,row,13));
			if (PQgetvalue(res,row,14)) ac->password=set_string(PQgetvalue(res,row,14));
			sscanf(PQgetvalue(res,row,15), "%u", &t); ac->status=t;
			
			u_short i=0;
			char *b=PQgetvalue(res,row,2); u_short k=strlen(b);
			while (i*7<k) {
				sscanf(b+i*7, "%06X", &t);
				if ((u=Units.getUnitById(t))) ac->AddUnit(u, ADD); else break;
				i++;
			}

			bAccounts->Add(ac);
	  	}  
 	} else 
		aDebug(DEBUG_BILLING, "No accounts are defined or table access problem\n");

	PQclear(res);
	return 1;
}

extern struct FeeCounters FC;

void pg_stBiLoadBdata(void *fd, Account *ac, char prefix) {
	char query[255];

	PGresult  *res;
	PGconn *conn=(PGconn*)fd;
	 	
	bstat *bs=NULL;
	time_t t_from=0;
	
	switch (prefix){
		case 'M': t_from=FC.mt; break;
		case 'W': t_from=FC.wt; break;
		case 'D': t_from=FC.dt; break;
		case 'H': t_from=FC.ht; break;
        }

	snprintf(query, 254, "SELECT subplan_oid, bytes_in, bytes_out, pay_in, pay_out FROM bdata WHERE account_oid=%u AND t_from=%lu AND prefix='%c'", ac->id, (unsigned long)t_from, prefix);

	res=PQexec(conn,query);
	if (PQresultStatus(res) != PGRES_TUPLES_OK) { 
		aLog(D_WARN, "Can't read data from database: %s\n",PQerrorMessage(conn));
		return; 
	}

	if(PQntuples(res)) {

		bSPlist *bsp;
		unsigned id;
		u_char i;

		for(int row=0;row < PQntuples(res); row++) {
			sscanf(PQgetvalue(res,row,0), "%u", &id);
			
			for(i=0,bsp=ac->plan->root;bsp!=NULL;bsp=bsp->next) {
				if(bsp->sp->id==id) {
					switch (prefix){
						case 'M': bs=&ac->data[i].m; break;
						case 'W': bs=&ac->data[i].w; break;
						case 'D': bs=&ac->data[i].d; break;
						case 'H': bs=&ac->data[i].h; break;
					}
					bs->from=t_from;

					sscanf(PQgetvalue(res,row,1), "%lld", &bs->in);
					sscanf(PQgetvalue(res,row,2), "%lld", &bs->out);
					sscanf(PQgetvalue(res,row,3), "%lf", &bs->pay_in);
					sscanf(PQgetvalue(res,row,4), "%lf", &bs->pay_out);
					break;
				}
				i++;
			}
		}
		aDebug(DEBUG_BILLING, "Account %s(%06X), '%c' bstat loaded\n", ac->name, ac->id, prefix);
	} else 
		aDebug(DEBUG_BILLING, "BDATA table access problem\n");

	PQclear(res);
}
#endif // billing
//////////////////////////////////////////////////////////////////////////
char *tr_func[ST_CONN_TYPES_NUM]={
NULL,
NULL,
"								\
CREATE FUNCTION netams_tr_summary() RETURNS \"trigger\"		\
    AS '							\
DECLARE								\
cnt int;							\
BEGIN								\
								\
cnt := (SELECT COUNT(*) FROM summary WHERE prefix=NEW.prefix	\
	AND unit_oid=NEW.unit_oid AND policy_oid=NEW.policy_oid	\
	AND t_from=NEW.t_from);					\
	if cnt <> 0 then					\
		DELETE FROM summary WHERE prefix=NEW.prefix 	\
		AND unit_oid=NEW.unit_oid 			\
		AND policy_oid=NEW.policy_oid 			\
		AND t_from=NEW.t_from;				\
	end if;							\
								\
RETURN NEW;							\
END;								\
'								\
    LANGUAGE plpgsql;						\
",			
NULL,
"								\
CREATE FUNCTION netams_tr_login() RETURNS \"trigger\"		\
    AS '							\
DECLARE								\
cnt int;							\
BEGIN								\
								\
cnt := (SELECT COUNT(*) FROM login WHERE unit_oid=NEW.unit_oid);\
        if cnt <> 0 then					\
            DELETE FROM login WHERE unit_oid=NEW.unit_oid;	\
        end if;							\
								\
RETURN NEW;							\
END;								\
'								\
    LANGUAGE plpgsql;						\
",		
"								\
CREATE FUNCTION netams_tr_quota() RETURNS \"trigger\"		\
    AS '							\
DECLARE								\
cnt int;							\
BEGIN								\
								\
cnt := (SELECT COUNT(*) FROM quota WHERE unit_oid=NEW.unit_oid 	\
	AND policy_oid=NEW.policy_oid);				\
        if cnt <> 0 then					\
            DELETE FROM quota WHERE unit_oid=NEW.unit_oid 	\
		AND policy_oid=NEW.policy_oid;			\
        end if;							\
								\
RETURN NEW;							\
END;								\
'								\
    LANGUAGE plpgsql;						\
",
NULL,
"								\
CREATE FUNCTION netams_tr_oids() RETURNS \"trigger\"		\
    AS '							\
DECLARE								\
cnt int;							\
BEGIN								\
								\
cnt := (SELECT COUNT(*) FROM oids WHERE object_oid=NEW.object_oid);\
        if cnt <> 0 then					\
            UPDATE oids SET 					\
		object_oid=NEW.object_oid, name=NEW.name	\
	    WHERE object_oid=NEW.object_oid;			\
	RETURN NULL;						\
        end if;							\
								\
RETURN NEW;							\
END;								\
'								\
    LANGUAGE plpgsql;						\
",
"								\
CREATE FUNCTION netams_tr_billing() RETURNS \"trigger\"		\
    AS '							\
DECLARE								\
cnt int;							\
BEGIN								\
								\
cnt := (SELECT COUNT(*) FROM billing WHERE account_oid=NEW.account_oid);\
        if cnt <> 0 then					\
            DELETE FROM billing WHERE account_oid=NEW.account_oid;\
        end if;							\
								\
RETURN NEW;							\
END;								\
'								\
    LANGUAGE plpgsql;						\
",
"								\
CREATE FUNCTION netams_tr_bdata() RETURNS \"trigger\"		\
    AS '							\
DECLARE								\
cnt int;							\
BEGIN								\
								\
cnt := (SELECT COUNT(*) FROM bdata WHERE 			\
	prefix=NEW.prefix AND account_oid=NEW.account_oid 	\
	AND subplan_oid=NEW.subplan_oid AND t_from=NEW.t_from);	\
        if cnt <> 0 then					\
            DELETE FROM bdata WHERE 				\
		prefix=NEW.prefix AND account_oid=NEW.account_oid \
		AND subplan_oid=NEW.subplan_oid AND t_from=NEW.t_from;\
        end if;							\
								\
RETURN NEW;							\
END;								\
'								\
    LANGUAGE plpgsql;						\
",
NULL
};
#endif // postgres
