/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: s_processor.c,v 1.109.4.5 2005/03/22 10:20:00 jura Exp $ */

#include "netams.h"

void *sProcessor(void *ss);
void sProcessorCancel(void *);
void sProcessorListCfgUnit(NetUnit *d, FILE *f, u_char nopasswords);
void sPConstructStoreMessages(oid netunit, policy_data *pd);
void sPMessagesGenerator(ServiceProcessor_cfg *cfg);
void sPMessagesMultiplexer(ServiceProcessor_cfg *cfg);

int cAutoAssignIP(Connection *conn, char *param[], u_char no_flag);
int cAutoUnits(Connection *conn, char *param[], u_char no_flag);
int cDefault(Connection *conn, char *param[], u_char no_flag);
int cRestrict(Connection *conn, char *param[], u_char no_flag);
int cRegister(Connection *conn, char *param[], u_char no_flag);


NetUnitsList Units;
PolicyList PolicyL;
MessageManager MsgMgr;
FIFO Mux_in(MAX_UNITS*5);
Service *Processor;
//////////////////////////////////////////////////////////////////////////
ServiceProcessor_cfg ProcessorCfg;
//////////////////////////////////////////////////////////////////////////
void sProcessorInit(Service *s){
	ServiceProcessor_cfg *cfg=&ProcessorCfg;
	cfg->def=NULL;
	cfg->delay=PROCESSOR_DELAY; // in seconds
	cfg->lifetime=PROCESSOR_LIFETIME;
	cfg->restrict_all=DROP;
	cfg->restrict_local=PASS;
	cfg->access_script=NULL;
	cfg->auto_assign=NULL;
	cfg->auto_units=NULL;
	
	//this pointers might be defined from others places, remember it
//	cfg->fifo=NULL;
//	cfg->st_root=NULL;

	Processor=s;
	s->cfg=(void*)cfg;
	pthread_create(&(s->t_id), NULL, &sProcessor, s);
}
//////////////////////////////////////////////////////////////////////////
void sProcessorProcessCfg(char *param[], Connection *conn, u_char no_flag){
	ServiceProcessor_cfg *cfg=(ServiceProcessor_cfg*)conn->service->cfg;

	if (!strcasecmp(param[0], "unit")) cUnit(conn, param, no_flag);
	else if (!strcasecmp(param[0], "policy")) cPolicy(conn, param, no_flag);
	else if (!strcasecmp(param[0], "default")) cDefault(conn, param, no_flag);
	else if (!strcasecmp(param[0], "restrict")) cRestrict(conn, param, no_flag);
	else if (!strcasecmp(param[0], "storage")) cRegister(conn, param, no_flag);
	else if (!strcasecmp(param[0], "lookup-delay")) {
		unsigned delay=strtol(param[1], NULL, 10);
		if (delay>=1 && delay<24*60*60) {
			aParse(conn, "lookup delay is set to %u seconds\n", delay);
			cfg->delay=delay;
		} 
		else aParse(conn, "lookup delay value invalid\n");
	}		
	else if (!strcasecmp(param[0], "flow-lifetime")) {
		unsigned lifetime=strtol(param[1], NULL, 10);
		if (lifetime>=1 && lifetime<24*60*60) {
			aParse(conn, "flow lifetime is set to %u seconds\n", lifetime);
			cfg->lifetime=lifetime;
		} 
		else aParse(conn, "flow lifetime value invalid\n");
	}		
	else if (!strcasecmp(param[0], "access-script")) {
		if(cfg->access_script) aFree(cfg->access_script);
		cfg->access_script=set_string(param[1]);
		aParse(conn, "access control script name is set to '%s'\n", param[1]);
	}		
	else if (!strcasecmp(param[0], "auto-assign")) cAutoAssignIP(conn, param, no_flag);
	else if (!strcasecmp(param[0], "auto-units")) cAutoUnits(conn, param, no_flag);
	else aParse(conn, "unknown processor command: %s\n", param[0]);
}
//////////////////////////////////////////////////////////////////////////
void *sProcessor(void *ss){
	Service *s=(Service*)ss;
	ServiceProcessor_cfg *cfg=(ServiceProcessor_cfg*)s->cfg;
	Service *st_s;
	st_unit *st;

	aLog(D_INFO, "service processor thread started\n");
	pthread_cleanup_push(sProcessorCancel, s);
	SET_CANCEL();
	Processor->t_id=pthread_self();
	
	// now we should sleep before starting actual processor
	if(!(s->flags&SERVICE_RUN)) Processor->Sleep();

	if (!cfg->st_root) aLog(D_WARN, "no storages registered!\n");
	//Initialize storages
	st_unit *p=NULL;
	for(st=cfg->st_root;st!=NULL;st=st->next) {
		if (!(st_s=Services.getService(SERVICE_STORAGE, st->id))) {
                	aLog(D_WARN, "storage %d registered but never started\n", st->id);
			if(st==cfg->st_root) cfg->st_root=st->next;
			else p->next=st->next;
                        aFree(st);
                } else {
                	st->in=((ServiceStorage_cfg*)st_s->cfg)->in;
                        st->out=((ServiceStorage_cfg*)st_s->cfg)->out;
                        st->s=st_s;
                        aLog(D_INFO, "using storage:%d, in=%p, out=%p, service_cfg=%p\n", st->id, st->in, st->out, st->s->cfg);
                        if(st==cfg->st_root) aLog(D_INFO, "using storage:%u as source for READ and STAT requests\n",st->id);
               	}
		p=st;	
	} 
	
	// loop for incoming data forever
	while(1) {
		//generate message for units
		sPMessagesGenerator(cfg);	
		
		//deliver generated messages to storages
		sPMessagesMultiplexer(cfg);

		Processor->Sleep(cfg->delay);
	} // infinite while
	// we will never reach this point
	pthread_cleanup_pop(0);
	return NULL;
}
//////////////////////////////////////////////////////////////////////////
void sProcessorCancel(void *v){
	Service *s=(Service*)v;
	ServiceProcessor_cfg *cfg=(ServiceProcessor_cfg*)s->cfg;
	
	cfg->lifetime=0; // this will flush all data when we will generate messages
	sPMessagesGenerator(cfg);
	aLog(D_INFO, "cancelling service processor, flushing %u messages\n", Mux_in.num_items);
	//deliver generated messages to storages
	sPMessagesMultiplexer(cfg);
	
	Processor=NULL;

	//perform cancellation
	st_unit *ptr,*st=cfg->st_root;
	while(st) {
		ptr=st;
		st=st->next;
		aFree(ptr);
	}
	cfg->st_root=NULL;
	if(cfg->access_script) aFree(cfg->access_script);
	if(cfg->def) delete cfg->def;
	PolicyL.DeleteAll();
	
	//clear it!
	bzero(&ProcessorCfg, sizeof(ServiceProcessor_cfg));
}
//////////////////////////////////////////////////////////////////////////
void sProcessorListCfg(Service *s, FILE *f, u_char nopasswords){
	ServiceProcessor_cfg *cfg=(ServiceProcessor_cfg*)s->cfg;
	
	if(cfg->delay != PROCESSOR_DELAY) fprintf(f, "lookup-delay %d\n", cfg->delay);
	if(cfg->lifetime != PROCESSOR_LIFETIME) fprintf(f, "flow-lifetime %d\n", cfg->lifetime);

	// first, policy rules
	{
	pthread_rwlock_rdlock(PolicyL.rwlock);
	Policy *d;
	char *buf;

	for (d=PolicyL.root; d!=NULL; d=d->next) {
		fprintf(f, "policy ");

		fprintf(f, "oid %06X ", d->id);
		if (d->name) fprintf(f, "name %s ", d->name);

		if (d->target.target_type) {
			buf=d->getTarget();
			fprintf(f, "target %-12s", buf);
			aFree(buf);
		}
		fprintf(f, "\n");
	}

	pthread_rwlock_unlock(PolicyL.rwlock);
	}

	// default parameters
	if (cfg->def && cfg->def->ap) { fprintf(f, "default acct-policy "); cfg->def->ap->ListForCfg(f); }
	if (cfg->def && cfg->def->fp) { fprintf(f, "default fw-policy "); cfg->def->fp->ListForCfg(f); } 

	// restrict parameters
	fprintf(f, "restrict all %s local %s\n", cfg->restrict_all?"drop":"pass", cfg->restrict_local?"drop":"pass");

	// auto-assign ip addresses
	for(AutoAssignEntry *e=cfg->auto_assign; e!=NULL; e=e->next ) {
		fprintf(f, "auto-assign %s", inet_ntoa(e->start));
		fprintf(f, " %s\n", inet_ntoa(e->stop));
	}

	// auto-units 
	for(AutoUnitsEntry *e=cfg->auto_units; e!=NULL; e=e->next ) {
		fprintf(f, "auto-units %u ", e->id);
		if (e->naming==AU_NAMING_BY_DNS) 
			fprintf(f, "type %s naming by-dns ", e->type==AU_TYPE_HOST?"host":"user");
		else 
			fprintf(f, "type %s naming prefix%u %s ", e->type==AU_TYPE_HOST?"host":"user", e->naming, e->prefix);
		if (e->put_to_group) if (Units.getUnit(e->put_to_group)) fprintf(f, "group %s", e->put_to_group);
		fprintf(f, "\n");
	}

	// next, print NetUnits
	{
	pthread_rwlock_rdlock(Units.rwlock);
	NetUnit *d;
	char buf[32];
	
        for (d=Units.root; d!=NULL && d->type==NETUNIT_GROUP; d=d->next) {
                fprintf(f, "unit group oid %06X ", d->id);
                if (d->name) fprintf(f, "name %s ", d->name);
                sProcessorListCfgUnit(d, f, nopasswords);
                fprintf(f, "\n");
        } //for

	
	for (d=Units.root; d!=NULL; d=d->next) {
		if(d->type==NETUNIT_GROUP) continue;
		fprintf(f, "unit %s oid %06X ", netunit_type_name[d->type], d->id);
		if (d->name) fprintf(f, "name %s ", d->name);
		if (d->bw) fprintf(f, "bw %s", getBW(d->bw, buf));

		switch (d->type) {
			case NETUNIT_HOST: {
				NetUnit_host *h = (NetUnit_host*)d;
				fprintf(f, "ip %s ", inet_ntoa(h->ip));
				} break;
			case NETUNIT_CLUSTER: { 
				NetUnit_host *t;
				NetUnit_cluster *h = (NetUnit_cluster*)d;
				for (t=h->root; t!=NULL; t=t->next)	fprintf(f, "ip %s ", inet_ntoa(t->ip));
				} break;
			case NETUNIT_GROUP: continue;
			case NETUNIT_NET: {
				NetUnit_net *h = (NetUnit_net*)d;
				fprintf(f, "ip %s ", inet_ntoa(h->ip));	
				fprintf(f, "mask %s ", inet_ntoa(h->mask));
				if (h->auto_units_id) fprintf(f, "auto-units %u ", h->auto_units_id);
				} break;
			case NETUNIT_USER: {
                                NetUnit_user *u = (NetUnit_user*)d;
				if(u->ip.s_addr!=0) fprintf(f, "ip %s ", inet_ntoa(u->ip));
				if(u->real_name) fprintf(f, "real_name \"%s\" ", u->real_name);
                                } break;
			default: 
				continue;
		} //switch

		sProcessorListCfgUnit(d, f, nopasswords);
		fprintf(f, "\n");
	} //for

	pthread_rwlock_unlock(Units.rwlock);
	} //netunits
	
	{ // data-sources and storages
	for(st_unit *st=cfg->st_root;st!=NULL;st=st->next) 
		if (st->s) { 
			fprintf(f, "storage %d", st->id);
			if (st->cat==ST_ALL) { 
				fprintf(f, " all\n"); 
				continue;
			}
			if (st->cat&ST_RAW) fprintf(f, " raw"); 
			if (st->cat&ST_SUMMARY) fprintf(f, " summary"); 
			if (st->cat&ST_UNITS) fprintf(f, " units");
			if (st->cat&ST_POLICY) fprintf(f, " policy");
			fprintf(f, "\n"); 
		}
	}

	if (cfg->access_script) fprintf(f, "access-script \"%s\"\n", cfg->access_script);

	fprintf(f, "\n");
}
//////////////////////////////////////////////////////////////////////////
void sProcessorListCfgUnit(NetUnit *d, FILE *f, u_char nopasswords){
	policy_data *cpd;

	if (d->email) fprintf(f, "email %s ", d->email);
	if (d->password) fprintf(f, "password %s ", nopasswords?"***":d->password);

	DSList *dsl=d->getDSList();
	if (dsl){
		fprintf(f, "ds-list ");
		for (; dsl!=NULL; dsl=dsl->next) {
			if (dsl!=d->dsl_root) fprintf(f, ",");
			fprintf(f, "%u", dsl->id);
		}
		fprintf(f, " ");
	}

	if (d->parent && d->parent->name) fprintf(f, "parent %s ", d->parent->name);
	if (d->sys_policy!=SP_NONE) { 
		//in config print only few sys_policies
		PrintSysPolicy(f, d->sys_policy&SP_DENY,d->sys_policy_perm);    
	}
	if (d->flags&NETUNIT_NLP) fprintf(f, "no-local-pass ");
	if (d->ap
#ifdef HAVE_BILLING
	&& !d->account
#endif
	) {
		u_char header_printed=0, already_default; 
		policy_data *cpx;

		for (cpd=d->ap->root; cpd!=NULL; cpd=cpd->next) {
			already_default=0;
			if (ProcessorCfg.def && ProcessorCfg.def->ap) {
				for (cpx=ProcessorCfg.def->ap->root; cpx!=NULL; cpx=cpx->next) 
					if (cpx->policy->id==cpd->policy->id) {
						already_default=1;
						break;
					}
			}
			if (!already_default) {
				if (!header_printed) { fprintf(f, "acct-policy "); header_printed=1; }
				fprintf(f, "%s%s%s ", (cpd->flags&POLICY_FLAG_INV)?"!":"", (cpd->flags&POLICY_FLAG_BRK)?"%":"", cpd->policy->name?cpd->policy->name:"");
			}
		}
	}
	if (d->fp) { 
		u_char header_printed=0, already_default; 
		policy_data *cpx;

		for (cpd=d->fp->root; cpd!=NULL; cpd=cpd->next) {
			already_default=0;
			if (ProcessorCfg.def && ProcessorCfg.def->fp) {
				for (cpx=ProcessorCfg.def->fp->root; cpx!=NULL; cpx=cpx->next) 
					if (cpx->policy->id==cpd->policy->id) {
						already_default=1;
						break;
					}
			}
			if (!already_default) {
				if (!header_printed) { fprintf(f, "fw-policy "); header_printed=1; }
				fprintf(f, "%s%s%s ", (cpd->flags&POLICY_FLAG_INV)?"!":"", (cpd->flags&POLICY_FLAG_BRK)?"%":"", cpd->policy->name?cpd->policy->name:"");
			}
		}
	}
}
//////////////////////////////////////////////////////////////////////////
void cShowProcessor(Connection *conn){
	FIFO *in, *out;
	ServiceProcessor_cfg *cfg=&ProcessorCfg;
	u_short hld_size=sizeof(MsgHolder);

	MsgMgr.Usage(conn);

	fprintf(conn->stream_w, "\nINPUT  Multiplexer\n");
	fprintf(conn->stream_w, "\t current: %u\tmax: %u\ttotal: %lu\t(%ub)\n", Mux_in.num_items, Mux_in.max_items, Mux_in.total_items,Mux_in.num_holders*hld_size); 
	fprintf(conn->stream_w, "OUTPUT Multiplexers\n");
	for(st_unit *st=cfg->st_root;st!=NULL;st=st->next) {
		if (st->s) {
			in=st->in;
			out=st->out;
			fprintf(conn->stream_w, "   Storage %d type ", st->s->instance);
			switch (((ServiceStorage_cfg*)(st->s->cfg))->type) {
				case HASH: fprintf(conn->stream_w, "HASH"); break;
				case MY_SQL: fprintf(conn->stream_w, "MySQL"); break;
				case POSTGRES: fprintf(conn->stream_w, "Postgres"); break;
				case ORACLE: fprintf(conn->stream_w, "Oracle"); break;
				default: fprintf(conn->stream_w, "UNKNOWN"); break;
			}
			fprintf(conn->stream_w, "\n");
			fprintf(conn->stream_w, "\t in current: %u\tmax: %u\ttotal: %lu\t(%ub)\n", in->num_items, in->max_items, in->total_items, in->num_holders*hld_size); 
			fprintf(conn->stream_w, "\tout current: %u\tmax: %u\ttotal: %lu\t(%ub)\n", out->num_items,	out->max_items, out->total_items, out->num_holders*hld_size); 
		}
	}
	
	if(cfg->fifo) {
		fprintf(conn->stream_w, "   Billing\n");
		fprintf(conn->stream_w, "\t current: %u\tmax: %u\ttotal: %lu\t(%ub)\n", cfg->fifo->num_items, cfg->fifo->max_items, cfg->fifo->total_items, cfg->fifo->num_holders*hld_size);
	}	
	
	if(cfg->def) {
		if(cfg->def->ap) {
			fprintf(conn->stream_w, "Default acct policy: ");
			cfg->def->ap->List(conn);
		}
		if(cfg->def->fp) {
			fprintf(conn->stream_w, "Default   fw policy: ");
			cfg->def->fp->List(conn);
		}
	}
}
//////////////////////////////////////////////////////////////////////////
void sPConstructStoreMessages(oid netunit, policy_data *pd){
	char prefix[5] = { 'F', 'H', 'D', 'W', 'M' };
	Message_Store *msg;
	
	for(u_char i=0;i<5;i++) {
		msg=(Message_Store*)MsgMgr.New(MSG_STORE);
		msg->ts=pd->to;
		msg->ap=pd->policy->id;
		msg->netunit=netunit;
		msg->prefix=prefix[i];
		switch(prefix[i]){
			case 'F':
				memcpy(msg->data, (struct pstat *)&pd->flow, sizeof (struct pstat));
				break;
			case 'M': 
				memcpy(msg->data, (struct pstat *)&pd->m, sizeof (struct pstat)); 
				break;
			case 'W':
				memcpy(msg->data, (struct pstat *)&pd->w, sizeof (struct pstat)); 
				break;
			case 'D': 
				memcpy(msg->data, (struct pstat *)&pd->d, sizeof (struct pstat)); 
				break;
			case 'H': 
				memcpy(msg->data, (struct pstat *)&pd->h, sizeof (struct pstat)); 
				break;
		}
		aDebug(DEBUG_PROC_MUX, "DS->P %c in unit:%06X acct:%06X from:%lu to:%lu in:%llu out:%llu\n",msg->prefix, msg->netunit, msg->ap, msg->data->from, msg->ts, msg->data->in, msg->data->out);
		Mux_in.Push((Message*)msg);
	}
}
//////////////////////////////////////////////////////////////////////////
int cAutoAssignIP(Connection *conn, char *param[], u_char no_flag) {
	in_addr start;
	in_addr stop;
	AutoAssignEntry *e,*p=NULL;
	
	if (param[1]) inet_aton(param[1], &start);
	if (param[2]) inet_aton(param[2], &stop);
	
	for(e=ProcessorCfg.auto_assign; e!=NULL; e=e->next) {
		if(e->start.s_addr==start.s_addr && e->stop.s_addr==stop.s_addr) break;
		p=e;
	}

	if(no_flag) {
		if(!e) return PARSE_OK; // nothing to remove
		if(ProcessorCfg.auto_assign==e) ProcessorCfg.auto_assign=e->next;
		else p->next=e->next;
		aFree(e);
		aParse(conn, "auto-assign %s - %s removed\n", param[1], param[2]);
	} else {
		if(e) return PARSE_OK; // already exist
		e=(AutoAssignEntry*)aMalloc(sizeof(AutoAssignEntry));
		e->start.s_addr=start.s_addr;
		e->stop.s_addr=stop.s_addr;
		if(ProcessorCfg.auto_assign==NULL) ProcessorCfg.auto_assign=e;
		else p->next=e;
		e->next=NULL;

		char buf1[32], buf2[32];
		strcpy(buf1, inet_ntoa(start)); strcpy(buf2, inet_ntoa(stop));
		aParse(conn, "auto-assign from %s to %s\n", buf1, buf2);
        }
	
	return PARSE_OK;
}
//////////////////////////////////////////////////////////////////////////
int cAutoUnits(Connection *conn, char *param[], u_char no_flag) {
	AutoUnitsEntry *e,*p=NULL;

	u_char id=strtol(param[1], NULL, 10);
	if (!id) return 0;

	for(e=ProcessorCfg.auto_units; e!=NULL; e=e->next) {
		if(e->id==id) break;
		p=e;
	}

	if (no_flag) {
		if(!e) return PARSE_OK; // nothing to remove
		if(ProcessorCfg.auto_units==e) ProcessorCfg.auto_units=e->next;
		else p->next=e->next;
		aFree(e->put_to_group);
		aFree(e->prefix);
		aFree(e);
		aParse(conn, "auto-units %d removed\n", id);
	} else {
		if(!e)
			e = (AutoUnitsEntry*)aMalloc(sizeof(AutoUnitsEntry));
 
		if (!strcasecmp("type", param[2])) {
			if (!strcasecmp("host", param[3])) e->type=AU_TYPE_HOST; 
			else if (!strcasecmp("user", param[3])) e->type=AU_TYPE_USER; 
		}

		int k=6;
		if (!strcasecmp("naming", param[4])) {
			if (!strcasecmp("by-dns", param[5])) e->naming=AU_NAMING_BY_DNS; 
			else {
				if (!strcasecmp("prefix1", param[5])) e->naming=AU_NAMING_PREFIX1; 
				else if (!strcasecmp("prefix2", param[5])) e->naming=AU_NAMING_PREFIX2; 
				if (param[6] != empty) { e->prefix=set_string(param[6]); k++; }
			}
		}

		// we cannot deal with group ID or check group name since units are not defined yet
		if (!strcasecmp("group", param[k])) if (param[k+1]!=empty) {
			e->put_to_group=set_string(param[k+1]);
		}
	
		if(e->id) 
			aParse(conn, "auto-units %u modified, type %u, naming %u, gr=%s\n", id, e->type, e->naming, e->put_to_group?e->put_to_group:"(no)");
		else {
			e->id=id;
			if(ProcessorCfg.auto_units==NULL) ProcessorCfg.auto_units=e;
			else p->next=e;
			e->next=NULL;
			aParse(conn, "auto-units %u added, type %u, naming %u, gr=%s\n", id, e->type, e->naming, e->put_to_group?e->put_to_group:"(no)");
        	}
	}

	return PARSE_OK;
}

void CreateAutoUnit(oid parent_id, in_addr addr) {
	AutoUnitsEntry *e;
	NetUnit *net=Units.getUnitById(parent_id);

	if(!net) return;
	
	//we need this to protect parent being removed in operation
	pthread_rwlock_rdlock(Units.rwlock);
	
	u_char id=((NetUnit_net*)net)->auto_units_id;
	
        for(e=ProcessorCfg.auto_units; e!=NULL; e=e->next) {
                if(e->id==id) break;
        }
	
	if(!e) {
		pthread_rwlock_unlock(Units.rwlock);
		return;
	}

	NetUnit *u=NULL;
	au_type_enum type = e->type;
        au_naming_enum naming = e->naming;
        char *prefix = e->prefix;

	if (type == AU_TYPE_HOST) {
		u=new NetUnit_host();
		((NetUnit_host*)u)->ip.s_addr=addr.s_addr;
                                
	}
	else if (type == AU_TYPE_USER) {
		u=new NetUnit_user();
		((NetUnit_user*)u)->ip.s_addr=addr.s_addr;
                                
	}
	
	u->id=newOid(0);
	char a[32], *b;
	char buf[32];
	
	bzero(a, 32);
	bzero(buf, 32);
	
	strcpy(a, inet_ntoa(addr));

	aLog(D_WARN, "auto-creating unit %06X, ip=\"%s\"\n", u->id, a);
	
	//set acct and fw policy acording of parent net 
	if(net->ap) net->ap->SetForUnit(POLICY_ACCT, u);
	if(net->fp) net->fp->SetForUnit(POLICY_FW, u);

	pthread_rwlock_unlock(Units.rwlock);

	if (e->put_to_group) {
		NetUnit *parent = Units.getUnit(e->put_to_group);
		if (parent) 
			if (parent->type==NETUNIT_GROUP)
				u->parent=parent;
	}
	                             
	if (naming == AU_NAMING_BY_DNS) {
		struct hostent *hp;
		hp=gethostbyaddr((const char *)&addr, sizeof addr, AF_INET);
		if (hp) {
			u->name=set_string(hp->h_name);
		} 
		else { // DNS failed!
			print_to_string( &u->name, "%s", a);
		}
	}
	else if (naming == AU_NAMING_PREFIX1) {
		strncpy(buf, strrchr(a, '.')+1, 3);
		print_to_string( &u->name, "%s%s", prefix, buf);
	}
	else if (naming == AU_NAMING_PREFIX2) {
		b=strchr(a, '.');
		strncpy(buf, strchr(b+1, '.')+1, 7);
		print_to_string( &u->name, "%s%s", prefix, buf);
	}
	
	aLog(D_WARN, "auto-creating unit %s (%06X) for net %s (%06X)\n", u->name, u->id, net->name, net->id);
	if (u->parent) aLog(D_WARN, "\tthis unit %s (%06X) was put to group %s (%06X)\n", u->name, u->id, u->parent->name?u->parent->name:"<\?\?>", u->parent->id);

	u->unit2trees(ADD);
	Units.Insert(u);
                                
}
//////////////////////////////////////////////////////////////////////////
int cDefault(Connection *conn, char *param[], u_char no_flag){

	u_char i=1;

	if (!strcmp(param[1], "acct-policy")) {
		if(!ProcessorCfg.def) ProcessorCfg.def = new NetUnit(NETUNIT_UNKNOWN);
		PolicyAdd(ProcessorCfg.def, &i, POLICY_ACCT, conn, param, no_flag);
	}
	else if (!strcmp(param[1], "fw-policy")) {
		if(!ProcessorCfg.def) ProcessorCfg.def = new NetUnit(NETUNIT_UNKNOWN);
		PolicyAdd(ProcessorCfg.def, &i, POLICY_FW, conn, param, no_flag);
	}
	else { aParse(conn, "default action name unspecified\n", param[1]); return PARSE_OK; }

	return PARSE_OK;
}
//////////////////////////////////////////////////////////////////////////
int cRestrict(Connection *conn, char *param[], u_char no_flag){
	ServiceProcessor_cfg *cfg=(ServiceProcessor_cfg*)conn->service->cfg;
	u_char i=1;

	while (param[i]!=empty) {
		if (!strcmp(param[i], "all")) {
			if (!strcmp(param[i+1], "drop")){
				cfg->restrict_all=DROP;
				aParse(conn, "restricting ALL traffic to DROP\n");
			}
			else if (!strcmp(param[i+1], "pass")){
				cfg->restrict_all=PASS;
				aParse(conn, "restricting ALL traffic to PASS\n");
			}
			else aParse(conn, "restrict ALL traffic command invalid!\n");
		}
		else if (!strcmp(param[i], "local")) {
			if (!strcmp(param[i+1], "drop")){
				cfg->restrict_local=DROP;
				aParse(conn, "restricting LOCAL traffic to DROP\n");
			}
			else if (!strcmp(param[i+1], "pass")){
				cfg->restrict_local=PASS;
				aParse(conn, "restricting LOCAL traffic to PASS\n");
			}
			else aParse(conn, "restrict LOCAL traffic command invalid!\n");
		}

		else aParse(conn, "restrict command '%s' invalid!\n", param[i]);

		i+=2;
	}
	FW_CHECK_CHANGED(time(NULL))
	return PARSE_OK;
}
//////////////////////////////////////////////////////////////////////////
int cRegister(Connection *conn, char *param[], u_char no_flag){
	ServiceProcessor_cfg *cfg=(ServiceProcessor_cfg*)conn->service->cfg;
	unsigned id=strtol(param[1], NULL, 10);
	st_unit *st,*last=NULL;
	for (st=cfg->st_root;st!=NULL;st=st->next) {
		if (st->id==id) break;
                last=st;
        }

        if(no_flag)
        	if(st) {
                	aParse(conn, "unregistering storage:%d\n", id);
                        aLog(D_INFO, "unregistering storage:%d\n", id);
                        if(st==cfg->st_root) {
                        	cfg->st_root=st->next;
                                if(cfg->st_root) {
                                	aParse(conn, "using storage:%u as source for READ and STAT requests\n",cfg->st_root->id);
                                	aLog(D_INFO, "using storage:%u as source for READ and STAT requests\n",cfg->st_root->id);
                        	}
                        } else
                        	last->next=st->next;
                        aFree(st);
                } else
                	aParse(conn, "no such storage %d or storage %d not registered\n",id,id);
	else {
       		if(st) {
               	 	aParse(conn, "reconfiguring storage %d\n",id);
			st->cat=0;
			for(u_char j=2;param[j]!=empty;j++) {
                        	if (!strcasecmp(param[j], "all")) {
					st->cat=ST_ALL;
					break;
				}
				if (!strcasecmp(param[j], "summary")) st->cat|=ST_SUMMARY;
                        	if (!strcasecmp(param[j], "raw")) st->cat|=ST_RAW;
				if (!strcasecmp(param[j], "units")) st->cat|=ST_UNITS;
				if (!strcasecmp(param[j], "policy")) st->cat|=ST_POLICY;
			}
                } else {
			Service *st_s=Services.getService(SERVICE_STORAGE, id);
			if((conn->service->flags&SERVICE_RUN) && !st_s) {
				aParse(conn, "storage %d registered but never started\n", id);
				return PARSE_OK;
			}
                	aParse(conn, "registering storage: %u\n", id);
                        st=(st_unit*)aMalloc(sizeof(st_unit));
                        st->id=id;
			st->s=st_s;
                       	if(conn->service->flags&SERVICE_RUN) {
				st->in=((ServiceStorage_cfg*)st_s->cfg)->in;
                       		st->out=((ServiceStorage_cfg*)st_s->cfg)->out;
                       		aLog(D_INFO, "using storage:%u, in=%p, out=%p, service_cfg=%p\n", st->id, st->in, st->out, st_s->cfg);
			}

                       	if(!cfg->st_root) {
                        	cfg->st_root=st;
                                aParse(conn, "using storage:%u as source for READ and STAT requests\n",id);
				aLog(D_INFO, "using storage:%u as source for READ and STAT requests\n",id);
                        } else
                        	last->next=st;
                        st->next=NULL;
			st->cat=0;

			for(u_char j=2;param[j]!=empty;j++) {
				if (!strcasecmp(param[j], "all")) {
					st->cat=ST_ALL;
					break;
				}
				if (!strcasecmp(param[j], "summary")) st->cat|=ST_SUMMARY;
				if (!strcasecmp(param[j], "raw")) st->cat|=ST_RAW;
				if (!strcasecmp(param[j], "units")) st->cat|=ST_UNITS;
				if (!strcasecmp(param[j], "policy")) st->cat|=ST_POLICY;
                        }
		}
	}
	return PARSE_OK;
}
//////////////////////////////////////////////////////////////////////////////////////////
void sPMessagesGenerator(ServiceProcessor_cfg *cfg) {
	u_char fill=0;
	struct time_counters tc;
	time_t current_time;
	struct timeval start, stop;
	unsigned long len;

	current_time=time(NULL);
	gettimeofday(&start, NULL);
	
	if (unsigned(current_time - aGetActual('H',current_time)) <= 2*cfg->delay) {
		fill=1;
		PrepareTimeCounters(&tc);
	}
	
	pthread_rwlock_rdlock(Units.rwlock);

	for (NetUnit *u=Units.root; u!=NULL; u=u->next) {
		if(!u->ap) continue; // no need to check unit without account policy
		pthread_rwlock_wrlock(u->ap->rwlock);
		for (policy_data *pd=u->ap->root; pd!=NULL; pd=pd->next) {
			if ( ( fill || (unsigned(current_time-pd->flow.from) >= cfg->lifetime) ) && 
			(pd->flow.in!=0 || pd->flow.out!=0 )) {
				//update counters for current flow
				len=pd->flow.in;
				pd->h.in+=len;
				pd->d.in+=len;
				pd->w.in+=len;
				pd->m.in+=len;

				len=pd->flow.out;
				pd->h.out+=len;
				pd->d.out+=len;
				pd->w.out+=len;
				pd->m.out+=len;

				//construct messages for policy_data pd of unit u
				sPConstructStoreMessages(u->id,pd);
					
				pd->flow.in=pd->flow.out=0;
				pd->flow.from=pd->to=0;
			}
			if (fill) FillTimeCounters(pd,&tc);
		}
		pthread_rwlock_unlock(u->ap->rwlock);
	}
	pthread_rwlock_unlock(Units.rwlock);

	gettimeofday(&stop, NULL);

	aDebug(DEBUG_PROC_MUX, "lookup takes %.4f  seconds\n", ((double)(stop.tv_usec-start.tv_usec))/1000000);
}
//////////////////////////////////////////////////////////////////////////
void sPMessagesMultiplexer(ServiceProcessor_cfg *cfg) {
	Message *msg;
	Message_Store *smsg;
	Message_Read  *rmsg;
	st_unit *st;

	while((msg=Mux_in.Pop())) {
		switch (msg->type) {
		case MSG_STORE:
			smsg=(Message_Store*)msg;
			if (smsg->prefix=='F' && cfg->fifo) 
				cfg->fifo->Push(msg); //pass F message to billing to be counted
			
			for (st=cfg->st_root;st!=NULL;st=st->next) {
				switch (smsg->prefix){
				case 'M':
				case 'W':
				case 'D':
				case 'H':
					if (st->cat&ST_SUMMARY) st->in->Push(msg);
					break;
				case 'F':
					if (st->cat&ST_RAW) st->in->Push(msg);
					break;
				} //switch
				aDebug(DEBUG_PROC_MUX, "P->ST %c st:%u unit:%06X acct:%06X from:%lu to:%lu\n", smsg->prefix, st->id, smsg->netunit, smsg->ap, smsg->data->from, smsg->ts);
			} // for all registered storages
			break;
		case MSG_READ:
			rmsg=(Message_Read*)msg;
			// only ONE storage process READ request, this default to st_root for now
			st=cfg->st_root;
			if (st) {
				switch (rmsg->prefix){
				case 'M':
				case 'W':
				case 'D':
				case 'H':
					if (st->cat&ST_SUMMARY) st->out->Push(msg);
					break;
				case 'F':
					if (st->cat&ST_RAW) st->out->Push(msg);
					break;
				} //switch
				aDebug(DEBUG_PROC_MUX, "P<-ST %c st:%u unit:%06X acct:%06X to now\n", rmsg->prefix, st->id, rmsg->netunit, rmsg->ap);
			}  else {
				aLog(D_WARN, "Can not perform READ request: no storages registered\n");
                        }
			break;
		default:
			aDebug(DEBUG_PROC_MUX, "out <%u> (unknown message type)\n", msg->type);
			break;
		} // this message processed
		
		//check if this message procesed
		if(!msg->Active()) MsgMgr.Delete(msg);

	} // all messages processed
	
	//wakeup all storages
	for (st_unit *st=cfg->st_root;st!=NULL;st=st->next) st->s->Wakeup();
}
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

