/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: policy.c,v 1.103.4.3 2005/02/24 09:58:13 jura Exp $ */

#include "netams.h"

char *policy_type_name[POLICY_TYPES_NUM]={ "unknown", "acct", "fw" };

time_t fw_check_changed;
/////////////////////////////////////////////////////////////////////////
u_char IsNetCheckBroken(NetUnit *u, Policy *p); //1-broken, 0-not
/////////////////////////////////////////////////////////////////////////
// Policy class
Policy::Policy() {
	id=0;
	name=NULL;
	next=NULL;
 	
	target.target_type=PT_UNKNOWN;
	target.check_type=PC_UNKNOWN;
	bzero(&target, sizeof(policy_target));
	
	target.unit=NULL; target.unit_id=0; target.unit_name=NULL;
	
	target.num_ports=0;
	for (u_char i=0; i<PC_MAX_PORTS; i++)
		target.src_ports[i]=target.dst_ports[i]=0;

	target.num_addrs=0;
	for (u_char i=0; i<PC_MAX_ADDRS; i++) 
		target.addr[i].s_addr=target.addr_mask[i].s_addr=INADDR_ANY;
	
	target.file=NULL;
	
	target.proto=IPPROTO_IP; //set default to IP
	
	target.t_begin=target.t_end=0;
	
	for (u_char i=0; i<7; i++) target.day_allowed[i]=1; // allow every day
	target.day_d1=target.day_d2=-1;
	
	target.logic=PL_OR; 
	for (u_char i=0; i<PC_MAX_POLICIES; i++) {
		target.list[i]=NULL;
		target.inversion[i]=0;
	}
}

Policy::~Policy() {
	if(name) aFree(name);
//do not unmark target.unit because it can be marked by another policy also
//	if (target.check_type&PC_UNIT && target.unit) target.unit->flags&=~NETUNIT_POLICY_TARGET;
	if (target.check_type&PC_FILE)  delete target.file;
}

void Policy::setName(char *n){
	if (name) aFree(name);
	name=set_string(n);
}

#define NO_TARGET(tgt) {		\
	if(no_flag) {			\
		target.check_type&=~tgt;\
		break;			\
	}				\
}
u_char Policy::setTarget(char **tgt, u_char *i, u_char no_flag){
	(*i)++;
	while(tgt[*i]!=empty) {
		if (!strncasecmp("proto", tgt[*i], 5)) {
			NO_TARGET(PC_IP_PROTO);

			struct protoent *proto;
			proto=getprotobyname(tgt[*i+1]);
			if(!proto) {
				u_short p=strtol(tgt[*i+1], NULL, 10);
				proto=getprotobynumber(p);
				if(!proto) return 0;
			}
			target.target_type|=PT_IP_TRAFFIC;
			target.check_type|=PC_IP_PROTO;
			target.proto=proto->p_proto;
			(*i)+=1;
		}
		else if (!strncasecmp("tos", tgt[*i], 3)) {
			NO_TARGET(PC_IP_TOS);
			
			target.target_type|=PT_IP_TRAFFIC;
			target.check_type|=PC_IP_TOS;
			target.ip_tos=strtol(tgt[*i+1], NULL, 10);
			(*i)+=1;
		}
		else if (!strncasecmp("time", tgt[*i], 4)) { // "target time 9-18" or "target time 00:40-21:30"   
			NO_TARGET(PC_TIME);

			target.target_type|=PT_IP_TRAFFIC;
			target.check_type|=PC_TIME;
			char *p = strchr(tgt[*i+1], '-'); if (!p) { (*i)+=1; continue;} 
			char *s1 = strchr(tgt[*i+1], ':'); if (s1>p) s1=NULL;
			char *s2 = strchr(p, ':');			
			int beg_h, beg_m=0, end_h, end_m=0;
			sscanf(tgt[*i+1], "%u", &beg_h);
			if (s1) sscanf(s1+1, "%u", &beg_m);
			sscanf(p+1, "%u", &end_h);
			if (s2) sscanf(s2+1, "%u", &end_m);
			target.t_begin=beg_h*60+beg_m;
			target.t_end=end_h*60+end_m;
			(*i)+=1;
		}
		else if (!strncasecmp("day", tgt[*i], 4)) { // "target day Mon-Fri" or "target day Sun"   
			NO_TARGET(PC_DAYOFWEEK);

			target.target_type|=PT_IP_TRAFFIC;
			target.check_type|=PC_DAYOFWEEK;
			char *p = strchr(tgt[*i+1], '-'); 
			int d2,d1=getdayofweek(tgt[*i+1]);
			if (p) d2=getdayofweek(p+1); else d2=d1;
			if (d1!=-1 && d2!=-1) {
				if (d1<=d2) 
					for (u_char k=0; k<7; k++) 
						if (k>=d1 && k<=d2) target.day_allowed[k]=1; 
						else target.day_allowed[k]=0;
				else 
					for (int k=0; k<7; k++)
						if (k<d1 && k>d2) target.day_allowed[k]=0;
						else target.day_allowed[k]=1;
				
				target.day_d1=d1; 
				target.day_d2=d2;
			}
			(*i)+=1;
		}
		else if (!strncasecmp("addr", tgt[*i], 4)) {
			NO_TARGET(PC_IP_ADDR);

			target.target_type|=PT_IP_TRAFFIC;
			target.check_type|=PC_IP_ADDR;
			u_char j=0;
			
			while (tgt[j+*i+1]!=empty && j<PC_MAX_ADDRS) {
				if(!isdigit(tgt[j+*i+1][0])) break;

				getAddr(tgt[j+*i+1], &target.addr[j], &target.addr_mask[j]);
				if(!strncasecmp("mask", tgt[j+*i+2], 4)) {
					inet_aton(tgt[j+*i+3], &target.addr_mask[j]);
					(*i)+=2;
				} else if (target.addr_mask[j].s_addr==INADDR_ANY)
					target.addr_mask[j].s_addr=HOST_MASK;
				
				target.addr[j].s_addr&=target.addr_mask[j].s_addr;
				j++;
			}
			target.num_addrs=j;
			if(j<PC_MAX_ADDRS) target.addr[j].s_addr=target.addr_mask[j].s_addr=0;
			(*i)+=j;
		}
		else if (!strncasecmp("ports", tgt[*i], 5)) {	// target ports s25 22 d24... proto defined before
			NO_TARGET(PC_IP_PORTS);

			target.target_type|=PT_IP_TRAFFIC;
			if(target.proto!=IPPROTO_TCP && target.proto!=IPPROTO_UDP)
				aLog(D_WARN, "ports undefined for this proto %s\n", getprotobynumber(target.proto)->p_name);
			else 
				target.check_type|=PC_IP_PORTS;
			u_char j=0;
			u_short ports;
			char *ptr;
			
			while (tgt[j+*i+1]!=empty && j<PC_MAX_PORTS) {
				if (tgt[j+*i+1][0]=='s' || tgt[j+*i+1][0]=='d' || tgt[j+*i+1][0]=='b') 
					ptr=tgt[j+*i+1]+1;
				else 
					ptr=tgt[j+*i+1];
				
				ports=strtol(ptr, NULL, 10);

				if (!ports) break;
				
				switch (tgt[j+*i+1][0]) {
					case 's':
						target.src_ports[j]=htons(ports);
						target.dst_ports[j]=0;
						break;
					case 'd':
						target.src_ports[j]=0;
						target.dst_ports[j]=htons(ports);
						break;
					default:
						target.src_ports[j]=htons(ports);
						target.dst_ports[j]=htons(ports);
						break;
				}
				j++;
			}
			target.num_ports=j;
			if(j<PC_MAX_PORTS) target.src_ports[j]=target.dst_ports[j]=0;
			(*i)+=j;
		}
		else if (!strncasecmp("units", tgt[*i], 5)) {	// syntax is follows: target units {oid 12345|name AAAA}
			NO_TARGET(PC_UNIT);

			target.target_type|=PT_IP_TRAFFIC;
			target.check_type|=PC_UNIT;
			
			if (!strcasecmp("oid", tgt[*i+1])) {
				target.unit_id=strtol(tgt[*i+2], NULL, 16);
			}
			else if (!strcasecmp("name", tgt[*i+1])) {
				target.unit_name=set_string(tgt[*i+2]);
			}
			else return 0;
			(*i)+=2;
		}
		else if (!strncasecmp("file", tgt[*i], 4)) {
			if(no_flag && target.file) delete target.file;  //special case
			NO_TARGET(PC_FILE);

			target.file = new PrefixFile();
			if(target.file->Load(tgt[*i+1])>=0) {
				target.target_type|=PT_IP_TRAFFIC;
				target.check_type|=PC_FILE;
			}
			(*i)+=1;
		}
		//it's unclear for me how this should work
		else if (!strncasecmp("mailstat", tgt[*i], 8)) {
			target.target_type|=PT_MAIL_STAT;
			target.check_type=PC_UNKNOWN;
		}
		else if (!strncasecmp("ifindex", tgt[*i], 7)) {   // target if s10 8 d7
			if(no_flag) target.target_type&=~PT_NETFLOW_TRAFFIC;
			NO_TARGET(PC_IFINDEX);

			target.target_type|=PT_NETFLOW_TRAFFIC;
			target.check_type|=PC_IFINDEX;
                                        
			u_char j=0;
			u_short ifs;
			char *ptr;		

                        while (tgt[j+*i+1]!=empty && j<PC_MAX_IFS) {
                                if (tgt[j+*i+1][0]=='s' || tgt[j+*i+1][0]=='d' || tgt[j+*i+1][0]=='b')
					ptr=tgt[j+*i+1]+1;
                                else 
					ptr=tgt[j+*i+1];
                                       
				ifs=strtol(ptr, NULL, 10);

				if(ifs==0 && ptr[0]!='0') break;

				switch (tgt[j+*i+1][0]) {
                                        case 's':
                                                target.src_ifs[j]=htons(ifs);
                                                target.dst_ifs[j]=-1;
                                                break;
                                        case 'd':
                                                target.src_ifs[j]=-1;
                                                target.dst_ifs[j]=htons(ifs);
                                                break;
                                        default:
                                                target.src_ifs[j]=htons(ifs);
                                                target.dst_ifs[j]=htons(ifs);
                                                break;
                                }
                                j++;
                        }

			if(j<PC_MAX_IFS) {
				target.src_ifs[j]=-1;
				target.dst_ifs[j]=-1;
			}
                        (*i)+=j;
		}
		else if (!strncasecmp("policy", tgt[*i], 6)) {
			if(no_flag) {
				target.target_type&=~PT_POLICY;	
				break;
			}	

			target.target_type|=PT_POLICY;
			target.check_type|=PC_UNKNOWN;
			if (!strncasecmp("-and", tgt[*i]+6, 4))	target.logic=PL_AND;
			else if (!strncasecmp("-or", tgt[*i]+6, 3)) target.logic=PL_OR;
			else aLog(D_WARN, "Unknown policy logic: %s (continue with OR)\n", tgt[*i]);
			u_char j=*i+1;
			u_char k;
			for (k=0; k<PC_MAX_POLICIES && target.list[k]!=NULL; k++);
			Policy *p;
			char *str;
			u_char inversion;
			aLog(D_INFO, "%u %s\n", k, tgt[j]);
			while (tgt[j]!=empty && k<PC_MAX_POLICIES) {
				str=tgt[j];
				if(str[0]=='!') {
					inversion=1;
					str++;
				}
				else inversion=0;

				p = PolicyL.getPolicy(str);
				if (p==NULL) p = PolicyL.getPolicyById(strtol(str, NULL, 16));
				if (p==NULL) break;
				target.list[k]=p;
				target.inversion[k]=inversion;
				k++;
				j++;
			}				
			(*i)=j+1;
		}
		else 
			aLog(D_WARN, "Wrong target command: %s\n", tgt[*i]);
		(*i)++;
	}

	return 1;
}

char *Policy::getTarget(){
	char *buf=(char*)aMalloc(256);

	if(target.target_type&PT_IP_TRAFFIC) {
		if(target.check_type&PC_IP_PROTO) {
			struct protoent *proto;
			proto=getprotobynumber(target.proto);
			if(proto) sprintf(buf, "proto %s ", proto->p_name);
			else sprintf(buf, "proto %u ", target.proto);
		}
		if (target.check_type&PC_IP_TOS) {
			sprintf(buf+strlen(buf), "tos %u ", target.ip_tos);
		}
		if (target.check_type&PC_TIME) {
			int beg_h, beg_m, end_h, end_m;
			beg_h=target.t_begin/60; beg_m=target.t_begin%60; end_h=target.t_end/60; end_m=target.t_end%60;  
			sprintf(buf+strlen(buf), "time %02d:%02d-%02d:%02d ", beg_h,beg_m,end_h,end_m);
		}
		if (target.check_type&PC_DAYOFWEEK && target.day_d1!=-1 && target.day_d2!=-1) {
			if (target.day_d1==target.day_d2) sprintf(buf+strlen(buf), "day %s ", daysofweek[target.day_d1]);
			else sprintf(buf+strlen(buf), "day %s-%s ", daysofweek[target.day_d1], daysofweek[target.day_d2]);
		}
		if (target.check_type&PC_IP_ADDR) {
			if(target.num_addrs) sprintf(buf+strlen(buf), "addr ");
			for (u_char i=0; i<target.num_addrs; i++) {
				sprintf(buf+strlen(buf), "%s", inet_ntoa(target.addr[i]));
				if(target.addr_mask[i].s_addr!=HOST_MASK) 
					sprintf(buf+strlen(buf), "/%u ", MASK2MLEN(&target.addr_mask[i]));
				else 
					sprintf(buf+strlen(buf), " ");
			}
		}
		if(target.check_type&PC_IP_PORTS) {
			if(target.num_ports) sprintf(buf+strlen(buf), "ports ");
			for (u_char i=0; i<target.num_ports; i++) {
				if (target.src_ports[i]==target.dst_ports[i]) 
					sprintf(buf+strlen(buf), "%u ", ntohs(target.src_ports[i]));
				else if (target.src_ports[i])
					sprintf(buf+strlen(buf), "s%u ", ntohs(target.src_ports[i]));
				else if (target.dst_ports[i]) 
					sprintf(buf+strlen(buf), "d%u ", ntohs(target.dst_ports[i]));
			} 
		}
		if (target.check_type&PC_UNIT) {
			if(!target.unit_id) {
				NetUnit *u;
				if(!(target.unit_name && (u=Units.getUnit(target.unit_name)))) {
					sprintf(buf+strlen(buf), "units name %s ",target.unit_name);
					goto SHOW_PC_FILE;
				}
				target.unit_id=u->id;
			}
			sprintf(buf+strlen(buf), "units oid %06X ",target.unit_id);
		}
SHOW_PC_FILE:
		if (target.check_type&PC_FILE) {	
			if(target.file) sprintf(buf+strlen(buf), "file %s ", target.file->filename);
		}
	}
	if(target.target_type&PT_NETFLOW_TRAFFIC) {
		if(target.check_type&PC_IFINDEX) {
			if(target.src_ifs[0]>=0 || target.dst_ifs[0]>=0)
				sprintf(buf+strlen(buf), "ifindex ");
			for (u_char i=0; i<PC_MAX_IFS && (target.src_ifs[i] >=0 || target.dst_ifs[i]>=0); i++) {
				if (target.src_ifs[i]==target.dst_ifs[i])
					sprintf(buf+strlen(buf), "%u ", ntohs(target.src_ifs[i]));
				else if (target.src_ifs[i]>=0)
					sprintf(buf+strlen(buf), "s%u ", ntohs(target.src_ifs[i]));
				else if (target.dst_ifs[i]>=0)
					sprintf(buf+strlen(buf), "d%u ", ntohs(target.dst_ifs[i]));
			}
		}

	}
	if(target.target_type & PT_MAIL_STAT) {
		sprintf(buf, "mailstat");
	}

	if(target.target_type&PT_POLICY) {
		sprintf(buf+strlen(buf), "policy-%s ", target.logic?"and":"or");
		for (u_char k=0; k<PC_MAX_POLICIES && target.list[k]!=NULL; k++) {
			if (target.inversion[k]) sprintf(buf+strlen(buf), "!");
			if (target.list[k]->name) sprintf(buf+strlen(buf), "%s ", target.list[k]->name);
			else sprintf(buf+strlen(buf), "%06X ", target.list[k]->id);
		}
	}
	return buf;
}

u_char Policy::Check(IPFlowStat *flow, match mf){
	u_char res=0;
	
	#ifdef DEBUG
		char buf1[32], buf2[32];
		strcpy(buf1, inet_ntoa(flow->srcaddr)); strcpy(buf2, inet_ntoa(flow->dstaddr));
		aDebug(DEBUG_POLICY, "Policy %s check: p %u s:%s:%u d:%s:%u mf=%u \n", name,flow->prot, buf1, ntohs(flow->srcport),buf2, ntohs(flow->dstport), mf);
                
	#endif

	if(target.target_type & PT_IP_TRAFFIC) {
		
		if(target.check_type&PC_IP_PROTO) {
			if(target.proto && target.proto!=flow->prot) return 0;
			res=1;
		}
		
		if(target.check_type&PC_IP_PORTS) {
			res=0;
			for (u_char i=0; i<target.num_ports; i++) {
				if (flow->srcport==target.src_ports[i] ||
				flow->dstport==target.dst_ports[i]) {
					res=1;
					break;
				}
			}
			if(!res) return 0;
		}
		
		if(target.check_type&PC_IP_TOS) {
			if(target.ip_tos!=flow->tos) return 0;
			res=1;
		}
		
		if(target.check_type&PC_TIME) {
			struct tm tm;
			time_t t=flow->First;
			localtime_r(&t, &tm);
			int flow_t = tm.tm_hour*60+tm.tm_min;
			if (target.t_begin<target.t_end) {
				if (target.t_begin>flow_t || flow_t>target.t_end) return 0;
			} else 
				if (target.t_begin>flow_t || flow_t<target.t_end) return 0;
			res=1;
		}

		if(target.check_type&PC_DAYOFWEEK) {
			struct tm tm;
			time_t t=flow->First;
			localtime_r(&t, &tm);
			int flow_dw = tm.tm_wday-1; if (flow_dw==-1) flow_dw=6;
			if (target.day_allowed[flow_dw]==0) return 0;
			res=1;
		}

		if(target.check_type&PC_IP_ADDR) {
			res=1;
			
			if(mf&MATCH_SRC) 
				for (u_char i=0; i<target.num_addrs; i++) 
					if(target.addr[i].s_addr==(flow->dstaddr.s_addr&target.addr_mask[i].s_addr))
						goto CHECK_PC_UNIT;
	
			if(mf&MATCH_DST)
				for (u_char i=0; i<target.num_addrs; i++)
					if(target.addr[i].s_addr==(flow->srcaddr.s_addr&target.addr_mask[i].s_addr))
						goto CHECK_PC_UNIT;
			return 0;
		}
CHECK_PC_UNIT:
		if (target.check_type&PC_UNIT) {
			if (target.unit_id || target.unit_name) {
				NetUnit *u=NULL;
				
				if (target.unit) u=target.unit;
				else if (target.unit_id) {
					if(!(u=Units.getUnitById(target.unit_id))) {
						aDebug(DEBUG_POLICY, "IP 'units' policy %s but target unit not exist,skip this check\n", name);
						goto CHECK_PC_FILE;
					}
					target.unit=u;
					u->flags|=NETUNIT_POLICY_TARGET;
				}
				else if (target.unit_name) {
					if(!(u=Units.getUnit(target.unit_name))) {
						aDebug(DEBUG_POLICY, "IP 'units' policy %s but target unit not exist,skip this check\n", name);
						goto CHECK_PC_FILE;
					}
					target.unit=u;
					u->flags|=NETUNIT_POLICY_TARGET;
				}
				
				if(!u) return 0;

				//this check if another point of connection is target unit
				match utmf=u->Check(flow);
				if(utmf==MATCH_NONE || ((mf|utmf)!=MATCH_BOTH)) return 0;
				res=1;
			}
		}
CHECK_PC_FILE: 
		if (target.check_type&PC_FILE) {
			if (target.file->list==NULL) {
				aDebug(DEBUG_POLICY, "prefix check but file is not loaded, skiping this check\n"); 
				goto CHECK_PT_NETFLOW;
			}
			res=1;		

			if(mf&MATCH_SRC && target.file->Check(flow->dstaddr.s_addr)) 
				goto CHECK_PT_NETFLOW;
			
			if(mf&MATCH_DST && target.file->Check(flow->srcaddr.s_addr))
				goto CHECK_PT_NETFLOW;
			
			return 0;
		}
	} //PT_IP_TRAFFIC 

CHECK_PT_NETFLOW:
	
	if(target.target_type&PT_NETFLOW_TRAFFIC) {
		if(target.check_type&PC_IFINDEX) {
			res=0;
			for (u_char i=0; i<PC_MAX_IFS && (target.src_ifs[i]>=0 || target.dst_ifs[i]>=0); i++) {
				if (flow->input==target.src_ifs[i] ||
				flow->output==target.dst_ifs[i]) {
					res=1;
					break;
				}
			}
			if(!res) return 0;
		}
	} //PT_NETFLOW_TRAFFIC

	if(target.target_type & PT_POLICY) {
		res=0;
		for (u_char k=0; k<PC_MAX_POLICIES && target.list[k]!=NULL; k++) {
			res=target.list[k]->Check(flow, mf);
			if(target.inversion[k]) res=(res?0:1);
			
			if (target.logic^res) break;
		}
		if(!res) return 0;
	}
	//if there is no successfull checks - result will be 0
	aDebug(DEBUG_POLICY, "matched\n");
	return res;
}
/////////////////////////////////////////////////////////////////////////
// PolicyList class
PolicyList::PolicyList(){
	root=last=NULL;
	num_policies=0;
	rwlock=(pthread_rwlock_t*)aMalloc(sizeof (pthread_rwlock_t));
	pthread_rwlock_init(rwlock, NULL);
}

PolicyList::~PolicyList(){
	pthread_rwlock_destroy(rwlock);
	aFree(rwlock);
}

void PolicyList::Insert(Policy *s){
	pthread_rwlock_wrlock(rwlock);
	Policy *d;
	for(d=root; d!=NULL; d=d->next)
		if (d==s) { 
			pthread_rwlock_unlock(rwlock);
			return;
		}
	if (root==NULL) root=s;
	else last->next=s;
	last=s; 
	num_policies++;
	pthread_rwlock_unlock(rwlock);
}

void PolicyList::Delete(Policy *s){
	pthread_rwlock_wrlock(rwlock);
	Policy *d, *p=NULL;
	for(d=root; d!=NULL; d=d->next)	{
		if (d==s) {
			if (s==root && s==last ) root=last=NULL;
			else if (s==root) root=s->next;
			else if (s==last) { last=p; last->next=NULL; }
			else p->next=s->next;

			num_policies--;
			Units.DeletePolicyElsewhere(s);
			break;
		}
		p=d; 
 	}
	pthread_rwlock_unlock(rwlock);
}

void PolicyList::DeleteAll(){
	pthread_rwlock_wrlock(rwlock);
	Policy *d,*p;
	for (d=root; d!=NULL; d=p) {
		p=d->next;
		delete d;
 	}
	pthread_rwlock_unlock(rwlock);
}

void PolicyList::Delete(oid id){
	Policy *p=getPolicyById(id);
	if (p) Delete(p);
}

void PolicyList::DeleteUnitFromTarget(NetUnit *u) {
	pthread_rwlock_rdlock(rwlock);
	Policy *d;
	for(d=root; d!=NULL; d=d->next)
		if (d->target.check_type&PC_UNIT && d->target.unit) d->target.unit=NULL;
	
	pthread_rwlock_unlock(rwlock);
}

Policy* PolicyList::getPolicy(char *name){
	pthread_rwlock_rdlock(rwlock);
	Policy *d;
	for(d=root; d!=NULL; d=d->next)	{
		if (!strcmp(d->name, name)) {
			pthread_rwlock_unlock(rwlock);
			return d;
		}
 	}
	pthread_rwlock_unlock(rwlock);
	return NULL;
}

Policy* PolicyList::getPolicyById(oid id){
	pthread_rwlock_rdlock(rwlock);
	Policy *d;
	for(d=root; d!=NULL; d=d->next)	{
		if (d->id==id) {
			pthread_rwlock_unlock(rwlock);
			return d;
		}
 	}
	pthread_rwlock_unlock(rwlock);
	return NULL;
}
/////////////////////////////////////////////////////////////////////////
int cPolicy(Connection *conn, char *param[], u_char no_flag){
	Policy *p=NULL;
	u_char i=1;
	oid id=0;
	
	//this is to support old policy command syntax
	if (!strcmp(param[1], "acct") || !strcmp(param[1], "fw")) i=2;

	if (!strcmp(param[i], "name")) 
		p=PolicyL.getPolicy(param[i+1]);
	else if (!strcmp(param[i], "oid")) {
		id=strtol(param[i+1], NULL, 16);
		p=PolicyL.getPolicyById(id);
	}
	else { aParse(conn, "policy %s name/oid unspecified\n", param[i]); return PARSE_OK; }
	
	if (!p && !no_flag) { 
		p = new Policy();
		p->id = newOid(id);
		aParse(conn, "policy %06X created\n", p->id); 
		PolicyL.Insert(p);
	} else if (!p && no_flag) {
		aParse(conn, "policy not exist\n");
		return PARSE_OK;
	}
	else if (p && no_flag){
		aParse(conn, "policy %06X deleted\n", p->id);
		PolicyL.Delete(p);
		delete p;
		return PARSE_OK;
	}

	while (param[i]!=empty) {
 		if (strcasecmp(param[i], "name")==0) { 
			p->setName(param[i+1]);
			aParse(conn, "policy %06X name set: %s\n", p->id, p->name);
		}
		else if (strcasecmp(param[i], "no")==0) {
			no_flag=1;
			i--;
		}
		else if (strcasecmp(param[i], "target")==0) {
			u_char j=p->setTarget(param, &i, no_flag); 
			if (j) {
				char *buf=p->getTarget();
				aParse(conn, "policy %06X target set: %s\n", p->id, buf);
				aFree(buf);
			}
			else aParse(conn, "policy %06X target invalid\n", p->id); 
		}
		else if (strcasecmp(param[i], "do")==0) {
			if (p->target.check_type&PC_FILE) {
		   		aParse(conn, "policy %06X file do: '%s'\n", p->id, param[i+1]);
		   		p->target.file->Do(conn, param[i+1], param[i+2]);
			}
	   		else aParse(conn, "policy %06X action unsupported!\n", p->id);	
		}
		else if (strcasecmp(param[i], "oid")==0) ;//do nothing
		else 
			aParse(conn, "policy %06X command unknown: %s\n", p->id, param[i]); 
			
		i=i+2;
	}
	return PARSE_OK;
}
/////////////////////////////////////////////////////////////////////////
int cShowPolicy(Connection *conn){
	pthread_rwlock_rdlock(PolicyL.rwlock);
	Policy *d;
	char *buf;

	fprintf(conn->stream_w, "%6s | %15s | %-20s\n", "OID", "NAME", "PARAMS");
	for (d=PolicyL.root; d!=NULL; d=d->next)	{
		fprintf(conn->stream_w, "%06X | %15s | ", d->id, d->name?d->name:"<\?\?>");
		if (d->target.target_type) {
			buf=d->getTarget();
			fprintf(conn->stream_w, "target: %-12s", buf);
			aFree(buf);
		}
		fprintf(conn->stream_w, "\n");
	}

	pthread_rwlock_unlock(PolicyL.rwlock);
	return PARSE_OK;
}

/////////////////////////////////////////////////////////////////////////
// PdList class
PdList::PdList(){
	root=last=NULL;
	num_policies=0;
	rwlock=(pthread_rwlock_t*)aMalloc(sizeof (pthread_rwlock_t));
        pthread_rwlock_init(rwlock, NULL);
}

PdList::~PdList(){
	policy_data *tmp;
	while(root) {
		tmp=root;
		root=root->next;
		aFree(tmp);
	}
	pthread_rwlock_destroy(rwlock);
	aFree(rwlock);
}

policy_data *PdList::Add(Policy *p, u_char flags, u_char place) {
	policy_data *d;	
	policy_data *after=NULL;
	u_char curr_place=0;
	if (!place) place=255;

	pthread_rwlock_wrlock(rwlock);
	for (d=root; d!=NULL; d=d->next) {
		if (d->policy==p && d->flags==flags) { 
			pthread_rwlock_unlock(rwlock);
			return NULL;
		}
		curr_place++;
		if (curr_place<place) after=d;
	}

	policy_data *np = (policy_data*)aMalloc(sizeof(policy_data));
	np->flags=flags;
	np->next=NULL;
	np->timestamp=0;		   
	np->policy=p;
	np->check=np->match=0; 
	
	struct time_counters tc;
	PrepareTimeCounters(&tc);
	FillTimeCounters(np,&tc);
	
	if (root==NULL) { root=np; last=np; }  // empty list
	else if (after==NULL) { np->next=root; root=np; } // first
	else if (after==last) { last->next=np; last=np; } // last
	else { np->next=after->next; after->next=np; }
	
	num_policies++;
	pthread_rwlock_unlock(rwlock);
	return np;
}

u_char PdList::Delete(Policy *s) {
	pthread_rwlock_wrlock(rwlock);
	policy_data *d, *p=NULL;
	for (d=root; d!=NULL; d=d->next)	{
		if (d->policy==s) {
			if (d==root && d==last ) root=last=NULL;
			else if (d==root) root=d->next;
			else if (d==last) { last=p; last->next=NULL; }
			else p->next=d->next;

			num_policies--;
			aFree(d);
			break;
		}
		p=d; 
	}
	pthread_rwlock_unlock(rwlock);
	return num_policies;
}

void PdList::List(Connection *conn){
	pthread_rwlock_rdlock(rwlock);
	policy_data *d;
	for (d=root; d!=NULL; d=d->next){
		fprintf(conn->stream_w, "%s(%06X) ", d->policy->name, d->policy->id);
	}
	fprintf(conn->stream_w, "\n");
	pthread_rwlock_unlock(rwlock);
}

void PdList::ListForCfg(FILE *f){
	pthread_rwlock_rdlock(rwlock);
	policy_data *d;
	for (d=root; d!=NULL; d=d->next){
		fprintf(f, "%s ", d->policy->name);
	}
	fprintf(f, "\n");
	pthread_rwlock_unlock(rwlock);
}

void PdList::SetForUnit(policy_type pt, NetUnit *u){
	policy_data *d;
	// notice!!! there is no deadlock between rwlock and Add(d->policy) 
	// they are from different units
	pthread_rwlock_rdlock(rwlock);
	for (d=root; d!=NULL; d=d->next){
		if (pt==POLICY_ACCT) {
			if(!u->ap) u->ap = new PdList();
		 	u->ap->Add(d->policy, d->flags); 
			aParse(cInternal, "unit %06X acct %s%spolicy %s added \n",
				u->id,
				(d->flags&POLICY_FLAG_INV)?"!":"",
				(d->flags&POLICY_FLAG_BRK)?"%":"",
				d->policy->name);

			sPConstructReadMessages(u->id, u->ap->last);
		} else if (pt==POLICY_FW) {
			if(!u->fp) u->fp = new PdList();
		 	u->fp->Add(d->policy, d->flags);
			aParse(cInternal, "unit %06X fw %s%spolicy %s added \n",
				u->id,
				(d->flags&POLICY_FLAG_INV)?"!":"",
				(d->flags&POLICY_FLAG_BRK)?"%":"",
				d->policy->name);
		}
	}
	pthread_rwlock_unlock(rwlock);
}

policy_data *PdList::Get(Policy *p){
	policy_data *d, *r=NULL;

	pthread_rwlock_rdlock(rwlock);
	for (d=root; d!=NULL; d=d->next) 
		if (d->policy==p) {
			r=d;
			break;
		}
	pthread_rwlock_unlock(rwlock);
	return r;
}

time_t PdList::LastUsed(){
	policy_data *d;
	time_t l=0;

	pthread_rwlock_rdlock(rwlock);
	for (d=root; d!=NULL; d=d->next) 
		if (d->timestamp > l) l=d->timestamp;
	pthread_rwlock_unlock(rwlock);
	return l;
}

void PdList::ClearLastUsed(){
	policy_data *d;
	pthread_rwlock_wrlock(rwlock);
	for (d=root; d!=NULL; d=d->next) d->timestamp=0;
	pthread_rwlock_unlock(rwlock);
}
/////////////////////////////////////////////////////////////////////////
// we need this fucntion to find out will be ip packet checked correctly for given policy
// for this unit
// see PR 59
u_char PdList::IsNetCheckBroken(NetUnit *u) {

	if(u->type!=NETUNIT_CLUSTER && u->type!=NETUNIT_NET) return 0; //all is ok with this units

	if(u->type==NETUNIT_NET) {
		unsigned mask=((NetUnit_net*)u)->mask.s_addr;
		if(mask==0 || mask==0xFF000000 || mask==0xFFFF0000 || mask==0xFFFFFF00 || mask==0xFFFFFFFF) return 0;
	}

	Policy *p;
	pthread_rwlock_rdlock(rwlock);

	for (policy_data *d=root; d!=NULL; d=d->next) {
		p=d->policy;
		if(p->target.target_type&PT_IP_TRAFFIC) {
		//this is only case we might perform wrong check
			if((p->target.check_type&PC_UNIT) || (p->target.check_type&PC_FILE)) {
				pthread_rwlock_unlock(rwlock);
				return 1;
			}
		}
	}
	pthread_rwlock_unlock(rwlock);
	return 0;
}
/////////////////////////////////////////////////////////////////////////
void PolicyDataUpdate(match m, policy_data *p, IPFlowStat *flow){
	unsigned long t=flow->Last;
	unsigned long len=flow->dOctets;

	p->to=p->timestamp=t;
	if (p->flow.from==0) p->flow.from=t;

	if (m&MATCH_DST) { 
		p->flow.in+=len;
	}

	if (m&MATCH_SRC) { 
		p->flow.out+=len;
	}
}
/////////////////////////////////////////////////////////////////////////
void PolicyAdd(NetUnit *u, u_char *i, policy_type po, Connection *conn, char *param[], u_char no_flag){
	Policy *p;
	char *c_param;
	policy_flag flags;
	u_char place=0;
	while (param[(*i)+1]!=empty) {
		flags=POLICY_FLAG_NONE;
		(*i)++; 
		c_param=param[*i];
		if (c_param[0]=='!') { flags|=POLICY_FLAG_INV; c_param++; }
		if (c_param[0]=='%') { flags|=POLICY_FLAG_BRK; c_param++; }
		if (c_param[0]=='!') { flags|=POLICY_FLAG_INV; c_param++; }
		
		if (!place) { 
			if ((place=strtol(c_param, NULL, 10))!=0) continue; else place=0;
		}
		
		p=PolicyL.getPolicy(c_param);
		policy_data *pdc;
		
		if (p)  {
			if (po==POLICY_ACCT) { 
				if (no_flag) {
					if(!u->ap) return;
					if(!u->ap->Delete(p)) {delete u->ap; u->ap=NULL; }
					aParse(conn, "unit %06X acct %spolicy %s removed\n", u->id, (flags&POLICY_FLAG_INV)?"inverted ":"", c_param); 
				} else {
					if(!u->ap) u->ap = new PdList();
					if ((pdc=u->ap->Add(p, flags, place))) {
						aParse(conn, "unit %06X acct %s%spolicy %s added <%u>\n", u->id, (flags&POLICY_FLAG_INV)?"inverted ":"", (flags&POLICY_FLAG_BRK)?"break ":"", c_param, place); 
						// we should construct a request to fill the policy_data structure
						sPConstructReadMessages(u->id, pdc);
					}
				}
				//set flags
				if(u->ap->IsNetCheckBroken(u)) u->flags |= NETUNIT_BROKEN_ACCT_MF;
				else u->flags &= ~NETUNIT_BROKEN_ACCT_MF;
			} else if (po==POLICY_FW) { 
				if (no_flag) {
					if(!u->fp) return;
					if(!u->fp->Delete(p)) {delete u->fp; u->fp=NULL;}
					aParse(conn, "unit %06X fw %spolicy %s removed\n", u->id, (flags&POLICY_FLAG_INV)?"inverted ":"", c_param); 
				} else {
					if(!u->fp) u->fp = new PdList();
					if ((pdc=u->fp->Add(p, flags, place))){
						aParse(conn, "unit %06X fw %s%spolicy %s added <%u>\n", u->id, (flags&POLICY_FLAG_INV)?"inverted ":"", (flags&POLICY_FLAG_BRK)?"break ":"", c_param, place); 
					}
				}
				//set flags
				if(u->fp->IsNetCheckBroken(u)) u->flags |= NETUNIT_BROKEN_FW_MF;
				else u->flags &= ~NETUNIT_BROKEN_FW_MF;
				FW_CHECK_CHANGED(time(NULL));
			}
		} else { // policy is defined
			aParse(conn, "unit %06X policy '%s' undefined\n", u->id, c_param);
			(*i)-=2;
			return;
		}
		place=0;
	}
}
/////////////////////////////////////////////////////////////////////////
void PrintSysPolicy(FILE *f, SysPolicy p, oid perm){
	if(perm) fprintf(f, "sys-%s-%06X ", (p&SP_DENY)?"deny":"allow", perm);
	else if(p&SP_DENY) fprintf(f, "sys-deny ");
	if(p&SP_DENY_MONEY) fprintf(f, "sys-deny-money ");
	if(p&SP_DENY_BLOCK) fprintf(f, "sys-deny-block ");
	if(p&SP_DENY_QUOTA) fprintf(f, "sys-deny-quota ");
	if(p&SP_DENY_LOGIN) fprintf(f, "sys-deny-login ");
	if(p&SP_DENY_AUTH) fprintf(f, "sys-deny-auth ");
}

void SetSysPolicy(char *p, NetUnit *u, Connection *conn){
	char *p1, *p2;
	char *s1=NULL, *s2=NULL;
	oid id=u->sys_policy_perm;
	SysPolicy prev=u->sys_policy;
	p1=p+4;
	if (!p1) { aParse(conn, "incorrect NetUnit sys-* value\n"); return; }

	p2=strchr(p1, '-'); 
	if (p2)  {
		p2++;
		s1=(char*)aMalloc(p2-p1+2); strncpy(s1, p1, p2-p1-1);
		s2=(char*)aMalloc(strlen(p2)+2); strncpy(s2, p2, strlen(p2));
		if(s2[0]=='0') {
			id=strtol(s2, NULL, 16); //here we depends that our oids looks like 0XXXXX
			NetUnit *pu=Units.getUnitById(id);
			if(!pu) {
				aParse(conn, "No such unit %06X\n", id);
				goto END;
			}
			else if(pu->type==NETUNIT_GROUP) {
				aParse(conn, "Unable to work with NetUnit group: %s(%06X)\n", pu->name,pu->id);
				goto END;
			}
		}
	}
	else {
		s1=(char*)aMalloc(strlen(p1)+2); strncpy(s1, p1, strlen(p1));
		s2=(char*)aMalloc(2); strncpy(s2, "", 1);
	}
	
	aDebug(DEBUG_PARSE, "SysPolicy got '%s' '%s'\n", s1, s2);
	
	if (strcasecmp(s1, "allow")==0) {
		if(id) u->sys_policy&=~SP_DENY;
		else if (strcasecmp(s2, "")==0) u->sys_policy=SP_NONE;
		else if (strcasecmp(s2, "money")==0) u->sys_policy&=~SP_DENY_MONEY;
		else if (strcasecmp(s2, "block")==0) u->sys_policy&=~SP_DENY_BLOCK;
		else if (strcasecmp(s2, "quota")==0) u->sys_policy&=~SP_DENY_QUOTA;
		else if (strcasecmp(s2, "login")==0) u->sys_policy&=~SP_DENY_LOGIN;
		else if (strcasecmp(s2, "auth")==0) u->sys_policy&=~SP_DENY_AUTH;
		else aParse(conn, "System policy for %06X allow: is incorrect\n", u->id);
	}
	else if (strcasecmp(s1, "deny")==0) {
		if(id) u->sys_policy|=SP_DENY;
		else if (strcasecmp(s2, "")==0) u->sys_policy|=SP_DENY;
		else if (strcasecmp(s2, "money")==0) u->sys_policy|=SP_DENY_MONEY;
		else if (strcasecmp(s2, "block")==0) u->sys_policy|=SP_DENY_BLOCK;
		else if (strcasecmp(s2, "quota")==0) u->sys_policy|=SP_DENY_QUOTA;
		else if (strcasecmp(s2, "login")==0) u->sys_policy|=SP_DENY_LOGIN;
		else if (strcasecmp(s2, "auth")==0) u->sys_policy|=SP_DENY_AUTH;
		else aParse(conn, "System policy for %06X deny: is incorrect\n", u->id);
	}
	else aParse(conn, "System policy for %06X *: is incorrect\n", u->id);
				
	if (prev!=u->sys_policy || id!=u->sys_policy_perm) {
		u->sys_policy_perm=id;
		if (s2[0]) aParse(conn, "System policy for %06X set to '%s-%s'\n", u->id, s1, s2);
		else aParse(conn, "System policy for %06X set to '%s'\n", u->id, s1);
	}

	FW_CHECK_CHANGED(time(NULL));

END:
	aFree(s1); aFree(s2);
}
/////////////////////////////////////////////////////////////////////////
void sPConstructReadMessages(oid netunit, policy_data *pdata) {
	char prefix[4] = { 'H', 'D', 'W', 'M' };
	Message_Read *msg;

	for(u_char i=0;i<4;i++) {
		msg=(Message_Read*)MsgMgr.New(MSG_READ);
		msg->pdata=pdata;
		msg->ap=pdata->policy->id;
		msg->netunit=netunit;
		msg->prefix=prefix[i];
		Mux_in.Push((Message*)msg);
	}
	aDebug(DEBUG_PROC_MUX, "P<-P H unit:%06X acct:%06X from:%lu\n", msg->netunit, msg->ap, msg->pdata->h.from);
	aDebug(DEBUG_PROC_MUX, "P<-P D unit:%06X acct:%06X from:%lu\n", msg->netunit, msg->ap, msg->pdata->d.from);
	aDebug(DEBUG_PROC_MUX, "P<-P W unit:%06X acct:%06X from:%lu\n", msg->netunit, msg->ap, msg->pdata->w.from);
	aDebug(DEBUG_PROC_MUX, "P<-P M unit:%06X acct:%06X from:%lu\n", msg->netunit, msg->ap, msg->pdata->m.from); 
	
	return;
}
//////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
