/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: messages_fifo.c,v 1.21 2005/01/17 13:13:21 jura Exp $ */

#include "netams.h"

//////////////////////////////////////////////////////////////////////////
// Message class

Message::Message(message_type t){
	type=t;
	next=NULL;
	active=0;
	lock=(pthread_mutex_t*)aMalloc(sizeof(pthread_mutex_t));
	pthread_mutex_init(lock, NULL);
}

Message::~Message(){
	pthread_mutex_destroy(lock);
	aFree(lock);
}

void Message::Push(){
	pthread_mutex_lock(lock);
	active++;
	pthread_mutex_unlock(lock);
}

void Message::Pop(){
	pthread_mutex_lock(lock);
	active--;
	pthread_mutex_unlock(lock);
}

u_char Message::Active(){
	return active;
}

//////////////////////////////////////////////////////////////////////////
Message_Read::Message_Read(): Message(MSG_READ) {
	ap=netunit=0;
	pdata=NULL;
}

Message_Read::~Message_Read() {
}
//////////////////////////////////////////////////////////////////////////
Message_Store::Message_Store(): Message(MSG_STORE) {
	ap=netunit=0;
	ts=0;
	data=(pstat*)aMalloc(sizeof(pstat));
}

Message_Store::~Message_Store() {
	aFree(data);
}
//////////////////////////////////////////////////////////////////////////
Message_Alert::Message_Alert(): Message(MSG_ALERT) {
	al=(alert*)aMalloc(sizeof(alert));
}

Message_Alert::~Message_Alert() {
	aFree(al);
}
//////////////////////////////////////////////////////////////////////////
Message_PVM_Billing::Message_PVM_Billing(): Message(MSG_PVM_BILLING) {
	account=NULL;
	text=NULL;
	ts=0;
}

Message_PVM_Billing::~Message_PVM_Billing() {
}
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
// FIFO class
FIFO::FIFO(unsigned max){
	root=last=ready=NULL;
	num_items=total_items=0;
	num_holders=0;
	max_items=max;
	lock=(pthread_mutex_t*)aMalloc(sizeof(pthread_mutex_t));
	pthread_mutex_init(lock, NULL);
}

FIFO::~FIFO(){
	MsgHolder *h;
	while(ready) {
		h=ready;
		ready=ready->next;
		aFree(h);
	}
	pthread_mutex_destroy(lock);
	aFree(lock);
}

MsgHolder* FIFO::newHolder() {
	MsgHolder *h;

	if(ready) {
		h=ready;
		ready=ready->next;
	} else {
		h=(MsgHolder*)aMalloc(sizeof(MsgHolder));
		num_holders++;
	}
	return h;
}

void FIFO::freeHolder(MsgHolder *h) {
	h->next=ready;
	ready=h;
}

u_char FIFO::Push(Message *msg){
	MsgHolder *h;

//	printf("FIFO push: root:%p last:%p\n", root, last);
	pthread_mutex_lock(lock);
	
	h=newHolder();
	msg->Push();
	h->msg=msg;
	if (root==NULL) root=h;
	else last->next=h;
	last=h;
	h->next=NULL; 
	num_items++;
	total_items++;
	
	if (num_items>max_items) {
		aLog(D_WARN, "FIFO %p overloaded, discarding message %p\n",this,root);
		msg=root->msg;
		h=root;
		root=root->next;
		num_items--;
		msg->Pop();
		if(!msg->Active()) MsgMgr.Delete(msg);
		freeHolder(h);
        }

	pthread_mutex_unlock(lock);
//	printf("FIFO push: %p\n", msg);
	return 1;
//	printf("FIFO push:  %p lock:%p\n", msg,lock);
}

Message* FIFO::Pop(){
	Message *m=NULL;

//	printf("FIFO pop: root:%p last:%p\n", root, last);
	pthread_mutex_lock(lock);
	
	if (root) {
		MsgHolder *h;

		m=root->msg;
		h=root;
		root=root->next;
		num_items--;
		m->Pop();
		freeHolder(h);
	}
	pthread_mutex_unlock(lock);
//	printf("FIFO pop:  %p lock:%p\n", m,lock);
	return m;
}

Message* FIFO::TryPop(){
        if(root) return root->msg;
	else return NULL;
}
//////////////////////////////////////////////////////////////////////////
// DS_FIFO class

DS_FIFO::DS_FIFO(unsigned max){
        root=last=ready=NULL;
        num_items=total_items=ready_items=0;
        max_items=max;
        lock=(pthread_mutex_t*)aMalloc(sizeof (pthread_mutex_t));
        pthread_mutex_init(lock, NULL);
}

DS_FIFO::~DS_FIFO(){
	IPStatMsg *m,*ptr=ready;
        ready=NULL;
        while(ptr) {
                m=ptr->next;
                aFree(ptr);
                ptr=m;
        }

        pthread_mutex_destroy(lock);
        aFree(lock);
}

IPStatMsg* DS_FIFO::Push(IPStatMsg *msg){
        pthread_mutex_lock(lock);
        if (root==NULL) root=msg;
        else last->next=msg;
        last=msg;
        msg->next=NULL;
        num_items++;
        total_items++;
        if (num_items>max_items) {
                aLog(D_WARN, "FIFO %p overloaded, discarding message %p\n",this,root);
                msg=root;
                root=root->next;
                aFree(msg);
                num_items--;
        }
	// prepare message
	if(ready) {
		msg=ready;
		ready=ready->next;
		ready_items--;
	} else {
		msg=(IPStatMsg*)aMalloc(sizeof(IPStatMsg));
	}
        pthread_mutex_unlock(lock);
//      printf("FIFO push: %p\n", msg);
        return msg;
}

IPStatMsg* DS_FIFO::Pop(IPStatMsg *msg){
        pthread_mutex_lock(lock);
	// put used message in ready
	if(msg) {
		msg->next=ready;
		ready=msg;
		ready_items++;
	}

        msg=root;
        if (root) {
		root=root->next;
        	num_items--;
	}
        pthread_mutex_unlock(lock);
//      printf("FIFO pop:  %p\n", m);
        return msg;
}


//////////////////////////////////////////////////////////////////////////
//MessageManager
MessageManager::MessageManager() {
	bzero(ready, NUM_MSG_TYPES*sizeof(Message*));;
	bzero(ready_items, NUM_MSG_TYPES*sizeof(unsigned));
	bzero(allocated, NUM_MSG_TYPES*sizeof(unsigned));
	
	total_ready_items=0;
	total_allocated=0;
	lock=(pthread_mutex_t*)aMalloc(sizeof(pthread_mutex_t));
	pthread_mutex_init(lock, NULL);
}

MessageManager::~MessageManager() {
	Message *m,*ptr;
	
	for(u_char t=1; t<NUM_MSG_TYPES; t++) {
		ptr=ready[t];
		ready[t]=NULL;
		while(ptr) {
			m=ptr->next;
			delete ptr;
			ptr=m;
		}
	}
	pthread_mutex_destroy(lock);
        aFree(lock);
}

Message* MessageManager::New(message_type t) {
	Message *m;
	pthread_mutex_lock(lock);
	if(ready[t]) {
		m=ready[t];
		ready[t]=ready[t]->next;
		ready_items[t]--;
		total_ready_items--;
	} else {
		switch(t) {
			case MSG_STORE:
				m=(Message*) new Message_Store();
				break;
			case MSG_PVM_BILLING:
				m=(Message*) new Message_PVM_Billing();
				break;
			case MSG_READ:
				m=(Message*) new Message_Read();
				break;
			case MSG_ALERT:
			case MSG_COMMAND:
				m=(Message*) new Message_Alert();
				break;
			default:
				aLog(D_WARN, "Trying to allocate wrong message type %u\n",t);
				pthread_mutex_unlock(lock);
				return NULL;
		}
		total_allocated++;
		allocated[t]++;
	}
	pthread_mutex_unlock(lock);
	return m;
}

void MessageManager::Delete(Message *m) {
	if(!m) return;
	pthread_mutex_lock(lock);
	message_type t=m->type;
	m->next=ready[t];
	ready[t]=m;
	ready_items[t]++;
	total_ready_items++;
	pthread_mutex_unlock(lock);
}
void MessageManager::Usage(Connection *conn) {
	fprintf(conn->stream_w, "Total messages:\n");
	fprintf(conn->stream_w, "\tStore: %u used, %u ready = %u bytes\n", allocated[1]-ready_items[1], ready_items[1], allocated[1]*sizeof(Message_Store));
	fprintf(conn->stream_w, "\tRead: %u used, %u ready = %u bytes\n", allocated[2]-ready_items[2], ready_items[2], allocated[2]*sizeof(Message_Read));
	fprintf(conn->stream_w, "\tAlert: %u used, %u ready = %u bytes\n", allocated[3]-ready_items[3], ready_items[3], allocated[3]*sizeof(Message_Alert));
	fprintf(conn->stream_w, "\tCommand: %u used, %u ready = %u bytes\n", allocated[4]-ready_items[4], ready_items[4], allocated[4]*sizeof(Message_Alert));
	fprintf(conn->stream_w, "\tPVM Billing: %u used, %u ready = %u bytes\n", allocated[5]-ready_items[5], ready_items[5], allocated[5]*sizeof(Message_PVM_Billing));
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////



