/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: flow.c,v 1.52.4.3 2005/01/31 23:57:16 anton Exp $ */

#include "netams.h"

//////////////////////////////////////////////////////////////////////////
u_char DSsystem(NetUnit *u, dsUlist *start);
u_char DSfw(NetUnit *u, IPFlowStat* flow, match mf);
//////////////////////////////////////////////////////////////////////////
Flow::Flow(Service *s, u_char fw_check) {
	//Flow should know about friends
	instance=s->instance;
	struct ServiceDS_cfg *cfg=(struct ServiceDS_cfg*)s->cfg;
	IPtree=cfg->IPtree;
	fifo=cfg->ds_fifo;
	
	if(fw_check) flags=FLOW_FW_CHECK;
	
	e_used=e_total=0;
	bw_used=bw_total=0;
	bwReady=NULL;

	t=t_now=t_msg=time(NULL);
	table=NULL;
	ready=NULL;
	last_id=0; 
	expired_id=0;
	sent_flows=0;
	sent_packets=0;
	table=(entry**)aMalloc(IPV4_HASH_SIZE*sizeof(entry*));
	aLog(D_INFO, "IPv4 HASH initialized with size %lu\n",IPV4_HASH_SIZE);
	message=(IPStatMsg*)aMalloc(sizeof(IPStatMsg));
	aLog(D_INFO, "Flow engine started\n");
}
//////////////////////////////////////////////////////////////////////////
Flow::~Flow() {
	aLog(D_INFO, "Flow engine stopped\n");
	FlushAll();
	entry *ptr=ready;
	while(ready) {
		ptr=ready;
		ready=ready->next;
		aFree(ptr);
	}
	
	bwUlist *bwu;
	while(bwReady) {
		bwu=bwReady;
		bwReady=bwReady->next;
		aFree(bwu);
	}

	aFree(message);
	aFree(table);
	aLog(D_INFO, "IPv4 HASH uninitialized\n");
}
//////////////////////////////////////////////////////////////////////////
void Flow::CheckExpire(entry *e){
    	if (unsigned(e->Last - e->First) > ACTIVE_TIMEOUT || 
	unsigned(t_now - e->Last) > INACTIVE_TIMEOUT)  e->flags|=ENTRY_EXPIRED;
#ifdef DEBUG 
   	if(e->flags&ENTRY_EXPIRED) aDebug(DEBUG_FLOW,"%u expired\n", e->id);
#endif
}
//////////////////////////////////////////////////////////////////////////
u_char Flow::Processpacket(struct ip *ip) {
    	unsigned hash;
	u_short sp=0, dp=0;
	in_addr_t src,dst;
	u_char proto;
	u_char tos;
	u_char tcp_flags=0;
    	entry *e;
	u_short chunk=0;
	
	dst = ip->ip_dst.s_addr;
	src = ip->ip_src.s_addr;
	proto = ip->ip_p;
	tos = ip->ip_tos;
	
	if (ntohs(ip->ip_off) & (IP_OFFMASK | IP_MF)) {
		//packet is fragmented. check obtained from /usr/src/sys/netinet/ip_fastfwd.c
		sp=dp=0;
		hash=IPV4_ADDR_HASH(src, dst);
	} else  
	if (proto==IPPROTO_TCP || proto==IPPROTO_UDP) {
		struct tcphdr *th;
		th=(struct tcphdr *)((unsigned char *)ip + ip->ip_hl*4);
		sp=th->th_sport;
		dp=th->th_dport;
		if(proto==IPPROTO_TCP) tcp_flags=th->th_flags;
		hash=IPV4_FULL_HASH(src, dst, sp, dp);
	} else {
		hash=IPV4_ADDR_HASH(src, dst);
	}
    	
	for(e=table[hash]; e!=NULL; e=e->next){
        	if (e->dstaddr.s_addr==dst && e->srcaddr.s_addr==src &&
		e->prot==proto && e->tos==tos &&
		e->srcport==sp && e->dstport==dp) {
			if(BWCheck(e, ntohs(ip->ip_len), tv)) return 0; //drop packet
            		e->dOctets+=ntohs(ip->ip_len);
                	e->dPkts++;
                	e->Last=t_now;
                	break;
        	}
		chunk++;
    	}
	
	if (e==NULL) {
		if(chunk>MAX_CHUNK) FlushAll(); //protection against DoS
                if(ready) { //we'll use ready empty entry
                        e=ready;
                        ready=ready->next;
                } else { // we must create a new flow record
                        e=(entry*)aMalloc(sizeof(entry));
			e_total++;
                }
		e_used++;
		e->dstaddr.s_addr=dst;
        	e->srcaddr.s_addr=src;
		e->dstport=dp;
		e->srcport=sp;
		e->dOctets=ntohs(ip->ip_len);
        	e->prot=proto;
        	e->tos=tos;
		e->tcp_flags=tcp_flags;
        	e->First=e->Last=t_now;
        	e->flags=0;
        	e->id=last_id;
        	e->dPkts=1;
        	last_id++;
        
		if((flags&FLOW_FW_CHECK) && !FW((IPFlowStat*)e)) e->flags|=ENTRY_BLOCKED;
		e->next=table[hash];
		table[hash]=e;
		
#ifdef DEBUG
                char buf1[32], buf2[32];
                strcpy(buf1, inet_ntoa(ip->ip_src));
                strcpy(buf2, inet_ntoa(ip->ip_dst));

                aDebug(DEBUG_FLOW,"[%u(%u)]-%p %u s:%s:%u d:%s:%u fw:%u\n", hash, e->id, e, proto, buf1, ntohs(sp), buf2, ntohs(dp), !(e->flags&ENTRY_BLOCKED));
#endif
    	}
	
	return !(e->flags&ENTRY_BLOCKED);
}
//////////////////////////////////////////////////////////////////////////
void Flow::Expiresearch(struct timeval *ttv){
    	unsigned i;
    	entry *e, *prev;
	entry *tmp;
	
	tv=ttv;

	t_now=tv->tv_sec;
	if(unsigned(t_now-t)<CHECK_EXPIRE) return;
	t=t_now;

    	aDebug(DEBUG_FLOW,"ExpireSearch t:[%u] a:[%u] at %lu: \n", last_id,last_id-expired_id, t_now);
    	
	for (i=0; i<IPV4_HASH_SIZE; i++) {
        	prev=NULL;
		e=table[i];
		while(e) {	
			tmp=e->next;
            		CheckExpire(e);
            		if (e->flags&ENTRY_EXPIRED) {
				// with high probability used packet might be used again thou ...
				if(e->dPkts) {
					if(!(e->flags&ENTRY_BLOCKED)) ToSend(e);
					//we need to recheck blocked status
					if((flags&FLOW_FW_CHECK) && IS_FW_CHECK_CHANGED(e->First)) {
						if(FW((IPFlowStat*)e))
							e->flags&=~ENTRY_BLOCKED;
						else {
							e->flags|=ENTRY_BLOCKED;
							FreeBWentry(e);
						}
					}
					e->dPkts=0;
					e->dOctets=0;	
					e->flags&= ~ENTRY_EXPIRED;
					e->First=e->Last=t_now;
//					e->id=last_id;
//                                      last_id++;
					prev=e;
                		} else {
					expired_id++;
					if (e==table[i])
                                        	table[i]=(entry *)e->next;
                                	else {
                                        	prev->next=e->next;
                                	}
					//this means we have flow that was unaccessible one round
					//move it to ready 
					e->next=ready;
					ready=e;
					e_used--;
					FreeBWentry(e);
				}
            		} else {
            			prev=e;
			}
			e=tmp;
        	}
    	}
	
	if(unsigned(t-t_msg) > MESSAGE_LIFETIME) DoSend();
}
//////////////////////////////////////////////////////////////////////////
void Flow::FlushAll() {
    	unsigned i;
    	entry *e;
	entry *tmp;

    	aDebug(DEBUG_FLOW,"FlushAll::begin\n");
        aDebug(DEBUG_FLOW,"FlushAll t:[%u] a:[%u] at %lu: ", last_id, last_id-expired_id, t_now);

    	for (i=0; i<IPV4_HASH_SIZE; i++) {
		e=table[i];
		while(e) {
                	aDebug(DEBUG_FLOW,"%u{%u} ", e->id, i);
            		expired_id++;
			tmp=e->next;

            		if(e->dPkts && !(e->flags&ENTRY_BLOCKED)) ToSend(e);
			e->next=ready;
			ready=e;
            		e_used--;
			FreeBWentry(e); //flush BW slots
            		
			e=tmp;
		}
		table[i]=NULL;
    	}
	// it's time to check if there is stalled unsent data in message
	DoSend();

    	aDebug(DEBUG_FLOW,"FlushAll::end\n");
}
//////////////////////////////////////////////////////////////////////////
void Flow::ToSend(entry *e) {
	
	memcpy(&message->records[message->header.count], e, sizeof(IPFlowStat));

        message->header.count++;
        aDebug(DEBUG_FLOW,"record %u in packet %llu prepared, %lu\n", message->header.count-1, sent_packets+1, e->dOctets);
        if (message->header.count==FLOWS_PER_PAK) DoSend();
}
//////////////////////////////////////////////////////////////////////////
void Flow::DoSend() {
	if(!message->header.count) return;

	message->header.version=FLOW_VERSION;
        message->header.engine_id=instance;
        message->header.engine_type=31;
        message->header.flow_sequence=sent_flows;

	sent_packets++;
        sent_flows+=message->header.count;
	aDebug(DEBUG_FLOW,"sending Flow packet with %u flows\n", message->header.count);
	//transfer to processor here
	message=fifo->Push(message);
	message->header.count=0;
	t_msg=t_now;
}
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
u_char Flow::FW(IPFlowStat *flow) {
        NetUnit *u, *up;
        dsUlist *start,*dsulist,*dsu;
        match mf;
        restrict_type fw_res=ProcessorCfg.restrict_all; // 0=pass, 1=drop

        pthread_rwlock_rdlock(IPtree->rwlock);

        //here we organize units that matches packet
        start=IPtree->checktree(flow, CHECK_FW);

        for(dsulist=start;dsulist!=NULL;dsulist=dsulist->link[CHECK_FW]) {
                if(fw_res==BRK) { //we know we drop packet, but we need to clear mf flag
                        dsulist->mf[CHECK_FW]=MATCH_NONE;
                        continue;
                }
                mf=dsulist->mf[CHECK_FW];
                for(dsu=dsulist;dsu!=NULL;dsu=dsu->next) {
                        u=dsu->u;

                        //check packet might be processed incorrectly due to wrong match flag
                        if(u->flags&NETUNIT_BROKEN_FW_MF) {
                                match mmf=u->Check(flow);
                                if(mmf!=mf && mf==MATCH_DST) continue;
                                mf=mmf;
                        } else
                                mf=dsulist->mf[CHECK_FW];

                        // yes, this unit corresponds to this data packet
                        aDebug(DEBUG_DS_IP, "packet matches unit %s (oid=%06X)\n", u->name, u->id);
                        // we will perform the fw check for this unit
                        if(DSsystem(u, start)) {
                                aDebug(DEBUG_DS_IP, " fw check for this unit: SYS-DROP\n");
                                fw_res=BRK;
                                break;
                        }
                        if(DSfw(u, flow, mf)) {
                                aDebug(DEBUG_DS_IP, " fw check for this unit: DROP\n");
                                fw_res=BRK;
                                break;
                        }
                        aDebug(DEBUG_DS_IP, " fw check for this unit: SYS-PASS\n");
                        
			// now check the fw policy for all parents (groups)
                        for (up=u->parent; up!=NULL; up=up->parent)
                                if (up->checkDSList(instance))  {
                                        // we will perform the fw check for this unit because it is parent
                                        aDebug(DEBUG_DS_IP, "packet matches parent %s (oid=%06X)\n", up->name, up->id);
                                        if(DSsystem(up, start)) {
                                                aDebug(DEBUG_DS_IP, " fw check for this parent: SYS-DROP\n");
                                                fw_res=BRK;
						goto NEXT;
                                        }
                                        if(DSfw(up, flow, mf)) {
                                                aDebug(DEBUG_DS_IP, " fw check for this parent: DROP\n");
                                                fw_res=BRK;
						goto NEXT;
                                        }
                                        aDebug(DEBUG_DS_IP, " fw check for this parent: PASS\n");
                                }
                        if(!(u->flags&NETUNIT_NLP)) fw_res=PASS; //thus we ignore no-local-pass

                        //init bw list
                        if(dsu->bwd) AddBW2entry(dsu->bwd, (entry*)flow, mf);
                }
NEXT:
                dsulist->mf[CHECK_FW]=MATCH_NONE;
        }

        pthread_rwlock_unlock(IPtree->rwlock);
        aDebug(DEBUG_DS_IP, "Forward decision = %u\n", !fw_res);

        return !fw_res;
}

u_char DSfw(NetUnit *u, IPFlowStat *flow, match mf){
        policy_data *cpd;
        u_char fw_res=ProcessorCfg.restrict_local;

        if(!u->fp) return fw_res;

        for (cpd=u->fp->root; cpd!=NULL; cpd=cpd->next) {
                cpd->check++;
                if(cpd->policy->Check(flow, mf)) {
                        if (cpd->flags&POLICY_FLAG_INV) {
                                cpd->match++;
                                fw_res=0;
                                if (cpd->flags&POLICY_FLAG_BRK) break;
                        } else {
                                cpd->match++;
                                fw_res=1;
                                break;
                        }
                 } else {
                        if (cpd->flags&POLICY_FLAG_INV) {
                                fw_res=1;
                                break;
                        } else {
                                fw_res=0;
                                if (cpd->flags&POLICY_FLAG_BRK) break;
                        }
                }
        }
        return fw_res;
}

u_char DSsystem(NetUnit *u, dsUlist *start) {
        if(!u->sys_policy_perm) return u->sys_policy;

        for(dsUlist *dsulist=start;dsulist!=NULL;dsulist=dsulist->link[CHECK_FW])
                if (dsulist->u!=u && u->sys_policy_perm==dsulist->u->id)
                        return (u->sys_policy&SP_DENY);

        return (u->sys_policy&~SP_DENY);
}
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
void Flow::AddBW2entry(BWdata *bw, entry *e, match mf) {
	bwUlist *bwu;

	if(bwReady) {
		bw_used++;
		bwu=bwReady;
		bwReady=bwReady->next;
	} else {
		bwu=(bwUlist*)aMalloc(sizeof(bwUlist));
		bw_total++;
	}

	bwu->bw=bw;
	bwu->bw->num_flows++;

	bwu->mf=mf;
	bwu->next=e->bwRoot;
	e->bwRoot=bwu;
//	printf("add bwdata[in=%lu,out=%lu,unit=%s] to entry %p, match=%d\n", bw->limit_in, bw->limit_out, bw->u->name, e, mf);
}

void Flow::FreeBWentry(entry *e){
	if(!e->bwRoot) return;

	bwUlist *bwu;
	u_char num=1;

	for(bwu=e->bwRoot; bwu->next!=NULL; bwu=bwu->next, num++){
		bwu->bw->num_flows--;
	}

	bwu->next=bwReady;
	bwReady=e->bwRoot;
	bw_used-=num;

	e->bwRoot=NULL;
}
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
