/*
 *  nwauth.c
 *
 *  Check a user/passwd against a NetWare server
 *
 *  Copyright (C) 1996 by Volker Lendecke
 *
 */

#include "config.h"

#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ncp/ncplib.h>

#include <libintl.h>
#define _(X) gettext(X)

extern int bindery_only;

static char *progname;

static void
usage(void)
{
	fprintf(stderr, _("usage: %s [options]\n"), progname);
}

static void
help(void)
{
	printf("\n");
	printf(_("usage: %s [options]\n"), progname);
	printf(_("\n"
	       "-h             Print this help text\n"
	       "-S server      Server name to be used\n"
	       "-U username    Username sent to server\n"
	       "-t type        Object type (decimal value)\n"
	       "-P password    User password\n"
	       "-b             Bindery only mode\n"
	       "\n"));
}

int
main(int argc, char *argv[])
{
	struct ncp_conn_spec *spec;
	struct ncp_conn *conn;
	char *server = NULL;
	char *object_name = NULL;
	char *object_pwd = NULL;
	int object_type = NCP_BINDERY_USER;
	union {
		struct sockaddr any;
#ifdef CONFIG_NATIVE_IPX
		struct sockaddr_ipx ipx;
#endif
#ifdef CONFIG_NATIVE_IP
		struct sockaddr_in inet;
#endif
		} addr;
	long err;
	int prterr;
	
	char *str;

	int opt;

	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);
	
	progname = argv[0];

	prterr = isatty(0);
	
	while ((opt = getopt(argc, argv, "h?S:U:t:P:b")) != EOF)
	{
		switch (opt)
		{
		case 'S':
			server = optarg;
			break;
		case 'U':
			object_name = optarg;
			break;
		case 't':
			object_type = atoi(optarg);
			break;
		case 'P':
			object_pwd = optarg;
			break;
		case 'b':
			bindery_only = 1;
			break;
		case 'h':
		case '?':
			help();
			exit(1);
		default:
			usage();
			exit(1);
		}
	}

	spec = ncp_find_conn_spec(server, object_name, "",
				  1, getuid(), &err);

	if (spec == NULL)
	{
		if (prterr)
			com_err(argv[0], err, _("when trying to find server"));
		exit(1);
	}
	if ((err = ncp_find_fileserver(spec->server, &addr.any, sizeof(addr))) != 0)
	{
		if (prterr)
			com_err(argv[0], err, _("when trying to find server"));
		exit(1);
	}
	spec->login_type = object_type;
	memset(spec->password, 0, sizeof(spec->password));

	if (object_pwd) {
		size_t l = strlen(object_pwd);
		if (l >= sizeof(spec->password)) {
			if (prterr)
				fprintf(stderr, _("Password too long\n"));
			exit(1);
		}
		memcpy(spec->password, object_pwd, l+1);
	} else if (isatty(0)) {
		str = getpass(_("Enter password: "));
		if (strlen(str) >= sizeof(spec->password))
		{
			printf(_("Password too long\n"));
			exit(1);
		}
		strcpy(spec->password, str);
	} else {
		fgets(spec->password, sizeof(spec->password), stdin);
	}

	str_upper(spec->password);

	if ((conn = ncp_open(spec, &err)) == NULL)
	{
		if (prterr)
			com_err(argv[0], err, _("when trying to open connection"));
		exit(1);
	}
	ncp_close(conn);
	return 0;
}
