/*
 *  ncpmount.c
 *
 *  Copyright (C) 1995, 1997 by Volker Lendecke
 *
 *  1/20/96 - Steven N. Hirsch (hirsch@emba.uvm.edu)
 *
 *  If the ncpfs support is not loaded and we are using kerneld to
 *  autoload modules, then we don't want to do it here.  I added
 *  a conditional which leaves out the test and load code.
 *
 *  Even if we _do_ want ncpmount to load the module, passing a
 *  fully-qualified pathname to modprobe causes it to bypass a 
 *  path search.  This may lead to ncpfs.o not being found on
 *  some systems.
 * 
 *  11/28/98 - Wolfram Pienkoss <wp@bsz.shk.th.schule.de>
 *  NLS added. Thanks Petr Vandrovec for the idea of charset select
 *  and many hints.
 *
 *  09/09/99 - Petr Vandrovec <vandrove@vc.cvut.cz>
 *  ttl option added
 *  added resetting behavior for SIGTERM
 *  reverted back to exec("/sbin/nwmsg") for incomming messages
 *     this allows tknwmsg work again...
 */

#include "config.h"

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>
#include <ncp/ext/socket.h>
#include <sys/param.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/errno.h>
#include <sys/time.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <stdlib.h>
#include <sys/mount.h>
#include <mntent.h>
#include <ncp/kernel/ipx.h>
#include <sys/ioctl.h>
#ifdef MOUNT3
#include <utmp.h>
#include <syslog.h>
#endif 

#include <ncp/kernel/ncp.h>
#include <ncp/kernel/ncp_fs.h>
#include <ncp/nwcalls.h>
#ifdef NDS_SUPPORT
#include <ncp/ndslib.h>
#endif
#if defined(MOUNT2) && defined(MOUNT3)
#define MULTIVERSION_MOUNT 1
#else
#define MULTIVERSION_MOUNT 0
#endif
#if MULTIVERSION_MOUNT
#include <sys/utsname.h>
#endif

#include <libintl.h>
#define _(X) gettext(X)

#ifndef NR_OPEN
/* getrlimit... */
#define NR_OPEN 1024
#endif

#ifndef MS_MANDLOCK
#define MS_MANDLOCK	64
#endif
#ifndef MS_NOATIME
#define MS_NOATIME	1024
#endif
#ifndef MS_NODIRATIME
#define MS_NODIRATIME	2048
#endif

#include "ncpmount.h"

static char *progname;
static char mount_point[MAXPATHLEN + 1];
static void usage(void);
static void help(void);

#ifdef MOUNT2
/* Returns 0 if the filesystem is in the kernel after this routine
   completes */
static int
load_ncpfs(void)
{
	FILE *ffs;
	char s[1024];
	char *p, *p1;
	pid_t pid;
	int status;

	/* Check if ncpfs is in the kernel */
	ffs = fopen("/proc/filesystems", "r");

	if (ffs == NULL)
	{
		perror(_("Error: \"/proc/filesystems\" could not be read:"));
		return -1;
	}
	p = NULL;
	while (!feof(ffs))
	{
		p1 = fgets(s, sizeof(s), ffs);
		if (p1)
		{
			p = strstr(s, "ncpfs");
			if (p)
			{
				break;
			}
		}
	}
	fclose(ffs);

	if (p)
	{
		return 0;
	}
	/* system() function without signal handling, from Stevens */

	if ((pid = fork()) < 0)
	{
		return 1;
	} else if (pid == 0)
	{
		/* child */
		execl("/sbin/modprobe", "modprobe", "ncpfs", NULL);
		_exit(127);	/* execl error */
	} else
	{
		/* parent */
		while (waitpid(pid, &status, 0) < 0)
		{
			if (errno != EINTR)
			{
				status = -1;
				break;
			}
		}
	}
	return status;
}
#endif	/* MOUNT2 */

struct ncp_mount_data_independent {
	int		ncp_fd;
	int		wdog_fd;
	int		message_fd;
	uid_t		mounted_uid;
	struct sockaddr_ipx serv_addr;
	unsigned char	*server_name;
	unsigned char   *mount_point;
	const unsigned char   *mounted_vol;
	unsigned int	time_out;
	unsigned int	retry_count;
	struct {
	unsigned int	mount_soft:1;
	unsigned int    mount_intr:1;
	unsigned int	mount_strong:1;
	unsigned int	mount_no_os2:1;
	unsigned int	mount_no_nfs:1;
	unsigned int	mount_extras:1;
	unsigned int	mount_symlinks:1;
		      } flags;
	uid_t		uid;
	gid_t		gid;
	mode_t		file_mode;
	mode_t		dir_mode;
};

#ifdef MOUNT3
static int process_connection(const struct ncp_mount_data_independent* mnt);
#endif 

#if MULTIVERSION_MOUNT
int getmountver(void) {
	struct utsname name;
	int maj, mid;
	int ver;

	if (uname(&name)) {
		fprintf(stderr, _("Cannot get kernel release\n"));
		exit(1);
	}
	if (sscanf(name.release, "%d.%d", &maj, &mid) != 2) {
		fprintf(stderr, _("Cannot convert kernel release \"%s\" to number\n"), name.release);
		exit(1);
	}
	ver = maj*0x10000 + mid*0x100;
	if (ver < 0x20100) {
#ifdef MOUNT2
		return 2;
#else
		fprintf(stderr, _("This kernel requires mount version 2 which is not available\n"));
		exit(1);
#endif
	}
#ifdef MOUNT3
	return 3;
#else
	fprintf(stderr, _("This kernel requires mount version 3 which is not available\n"));
	exit(1);
#endif
}
#else
#ifdef MOUNT3
#define getmountver() 3
#else
#ifdef MOUNT2
#define getmountver() 2
#else
#error "You must define at least one of MOUNT2, MOUNT3"
#endif
#endif
#endif

/* Check whether user is allowed to mount on the specified mount point */
static int
mount_ok(struct stat *st)
{
	if (!S_ISDIR(st->st_mode))
	{
		errno = ENOTDIR;
		return -1;
	}
	if ((getuid() != 0)
	    && ((getuid() != st->st_uid)
		|| ((st->st_mode & S_IRWXU) != S_IRWXU)))
	{
		errno = EPERM;
		return -1;
	}
	return 0;
}

#ifdef MOUNT3
/*
 * This function changes the processes name as shown by the system.
 * Stolen from Marcin Dalecki's modald :-)
 */
static void inststr(char *dst[], int argc, const char *src)
{
	/* stolen from the source to perl 4.036 (assigning to $0) */
	char *ptr, *ptr2;
	int count;
	ptr = dst[0] + strlen(dst[0]);
	for (count = 1; count < argc; count++) {
		if (dst[count] == ptr + 1)
			ptr += strlen(++ptr);
	}
	if (environ[0] == ptr + 1) {
		for (count = 0; environ[count]; count++)
			if (environ[count] == ptr + 1)
				ptr += strlen(++ptr);
	}
	count = 0;
	for (ptr2 = dst[0]; ptr2 <= ptr; ptr2++) {
		*ptr2 = '\0';
		count++;
	}
	strncpy(dst[0], src, count);
	for (count = 1; count < argc; count++) {
		dst[count] = NULL;
	}
}
#endif 

#ifdef MOUNT2
int ncp_mount_v2(const char* mount_name, unsigned long flags, const struct ncp_mount_data_independent* data) {
	struct ncp_mount_data_v2 datav2;

	if (data->serv_addr.sipx_family != AF_IPX) {
		errno = EPROTONOSUPPORT;
		return -1;
	}
	datav2.version    = NCP_MOUNT_VERSION_V2;
	datav2.ncp_fd     = data->ncp_fd;
	datav2.wdog_fd    = data->wdog_fd;
	datav2.message_fd = data->message_fd;
	datav2.mounted_uid = data->mounted_uid;
	memcpy(&datav2.serv_addr, &data->serv_addr, sizeof(datav2.serv_addr));
	strncpy(datav2.server_name, data->server_name, sizeof(datav2.server_name));
	strncpy(datav2.mount_point, data->mount_point, sizeof(datav2.mount_point));
	strncpy(datav2.mounted_vol, data->mounted_vol, sizeof(datav2.mounted_vol));
	datav2.time_out    = data->time_out;
	datav2.retry_count = data->retry_count;
	datav2.flags       = 0;
	datav2.flags      |= data->flags.mount_soft?NCP_MOUNT2_SOFT:0;
	datav2.flags      |= data->flags.mount_intr?NCP_MOUNT2_INTR:0;
	datav2.flags      |= data->flags.mount_strong?NCP_MOUNT2_STRONG:0;
	datav2.flags      |= data->flags.mount_no_os2?NCP_MOUNT2_NO_OS2:0;
	datav2.flags      |= data->flags.mount_no_nfs?NCP_MOUNT2_NO_NFS:0;
	if (data->flags.mount_extras || data->flags.mount_symlinks) {
		errno = EINVAL;
		return -1;
	}
	datav2.uid         = data->uid;
	datav2.gid         = data->gid;
        datav2.file_mode   = data->file_mode;
        datav2.dir_mode    = data->dir_mode;
	return mount(mount_name, data->mount_point, "ncpfs", flags, (void*) &datav2);
}	
#endif

#ifdef MOUNT3	
int ncp_mount_v3(const char* mount_name, unsigned long flags, const struct ncp_mount_data_independent* data, int argc, char* argv[]) {
	struct ncp_mount_data_v3 datav3;
	int err;

	datav3.version    = NCP_MOUNT_VERSION_V3;
	datav3.ncp_fd     = data->ncp_fd;
	datav3.mounted_uid = data->mounted_uid;
	strncpy(datav3.mounted_vol, data->mounted_vol, sizeof(datav3.mounted_vol));
	datav3.time_out    = data->time_out;
	datav3.retry_count = data->retry_count;
	datav3.flags       = 0;
	datav3.flags      |= data->flags.mount_soft?NCP_MOUNT3_SOFT:0;
	datav3.flags      |= data->flags.mount_intr?NCP_MOUNT3_INTR:0;
	datav3.flags      |= data->flags.mount_strong?NCP_MOUNT3_STRONG:0;
	datav3.flags      |= data->flags.mount_no_os2?NCP_MOUNT3_NO_OS2:0;
	datav3.flags      |= data->flags.mount_no_nfs?NCP_MOUNT3_NO_NFS:0;
	datav3.flags	  |= data->flags.mount_extras?NCP_MOUNT3_EXTRAS:0;
	datav3.flags	  |= data->flags.mount_symlinks?NCP_MOUNT3_SYMLINKS:0;
	datav3.uid         = data->uid;
	datav3.gid         = data->gid;
        datav3.file_mode   = data->file_mode;
        datav3.dir_mode    = data->dir_mode;
	connect(datav3.ncp_fd, (const struct sockaddr *)&data->serv_addr, sizeof(data->serv_addr));
	datav3.wdog_pid = fork();
	if (datav3.wdog_pid < 0) {
		fprintf(stderr, _("could not fork: %s\n"), strerror(errno));
		exit(1);
	}
	if (datav3.wdog_pid == 0) {
		/* Child */
		inststr(argv, argc, "ncpd");
		process_connection(data);
		exit(0);	/* Should not return from process_connection */
	}
	err=mount(mount_name, data->mount_point, "ncpfs", flags, (void*) &datav3);
	if (err) {
		/* Mount unsuccesful so we have to kill daemon                */
		/* If mount success, kernel kills daemon (at least in 2.1.79) */
		kill(datav3.wdog_pid, SIGTERM);
	}
	return err;
}	
#endif

int
ncp_mount_specific(struct ncp_conn* conn, int pathNS, const unsigned char* NWpath, int pathlen) {
	int result;

#ifdef NCP_IOC_SETROOT
	{
		struct ncp_setroot_ioctl sr;

		if (pathlen == 1) {
			sr.volNumber = -1;
			sr.name_space = -1;
			sr.dirEntNum = 0;
		} else {
			struct nw_info_struct dirinfo;
	
			result = ncp_obtain_file_or_subdir_info2(conn, pathNS, NW_NS_DOS,
				0x8006, RIM_ALL, 0xFF, 0, 0, NWpath, pathlen, &dirinfo);
			if (result) {
				return -ENOENT;
			}
			if (!(dirinfo.attributes & htonl(0x10000000))) {
				return -ENOTDIR;
			}
			sr.volNumber = dirinfo.volNumber;
			sr.name_space = NW_NS_DOS;
			sr.dirEntNum = dirinfo.dirEntNum;
		}
		result = ioctl(ncp_get_fid(conn), NCP_IOC_SETROOT, &sr);
		if (!result) {
			return 0;
		}
	}
#endif
	if ((pathlen != 1) && (*NWpath != 1)) {
		fprintf(stderr,
#ifdef NCP_IOC_SETROOT
			_("Remote directory is specified but kernel"
				  " does not support subdir mounts\n")
#else
			_("Remote directory is specified but ncpmount"
				  " does not support subdir mounts\n")
#endif
		);
		return -ENOPKG;
	}
	if (ioctl(ncp_get_fid(conn), NCP_IOC_CONN_LOGGED_IN, NULL) != 0) {
		return -errno;
	}
	return 0;
}

static int get_passwd(const char* file, const char* server, const char* user, const char** passwd) {
	FILE *f;
	char b[2048];

	if (*passwd) return 0;	/* you have password */
	if (!file) return 0;	/* you have not password file */
	if (!server) return 0;	/* something is wrong */
	if (!user) return 0;
	f = fopen(file, "r");
	if (!f) return -1;	/* NOENT */
	while (fgets(b, sizeof(b), f)) {
		char* s;
		char* u;
		char* p;
		int l;

		b[sizeof(b) - 1] = 0;
		l = strlen(b);
		while ((l > 0) && (b[l-1] == '\n')) l--;
		b[l] = 0;
		s = strtok(b, "/");
		if (!s) continue;	/* malformed line */
		u = strtok(NULL, ":");
		if (!u) continue;	/* malformed line */
		p = strtok(NULL, ":");
		if (!p) continue;
		if (*s && strcmp(server, s)) continue;
		if (strcmp(user, u)) continue;
		*passwd = strdup(p);
		fclose(f);
		return 0;
	}
	fclose(f);
	return 0;
}

struct ncp_mount_info {
	struct ncp_mount_data_independent mdata;
	struct ncp_nls_ioctl nls_cs;
	int		upcase_password;
	unsigned int	flags;
	const char*	server;
	const char*	user;
	const char*	password;
	const char*	server_name;
	const char*	passwd_file;
	int		allow_multiple_connections;
	int		sig_level;
	int		force_bindery_login;
	const char*	remote_path;
	int		dentry_ttl;
	unsigned int	pathlen;
	unsigned char	NWpath[512];
};

struct optinfo {
	char		shortopt;
	const char*	option;
#define FN_NOPARAM	1
#define FN_INT		2
#define FN_STRING	4
	int		flags;
	void		*proc;
	unsigned int	param;
};

static void opt_set_flags(struct ncp_mount_info* info, unsigned int param) {
	info->flags |= param;
}

static void opt_clear_flags(struct ncp_mount_info* info, unsigned int param) {
	info->flags &= ~param;
}

static void opt_remount(struct ncp_mount_info* info, unsigned int param) {
	fprintf(stderr, _("Remounting not supported, sorry\n"));
	exit(1);
}

static void opt_set_mount_soft(struct ncp_mount_info* info, unsigned int param) {
	info->mdata.flags.mount_soft = param;
}

static void opt_set_mount_timeout(struct ncp_mount_info* info, unsigned int param) {
	if ((param < 1) || (param > 900)) {
		fprintf(stderr, _("Timeout must be between 1 and 900 secs inclusive\n"));
		exit(1);
	}
	info->mdata.time_out = param;
}

static void opt_set_mount_retry(struct ncp_mount_info* info, unsigned int param) {
	if ((param < 1) || (param > 0x10000)) {
		fprintf(stderr, _("Retry count must be between 1 and 65536 inclusive\n"));
		exit(1);
	}
	info->mdata.retry_count = param;
}

static void opt_set_mount_strong(struct ncp_mount_info* info, unsigned int param) {
	info->mdata.flags.mount_strong = param;
}

static void opt_set_mount_extras(struct ncp_mount_info* info, unsigned int param) {
	info->mdata.flags.mount_extras = param;
}

static void opt_set_mount_symlinks(struct ncp_mount_info* info, unsigned int param) {
	info->mdata.flags.mount_symlinks = param;
}

static void opt_set_mount_no_nfs(struct ncp_mount_info* info, unsigned int param) {
	info->mdata.flags.mount_no_nfs = param;
}

static void opt_set_mount_no_os2(struct ncp_mount_info* info, unsigned int param) {
	info->mdata.flags.mount_no_os2 = param;
}

static void opt_set_mounted_uid(struct ncp_mount_info* info, const char* param) {
	if (isdigit(param[0])) {
		char* end;

		info->mdata.mounted_uid = strtoul(param, &end, 10);
		if (end && *end && !isspace(*end)) {
			fprintf(stderr, _("owner parameter `%s' is not valid decimal number\n"), param);
			exit(1);
		}
	} else {
		struct passwd *pwd = getpwnam(param);

		if (!pwd) {
			fprintf(stderr, _("User `%s' does not exist on this machine\n"), param);
			exit(1);
		}
		info->mdata.mounted_uid = pwd->pw_uid;
	}
}

static void opt_set_uid(struct ncp_mount_info* info, const char* param) {
	if (isdigit(param[0])) {
		char* end;

		info->mdata.uid = strtoul(param, &end, 10);
		if (end && *end && !isspace(*end)) {
			fprintf(stderr, _("uid parameter `%s' is not valid decimal number\n"), param);
			exit(1);
		}
	} else {
		struct passwd *pwd = getpwnam(param);

		if (!pwd) {
			fprintf(stderr, _("User `%s' does not exist on this machine\n"), param);
			exit(1);
		}
		info->mdata.uid = pwd->pw_uid;
	}
}

static void opt_set_gid(struct ncp_mount_info* info, const char* param) {
	if (isdigit(param[0])) {
		char* end;

		info->mdata.gid = strtoul(param, &end, 10);
		if (end && *end && !isspace(*end)) {
			fprintf(stderr, _("gid parameter `%s' is not valid decimal number\n"), param);
			exit(1);
		}
	} else {
		struct group* grp = getgrnam(param);

		if (!grp) {
			fprintf(stderr, _("Group `%s' does not exist on this machine\n"), param);
			exit(1);
		}
		info->mdata.gid = grp->gr_gid;
	}
}

static void opt_set_passwd(struct ncp_mount_info* info, const char* param) {
	if (!param) param = "";
	
	if (strlen(param) >= sizeof(((const struct ncp_conn_spec*)param)->password)) {
		fprintf(stderr, _("Specified password is too long\n"));
		exit(1);
	}
	info->password = param;
}

static void opt_set_user(struct ncp_mount_info* info, const char* param) {
	info->user = param;
}

static void opt_set_server(struct ncp_mount_info* info, const char* param) {
	if (strlen(param) >= sizeof(((const struct ncp_conn_spec*)param)->server)) {
		fprintf(stderr, _("Specified server name `%s' is too long\n"), param);
		exit(1);
	}
	info->server = param;
}

static void opt_set_bindery(struct ncp_mount_info* info, unsigned int param) {
	info->force_bindery_login = param;
}

static void opt_set_server_name(struct ncp_mount_info* info, const char* param) {
	info->server_name = param;
}

static void opt_set_volume(struct ncp_mount_info* info, const char* param) {
	info->pathlen = ncp_path_to_NW_format(param, info->NWpath, sizeof(info->NWpath));
	info->remote_path = param;
	if (info->pathlen < 0) {
		fprintf(stderr, _("Volume path `%s' is invalid: `%s'\n"), param, strerror(-info->pathlen));
		exit(1);
	};
	if (info->pathlen == 1) {
		info->mdata.mounted_vol = "";
		info->remote_path = "/";
	} else if (info->NWpath[0] != 1) {
		info->mdata.mounted_vol = "dummy";
	} else if (strlen(param) > NCP_VOLNAME_LEN) {
		fprintf(stderr, _("Volume name `%s' is too long\n"), param);
		exit(1);
	} else {
		info->mdata.mounted_vol=param;
	}
}

static void opt_set_sig_level(struct ncp_mount_info* info, unsigned int param) {
	if ((param < 0) || (param > 3)) {
		fprintf(stderr, _("NCP signature level must be number between 0 and 3. You specified %u\n"), param);
		exit(1);
	}
	info->sig_level = param;
}

static void opt_set_ttl(struct ncp_mount_info* info, unsigned int param) {
	if (param > 20000) {
		fprintf(stderr, _("NCP cache time to live must be less than 20000 ms. You specified %u\n"), param);
		exit(1);
	}
	info->dentry_ttl = param;
}

static void opt_set_allow_multiple(struct ncp_mount_info* info, unsigned int param) {
	info->allow_multiple_connections = param;
}

static void opt_set_file_mode(struct ncp_mount_info* info, const char* param) {
	char* end;

	info->mdata.file_mode = strtoul(param, &end, 8);
	if (end && *end && !isspace(*end)) {
		fprintf(stderr, _("File mode `%s' is not valid octal number\n"), param);
		exit(1);
	}
}

static void opt_set_dir_mode(struct ncp_mount_info* info, const char* param) {
	char* end;

	info->mdata.dir_mode = strtoul(param, &end, 8);
	if (end && *end && !isspace(*end)) {
		fprintf(stderr, _("Directory mode `%s' is not valid octal number\n"), param);
		exit(1);
	}
}

static void opt_set_passwd_file(struct ncp_mount_info* info, const char* param) {
	info->passwd_file = param;
}

static void opt_set_iocharset(struct ncp_mount_info* info, const char* param) {
	if (strlen(param) >= sizeof(info->nls_cs.iocharset)) {
		fprintf(stderr, _("I/O charset name `%s' is too long\n"), param);
		exit(128);
	}
	strcpy(info->nls_cs.iocharset, param);
}

static void opt_set_codepage(struct ncp_mount_info* info, const char* param) {
	if (strlen(param) >= sizeof(info->nls_cs.codepage)) {
		fprintf(stderr, _("Codepage name `%s' is too long\n"), param);
		exit(128);
	}
	strcpy(info->nls_cs.codepage, param);
}

static void opt_set_upcase_passwd(struct ncp_mount_info* info, unsigned int param) {
	info->upcase_password = param;
}

static void opt_not_implemented(struct ncp_mount_info* info, unsigned int param) {
	/* noop */
}

static struct optinfo opts[] = {
	{0,   "ro",		FN_NOPARAM,	opt_set_flags,		MS_RDONLY},
	{0,   "rw",		FN_NOPARAM,	opt_clear_flags,	MS_RDONLY},
	{0,   "nosuid",		FN_NOPARAM,	opt_set_flags,		MS_NOSUID},
	{0,   "suid",		FN_NOPARAM,	opt_clear_flags,	MS_NOSUID},
	{0,   "nodev",		FN_NOPARAM,	opt_set_flags,		MS_NODEV},
	{0,   "dev",		FN_NOPARAM,	opt_clear_flags,	MS_NODEV},
	{0,   "noexec",		FN_NOPARAM,	opt_set_flags,		MS_NOEXEC},
	{0,   "exec",		FN_NOPARAM,	opt_clear_flags,	MS_NOEXEC},
	{0,   "sync",		FN_NOPARAM,	opt_set_flags,		MS_SYNCHRONOUS},
	{0,   "async",		FN_NOPARAM,	opt_clear_flags,	MS_SYNCHRONOUS},
	{0,   "mand",		FN_NOPARAM,	opt_set_flags,		MS_MANDLOCK},
	{0,   "nomand",		FN_NOPARAM,	opt_clear_flags,	MS_MANDLOCK},
	{0,   "noatime",	FN_NOPARAM,	opt_set_flags,		MS_NOATIME},
	{0,   "atime",		FN_NOPARAM,	opt_clear_flags,	MS_NOATIME},
	{0,   "nodiratime",	FN_NOPARAM,	opt_not_implemented,	MS_NODIRATIME},	/* does not exist in 2.0 kernel */
	{0,   "remount",	FN_NOPARAM,	opt_remount,		0},
	{0,   "soft",		FN_NOPARAM,	opt_set_mount_soft,	1},
	{0,   "hard",		FN_NOPARAM,	opt_set_mount_soft,	0},
	{'t', "timeo",		FN_INT,		opt_set_mount_timeout,	60},
	{'t', "timeout",	FN_INT,		opt_set_mount_timeout,	60},
	{'r', "retry",		FN_INT,		opt_set_mount_retry,	5},
	{'u', "uid",		FN_STRING,	opt_set_uid,		0},
	{'g', "gid",		FN_STRING,	opt_set_gid,		0},
	{'c', "owner",		FN_STRING,	opt_set_mounted_uid,	0},
	{0,   "nopasswd",	FN_NOPARAM,	opt_set_passwd,		0}, /* 'n' without mount.ncp */
	{'P', "passwd",		FN_STRING,	opt_set_passwd,		0},
	{0,   "passwdfile",	FN_STRING,	opt_set_passwd_file,	0},
	{'U', "user",		FN_STRING,	opt_set_user,		0},
	{'S', "server",		FN_STRING,	opt_set_server,		0},
	{'s', "strong",		FN_NOPARAM,	opt_set_mount_strong,	1},
	{0,   "nostrong",	FN_NOPARAM,	opt_set_mount_strong,	0},
	{0,   "symlinks",	FN_NOPARAM,	opt_set_mount_symlinks,	1},
	{0,   "nosymlinks",	FN_NOPARAM,	opt_set_mount_symlinks,	0},
	{0,   "extras",		FN_NOPARAM,	opt_set_mount_extras,	1},
	{0,   "noextras",	FN_NOPARAM,	opt_set_mount_extras,	0},
	{0,   "nonfs",		FN_NOPARAM,	opt_set_mount_no_nfs,	1},
	{0,   "noos2",		FN_NOPARAM,	opt_set_mount_no_os2,	1},
	{0,   "nolong",		FN_NOPARAM,	opt_set_mount_no_os2,	1},
	{'b', "bindery",	FN_NOPARAM,	opt_set_bindery,	1},
	{'A', "ipserver",	FN_STRING,	opt_set_server_name,	0},
	{'V', "volume",		FN_STRING,	opt_set_volume,		0},
	{'i', "signature",	FN_INT,		opt_set_sig_level,	0},
	{'m', "multiple",	FN_NOPARAM,	opt_set_allow_multiple,	1},
	{'f', "mode",		FN_STRING,	opt_set_file_mode,	0},
	{'f', "filemode",	FN_STRING,	opt_set_file_mode,	0},
	{'d', "dirmode",	FN_STRING,	opt_set_dir_mode,	0},
	{'y', "iocharset",	FN_STRING,	opt_set_iocharset,	0},
	{'p', "codepage",	FN_STRING,	opt_set_codepage,	0},
	{'C', "noupcasepasswd",	FN_NOPARAM,	opt_set_upcase_passwd,	0},
	{0,   "ttl",            FN_INT,		opt_set_ttl,		0},
	{0,   NULL,		0,		NULL,			0}
};

static void proc_option(struct ncp_mount_info* info, const char* opt, const char* param) {
	struct optinfo* optr;

	for (optr = opts; optr->option; optr++) {
		if (!strcmp(optr->option, opt) || ((opt[0] == optr->shortopt) && (opt[1] == 0))) {
			if (param) {
				if (optr->flags & FN_STRING)
					((void (*)(struct ncp_mount_info*, const char*))(optr->proc))(info, param);
				else if (optr->flags & FN_INT) {
					unsigned int i;
					char* end;

					i = strtoul(param, &end, 10);
					if (end && *end && !isspace(*end)) {
						fprintf(stderr, _("Value `%s' for option `%s' is not a number\n"), param, opt);
						exit(1);
					}
					((void (*)(struct ncp_mount_info*, unsigned int))(optr->proc))(info, i);
				} else {
					fprintf(stderr, _("Ignoring unneeded value for option `%s'\n"), opt);
					((void (*)(struct ncp_mount_info*, unsigned int))(optr->proc))(info, optr->param);
				}
			} else {
				if (!(optr->flags & FN_NOPARAM)) {
					fprintf(stderr, _("Required parameter for option `%s' missing\n"), opt);
					exit(1);
				}
				((void (*)(struct ncp_mount_info*, unsigned int))(optr->proc))(info, optr->param);
			}
			return;
		}
	}
	fprintf(stderr, _("Unknown option `%s', ignoring it\n"), opt);
}

struct smntflags {
	unsigned int	flag;
	const char*	name;
		} mntflags[] = {
       {MS_NOEXEC,	"noexec"},
       {MS_NOSUID,	"nosuid"},
       {MS_NODEV,	"nodev"},
       {MS_SYNCHRONOUS,	"sync"},
       {MS_MANDLOCK,	"mand"},
       {MS_NOATIME,	"noatime"},
       {MS_NODIRATIME,	"nodiratime"},
       {0,		NULL}};

int
main(int argc, char *argv[])
{
	struct ncp_mount_info info;
	struct stat st;
	char mount_name[256];

	int fd, result;
	struct sockaddr_ipx addr;

	long err;

	int um;

	struct mntent ment;
	FILE *mtab;

	struct ncp_conn_spec *spec;

	struct ncp_conn *conn;

	int opt;

#ifdef CONFIG_NATIVE_IP
	struct sockaddr_in server_in;
#endif

	char *tmp_mount;

	int mount_protocol_version = -1;

	int mount_ncp = 0;
	int opt_n = 0;
	int opt_v = 0;

	progname = argv[0];

	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);
	
	memset(&info, 0, sizeof(info));
	memset(&spec, 0, sizeof(spec));

	info.flags = MS_MGC_VAL;
	info.sig_level = -1;
	info.remote_path = "/";
	info.dentry_ttl = 0;
	info.NWpath[0] = 0;
	info.pathlen = 1;

	info.mdata.mounted_uid = getuid();

	if (geteuid() != 0)
	{
		fprintf(stderr, _("%s must be installed suid root\n"), progname);
		exit(1);
	}
	info.mdata.uid = getuid();
	info.mdata.gid = getgid();
	um = umask(0);
	umask(um);
	info.mdata.file_mode = (S_IRWXU | S_IRWXG | S_IRWXO) & ~um;
	info.mdata.dir_mode = 0;
	info.mdata.flags.mount_soft = 1;
	info.mdata.flags.mount_intr = 0;
	info.mdata.flags.mount_strong = 0;
	info.mdata.flags.mount_no_os2 = 0;
	info.mdata.flags.mount_no_nfs = 0;
	info.mdata.flags.mount_extras = 0;
	info.mdata.flags.mount_symlinks = 0;
	info.mdata.time_out = 60;
	info.mdata.retry_count = 5;
	info.mdata.mounted_vol = "";

	info.upcase_password = 1;

	while ((opt = getopt(argc, argv, "CS:U:c:u:g:f:d:P:nh?vV:t:r:o:"
		"sN:y:p:bi:mA:"
#ifdef MOUNT2
		"2"
#endif
#ifdef MOUNT3
		"3"
#endif
	        )) != EOF)
	{
		char tmpopt[2];

		switch (opt)
		{
		case 'C':
		case 's':
		case 'b':
		case 'm':
			optarg = NULL;
		case 'S':
		case 'U':
		case 'c':
		case 'u':
		case 'g':
		case 'f':
		case 'd':
		case 'V':
		case 't':
		case 'r':
		case 'i':
		case 'A':
		case 'y':
		case 'p':
			tmpopt[0] = opt;
			tmpopt[1] = 0;
			proc_option(&info, tmpopt, optarg);
			break;
		case 'P':
			if (!mount_ncp) opt_n = 0;	/* clear -n (nopasswd) */
			tmpopt[0] = opt;
			tmpopt[1] = 0;
			proc_option(&info, tmpopt, optarg);
			break;
		case 'n':
			opt_n = 1;	/* no passwd or no update /etc/fstab */
			break;
		case 'h':
		case '?':
			help();
			exit(1);
		case 'v':
			opt_v = 1;	/* version or verbose */
			break;	
#ifdef MOUNT2
		case '2':
			mount_protocol_version = 2;
			break;
#endif /* MOUNT2 */
#ifdef MOUNT3
		case '3':
			mount_protocol_version = 3;
			break;
#endif /* MOUNT3 */
		case 'N':
			{
				char *inp;
				char *out;
				char *dup;

				dup = inp = strdup(optarg);

				if (!dup) {
					fprintf(stderr, _("%s: out of memory\n"), progname);
					exit(1);
				}
				while ((out = strtok(inp, ",;:"))!=NULL) {
					inp=NULL;
					if (!strcasecmp(out, "OS2"))
						info.mdata.flags.mount_no_os2=1;
					else if (!strcasecmp(out, "LONG"))
						info.mdata.flags.mount_no_os2=1;
					else if (!strcasecmp(out, "NFS"))
						info.mdata.flags.mount_no_nfs=1;
					else {
						fprintf(stderr, _("Unknown namespace \"%s\"\n"), out);
						return 128;
					}
				}
				free(dup);
			};
			break;
		case 'o':
			{
				char* dup;
				char* arg;

				arg = dup = strdup(optarg);	/* loose memory... but who cares */
				if (!dup) {
					fprintf(stderr, _("%s: out of memory\n"), progname);
					exit(1);
				}
				mount_ncp = 1;
				do {
					char* end;
					char* parm;

					arg += strspn(arg, " \t");
					if (!*arg) break;
					end = strchr(arg, ',');
					if (end) *end++ = 0;

					parm = strchr(arg, '=');
					if (parm) *parm++ = 0;

					proc_option(&info, arg, parm);
					arg = end;
				} while (arg);
			}
			break;
		default:
			usage();
			return -1;
		}
	}
	
	if (opt_n && !mount_ncp) {
		opt_n = 0;
		opt_set_passwd(&info, "");
	}
	if (opt_v && !mount_ncp) {
		fprintf(stderr, _("ncpfs version %s\n"), NCPFS_VERSION);
		exit(1);
	}
	if (optind < argc - 1) {
		char* s;
		char* u;

		s = strdup(argv[optind++]);
		if (s) {
			u = strchr(s, '/');
			if (u) *u++ = 0;
		} else
			u = NULL;
		if (!info.server) info.server = s;
		if (!info.user) info.user = u;
	}

	if (mount_protocol_version < 0) {
		mount_protocol_version = getmountver();
	}
	if (optind != argc - 1)
	{
		usage();
		return -1;
	}
	realpath(argv[optind], mount_point);

	if (stat(mount_point, &st) == -1)
	{
		fprintf(stderr, _("could not find mount point %s: %s\n"),
			mount_point, strerror(errno));
		exit(1);
	}
	if (mount_ok(&st) != 0)
	{
		fprintf(stderr, _("cannot to mount on %s: %s\n"),
			mount_point, strerror(errno));
		exit(1);
	}

	get_passwd(info.passwd_file, info.server, info.user, &info.password);
	
	if ((spec = ncp_find_conn_spec2(info.server, info.user, info.password, 1, info.mdata.uid, info.allow_multiple_connections, &err))
	    == NULL)
	{
		com_err(progname, err, _("in find_conn_spec"));
		exit(1);
	}
	if (info.upcase_password != 0)
	{
		str_upper(spec->password);
	}

#ifdef MOUNT2
	if (mount_protocol_version < 3) {
		/* Check if the ncpfs filesystem is in the kernel.  If not, attempt
		 * to load the ncpfs module */
		if (load_ncpfs() != 0)
		{
			fprintf(stderr, _("Error: Unable to load ncpfs, exiting...\n"));
			exit(1);
		}
	}
#endif /* MOUNT2 */

	info.mdata.server_name = spec->server;

	if (info.mdata.dir_mode == 0)
	{
		info.mdata.dir_mode = info.mdata.file_mode;
		if ((info.mdata.dir_mode & S_IRUSR) != 0)
			info.mdata.dir_mode |= S_IXUSR;
		if ((info.mdata.dir_mode & S_IRGRP) != 0)
			info.mdata.dir_mode |= S_IXGRP;
		if ((info.mdata.dir_mode & S_IROTH) != 0)
			info.mdata.dir_mode |= S_IXOTH;
	}
#ifdef CONFIG_NATIVE_IP
	if (info.server_name) {
		struct hostent* h;

		h = gethostbyname(info.server_name);
		if (!h) {
			fprintf(stderr, _("Get host address `%s': "), info.server_name);
			herror(NULL);
			return 1;
		}
		if (h->h_addrtype != AF_INET) {
			fprintf(stderr, _("Get host address `%s': Not AF_INET\n"), info.server_name);
			return 1;
		}
		if (h->h_length != 4) {
			fprintf(stderr, _("Get host address `%s': Bad address length\n"), info.server_name);
			return 1;
		}
		server_in.sin_family = h->h_addrtype;
		memcpy(&server_in.sin_addr.s_addr, h->h_addr, 4);
		server_in.sin_port = htons(0x020C);
		memcpy(&info.mdata.serv_addr, &server_in, sizeof(server_in));
	} else 
#endif	/* CONFIG_NATIVE_IP */
	{
		if ((!info.allow_multiple_connections)&&
		    ((tmp_mount = ncp_find_permanent(spec)) != NULL))
		{
			fprintf(stderr,
				_("You already have mounted server %s\nas user "
				"%s\non mount point %s\n"), spec->server, spec->user,
				tmp_mount);
			exit(1);
		}
		if ((err = ncp_find_fileserver(spec->server, (struct sockaddr*)&info.mdata.serv_addr, sizeof(info.mdata.serv_addr))) != 0)
		{
			com_err("ncpmount", err, _("when trying to find %s"),
				spec->server);
			exit(1);
		}
	}

#ifdef CONFIG_NATIVE_IP
	if (info.mdata.serv_addr.sipx_family == AF_INET) {
		info.mdata.ncp_fd = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
		if (info.mdata.ncp_fd == -1) {
			com_err("ncpmount", err, _("opening ncp_socket"));
			exit(1);
		}
		info.mdata.wdog_fd = -1;
		info.mdata.message_fd = -1;
		memset(&addr, 0, sizeof(addr));
		addr.sipx_family = AF_INET;
	} else 
#endif	/* CONFIG_NATIVE_IP */
#ifdef CONFIG_NATIVE_IPX
	if (info.mdata.serv_addr.sipx_family == AF_IPX) {
		info.mdata.ncp_fd = socket(AF_IPX, SOCK_DGRAM, PF_IPX);
		if (info.mdata.ncp_fd == -1)
		{
			com_err("ncpmount", err, _("opening ncp_socket"));
			exit(1);
		}
		info.mdata.wdog_fd = socket(AF_IPX, SOCK_DGRAM, PF_IPX);
		if (info.mdata.wdog_fd == -1)
		{
			fprintf(stderr, _("could not open wdog socket: %s\n"),
				strerror(errno));
			exit(1);
		}
		memset(&addr, 0, sizeof(addr));
		addr.sipx_family = AF_IPX;
		addr.sipx_type = NCP_PTYPE;
	} else
#endif	/* CONFIG_NATIVE_IPX */
	{
		fprintf(stderr, _("\nNo transport available\n"));
		exit(2);
	}
	if (bind(info.mdata.ncp_fd, (struct sockaddr *) &addr, sizeof(addr)) == -1)
	{
		fprintf(stderr, _("\nbind: %s\n"),
			strerror(errno));
		fprintf(stderr,
			_("\nMaybe you want to use \n"
			  "ipx_configure --auto_interface=on --auto_primary=on\n"
			  "and try again after waiting a minute.\n\n"));
		exit(1);
	}

#ifdef CONFIG_NATIVE_IPX
	if (info.mdata.serv_addr.sipx_family == AF_IPX) {
		socklen_t addrlen;

		addrlen = sizeof(addr);

		if (getsockname(info.mdata.ncp_fd, (struct sockaddr *) &addr, &addrlen) == -1)
		{
			perror(_("getsockname ncp socket"));
			close(info.mdata.ncp_fd);
			close(info.mdata.wdog_fd);
			exit(1);
		}
		addr.sipx_port = htons(ntohs(addr.sipx_port) + 1);
	
		if (bind(info.mdata.wdog_fd, (struct sockaddr *) &addr, sizeof(addr)) == -1)
		{
			fprintf(stderr, _("bind(wdog_sock, ): %s\n"),
				strerror(errno));
			exit(1);
		}

		info.mdata.message_fd = socket(AF_IPX, SOCK_DGRAM, PF_IPX);
		if (info.mdata.message_fd == -1)
		{
			fprintf(stderr, _("could not open message socket: %s\n"),
				strerror(errno));
			exit(1);
		}
		addr.sipx_port = htons(ntohs(addr.sipx_port) + 1);
	
		if (bind(info.mdata.message_fd, (struct sockaddr *) &addr, sizeof(addr)) == -1)
		{
			fprintf(stderr, _("bind(message_sock, ): %s\n"),
				strerror(errno));
			exit(1);
		}
	}
#endif	/* CONFIG_NATIVE_IPX */

	info.mdata.mount_point = mount_point;

	strcpy(mount_name, spec->server);
	strcat(mount_name, "/");
	strcat(mount_name, spec->user);

	switch (mount_protocol_version) {
#ifdef MOUNT2
		case 2:
			result = ncp_mount_v2(mount_name, info.flags, &info.mdata);
			break;
#endif /* MOUNT2 */
#ifdef MOUNT3
		case 3:
			result = ncp_mount_v3(mount_name, info.flags, &info.mdata, argc, argv);
			break;
#endif /* MOUNT3 */
		default:
			fprintf(stderr, _("Unsupported mount protocol version %d\n"), mount_protocol_version);
			exit(2);
	}
	if (result < 0)
	{
		com_err("ncpmount", errno, _("in mount(2)"));
		exit(1);
	}
	if ((err = ncp_open_mount(mount_point, &conn)) != 0)
	{
		umount(mount_point);
		com_err("ncpmount", err, _("attempt to open mount point"));
		exit(1);
	}

	if ((info.nls_cs.codepage[0] != 0) || (info.nls_cs.iocharset[0] != 0)) {
		int i;
		
		i = ioctl(ncp_get_fid(conn), NCP_IOC_SETCHARSETS,  &info.nls_cs);
		if (i && (errno == EINVAL)) {
			struct ncp_nls_ioctl_old old_nls;
			char* p = info.nls_cs.codepage;
			
			/* skip `cp' */
			while (*p && !isdigit(*p)) p++;
			old_nls.codepage = strtoul(p, NULL, 0);
			strcpy(old_nls.iocharset, info.nls_cs.iocharset);
			i = ioctl(ncp_get_fid(conn), NCP_IOC_SETCHARSETS_OLD, &old_nls);
		}
		if (i) {
			if (errno == EINVAL) {
				fprintf(stderr, _("Your kernel does not support character mapping. You should upgrade to latest version.\n"));
			} else {
				perror(_("Warning: Unable to load NLS charsets"));
			}
		}
	}
	if (info.dentry_ttl) {
		int err = ncp_set_dentry_ttl(conn, info.dentry_ttl);
		if (err == EINVAL) {
			fprintf(stderr, _("Your kernel does not support filename caching. You should upgrade to latest kernel version.\n"));
		} else if (err != 0) {
			fprintf(stderr, _("Warning: Cannot enable filename caching: %s\n"), strnwerror(err));
		}
	}
	
	if (info.sig_level >= 0) {
#ifdef SIGNATURES
		int err = 0;
		size_t ln;
		u_int32_t flags;
		
		err = NWCCGetConnInfo(conn, NWCC_INFO_MAX_PACKET_SIZE, sizeof(ln), &ln);
		if (!err)
			err = ncp_renegotiate_connparam(conn, ln, (info.sig_level > 1)? 2:0);
		if (!err)
			err = NWCCGetConnInfo(conn, NWCC_INFO_SECURITY, sizeof(flags), &flags);
#if 0
		if (flags & NWCC_SECUR_LEVEL_SIGN_HEADERS) {
			if (info.sig_level < 2) {
				err = ncp_renegotiate_connparam(conn, ln, 0);
			}
		} else {
			if (info.sig_level >= 2) {
				err = ncp_renegotiate_connparam(conn, ln, 2);
			}
		}
#endif /* 0 */
		if (err || ((info.sig_level == 0) && (flags & NWCC_SECUR_LEVEL_SIGN_HEADERS))
			|| ((info.sig_level == 3) && !(flags & NWCC_SECUR_LEVEL_SIGN_HEADERS))) {
			fprintf(stderr, _("Unable to negotiate requested security level\n"));
			umount(mount_point);
			exit(1);
		}
#else /* SIGNATURES */
		fprintf(stderr, _("Packet signing is required, but is not supported by this ncpmount\n"));
		umount(mount_point);
		exit(1);
#endif /* SIGNATURES */
	}
#ifdef NDS_SUPPORT
	if ((!info.force_bindery_login) && (!nds_get_tree_name(conn, NULL, 0)))
	{
		if ((err = nds_login_auth(conn, spec->user, spec->password)))
		{
			if (err != NWE_PASSWORD_EXPIRED) {
				com_err("ncpmount", err, _("in nds login"));
				fprintf(stderr, _("Login denied.\n"));
				ncp_close(conn);
				umount(mount_point);
				exit(1);
			}
	    		fprintf(stderr, _("Your password has expired\n"));
		}
	}
	else
	{
#endif	
	if ((err = ncp_login_user(conn, spec->user, spec->password)) != 0)
	{
		struct nw_property p;
		struct ncp_prop_login_control *l
		= (struct ncp_prop_login_control *) &p;

		if (err != NWE_PASSWORD_EXPIRED)
		{
			com_err("ncpmount", err, _("in login"));
			fprintf(stderr, _("Login denied\n"));
			ncp_close(conn);
			umount(mount_point);
			exit(1);
		}
		fprintf(stderr, _("Your password has expired\n"));

		if ((err = ncp_read_property_value(conn, NCP_BINDERY_USER,
						   spec->user, 1,
						   "LOGIN_CONTROL", &p)) == 0)
		{
			fprintf(stderr, _("You have %d login attempts left\n"),
				l->GraceLogins);
		}
	}
#ifdef NDS_SUPPORT
	}
#endif
	if ((err = ncp_mount_specific(conn, NW_NS_DOS, info.NWpath, info.pathlen)) != 0) 
	{
		fprintf(stderr, _("Cannot access path \"%s\": %s\n"), info.remote_path, strerror(-err));
		ncp_close(conn);
		umount(mount_point);
		exit(1);
	}
	ncp_close(conn);

	if (!opt_n) {
		struct smntflags* sf;
		char mnt_opts[80];
		char* p;

		ment.mnt_fsname = mount_name;
		ment.mnt_dir = mount_point;
		ment.mnt_type = "ncpfs";
		ment.mnt_opts = mnt_opts;
		ment.mnt_freq = 0;
		ment.mnt_passno = 0;

		p = mnt_opts;
		*p++ = 'r';
		*p++ = (info.flags & MS_RDONLY)?'o':'w';
		for (sf = mntflags; sf->flag; sf++) {
			if (info.flags & sf->flag) {
				*p++ = ',';
				strcpy(p, sf->name);
				p += strlen(p);
			}
		}
		*p = 0;

		if ((fd = open(MOUNTED "~", O_RDWR | O_CREAT | O_EXCL, 0600)) == -1)
		{
			fprintf(stderr, _("Can't get %s~ lock file\n"), MOUNTED);
			exit(1);
		}
		close(fd);

		if ((mtab = setmntent(MOUNTED, "a+")) == NULL)
		{
			fprintf(stderr, _("Can't open %s\n"), MOUNTED);
			exit(1);
		}
		if (addmntent(mtab, &ment) == 1)
		{
			fprintf(stderr, _("Can't write mount entry\n"));
			exit(1);
		}
		if (fchmod(fileno(mtab), 0644) == -1)
		{
			fprintf(stderr, _("Can't set perms on %s\n"), MOUNTED);
			exit(1);
		}
		endmntent(mtab);

		if (unlink(MOUNTED "~") == -1)
		{
			fprintf(stderr, _("Can't remove %s~\n"), MOUNTED);
			exit(1);
		}
	}
	return 0;
}

static void usage(void)
{
	printf(_("usage: %s [options] mount-point\n"), progname);
	printf(_("Try `%s -h' for more information\n"), progname);
}

static void
help(void)
{
	printf(_("\n"
		 "usage: %s [options] mount-point\n"), progname);
	printf(_("\n"
	       "-S server      Server name to be used\n"
	       "-U username    Username sent to server\n"
	       "-V volume      Volume to mount, for NFS re-export\n"
	       "-u uid         uid the mounted files get\n"
	       "-g gid         gid the mounted files get\n"
	       "-f mode        permission the files get (octal notation)\n"
	       "-d mode        permission the dirs get (octal notation)\n"
	       "-c uid         uid to identify the connection to mount on\n"
	       "               Only makes sense for root\n"
	       "-t time_out    Waiting time (in 1/100s) to wait for\n"
	       "               an answer from the server. Default: 60\n"
	       "-r retry_count Number of retry attempts. Default: 5\n"
	       "-C             Don't convert password to uppercase\n"
	       "-P password    Use this password\n"
	       "-n             Do not use any password\n"
	       "               If neither -P nor -n are given, you are\n"
	       "               asked for a password.\n"
	       "-s             Enable renaming/deletion of read-only files\n"
	       "-h             print this help text\n"
	       "-v             print ncpfs version number\n"
	       "%s%s"
	       "-m             Allow multiple logins to server\n"
	       "-N os2,nfs     Do not use specified namespaces on mounted volume\n"
	       "-y charset     character set used for input and display\n"
	       "-p codepage    codepage used on volume, including letters `cp'\n"
	       "\n"),
#ifdef NDS_SUPPORT
	       _("-b             Force bindery login to NDS servers\n")
#else
	       ""
#endif /* NDS_SUPPORT */
	       ,
#ifdef SIGNATURES
	       _("-i level       Signature level, 0=never, 1=supported, 2=preferred, 3=required\n")
#else
	       ""
#endif /* SIGNATURES */
	       );
}

#ifdef MOUNT3
static void 
msg_received (void)
{
	struct ncp_conn *conn;
	int err;
	
	err = fork();
	if (err > 0)
		return;
	if (err == 0)
		execl("/sbin/nwmsg", "/sbin/nwmsg", mount_point, NULL);
	/* fork or exec failed; get rid of message */
	if (ncp_open_mount(mount_point, &conn) == 0) {
		char message[256];
		
		if (!ncp_get_broadcast_message(conn, message)) {
			/* drop message to wastebasket */
		}
		ncp_close(conn);
	}
	exit(0);
}

/* MSG_DONTWAIT defined and module can run only on 2.1.x kernel */
#if defined(MSG_DONTWAIT) && !defined(MOUNT2)
#define recvfrom_notm(fd,buf,ln,sender,addrlen) recvfrom(fd,buf,ln,MSG_DONTWAIT,sender,addrlen)
#else
int recvfrom_notm(int fd, void* buf, size_t len, struct sockaddr* sender, socklen_t* addrlen) {
	int ret;
	int flg;

	flg = fcntl(fd, F_GETFL);
	if (flg == -1) return -1;
	fcntl(fd, F_SETFL, flg | O_NONBLOCK);
	ret = recvfrom(fd, buf, len, 0, sender, addrlen);
	fcntl(fd, F_SETFL, flg);
	return ret;
}
#endif /* MSG_DONTWAIT && !MOUNT2 */

static void 
process_msg_packet (int msg_fd)
{
	struct sockaddr_ipx sender;
	socklen_t addrlen = sizeof(sender);
	char buf[1024];

	if (recvfrom_notm(msg_fd, buf, sizeof(buf), 
		     (struct sockaddr *) &sender, &addrlen) <= 0) {
		return;
	}
	msg_received();
}

static void 
process_wdog_packet (int wdog_fd)
{
	struct sockaddr_ipx sender;
	socklen_t addrlen = sizeof(sender);
	char buf[2];

	if (recvfrom_notm(wdog_fd, buf, sizeof(buf),
		  (struct sockaddr *) &sender, &addrlen) < (int)sizeof(buf)) {
		return;
	}
	if (buf[1] != '?') {
		return;
	}
	buf[1] = 'Y';
	sendto(wdog_fd, buf, 2, 0, (struct sockaddr *) &sender, addrlen);
}

void reapChld(int dummy) {
	signal(SIGCHLD, reapChld);
	/* reap all exited childrens... */
	while (wait4(-1, NULL, WNOHANG, NULL) > 0) ;
}

static int 
process_connection (const struct ncp_mount_data_independent* mnt)
{
	int i;
	int result;
	int max;
	int wdog_fd = mnt->wdog_fd;
	int msg_fd = mnt->message_fd;

	chdir("/");
	setsid();
	{
		sigset_t newsigset;
		sigset_t oldsigset;
	
		sigemptyset(&newsigset);
		sigaddset(&newsigset, SIGTERM);
		sigaddset(&newsigset, SIGCHLD);
		sigprocmask(SIG_UNBLOCK,&newsigset, &oldsigset);
		signal(SIGCHLD, reapChld);
		signal(SIGTERM, SIG_DFL);
	}
	
#ifdef CONFIG_NATIVE_IP
	if (wdog_fd == -1) {
		unsigned char buf[]={0x3E, 0x3E, 0, 0, 0, 0, 0, 0, 0, 'Y'};
		struct ncp_fs_info fsinfo;
		int fd, err;
		int ncp_fd = mnt->ncp_fd;

		for (i = 0; i < NR_OPEN; i++) {
			if (i == ncp_fd) continue;
			close(i);
		}
		/* we sleep here because of mount(2) must have enough time to run */
		sleep(180);	/* 3 min. */
		fd = open(mnt->mount_point, O_RDONLY);
		if (fd < 0) return 1;	/* Give up */
		fsinfo.version = NCP_GET_FS_INFO_VERSION;
		err = ioctl(fd, NCP_IOC_GET_FS_INFO, &fsinfo);
		close(fd);
		if (err) return 1;	/* Give up */
		buf[3] = buf[8] = fsinfo.connection&0xFF;
		buf[5] = buf[7] = (fsinfo.connection>>8)&0xFF;

		while (1) {
			send(ncp_fd, buf, sizeof(buf), 0);
			sleep(180);	/* 3 min. */
		}
		return 0;
	}
#endif /* CONFIG_NATIVE_IP */

	for (i = 0; i < NR_OPEN; i++) {
		if ((i == wdog_fd) || (i == msg_fd)) {
			continue;
		}
		close(i);
	}

	max = (wdog_fd > msg_fd ? wdog_fd : msg_fd) + 1;

	while (1) {
		fd_set rd;

		FD_ZERO(&rd);
		FD_SET(wdog_fd, &rd);
		FD_SET(msg_fd, &rd);

		if ((result = select(max, &rd, NULL, NULL, NULL)) == -1) {
			exit(0);
		}
		if (FD_ISSET(wdog_fd, &rd)) {
			process_wdog_packet(wdog_fd);
		}
		if (FD_ISSET(msg_fd, &rd)) {
			process_msg_packet(msg_fd);
		}
	}
	return 0;
}
#endif /* MOUNT3 */
