/*
 *  ncplib.h
 *
 *  Copyright (C) 1995, 1996 by Volker Lendecke
 *
 */

#ifndef _NCPLIB_H
#define _NCPLIB_H

#include <ncp/ncp.h>
#include <ncp/ext/socket.h>
#include <sys/param.h>
#include <stdio.h>
#include <time.h>

#include <ncp/ipxlib.h>

typedef u_int8_t byte;
typedef u_int16_t word;
typedef u_int32_t dword;

typedef int32_t NWCCODE;

typedef enum NET_ADDRESS_TYPE {
	NT_IPX = 0,
	NT_UDP = 8,
	NT_TCP = 9
} NET_ADDRESS_TYPE;

#define BVAL(buf,pos) (((const u_int8_t *)(buf))[pos])
#define BWVAL(buf,pos) (((u_int8_t*)(buf))[pos])
#define PVAL(buf,pos) ((unsigned int)BVAL(buf,pos))
#define BSET(buf,pos,val) (BWVAL(buf,pos) = (val))

#ifdef __cplusplus
extern "C" {
#endif

static inline word
WVAL_HL(const void * buf, int pos)
{
	return PVAL(buf, pos) << 8 | PVAL(buf, pos + 1);
}
static inline dword
DVAL_HL(const void * buf, int pos)
{
	return WVAL_HL(buf, pos) << 16 | WVAL_HL(buf, pos + 2);
}
static inline void
WSET_HL(void * buf, int pos, word val)
{
	BSET(buf, pos, val >> 8);
	BSET(buf, pos + 1, val & 0xff);
}
static inline void
DSET_HL(void * buf, int pos, dword val)
{
	WSET_HL(buf, pos, val >> 16);
	WSET_HL(buf, pos + 2, val & 0xffff);
}


/* we know that the 386 can handle misalignment and has the "right" 
   byteorder */
#if defined(__i386__)

static inline word
WVAL_LH(const void * buf, int pos)
{
	return *((const word *) (((const u_int8_t*)buf) + pos));
}
static inline dword
DVAL_LH(const void * buf, int pos)
{
	return *((const dword *) (((const u_int8_t*)buf) + pos));
}
static inline void
WSET_LH(void * buf, int pos, word val)
{
	*((word *) (((u_int8_t*)buf) + pos)) = val;
}
static inline void
DSET_LH(void * buf, int pos, dword val)
{
	*((dword *) (((u_int8_t*)buf) + pos)) = val;
}

#else

static inline word
WVAL_LH(const void * buf, int pos)
{
	return PVAL(buf, pos) | PVAL(buf, pos + 1) << 8;
}
static inline dword
DVAL_LH(const void * buf, int pos)
{
	return WVAL_LH(buf, pos) | WVAL_LH(buf, pos + 2) << 16;
}
static inline void
WSET_LH(void * buf, int pos, word val)
{
	BSET(buf, pos, val & 0xff);
	BSET(buf, pos + 1, val >> 8);
}
static inline void
DSET_LH(void * buf, int pos, dword val)
{
	WSET_LH(buf, pos, val & 0xffff);
	WSET_LH(buf, pos + 2, val >> 16);
}

#endif

void
str_upper(char *name);

enum connect_state
{
	NOT_CONNECTED = 0,
	CONN_PERMANENT,
	CONN_TEMPORARY
};

#define NCP_CONN_INVALID	0
#define NCP_CONN_PERMANENT	1
#define NCP_CONN_TEMPORARY	2

#define NCPFS_MAX_CFG_USERNAME	256

/* This is abstract type now. Use NWCCGetConnInfo instead */
struct ncp_conn;

#ifdef MAKE_NCPLIB
int ncp_get_fid(struct ncp_conn*);
#endif

struct ncp_conn_spec
{
	char server[NCP_BINDERY_NAME_LEN];
	char user[NCPFS_MAX_CFG_USERNAME];
	uid_t uid;
	int login_type;		/* NCP_BINDERY_USER / NCP_BINDERY_PSERVER */
	char password[NCP_BINDERY_NAME_LEN];
};

struct ncp_search_seq
{
	struct nw_search_sequence s;
	int name_space;		/* RYP: namespace is reserved word for new C++ compilers ( gcc 2.7x, egcc 2.9x, borland cbuilder */
};

struct ncp_property_info
{
	u_int8_t property_name[16];
	u_int8_t property_flags;
	u_int8_t property_security;
	u_int32_t search_instance;
	u_int8_t value_available_flag;
	u_int8_t more_properties_flag;
};

/* ncp_initialize is the main entry point for user programs which want
   to connect to a NetWare Server. It looks for -S, -U, -P and -n in
   the argument list, opens the connection and removes the arguments
   from the list. It was designed after the X Windows init
   functions. */
struct ncp_conn *
 ncp_initialize(int *argc, char **argv,
		int login_necessary, long *err);

/* You can login as another object by this procedure. As a first use
   pserver comes to mind. */
struct ncp_conn *
 ncp_initialize_as(int *argc, char **argv,
		   int login_necessary, int login_type, long *err);

/* You can login as another object by this procedure. As a first use
   pserver comes to mind. If required = 0 and none of -S,-U,-P is
   specified, NULL is returned regardless of configuration files */
struct ncp_conn *
 ncp_initialize_2(int *argc, char **argv, int login_necessary, 
		  int login_type, long *err, int required);


/* Open a connection */
struct ncp_conn *
ncp_open(const struct ncp_conn_spec *spec, long *err);

/* Open a connection on an existing fd - it accepts only ncpfs fd now.
   In future, it can accept also connected IPX, IP and NCP socket (I'll see) */
int
ncp_open_fd(int fd, struct ncp_conn** conn);

/* Open a connection on an existing mount point */
int 
ncp_open_mount(const char *mount_point, struct ncp_conn** conn);

/* Find a permanent connection that fits the spec, return NULL if
 * there is none. */
char *
ncp_find_permanent(const struct ncp_conn_spec *spec);

/* Find the address of a file server */
long
ncp_find_fileserver(const char *server_name, struct sockaddr* addr, socklen_t addrlen);

/* Find the address of a server */
long
ncp_find_server(const char **server_name, int type, struct sockaddr* addr, socklen_t addrlen);

/* Detach from a permanent connection or destroy a temporary
   connection */
long
ncp_close(struct ncp_conn *conn);

/* like getmntent, ncp_get_conn_ent scans /etc/mtab for usable
   connections */

struct ncp_conn_ent
{
	char server[NCP_BINDERY_NAME_LEN];
	char* user;
	uid_t uid;
	char mount_point[MAXPATHLEN];
};

struct ncp_conn_ent *
ncp_get_conn_ent(FILE * filep);

#define NWCLIENT (".nwclient")
#define NWC_NOPASSWORD ("-")

/* find an appropriate connection */

struct ncp_conn_spec *
 ncp_find_conn_spec(const char *server, const char *user, const char *password,
		    int login_necessary, uid_t uid, long *err);

struct ncp_conn_spec *
 ncp_find_conn_spec2(const char *server, const char *user, const char *password,
		    int login_necessary, uid_t uid, int allow_multiple_conns, 
		    long *err);

long
 ncp_get_file_server_description_strings(struct ncp_conn *conn,
					 char target[512]);

long
 ncp_get_file_server_time(struct ncp_conn *conn, time_t * target);
long
 ncp_set_file_server_time(struct ncp_conn *conn, time_t * source);

struct ncp_file_server_info
{
	u_int8_t ServerName[48] __attribute__((packed));
	u_int8_t FileServiceVersion __attribute__((packed));
	u_int8_t FileServiceSubVersion __attribute__((packed));
	u_int16_t MaximumServiceConnections __attribute__((packed));
	u_int16_t ConnectionsInUse __attribute__((packed));
	u_int16_t NumberMountedVolumes __attribute__((packed));
	u_int8_t Revision __attribute__((packed));
	u_int8_t SFTLevel __attribute__((packed));
	u_int8_t TTSLevel __attribute__((packed));
	u_int16_t MaxConnectionsEverUsed __attribute__((packed));
	u_int8_t AccountVersion __attribute__((packed));
	u_int8_t VAPVersion __attribute__((packed));
	u_int8_t QueueVersion __attribute__((packed));
	u_int8_t PrintVersion __attribute__((packed));
	u_int8_t VirtualConsoleVersion __attribute__((packed));
	u_int8_t RestrictionLevel __attribute__((packed));
	u_int8_t InternetBridge __attribute__((packed));
	u_int8_t Reserved[60] __attribute__((packed));
};

long
 ncp_get_file_server_information(struct ncp_conn *conn,
				 struct ncp_file_server_info *target);

long
 ncp_get_connlist(struct ncp_conn *conn,
		  u_int16_t object_type, const char *object_name,
		  int *returned_no, u_int8_t conn_numbers[256]);

long
 ncp_get_stations_logged_info(struct ncp_conn *conn,
			      u_int32_t connection,
			      struct ncp_bindery_object *target,
			      time_t * login_time);

long
 ncp_get_internet_address(struct ncp_conn *conn,
			  u_int32_t connection,
			  struct sockaddr *target,
			  u_int8_t * conn_type);

long
 ncp_send_broadcast(struct ncp_conn *conn,
		    u_int8_t no_conn, const u_int8_t * connections,
		    const char *message);

long
 ncp_send_broadcast2(struct ncp_conn *conn,
		     unsigned int conns, const unsigned int* connlist,
		     const char* message);

long
 ncp_get_encryption_key(struct ncp_conn *conn,
			char *target);
long
 ncp_get_bindery_object_id(struct ncp_conn *conn,
			   NWObjectType object_type,
			   const char *object_name,
			   struct ncp_bindery_object *target);
long
 ncp_get_bindery_object_name(struct ncp_conn *conn,
			     NWObjectID object_id,
			     struct ncp_bindery_object *target);
long
 ncp_scan_bindery_object(struct ncp_conn *conn,
			 NWObjectID last_id, NWObjectType object_type,
			 char *search_string,
			 struct ncp_bindery_object *target);
long
 ncp_create_bindery_object(struct ncp_conn *conn,
			   NWObjectType object_type,
			   const char *object_name,
			   u_int8_t object_security,
			   u_int8_t object_status);
long
 ncp_delete_bindery_object(struct ncp_conn *conn,
			   NWObjectType object_type,
			   const char *object_name);

long
 ncp_change_object_security(struct ncp_conn *conn,
			    u_int16_t object_type,
			    const char *object_name,
			    u_int8_t security);

struct ncp_station_addr
{
	u_int32_t NetWork __attribute__((packed));
	u_int8_t Node[6] __attribute__((packed));
	u_int16_t Socket __attribute__((packed));
};

struct ncp_prop_login_control
{
	u_int8_t AccountExpireDate[3] __attribute__((packed));
	u_int8_t Disabled __attribute__((packed));
	u_int8_t PasswordExpireDate[3] __attribute__((packed));
	u_int8_t GraceLogins __attribute__((packed));
	u_int16_t PasswordExpireInterval __attribute__((packed));
	u_int8_t MaxGraceLogins __attribute__((packed));
	u_int8_t MinPasswordLength __attribute__((packed));
	u_int16_t MaxConnections __attribute__((packed));
	u_int8_t ConnectionTimeMask[42] __attribute__((packed));
	u_int8_t LastLogin[6] __attribute__((packed));
	u_int8_t RestrictionMask __attribute__((packed));
	u_int8_t reserved __attribute__((packed));
	u_int32_t MaxDiskUsage __attribute__((packed));
	u_int16_t BadLoginCount __attribute__((packed));
	u_int32_t BadLoginCountDown __attribute__((packed));
	struct ncp_station_addr LastIntruder __attribute__((packed));
};

long
 ncp_read_property_value(struct ncp_conn *conn,
			 NWObjectType object_type, const char *object_name,
			 int segment, const char *prop_name,
			 struct nw_property *target);
long
 ncp_scan_property(struct ncp_conn *conn,
		   NWObjectType object_type, const char *object_name,
		   NWObjectID last_id, const char *search_string,
		   struct ncp_property_info *property_info);
long
 ncp_add_object_to_set(struct ncp_conn *conn,
		       NWObjectType object_type, const char *object_name,
		       const char *property_name,
		       NWObjectType member_type,
		       const char *member_name);
long
 ncp_change_property_security(struct ncp_conn *conn,
			      NWObjectType object_type, const char *object_name,
			      const char *property_name,
			      u_int8_t property_security);
long
 ncp_create_property(struct ncp_conn *conn,
		     NWObjectType object_type, const char *object_name,
		     const char *property_name,
		     u_int8_t property_flags,
		     u_int8_t property_security);
long
 ncp_delete_object_from_set(struct ncp_conn *conn,
			    NWObjectType object_type, const char *object_name,
			    const char *property_name,
			    NWObjectType member_type,
			    const char *member_name);
long
 ncp_delete_property(struct ncp_conn *conn,
		     NWObjectType object_type, const char *object_name,
		     const char *property_name);
long
 ncp_write_property_value(struct ncp_conn *conn,
			  NWObjectType object_type, const char *object_name,
			  const char *property_name,
			  u_int8_t segment,
			  const struct nw_property *property_value);

/* Bit masks for security flag */
#define NCP_SEC_CHECKSUMMING_REQUESTED        (1)
#define NCP_SEC_SIGNATURE_REQUESTED           (2)
#define NCP_SEC_COMPLETE_SIGNATURES_REQUESTED (4)
#define NCP_SEC_ENCRYPTION_REQUESTED          (8)
#define NCP_SEC_LIP_DISABLED                (128)

long
 ncp_get_big_ncp_max_packet_size(struct ncp_conn *conn,
				 u_int16_t proposed_max_size,
				 u_int8_t proposed_security_flag,
				 u_int16_t * accepted_max_size,
				 u_int16_t * echo_socket,
				 u_int8_t * accepted_security_flag);

long
 ncp_login_encrypted(struct ncp_conn *conn,
		     const struct ncp_bindery_object *object,
		     const unsigned char *key,
		     const unsigned char *passwd);

long
 ncp_login_unencrypted(struct ncp_conn *conn,
		       NWObjectType object_type, const char *object_name,
		       const unsigned char *passwd);

long
 ncp_change_login_passwd(struct ncp_conn *conn,
			 const struct ncp_bindery_object *object,
			 const unsigned char *key,
			 const unsigned char *oldpasswd,
			 const unsigned char *newpasswd);

#define NCP_GRACE_PERIOD (0xdf)

#define NCPLIB_ERROR			(0x8700)
/* ~/.nwclient is group/world writeable or readable */
#define NCPLIB_INVALID_MODE		(NCPLIB_ERROR | 0x01)
/* ncp_get_namespace_info_element called with itemid not in nsrim */
#define NCPLIB_INFORMATION_NOT_KNOWN	(NCPLIB_ERROR | 0x02)
/* neither fixed nor variable bit requested */
#define NCPLIB_NSFORMAT_INVALID		(NCPLIB_ERROR | 0x03)

#define NWE_REQUESTER_ERROR		(0x8800)
#define NWE_BUFFER_OVERFLOW		(NWE_REQUESTER_ERROR | 0x0E)
#define NWE_SERVER_NO_CONN		(NWE_REQUESTER_ERROR | 0x0F)
#define NWE_INVALID_NCP_PACKET_LENGTH	(NWE_REQUESTER_ERROR | 0x16)
#define NWE_BUFFER_INVALID_LEN		(NWE_REQUESTER_ERROR | 0x33)	/* buffer underflow */
#define NWE_USER_NO_NAME		(NWE_REQUESTER_ERROR | 0x34)
#define NWE_PARAM_INVALID		(NWE_REQUESTER_ERROR | 0x36)
#define NWE_SERVER_NOT_FOUND		(NWE_REQUESTER_ERROR | 0x47)
#define NWE_SIGNATURE_LEVEL_CONFLICT	(NWE_REQUESTER_ERROR | 0x61)
#define NWE_INVALID_LEVEL		(NWE_REQUESTER_ERROR | 0x6B)
#define NWE_UNSUPPORTED_TRAN_TYPE	(NWE_REQUESTER_ERROR | 0x70)
#define NWE_REQUESTER_FAILURE		(NWE_REQUESTER_ERROR | 0xFF)

#define NWE_SERVER_ERROR		(0x8900)
#define NWE_Q_NO_RIGHTS			(NWE_SERVER_ERROR | 0xD3)
#define NWE_LOGIN_MAX_EXCEEDED		(NWE_SERVER_ERROR | 0xD9)
#define NWE_LOGIN_UNAUTHORIZED_TIME	(NWE_SERVER_ERROR | 0xDA)
#define NWE_LOGIN_UNAUTHORIZED_STATION	(NWE_SERVER_ERROR | 0xDB)
#define NWE_ACCT_DISABLED		(NWE_SERVER_ERROR | 0xDC)
#define NWE_PASSWORD_INVALID		(NWE_SERVER_ERROR | 0xDE)
#define NWE_PASSWORD_EXPIRED		(NWE_SERVER_ERROR | 0xDF)
#define NWE_BIND_MEMBER_ALREADY_EXISTS	(NWE_SERVER_ERROR | 0xE9)
#define NWE_NCP_NOT_SUPPORTED		(NWE_SERVER_ERROR | 0xFB)
#define NWE_SERVER_UNKNOWN		(NWE_SERVER_ERROR | 0xFC)
#define NWE_CONN_NUM_INVALID		(NWE_SERVER_ERROR | 0xFD)
#define NWE_SERVER_FAILURE		(NWE_SERVER_ERROR | 0xFF)

const char* strnwerror(int err);

long
 ncp_get_bindery_object_id(struct ncp_conn *conn,
			   NWObjectType object_type,
			   const char *object_name,
			   struct ncp_bindery_object *target);

long
 ncp_login_user(struct ncp_conn *conn,
		const unsigned char *username,
		const unsigned char *password);

long
 ncp_get_volume_info_with_number(struct ncp_conn *conn, int n,
				 struct ncp_volume_info *target);

long
 ncp_get_volume_number(struct ncp_conn *conn, const char *name,
		       int *target);

long
 ncp_file_search_init(struct ncp_conn *conn,
		      int dir_handle, const char *path,
		      struct ncp_filesearch_info *target);

long
 ncp_file_search_continue(struct ncp_conn *conn,
			  struct ncp_filesearch_info *fsinfo,
			  int attributes, const char *path,
			  struct ncp_file_info *target);

long
 ncp_get_finfo(struct ncp_conn *conn,
	       int dir_handle, const char *path, const char *name,
	       struct ncp_file_info *target);

long
 ncp_open_file(struct ncp_conn *conn,
	       int dir_handle, const char *path,
	       int attr, int access,
	       struct ncp_file_info *target);
long
 ncp_close_file(struct ncp_conn *conn, const char *file_id);

long
 ncp_create_newfile(struct ncp_conn *conn,
		    int dir_handle, const char *path,
		    int attr,
		    struct ncp_file_info *target);

long
 ncp_create_file(struct ncp_conn *conn,
		 int dir_handle, const char *path,
		 int attr,
		 struct ncp_file_info *target);

long
 ncp_erase_file(struct ncp_conn *conn,
		int dir_handle, const char *path,
		int attr);

long
 ncp_rename_file(struct ncp_conn *conn,
		 int old_handle, const char *old_path,
		 int attr,
		 int new_handle, const char *new_path);

long
 ncp_create_directory(struct ncp_conn *conn,
		      int dir_handle, const char *path,
		      int inherit_mask);

long
 ncp_delete_directory(struct ncp_conn *conn,
		      int dir_handle, const char *path);

long
 ncp_rename_directory(struct ncp_conn *conn,
		      int dir_handle,
		      const char *old_path, const char *new_path);

long
 ncp_get_trustee(struct ncp_conn *conn, NWObjectID object_id,
		 u_int8_t vol, char *path,
		 u_int16_t * trustee, u_int16_t * contin);

long
 ncp_add_trustee(struct ncp_conn *conn,
		 int dir_handle, const char *path,
		 NWObjectID object_id, u_int8_t rights);

long
 ncp_delete_trustee(struct ncp_conn *conn,
		    int dir_handle, const char *path,
		    NWObjectID object_id);

long
 ncp_read(struct ncp_conn *conn, const char *file_id,
	  off_t offset, size_t count, char *target);

long
 ncp_write(struct ncp_conn *conn, const char *file_id,
	   off_t offset, size_t count, const char *source);

long
 ncp_copy_file(struct ncp_conn *conn,
	       const char source_file[6],
	       const char target_file[6],
	       u_int32_t source_offset,
	       u_int32_t target_offset,
	       u_int32_t count,
	       u_int32_t * copied_count);

#define SA_NORMAL	(0x0000)
#define SA_HIDDEN	(0x0002)
#define SA_SYSTEM	(0x0004)
#define SA_SUBDIR_ONLY	(0x0010)
#define SA_SUBDIR_FILES	(0x8000)
#define SA_ALL		(SA_SUBDIR_FILES | SA_SYSTEM | SA_HIDDEN)
#define SA_SUBDIR_ALL	(SA_SUBDIR_ONLY | SA_SYSTEM | SA_HIDDEN)
#define SA_FILES_ALL	(SA_NORMAL | SA_SYSTEM | SA_HIDDEN)

#define NCP_DIRSTYLE_HANDLE     0x00
#define NCP_DIRSTYLE_DIRBASE    0x01
#define NCP_DIRSTYLE_NOHANDLE   0xFF

long
 ncp_obtain_file_or_subdir_info(struct ncp_conn *conn,
				u_int8_t source_ns, u_int8_t target_ns,
				u_int16_t search_attribs, u_int32_t rim,
				u_int8_t vol, u_int32_t dirent, 
				const char *path,
				struct nw_info_struct *target);

#define NCP_PERM_READ   (0x001)
#define NCP_PERM_WRITE  (0x002)
#define NCP_PERM_OPEN   (0x004)
#define NCP_PERM_CREATE (0x008)
#define NCP_PERM_DELETE (0x010)
#define NCP_PERM_OWNER  (0x020)
#define NCP_PERM_SEARCH (0x040)
#define NCP_PERM_MODIFY (0x080)
#define NCP_PERM_SUPER  (0x100)

long
 ncp_get_eff_directory_rights(struct ncp_conn *conn,
			      u_int8_t source_ns,
			      u_int8_t target_ns,
			      u_int16_t search_attribs,
			      u_int8_t vol, u_int32_t dirent, const char *path,
			      u_int16_t * my_effective_rights);

long
ncp_do_lookup2(struct ncp_conn *conn,
	       u_int8_t _source_ns,
	       struct nw_info_struct *dir,
	       const char *path,       /* may only be one component */
	       u_int8_t _target_ns,
	       struct nw_info_struct *target);

long
ncp_do_lookup(struct ncp_conn *conn,
	      struct nw_info_struct *dir,
	      const char *path,	/* may only be one component */
	      struct nw_info_struct *target);

long
 ncp_modify_file_or_subdir_dos_info(struct ncp_conn *conn,
				    struct nw_info_struct *file,
				    u_int32_t info_mask,
				    struct nw_modify_dos_info *info);

long
 ncp_del_file_or_subdir(struct ncp_conn *conn,
			struct nw_info_struct *dir, char *name);


long
 ncp_open_create_file_or_subdir(struct ncp_conn *conn,
				struct nw_info_struct *dir, char *name,
				int open_create_mode,
				u_int32_t create_attributes,
				int desired_acc_rights,
				struct nw_file_info *target);

long
 ncp_initialize_search(struct ncp_conn *conn,
		       const struct nw_info_struct *dir,
		       int name_space,
		       struct ncp_search_seq *target);

long
ncp_search_for_file_or_subdir2(struct ncp_conn *conn,
			       int search_attributes,
			       u_int32_t RIM,
			       struct ncp_search_seq *seq,
			       struct nw_info_struct *target);

long
 ncp_search_for_file_or_subdir(struct ncp_conn *conn,
			       struct ncp_search_seq *seq,
			       struct nw_info_struct *target);

long
 ncp_ren_or_mov_file_or_subdir(struct ncp_conn *conn,
			       struct nw_info_struct *old_dir, char *old_name,
			       struct nw_info_struct *new_dir, char *new_name);

long
 ncp_create_queue_job_and_file(struct ncp_conn *conn,
			       NWObjectID queue_id,
			       struct queue_job *job);

long
 ncp_get_queue_length(struct ncp_conn *conn,
                      NWObjectID queue_id,
                      u_int32_t *queue_length);

long 
 ncp_get_queue_job_ids(struct ncp_conn *conn,
                       NWObjectID queue_id,
                       u_int32_t queue_section,
                       u_int32_t *length1,
                       u_int32_t *length2,
                       u_int32_t ids[]);
long 
 ncp_get_queue_job_info(struct ncp_conn *conn,
                        NWObjectID queue_id,
                        u_int32_t job_id,
                        struct nw_queue_job_entry *jobdata);

long
NWRemoveJobFromQueue2(struct ncp_conn* conn, NWObjectID queue_id,
			u_int32_t job_id);
                        
long
 ncp_close_file_and_start_job(struct ncp_conn *conn,
			      NWObjectID queue_id,
			      struct queue_job *job);

long
 ncp_attach_to_queue(struct ncp_conn *conn, NWObjectID queue_id);

long
 ncp_detach_from_queue(struct ncp_conn *conn, NWObjectID queue_id);

long
 ncp_service_queue_job(struct ncp_conn *conn, NWObjectID queue_id, 
 		       u_int16_t job_type, struct queue_job *job);

long
 ncp_finish_servicing_job(struct ncp_conn *conn, NWObjectID queue_id,
			  u_int32_t job_number, u_int32_t charge_info);

long
 ncp_abort_servicing_job(struct ncp_conn *conn, NWObjectID queue_id,
			 u_int32_t job_number);

long
 ncp_get_broadcast_message(struct ncp_conn *conn, char message[256]);

long
 ncp_dealloc_dir_handle(struct ncp_conn *conn, u_int8_t dir_handle);

#define NCP_ALLOC_PERMANENT (0x0000)
#define NCP_ALLOC_TEMPORARY (0x0001)
#define NCP_ALLOC_SPECIAL   (0x0002)

long
 ncp_alloc_short_dir_handle2(struct ncp_conn *conn,
			     u_int8_t _namespace,
			     struct nw_info_struct *dir,
			     u_int16_t alloc_mode,
			     u_int8_t * target);

long
 ncp_alloc_short_dir_handle(struct ncp_conn *conn,
			    struct nw_info_struct *dir,
			    u_int16_t alloc_mode,
			    u_int8_t * target);

long
 ncp_get_effective_dir_rights(struct ncp_conn *conn,
			      struct nw_info_struct *file,
			      u_int16_t * target);

struct ncp_trustee_struct
{
	NWObjectID object_id;
	u_int16_t rights;
};

long
ncp_add_trustee_set(struct ncp_conn *conn,
		     u_int8_t volume_number, u_int32_t dir_entry,
		     u_int16_t rights_mask,
		     int object_count, 
		     const struct ncp_trustee_struct *rights);

struct ncp_deleted_file
{
	int32_t		seq;
	u_int32_t	vol;
	u_int32_t	base;
};

long
ncp_ns_scan_salvageable_file(struct ncp_conn* conn, u_int8_t src_ns,
			     int dirstyle, u_int8_t vol_num, 
			     u_int32_t dir_base,
			     const unsigned char* encpath, int pathlen,
			     struct ncp_deleted_file* finfo,
			     char* name, int maxnamelen);

long
ncp_ns_purge_file(struct ncp_conn* conn, const struct ncp_deleted_file* finfo);

long
ncp_ns_get_full_name(struct ncp_conn* conn, u_int8_t src_ns, u_int8_t dst_ns,
	             int dirstyle, u_int8_t vol_num, u_int32_t dir_base,
		     const unsigned char* encpath, size_t pathlen,
	             char* name, size_t maxnamelen);

int
ncp_get_conn_type(struct ncp_conn* conn);

int
ncp_get_conn_number(struct ncp_conn* conn);

NWCCODE
ncp_get_dentry_ttl(struct ncp_conn* conn, unsigned int* ttl);

NWCCODE
ncp_set_dentry_ttl(struct ncp_conn* conn, unsigned int ttl);

/* What to do with them?! Simply do not use them if you compiled libncp
   without NDS support */
long
ncp_send_nds_frag(struct ncp_conn *conn,
    int ndsverb,
    const char *inbuf, size_t inbuflen,
    char *outbuf, size_t outbufsize, size_t *outbuflen);

long
ncp_send_nds(struct ncp_conn *conn, int fn,
 const char *data_in, size_t data_in_len, 
 char *data_out, size_t data_out_max, size_t *data_out_len);

long
ncp_change_conn_state(struct ncp_conn *conn, int new_state);
/* end of NDS specific... */

struct ncp_conn *
ncp_open_addr(const struct sockaddr *target, long *err);

int
ncp_path_to_NW_format(const char* path, unsigned char* buff, int buffsize);

long
ncp_obtain_file_or_subdir_info2(struct ncp_conn* conn, u_int8_t source_ns,
			u_int8_t target_ns, u_int16_t search_attribs, 
			u_int32_t rim, int dir_style, u_int8_t vol, 
			u_int32_t dirent, 
			const unsigned char* path, int pathlen,
			struct nw_info_struct* target);

typedef unsigned int NWVOL_NUM;
typedef unsigned int NWDIR_HANDLE;
typedef u_int32_t NWDIR_ENTRY;

struct NSI_Name {
	size_t		NameLength;
	char		Name[256];
};
	
struct NSI_Attributes {
	u_int32_t	Attributes;
	u_int16_t	Flags;
};

struct NSI_TotalSize {
	u_int32_t	TotalAllocated;
	size_t		Datastreams;
};

struct NSI_ExtAttrInfo {
	u_int32_t	DataSize;
	u_int32_t	Count;
	u_int32_t	KeySize;
};

struct NSI_Change {
	u_int16_t	Date;
	u_int16_t	Time;
	NWObjectID	ID;
};

struct NSI_Directory {
	NWDIR_ENTRY	dirEntNum;
	NWDIR_ENTRY	DosDirNum;
	NWVOL_NUM	volNumber;
};

struct NSI_DatastreamFATInfo {
	u_int32_t	Number;
	u_int32_t	FATBlockSize;
};

struct NSI_DatastreamSizes {
	size_t	NumberOfDatastreams;
	struct NSI_DatastreamFATInfo ds[0];
};

struct NSI_DatastreamInfo {
	u_int32_t	Number;
	u_int32_t	Size;	/* 64bits? */
};

struct NSI_DatastreamLogicals {
	size_t	NumberOfDatastreams;
	struct NSI_DatastreamInfo ds[0];
};

struct NSI_DOSName {
	size_t	NameLength;	/* 0..14 */
	char	Name[16];	/* align... why not... */
};

struct NSI_MacTimes {
	int32_t CreateTime;
	int32_t BackupTime;
};

typedef u_int64_t ncp_off64_t;

struct nw_info_struct2 {
	ncp_off64_t	SpaceAllocated;
	struct NSI_Attributes Attributes;
	ncp_off64_t	DataSize;
	struct NSI_TotalSize TotalSize;
	struct NSI_ExtAttrInfo ExtAttrInfo;
	struct NSI_Change Archive;
	struct NSI_Change Modify;
	struct NSI_Change Creation;
	struct {
	u_int16_t	Date;
	u_int16_t	Time;	/* NW5.00 */
		}	LastAccess;
	u_int32_t	OwningNamespace;
	struct NSI_Directory Directory;
	u_int16_t	Rights;	/* inherited */
			
	u_int16_t	ReferenceID;
	u_int32_t	NSAttributes;
	/* datastream actual cannot be retrieved this way... */
	/* datastream logical ... dtto ... */
	int32_t		UpdateTime; /* seconds, relative to year 2000... */ /* NW4.11 */
	struct NSI_DOSName DOSName;	/* NW4.11 */
	u_int32_t	FlushTime;	/* NW4.11 */
	u_int32_t	ParentBaseID;	/* NW4.11 */
	u_int8_t	MacFinderInfo[32]; /* NW4.11 */
	u_int32_t	SiblingCount;	/* NW4.11 */
	u_int32_t	EffectiveRights; /* NW4.11 */
	struct NSI_MacTimes MacTimes;	/* NW4.11 */
	u_int32_t	reserved1[8];	/* future expansion... */
	struct NSI_Name	Name;
	u_int8_t	reserved2[512];	/* future expansion... (unicode name?) */
};

struct ncp_dos_info {
	u_int32_t	Attributes;
	struct NSI_Change Creation;
	struct NSI_Change Modify;
	struct NSI_Change Archive;
	struct {
	u_int16_t	Date;
	u_int16_t	reserved;	/* LastAccess time... */
		      } LastAccess;
	struct {
	u_int16_t	Grant;
	u_int16_t	Revoke;
		      } Rights;
	u_int32_t	MaximumSpace;	/* 64bit? It is in 4KB blocks... And NSS does not support this... */
};

struct ncp_namespace_format {
	unsigned int	Version;	/* used only by library */
	struct {
	u_int32_t	fixed;
	u_int32_t	variable;
	u_int32_t	huge;
		      } BitMask;
	struct {
	size_t		fixed;
	size_t		variable;
	size_t		huge;
		      } BitsDefined;
	size_t		FieldsLength[32];
};

NWCCODE
ncp_ns_open_create_entry(struct ncp_conn *conn,
				/* entry info */
				unsigned int ns,
				unsigned int search_attributes,
				int dirstyle,
				unsigned int vol,
				u_int32_t dirent,
				const unsigned char* encpath, size_t pathlen,
				/*  what to do with entry */
				int datastream,
				int open_create_mode,
				u_int32_t create_attributes,
				u_int16_t desired_access_rights,
				/* what to return */
				u_int32_t rim,
				/* returned */
				struct nw_info_struct2* target, size_t sizeoftarget,
				u_int8_t* oc_action,
				u_int8_t* oc_callback,
				u_int8_t* file_handle	/* ?? u_int32_t* or NW_FILE_HANDLE* ?? */
				);

NWCCODE
ncp_ns_obtain_entry_info(struct ncp_conn* conn, 
				/* entry info */
				unsigned int source_ns,
				unsigned int search_attributes, 
				int dirstyle, 
				unsigned int vol, 
				u_int32_t dirent, 
				const unsigned char* encpath, size_t pathlen,
				/* what to do with entry */
				unsigned int target_ns,
				/* what to return */
				u_int32_t rim,
				/* returned */
				struct nw_info_struct2* target, size_t sizeoftarget);

NWCCODE
ncp_ns_modify_entry_dos_info(struct ncp_conn* conn,
				    /* entry info */
				    unsigned int ns,
				    unsigned int search_attributes,
				    int dirstyle,
				    unsigned int vol,
				    u_int32_t dirent,
				    const unsigned char* encpath, size_t pathlen,
				    /* what to do with entry */
				    u_int32_t mim,
				    const struct ncp_dos_info* info
				    /* nothing to return */
				    /* nothing returned */
				    );

NWCCODE
ncp_ns_obtain_namespace_info_format(struct ncp_conn* conn,
				  /* entry info */
				  unsigned int vol,
				  /* what to do with entry */
				  unsigned int target_ns,
				  /* returned */
				  struct ncp_namespace_format* format, size_t sizeofformat);

NWCCODE
ncp_ns_obtain_entry_namespace_info(struct ncp_conn* conn,
					  /* entry info */
					  unsigned int source_ns,
					  unsigned int vol,
					  u_int32_t dirent,
					  /* what to do with entry */
					  unsigned int target_ns,
					  /* rim */
					  u_int32_t nsrim,
					  /* returned */
					  void* buffer, size_t* len, size_t maxlen);
					  
NWCCODE
ncp_ns_modify_entry_namespace_info(struct ncp_conn* conn,
					  /* entry info */
					  unsigned int source_ns,
					  unsigned int vol,
					  u_int32_t dirent,
					  /* what to do with entry */
					  unsigned int target_ns,
					  /* rim */
					  u_int32_t nsrim,
					  /* returned */
					  const void* buffer, size_t buflen);
					  
NWCCODE
ncp_ns_get_namespace_info_element(const struct ncp_namespace_format* nsformat,
			       u_int32_t nsrim, 
			       const void* buffer,
			       size_t bufferlen,
			       unsigned int itemid,
			       void* itembuffer, size_t* itemlen, size_t itemmaxlen);

NWCCODE
ncp_ns_alloc_short_dir_handle(struct ncp_conn* conn,
			      /* input */
			      unsigned int ns,
			      int dirstyle,
			      unsigned int vol,
			      u_int32_t dirent,
			      const unsigned char* encpath, size_t pathlen,
			      /* alloc short dir handle specific */
			      unsigned int allocate_mode,
			      /* output */
			      NWDIR_HANDLE* dirhandle,
			      NWVOL_NUM* ovol);
			 
typedef struct ncp_directory_list_handle* NWDIRLIST_HANDLE;

NWCCODE
ncp_ns_search_init(struct ncp_conn* conn,
		   /* input */
		   unsigned int ns,
		   unsigned int search_attributes,
		   unsigned int dirstyle,
		   unsigned int vol,
		   NWDIR_ENTRY dirent,
		   const unsigned char* encpath, size_t enclen,
		   /* search specific */
		   int datastream,
		   const unsigned char* pattern, size_t patlen,
		   u_int32_t rim,
		   /* handle */
		   NWDIRLIST_HANDLE* handle);
		   
NWCCODE
ncp_ns_search_next(NWDIRLIST_HANDLE handle,
		   struct nw_info_struct2* target, size_t sizeoftarget);

NWCCODE
ncp_ns_search_end(NWDIRLIST_HANDLE handle);

typedef struct {
	NWObjectID	objectID;
	u_int16_t	objectRights;
} TRUSTEE_INFO;

NWCCODE
ncp_ns_trustee_add(struct ncp_conn* conn,
		   /* input */
		   unsigned int ns,
		   unsigned int search_attributes,
		   int dirstyle,
		   unsigned int vol,
		   u_int32_t dirent,
		   const unsigned char* encpath, size_t pathlen,
		   /* trustee add specific */
		   const TRUSTEE_INFO* trustees,
		   unsigned int object_count,
		   u_int16_t rights_mask);

NWCCODE
ncp_ns_trustee_del(struct ncp_conn* conn,
		   /* input */
		   unsigned int ns,
		   int dirstyle,
		   unsigned int vol,
		   u_int32_t dirent,
		   const unsigned char* encpath, size_t pathlen,
		   /* trustee del specific */
		   const TRUSTEE_INFO* trustees,
		   unsigned int object_count);

NWCCODE
ncp_ns_trustee_scan(struct ncp_conn* conn,
		    /* input */
		    unsigned int ns,
		    unsigned int search_attributes,
		    int dirstyle,
		    unsigned int vol,
		    u_int32_t dirent,
		    const unsigned char* encpath, size_t pathlen,
		    /* trustee scan specific */
		    u_int32_t* iter,
		    TRUSTEE_INFO* trustees,
		    unsigned int* object_count);

typedef struct ncp_volume_list_handle* NWVOL_HANDLE;

NWCCODE
ncp_volume_list_init(struct ncp_conn* conn,
		     /* input */
		     unsigned int ns,
		     unsigned int reqflags,
		     /* output */
		     NWVOL_HANDLE* handle);
		     
NWCCODE
ncp_volume_list_next(NWVOL_HANDLE handle,
		     unsigned int *volnum,
		     char* volname, size_t maxlen);
		     
NWCCODE
ncp_volume_list_end(NWVOL_HANDLE handle);
		     
//NWCCODE ncp_extract_file_info2(u_int32_t rim, const u_int8_t*, size_t,
//		struct nw_info_struct2*);

int
ncp_get_mount_uid(int fid, uid_t* uid);

void
com_err(const char* program, int error, const char* msg, ...);

#ifdef __MAKE_SULIB__
long
ncp_renegotiate_connparam(struct ncp_conn* conn, int buffsize, int options);
#endif	/* __MAKE_SULIB__ */

#ifdef __cplusplus
}
#endif

#endif	/* _NCPLIB_H */
