//////////////////////////////////////////////////////////////////
//
// GkStatus.h	thread listening for connections to receive
//				status updates from the gatekeeper
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// History:
// 	990913	initial version (Jan Willamowius)
//	991025	Added command thread (Ashley Unitt)
//
//////////////////////////////////////////////////////////////////

#ifndef GKSTATUS_H
#define GKSTATUS_H "#(@) $Id: GkStatus.h,v 1.8.6.14 2004/05/12 17:46:33 zvision Exp $"

#if (_MSC_VER >= 1200)  
#pragma warning( disable : 4786 ) // warning about too long debug symbol off
#endif

#include <list>
#include "singleton.h"

class PTelnetSocket;
class GkStatus : public PThread, public Singleton<GkStatus>
{
	PCLASSINFO(GkStatus, PThread)
public:
	GkStatus();
	virtual ~GkStatus();

	void Initialize(const PIPSocket::Address &);

	void Close(void);

	/** controls wether there may be a client to delete or not */
	void SetDirty(bool isDirty) { m_IsDirty = isDirty; }
	bool IsDirty() const { return m_IsDirty; }

	/** called frequently to erase clients from the list */
	void CleanupClients();

	/** #level# is the 'status trace level'  */
	void SignalStatus(const PString & Message, int level = 0);


	enum enumCommands {
		e_PrintAllRegistrations,
		e_PrintAllRegistrationsVerbose,/// extra line per reg starting with '#'. yeah #.
		e_PrintAllCached,
		e_PrintCurrentCalls,
		e_PrintCurrentCallsVerbose,    /// extra line per call starting with '#'. yeah #.
		e_Find,                        /// find an endpoint
		e_FindVerbose,
		e_DisconnectIp,                /// disconnect a call by endpoint IP number
		e_DisconnectAlias,             /// disconnect a call by endpoint alias
		e_DisconnectCall,              /// disconnect a call by call number
		e_DisconnectEndpoint,          /// disconnect a call by endpoint ID
		e_DisconnectSession,		   /// disconnect a user from status port
		e_ClearCalls,                  /// disconnect all calls
		e_UnregisterAllEndpoints,      /// force unregisterung of all andpoints
		e_UnregisterAlias,             /// force unregisterung of one andpoint by alias
		e_UnregisterIp,                /// force unregisterung of one andpoint by IP number
		e_TransferCall,                /// transfer call from one endpoint to another
		e_MakeCall,                    /// establish a new call from endpoint A to endpoint B
		e_Yell,                        /// write a message to all status clients
		e_Who,                         /// list who is logged on at a status port
		e_GK,                          /// show my parent gatekeeper
		e_Help,                        /// List all commands
		e_Version,                     /// GkStatus Protocol Info
		e_Debug,                       /// Debugging commands
		e_Statistics,                  /// Show Statistics
		e_Exit,                        /// Close Connection
		e_Reload,                      /// Reload Config File
		e_Shutdown,                    /// Shutdown the program
		e_RouteToAlias,                /// Route a call upon ARQ to a specified alias eg. a free CTI agent
		e_RouteReject                  /// Reject to Route a call upon ARQ (send ARJ)
		/// Number of different strings
	};
	static const int NumberOfCommandStrings;


// Visual C++ doesn't grok this:
// protected:
	// Client class handles status commands
	class Client : public PThread
	{
		PCLASSINFO(Client, PThread)
	public:
		Client( GkStatus * _StatusThread, PTelnetSocket * _Socket );
		virtual ~Client();

		/* 
		BOOL Write( const char * Message, size_t MsgLen ); 
		*/
		bool WriteString(const PString & Message, int level = 0);
		bool Close();
		
		PString WhoAmI() const;
	
		int GetInstanceNo() const { return InstanceNo; }	
		int GetTraceLevel() const { return TraceLevel; }
		bool IsDeletable() const { return PleaseDelete; }
		bool IsLoggedIn() const { return LoggedIn; }

	private:
		int	InstanceNo;
		int	TraceLevel;
		int	ErrorCnt;
		bool	PleaseDelete;
		bool	LoggedIn;
		PString	UserLoggedIn;
		static int   StaticInstanceNo;

		/** mutex to protect writing to the socket */
		PMutex       Mutex;

		virtual void Main();

		bool AuthenticateClient();
		bool CheckRule(const PString &);
		bool AuthenticateUser();
		PString GetPassword(const PString & UserName) const;

		bool ReadCommand(PString& cmd);

		PTelnetSocket * Socket;
		GkStatus   * StatusThread;
		
		/// prints out the GkStatus commands to the client
		void PrintHelp();

		/// handles the 'Debug' command. #Args# is the whole tokenised command line.
		void DoDebug(const PStringArray &Args);

		/// map for fast (and easy) 'parsing' the commands from the user
		static PStringToOrdinal Commands;

	};

	/** disconnect a given session  */
	bool DisconnectSession(int session, Client *);

	void ShowUsers(Client *) const;

private:
	virtual void Main();

	PIPSocket::Address GKHome;
	PTCPSocket StatusListener;
	std::list<Client *> Clients;
	mutable PReadWriteMutex ClientSetLock;
	bool m_IsDirty;
};

#endif // GKSTATUS_H
