/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "BasicConfigDialog.h"
#include "Application.h"
#include "OperationLog.h"
#include "GlobalState.h"
#include "Conf.h"
#include "ConfigFile.h"
#include "StringUtils.h"
#include "ProxyDescriptor.h"
#include "InetAddr.h"
#include "EffectiveFileTimestamps.h"
#include "types.h"
#include <ace/config.h>
#include <boost/lexical_cast.hpp>
#include <wx/textctrl.h>
#include <wx/checkbox.h>
#include <wx/sizer.h>
#include <wx/msgdlg.h>
#include <list>
#include <sstream>
#include <time.h>

using namespace std;

BEGIN_EVENT_TABLE(BasicConfigDialog, wxDialog)
	EVT_CLOSE(onWindowClose)
	EVT_BUTTON(wxID_OK, onApply)
END_EVENT_TABLE()

BasicConfigDialog* BasicConfigDialog::m_spInstance = 0;

BasicConfigDialog::BasicConfigDialog()
:	wxDialog(wxGetApp().GetTopWindow(), -1, _T("Basic Configuration"), wxDefaultPosition),
	m_pListenPort(0),
	m_pForwarding(0),
	m_pProxyHost(0),
	m_pProxyPort(0)
{
	wxBoxSizer *topsizer = new wxBoxSizer(wxVERTICAL);
	SetSizer(topsizer);
	
	wxBoxSizer *hsz1 = new wxBoxSizer(wxHORIZONTAL);
	topsizer->Add(hsz1, 0, wxALIGN_CENTRE|wxALL, 5 );
	hsz1->Add(new wxStaticText(this, -1, _T("Listen Port")), 0, wxALIGN_CENTRE|wxALL, 5);
	m_pListenPort = new wxTextCtrl(this, -1, wxEmptyString, wxDefaultPosition, wxSize(45, -1));
	hsz1->Add(m_pListenPort, 0, wxALIGN_CENTRE|wxALL, 5);
	
	wxStaticBoxSizer *vsz2 = new wxStaticBoxSizer(
		new wxStaticBox(this, -1, _T("Forwarding")),
		wxVERTICAL
	);
	topsizer->Add(vsz2, 0, wxGROW|wxALIGN_CENTER_VERTICAL|wxLEFT|wxRIGHT, 5);
	m_pForwarding = new wxCheckBox(this, -1, _T("Forward to another proxy"));
	vsz2->Add(m_pForwarding, 0, wxALIGN_CENTER_VERTICAL|wxALL, 3);
	
	wxBoxSizer *hsz21 = new wxBoxSizer(wxHORIZONTAL);
	vsz2->Add(hsz21, 0, wxALIGN_CENTRE|wxALL, 2);
	m_pProxyHost = new wxTextCtrl(this, -1, wxEmptyString, wxDefaultPosition, wxSize(120,-1));
	hsz21->Add(m_pProxyHost, 0, wxALIGN_CENTRE|wxALL, 5);
	hsz21->Add(new wxStaticText(this, -1, _T(":")), 0, wxALIGN_CENTRE);
	m_pProxyPort = new wxTextCtrl(this, -1, wxEmptyString, wxDefaultPosition, wxSize(45,-1));
	hsz21->Add(m_pProxyPort, 0, wxALIGN_CENTRE|wxALL, 5 );
	
	m_pTrayAnimation = new wxCheckBox(this, -1, _T("Tray icon animation"));
	topsizer->Add(m_pTrayAnimation, 0, wxLEFT|wxRIGHT|wxTOP, 5);
	
	wxBoxSizer *hsz3 = new wxBoxSizer(wxHORIZONTAL);
	topsizer->Add(hsz3, 0, wxALIGN_CENTRE|wxALL, 5);
	hsz3->Add(new wxButton(this, wxID_OK, _T("OK")), 0, wxALIGN_CENTRE|wxALL, 5);
	wxButton* cancel_btn = new wxButton(this, wxID_CANCEL, _T("Cancel"));
	hsz3->Add(cancel_btn, 0, wxALIGN_CENTRE|wxALL, 5);
	cancel_btn->SetDefault();
	
	topsizer->SetSizeHints(this);
	
	loadConfig();
}

BasicConfigDialog::~BasicConfigDialog()
{
}

void
BasicConfigDialog::show()
{
	if (!m_spInstance) {
		m_spInstance = new BasicConfigDialog;
		m_spInstance->Show();
		m_spInstance->m_pListenPort->SetSelection(0, 0);
		// Why the hell it gets selected?
		// And why unselecting it only works after Show()?
	} else {
		m_spInstance->Show();
		m_spInstance->Raise();
	}
}

void
BasicConfigDialog::onWindowClose(wxCloseEvent& evt)
{
	if (m_spInstance == this) {
		m_spInstance = 0;
	}
	Destroy();
}

void
BasicConfigDialog::loadConfig()
{
	Config config = GlobalState::ReadAccessor()->config();
	list<InetAddr> listen_addrs = config.getListenAddresses();
	if (listen_addrs.size() > 1) {
		m_pListenPort->SetValue(wxEmptyString);
		showWarning(
			_T("Your current config has multiple listen addresses.\n")
			_T("You should be using the Advanced Configuration dialog.")
		);
	} else if (listen_addrs.size() == 1) {
		wxString val;
		val << listen_addrs.front().getPort();
		m_pListenPort->SetValue(val);
	} else {
		m_pListenPort->SetValue(wxEmptyString);
	}
	m_pTrayAnimation->SetValue(config.isTrayAnimationEnabled());
	m_pForwarding->SetValue(config.isNextHopProxyEnabled());
	ProxyDescriptor pdesc = config.getNextHopProxy();
	m_pProxyHost->SetValue(pdesc.getProxyAddr().getHost().c_str());
	if (pdesc.getProxyAddr().getPort() <= 0) {
		m_pProxyPort->SetValue(wxEmptyString);
	} else {
		wxString val;
		val << pdesc.getProxyAddr().getPort();
		m_pProxyPort->SetValue(val);
	}
}

void
BasicConfigDialog::onApply(wxCommandEvent& evt)
{
	bool forwarding = m_pForwarding->GetValue();
	uint16_t listen_port = 0;
	try {
		string port = StringUtils::trimStr(m_pListenPort->GetValue().mb_str());
		listen_port = boost::lexical_cast<uint16_t>(port);
	} catch (boost::bad_lexical_cast& e) {}
	if (listen_port == 0) {
		showError(_T("Listen port must be an integer in range of [1, 65535]"));
		m_pListenPort->SetFocus();
		return;
	}
	
	string proxy_host = StringUtils::trimStr(m_pProxyHost->GetValue().mb_str());	
	if (forwarding && proxy_host.empty()) {
		showError(_T("Remote proxy host is not given"));
		m_pProxyHost->SetFocus();
		return;
	}
	
	uint16_t proxy_port = 0;
	try {
		string port = StringUtils::trimStr(m_pProxyPort->GetValue().mb_str());
		proxy_port = boost::lexical_cast<uint16_t>(port);
	} catch (boost::bad_lexical_cast& e) {}
	if (forwarding && proxy_port == 0) {
		showError(_T("Remote proxy port must be an integer in range of [1, 65535]"));
		m_pProxyPort->SetFocus();
		return;
	}
	
	Config const old_config(GlobalState::ReadAccessor()->config());
	Config new_config(old_config);
	list<InetAddr> listen_addrs;
	listen_addrs.push_back(InetAddr("127.0.0.1", listen_port));
	new_config.setListenAddresses(listen_addrs);
	new_config.setTrayAnimationEnabled(m_pTrayAnimation->GetValue());
	new_config.setNextHopProxyEnabled(forwarding);
	InetAddr proxy_addr(proxy_host, -1);
	if (proxy_port > 0) {
		proxy_addr.setPort(proxy_port);
	}
	new_config.setNextHopProxy(ProxyDescriptor(proxy_addr));
	
	Application& app = wxGetApp();
	Log* log = OperationLog::instance();
	log->appendRecord(_T("Applying new config ... "));
	size_t num_records = log->getNumRecords();
#if 0
	if (new_config == old_config) {
		log->appendToLastRecord(
			_T("no changes"),
			log->getSuccessStyle()
		);
		Close();
		return;
	}
#endif
	
	if (!app.applyConfig(new_config)) {
		restoreOldConfig(old_config);
		app.showLogDialog();
		return;
	}
	
	ConfigFile new_config_file(app.configFile());
	new_config_file.updateWith(new_config);
	ostringstream content;
	content << new_config_file;
	
	time_t mtime = 0;
	if (!app.writeFile(app.getConfigFileName(), content.str(), &mtime)) {
		restoreOldConfig(old_config);
		app.showLogDialog();
		return;
	}
	
	app.configFile().swap(new_config_file);
	EffectiveFileTimestamps::config = mtime;
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord(_T("done"), log->getSuccessStyle());
	}
	
	Close();
}

bool
BasicConfigDialog::restoreOldConfig(Config const& config)
{
	Application& app = wxGetApp();
	Log* log = OperationLog::instance();
	log->appendRecord(_T("Restoring the old config ... "));
	size_t num_records = log->getNumRecords();
	
	if (!app.applyConfig(config)) {
		return false;
	}
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord(_T("done"), log->getSuccessStyle());
	}
	return true;
}

void
BasicConfigDialog::showError(wxString const& msg)
{
	wxMessageDialog* dialog = new wxMessageDialog(
		this, msg, _T("Error"), wxOK|wxICON_ERROR
	);
	dialog->ShowModal();
}

void
BasicConfigDialog::showWarning(wxString const& msg)
{
	wxMessageDialog* dialog = new wxMessageDialog(
		this, msg, _T("Warning"), wxOK|wxICON_EXCLAMATION
	);
	dialog->ShowModal();
}
