/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "DebugConnection.h"
#include "AutoScrollingWindow.h"
#include "DebugLayout.h"
#include "DebugMessage.h"
#include "ClientRequestDescriptor.h"
#include "RefCountable.h"
#include "RefCounter.h"
#include "SplittableBuffer.h"
#include "StringUtils.h"
#include <sigc++/sigc++.h>
#include <gtkmm/window.h>
#include <gtkmm/fixed.h>
#include <gtkmm/drawingarea.h>
#include <gtkmm/notebook.h>
#include <gtkmm/textbuffer.h>
#include <gtkmm/textview.h>
#include <gtkmm/texttag.h>
#include <gtkmm/frame.h>
#include <gtkmm/button.h>
#include <gtkmm/checkbutton.h>
#include <gtkmm/comboboxtext.h>
#include <gtkmm/entry.h>
#include <gtkmm/box.h>
#include <gtkmm/alignment.h>
#include <gtkmm/adjustment.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/filechooserdialog.h>
#include <gtkmm/stock.h>
#include <gdkmm/color.h>
#include <memory>
#include <fstream>
#include <algorithm>
#include <stddef.h>

using namespace std;

class DebugConnection::Traffic :
	public RefCountable<RefCounter<ACE_NULL_SYNCH> >
{
public:
	SplittableBuffer data;
	size_t totalSize;
	size_t storedSize; // equals to data.size()
	sigc::signal<void> updateSignal;
	
	Traffic();
	
	virtual ~Traffic();
};


class DebugConnection::Channel
:	public RefCountable<RefCounter<ACE_NULL_SYNCH> >
{
public:
	Channel(int left);
	
	virtual ~Channel();
	
	int getLeft() const { return m_left; }
	
	void growBy(int increment);
	
	void ensureClosed();
	
	void consumeTraffic(SplittableBuffer& new_data, size_t traf_limit);
	
	DebugMessage* getLastMessage() const { return m_pLastMessage; }
	
	void setLastMessage(DebugMessage* msg) { m_pLastMessage = msg; }
	
	IntrusivePtr<Traffic> const& getTraffic() const { return m_ptrTraffic; }
private:
	int m_left;
	DebugMessage* m_pLastMessage;
	IntrusivePtr<Traffic> m_ptrTraffic;
};


class DebugConnection::ViewWindow : public Gtk::Window
{
public:
	ViewWindow(Glib::RefPtr<Gtk::TextBuffer> const& debug_text);
	
	virtual ~ViewWindow();
	
	ViewWindow* access() { return this; }
	
	void addTrafficChannel(
		Glib::ustring const& name,
		IntrusivePtr<Traffic> const& traffic);
	
	void openFirstPage();
private:
	virtual void on_hide();
	
	Gtk::Notebook m_notebook;
	Gtk::TextView m_dbgTextView;
	AutoScrollingWindow m_dbgTextScrollWnd;
};


class DebugConnection::TrafficPage : public Gtk::Frame
{
public:
	TrafficPage(IntrusivePtr<Traffic> const& traffic);
	
	virtual ~TrafficPage();
private:
	typedef sigc::slot<SaveTrafficDialog*> SaveDialogAccessor;
	
	void onTrafficUpdate();
	
	void onEncodingChange();
	
	void updateTraffic();
	
	void writeTraffic();
	
	void showSaveDialog();
	
	static std::string trafficToString(size_t traffic);
	
	IntrusivePtr<Traffic> m_ptrTraffic;
	Glib::RefPtr<Gtk::TextBuffer> m_ptrTrafBuffer;
	Glib::RefPtr<Gtk::TextTag> m_ptrSkippedTag;
	Gtk::TextView m_trafView;
	Gtk::ScrolledWindow m_trafScrollWindow;
	Gtk::Button m_updateButton;
	Gtk::Button m_saveButton;
	Gtk::ComboBoxText m_encodingCombo;
	Gtk::Alignment m_encodingComboAlignment;
	Gtk::Label m_trafTotalSizeLabel;
	Gtk::Label m_trafStoredSizeLabel;
	Gtk::VBox m_vbox1;
	Gtk::HBox m_hbox1;
	Gtk::HBox m_hbox2;
	SaveDialogAccessor m_saveDialogAccessor;
};


class DebugConnection::SaveTrafficDialog : public Gtk::FileChooserDialog
{
public:
	SaveTrafficDialog(IntrusivePtr<Traffic> const& traffic);
	
	virtual ~SaveTrafficDialog();
	
	SaveTrafficDialog* access() { return this; }
private:
	virtual void on_hide();
	
	void onResponse(int response);
	
	bool saveToFile(std::string const& fname);
	
	IntrusivePtr<Traffic> m_ptrTraffic;
	Gtk::CheckButton m_blockBoundriesCB;
};


/* ========================= DebugConnection ============================*/

DebugConnection::DebugConnection(DebugLayout& layout, int top, int left, int right)
:	m_rLayout(layout),
	m_top(top),
	m_left(left),
	m_right(right),
	m_height(DebugLayout::CONN_INITIAL_HEIGHT),
	m_isClosed(false),
	m_ptrDebugText(Gtk::TextBuffer::create())
{
	typedef DebugLayout DL;
	
	add(m_container);
	m_container.set_size_request(DL::CONN_WIDTH, m_height);
	m_container.show();
	
	int offset = DL::CONN_BORDER + DL::CONN_PADDING;
	for (int i = 0; i < NUM_CHANNELS; ++i) {
		m_channels[i].reset(new Channel(offset));
		offset += DL::MSG_OUTER_WIDTH + DL::MSG_HOR_SPACING;
	}
	
	Gdk::Color border_color("#ac40ef");
	for (int i = 0; i < 4; ++i) {
		m_borders[i].modify_bg(Gtk::STATE_NORMAL, border_color);
		if (i != 3) {
			m_borders[i].show();
		}
	}
	for (int i = 0; i < 2; ++i) {
		// 0 and 1 are vertical borders
		m_borders[i].set_size_request(DL::CONN_BORDER, m_height - DL::CONN_BORDER*2);
	}
	for (int i = 2; i < 4; ++i) {
		// 2 and 3 are horizontal borders, full width
		m_borders[i].set_size_request(DL::CONN_WIDTH, DL::CONN_BORDER);
	}
	
	m_container.put(m_borders[0], 0, DL::CONN_BORDER);
	m_container.put(m_borders[1], DL::CONN_WIDTH - DL::CONN_BORDER, DL::CONN_BORDER);
	m_container.put(m_borders[2], 0, DebugLayout::CONN_BORDER);
	m_container.put(m_borders[3], 0, m_height);
	
	signal_button_press_event().connect(
		sigc::mem_fun(*this, &DebugConnection::onButtonPress)
	);
	
	show();
}

DebugConnection::~DebugConnection()
{
	m_destroySignal.emit();
}

void
DebugConnection::growBy(int increment)
{
	typedef DebugLayout DL;
	
	m_height += increment;
	m_container.set_size_request(DL::CONN_WIDTH, m_height);
	for (int i = 0; i < 2; ++i) {
		m_borders[i].set_size_request(DL::CONN_BORDER, m_height - DL::CONN_BORDER*2);
	}
	m_container.move(m_borders[3], 0, m_height - DL::CONN_BORDER);
	
	for (int i = 0; i < NUM_CHANNELS; ++i) {
		m_channels[i]->growBy(increment);
	}
}

void
DebugConnection::finish(bool error)
{
	typedef DebugLayout DL;
	
	if (m_isClosed) {
		return;
	}
	
	for (int i = 0; i < NUM_CHANNELS; ++i) {
		m_channels[i]->ensureClosed();
	}
	
	m_rLayout.growBy(DL::CONN_PADDING + DL::CONN_BORDER);
	
	m_isClosed = true;
	m_rLayout.setNextObjectBeginPos(getBottom());
	m_rLayout.setNextObjectEndPos(getBottom());
	
	if (!error) {
		m_borders[3].show();
	}
}

void
DebugConnection::showMessagesInRange(int top, int bottom)
{
	typedef DebugLayout DL;
	
	int left = std::max(m_left - DL::CONN_HOR_SPACING, 0);
	int right = std::min(m_right + DL::CONN_HOR_SPACING, m_rLayout.getWidth());
	m_rLayout.get_hadjustment()->clamp_page(left, right);
	
	int abs_top = m_top + top;
	int abs_bottom = m_top + bottom;
	abs_top = std::max(abs_top - DL::MSG_VER_SPACING, 0);
	abs_bottom = std::min(abs_bottom + DL::MSG_VER_SPACING, m_rLayout.getHeight());
	m_rLayout.get_vadjustment()->clamp_page(abs_top, abs_bottom);
}

void
DebugConnection::addClientRequestDescriptor(
	int id, IntrusivePtr<ClientRequestDescriptor> const& desc)
{
	m_requestMap.insert(RequestMap::value_type(id, desc));
}

void
DebugConnection::httpMessageBegin(int request_id,
	HttpMessageType type, std::string const& headers)
{
	typedef DebugLayout DL;
	
	Channel& channel = *m_channels[getChannelFor(type)];
	channel.ensureClosed();
	DebugMessage* last_msg = channel.getLastMessage();
		
	int top = m_rLayout.getNextObjectBeginPos() - m_top;
	// top is relative to m_top
	if (last_msg) {
		top = std::max(top, last_msg->getBottom() + DL::MSG_VER_SPACING);
	}
	
	int bottom = top + DL::MSG_INITIAL_OUTER_HEIGHT;
	int grow_by = bottom - m_height;
	if (grow_by > 0) {
		m_rLayout.growBy(grow_by);
	}
	
	DebugMessage* msg = manage(new DebugMessage(
		this, top, headers, type,
		getClientRequestDescriptor(request_id)
	));
	m_container.put(*msg, channel.getLeft(), top);
	
	channel.setLastMessage(msg);
	
	int next_pos = m_top + top + DL::MSG_SHIFT;
	m_rLayout.setNextObjectBeginPos(next_pos);
	m_rLayout.setNextObjectEndPos(next_pos);
}

void
DebugConnection::httpMessageEnd(HttpMessageType type, bool error)
{
	typedef DebugLayout DL;
	
	Channel& channel = *m_channels[getChannelFor(type)];
	DebugMessage* msg = channel.getLastMessage();
	
	if (!msg || !msg->isOpen()) {
		return;
	}
	
	int grow_by = m_rLayout.getNextObjectEndPos() - (m_top + msg->getBottom());
	if (grow_by > 0) {
		m_rLayout.growBy(grow_by); // this alters last_msg->getBottom();
	}
	msg->finish(error);
	
	int next_begin = m_top + msg->getBottom();
	m_rLayout.setNextObjectBeginPos(next_begin);
	m_rLayout.setNextObjectEndPos(next_begin + DL::MSG_SHIFT);
}

void
DebugConnection::logMessage(std::string const& msg)
{
	try {
		Glib::ustring umsg = Glib::convert(msg, "utf-8", "iso-8859-1");
		umsg += '\n';
		m_ptrDebugText->insert(m_ptrDebugText->end(), umsg);
	} catch (Glib::ConvertError&) {}
}

void
DebugConnection::logTraffic(SplittableBuffer& traf, TrafficDirection dir)
{
	Channel& channel = *m_channels[getChannelFor(dir)];	
	channel.consumeTraffic(traf, m_rLayout.getTrafficLimit());
}

int
DebugConnection::getChannelFor(HttpMessageType msg_type)
{
	typedef AbstractDebugAgent ADA;
	switch (msg_type) {
		case ADA::INCOMING_REQUEST:
		return INCOMING_REQUEST_CHANNEL;
		case ADA::OUTGOING_REQUEST:
		return OUTGOING_REQUEST_CHANNEL;
		case ADA::INCOMING_RESPONSE:
		return INCOMING_RESPONSE_CHANNEL;
		case ADA::SCRIPT_REQUEST:
		case ADA::SCRIPT_RESPONSE:
		return SCRIPT_CHANNEL;
		default:
		return OUTGOING_RESPONSE_CHANNEL;
	}
}

int
DebugConnection::getChannelFor(TrafficDirection traf_dir)
{
	typedef AbstractDebugAgent ADA;
	switch (traf_dir) {
		case ADA::FROM_CLIENT:
		return INCOMING_REQUEST_CHANNEL;
		case ADA::TO_CLIENT:
		return OUTGOING_RESPONSE_CHANNEL;
		case ADA::FROM_SERVER:
		return INCOMING_RESPONSE_CHANNEL;
		case ADA::TO_SERVER:
		return OUTGOING_REQUEST_CHANNEL;
		default:
		return SCRIPT_CHANNEL;
	}
}

bool
DebugConnection::onButtonPress(GdkEventButton* event)
{
	if (event->button == 1 && event->type == GDK_2BUTTON_PRESS) {
		if (ViewWindow* wnd = m_viewWindowAccessor()) {
			wnd->present();
		} else {
			createViewWindow();
		}
	}
	return true; // event has been handled
}

void
DebugConnection::createViewWindow()
{
	auto_ptr<ViewWindow> wnd(new ViewWindow(m_ptrDebugText));
	m_viewWindowAccessor = sigc::mem_fun(
		*wnd, &DebugConnection::ViewWindow::access
	);
	
	wnd->addTrafficChannel(
		"From Client",
		m_channels[INCOMING_REQUEST_CHANNEL]->getTraffic()
	);
	wnd->addTrafficChannel(
		"To Server",
		m_channels[OUTGOING_REQUEST_CHANNEL]->getTraffic()
	);
	wnd->addTrafficChannel(
		"From Server",
		m_channels[INCOMING_RESPONSE_CHANNEL]->getTraffic()
	);
	wnd->addTrafficChannel(
		"To Client",
		m_channels[OUTGOING_RESPONSE_CHANNEL]->getTraffic()
	);
	wnd->addTrafficChannel(
		"Scripts",
		m_channels[SCRIPT_CHANNEL]->getTraffic()
	);
	wnd->openFirstPage();
	
	wnd->show();
	wnd.release();
}

IntrusivePtr<ClientRequestDescriptor>
DebugConnection::getClientRequestDescriptor(int id) const
{
	RequestMap::const_iterator it = m_requestMap.find(id);
	if (it != m_requestMap.end()) {
		return it->second;
	}
	return IntrusivePtr<ClientRequestDescriptor>();
}


/*=====================DebugConenction::Traffic ========================*/

DebugConnection::Traffic::Traffic()
:	totalSize(0),
	storedSize(0)
{
}

DebugConnection::Traffic::~Traffic()
{
}


/*==================== DebugConnection::Channel ========================*/

DebugConnection::Channel::Channel(int left)
:	m_left(left),
	m_pLastMessage(0),
	m_ptrTraffic(new Traffic)
{
}

DebugConnection::Channel::~Channel()
{
}

void
DebugConnection::Channel::growBy(int increment)
{
	if (m_pLastMessage && m_pLastMessage->isOpen()) {
		m_pLastMessage->growBy(increment);
	}
}

void
DebugConnection::Channel::ensureClosed()
{
	if (m_pLastMessage) {
		m_pLastMessage->ensureClosed();
	}
}

void
DebugConnection::Channel::consumeTraffic(
	SplittableBuffer& new_data, size_t traf_limit)
{
	if (new_data.empty()) {
		return;
	}
	
	size_t size = new_data.size();
	Traffic& traf = *m_ptrTraffic;
	traf.data.appendDestructive(new_data);
	traf.totalSize += size;
	traf.storedSize += size;
	if (traf.storedSize > traf_limit) {
		traf.data.trimFront(traf.storedSize - traf_limit);
		traf.storedSize = traf_limit;
	}
	traf.updateSignal.emit();
}


/*==================== DebugConnection::ViewWindow ====================*/

DebugConnection::ViewWindow::ViewWindow(
	Glib::RefPtr<Gtk::TextBuffer> const& debug_text)
:	m_dbgTextView(debug_text)
{
	set_title("Client Connection");
	set_size_request(500, 350);
	add(m_notebook);
	
	m_dbgTextView.set_editable(false);
	m_dbgTextView.set_cursor_visible(false);
	
	m_dbgTextScrollWnd.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	m_dbgTextScrollWnd.set_shadow_type(Gtk::SHADOW_IN);
	m_dbgTextScrollWnd.add(m_dbgTextView);
	
	m_notebook.append_page(m_dbgTextScrollWnd, "Debug Log");
	
	show_all_children();
}

DebugConnection::ViewWindow::~ViewWindow()
{
}

void
DebugConnection::ViewWindow::addTrafficChannel(
	Glib::ustring const& name,
	IntrusivePtr<Traffic> const& traffic)
{
	m_notebook.append_page(*manage(new TrafficPage(traffic)), name);
}

void
DebugConnection::ViewWindow::openFirstPage()
{
	m_notebook.set_current_page(0);
}

void
DebugConnection::ViewWindow::on_hide()
{
	delete this;
}


/*=================== DebugConnection::TrafficPage ====================*/

DebugConnection::TrafficPage::TrafficPage(IntrusivePtr<Traffic> const& traffic)
:	m_ptrTraffic(traffic),
	m_ptrTrafBuffer(Gtk::TextBuffer::create()),
	m_ptrSkippedTag(m_ptrTrafBuffer->create_tag()),
	m_trafView(m_ptrTrafBuffer),
	m_updateButton("Update"),
	m_saveButton(Gtk::Stock::SAVE),
	m_encodingComboAlignment(1.0, 0.5, 0.0, 0.0),
	m_trafTotalSizeLabel(trafficToString(traffic->totalSize)),
	m_trafStoredSizeLabel(trafficToString(traffic->storedSize))
{
	set_shadow_type(Gtk::SHADOW_NONE);
	add(m_vbox1);
	
	m_vbox1.pack_start(m_hbox1, Gtk::PACK_SHRINK);
	m_hbox1.set_border_width(2);
	m_hbox1.set_spacing(3);
	
	m_hbox1.pack_start(m_hbox2, Gtk::PACK_SHRINK);
	m_hbox2.pack_start(*manage(new Gtk::Label("Traffic size: ")), Gtk::PACK_SHRINK);
	m_hbox2.pack_start(m_trafStoredSizeLabel, Gtk::PACK_SHRINK);
	m_hbox2.pack_start(*manage(new Gtk::Label("/")), Gtk::PACK_SHRINK);
	m_hbox2.pack_start(m_trafTotalSizeLabel, Gtk::PACK_SHRINK);
	m_hbox2.pack_start(*manage(new Gtk::Label(" KB")), Gtk::PACK_SHRINK);
	m_hbox1.pack_start(m_updateButton, Gtk::PACK_SHRINK);
	m_hbox1.pack_start(m_saveButton, Gtk::PACK_SHRINK);
	m_hbox1.pack_start(m_encodingComboAlignment);
	m_encodingComboAlignment.add(m_encodingCombo);
	
	m_encodingCombo.append_text("ISO-8859-1");
	m_encodingCombo.append_text("KOI8-R");
	m_encodingCombo.append_text("CP1251");
	m_encodingCombo.set_active(0);
	/*
	Multibyte encodings such as UTF-8 won't work here.
	The data stream can contain a mix of encodings and even binary data,
	so Glib::convert will just thrown an exception when it encounteres
	invalid sequences of characters. Actually, even single-byte encodings
	somtimes refuse to work.
	*/
	
	m_encodingCombo.signal_changed().connect(
		sigc::mem_fun(*this, &TrafficPage::onEncodingChange)
	);
	
	m_updateButton.signal_clicked().connect(
		sigc::mem_fun(*this, &TrafficPage::updateTraffic)
	);
	m_saveButton.signal_clicked().connect(
		sigc::mem_fun(*this, &TrafficPage::showSaveDialog)
	);
	
	m_ptrSkippedTag->property_foreground() = "#0000ff";
	m_trafView.set_editable(false);
	m_trafView.set_cursor_visible(false);
	
	m_trafScrollWindow.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	m_trafScrollWindow.set_shadow_type(Gtk::SHADOW_IN);
	m_trafScrollWindow.add(m_trafView);
	m_vbox1.pack_start(m_trafScrollWindow);
	
	m_ptrTraffic->updateSignal.connect(
		sigc::mem_fun(*this, &DebugConnection::TrafficPage::onTrafficUpdate)
	);
	
	updateTraffic();
	
	show_all();
}

DebugConnection::TrafficPage::~TrafficPage()
{
}

void
DebugConnection::TrafficPage::onTrafficUpdate()
{
	m_trafTotalSizeLabel.set_text(trafficToString(m_ptrTraffic->totalSize));
	m_trafStoredSizeLabel.set_text(trafficToString(m_ptrTraffic->storedSize));
}

void
DebugConnection::TrafficPage::onEncodingChange()
{
	updateTraffic();
}

void
DebugConnection::TrafficPage::updateTraffic()
{
	m_ptrTrafBuffer->set_text(Glib::ustring());
	
	if (m_ptrTraffic->totalSize > m_ptrTraffic->storedSize) {
		size_t skipped = m_ptrTraffic->totalSize - m_ptrTraffic->storedSize;
		Glib::ustring msg("<");
		msg += StringUtils::fromNumber(skipped);
		msg += " bytes skipped>\n";
		m_ptrTrafBuffer->insert_with_tag(
			m_ptrTrafBuffer->end(), msg, m_ptrSkippedTag
		);
	}
	
	writeTraffic();
	m_updateButton.set_sensitive(false);
}

void
DebugConnection::TrafficPage::writeTraffic()
{
	struct CharTranslator
	{
		char table[256];
		CharTranslator() {
			for (int i = 0; i < 32; i++) {
				table[i] = '?';
			}
			for (int i = 32; i < 256; i++) {
				table[i] = i;
			}
			table[(unsigned char)'\n'] = '\n';
			table[(unsigned char)'\r'] = '\r';
			table[(unsigned char)'\t'] = '\t';
		}
	};
	static CharTranslator const translator;
	
	string traf_str = m_ptrTraffic->data.toString();
	string::iterator it = traf_str.begin();
	string::iterator const end = traf_str.end();
	for (; it != end; ++it) {
		*it = translator.table[(unsigned char)*it];
	}
	
	Glib::ustring traf_ustr;
	try {
		traf_ustr = Glib::convert_with_fallback(
			traf_str, "utf-8",
			m_encodingCombo.get_active_text(), "?"
		);
	} catch (Glib::ConvertError&) {}
	
	m_ptrTrafBuffer->insert(m_ptrTrafBuffer->end(), traf_ustr);
}

void
DebugConnection::TrafficPage::showSaveDialog()
{
	if (SaveTrafficDialog* dlg = m_saveDialogAccessor()) {
		dlg->present();
	} else {
		auto_ptr<SaveTrafficDialog> dlg(
			new SaveTrafficDialog(m_ptrTraffic)
		);
		m_saveDialogAccessor = sigc::mem_fun(
			*dlg, &SaveTrafficDialog::access
		);
		dlg->show();
		dlg.release();
	}
}

std::string
DebugConnection::TrafficPage::trafficToString(size_t traffic)
{
	return StringUtils::fromNumber((traffic + 1023) / 1024);
}


/*================ DebugConnection::SaveTrafficDialog ===================*/

DebugConnection::SaveTrafficDialog::SaveTrafficDialog(
	IntrusivePtr<Traffic> const& traffic)
:	FileChooserDialog("Save Traffic", Gtk::FILE_CHOOSER_ACTION_SAVE),
	m_ptrTraffic(traffic),
	m_blockBoundriesCB("Include block boundry (<<EOB>>) marks")
{
	set_extra_widget(m_blockBoundriesCB);
	add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_OK);
	
	signal_response().connect(
		sigc::mem_fun(*this, &SaveTrafficDialog::onResponse)
	);
}

DebugConnection::SaveTrafficDialog::~SaveTrafficDialog()
{
}

void
DebugConnection::SaveTrafficDialog::on_hide()
{
	delete this;
}

void
DebugConnection::SaveTrafficDialog::onResponse(int response)
{
	if (response == Gtk::RESPONSE_OK) {
		if (saveToFile(get_filename())) {
			hide();
		}
	} else {
		hide(); // will cause the destruction of the dialog
	}
}

bool
DebugConnection::SaveTrafficDialog::saveToFile(std::string const& fname)
{
	bool const write_block_boundries = m_blockBoundriesCB.get_active();
	
	ofstream file(fname.c_str());
	if (!file.good()) {
		Gtk::MessageDialog dlg(
			"Error opening file for writing",
			false, // use_markup
			Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK,
			true // modal
		);
		dlg.run();
		return false;
	}
	
	SplittableBuffer::SpanIterator it = m_ptrTraffic->data.spanBegin();
	SplittableBuffer::SpanIterator const end = m_ptrTraffic->data.spanEnd();
	for (; it != end; ++it) {
		file.write(it->begin(), it->size());
		if (write_block_boundries) {
			file << "<<EOB>>";
		}
	}
	file.flush();
	
	if (file.bad()) {
		Gtk::MessageDialog dlg(
			"Error writing to file",
			false, // use_markup
			Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK,
			true // modal
		);
		dlg.run();
		return false;
	}
	
	return true;
}
