/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "RegexFilterDescriptor.h"
#include "TextPattern.h"
#include "BString.h"
#include <ace/config.h>
#include <boost/regex.hpp>
#include <algorithm>

using namespace std;

struct RegexFilterDescriptor::ApSwap
{
	// on some platforms std::swap doesn't work for auto_ptr
	template<typename T>
	static void swap(auto_ptr<T>& o1, auto_ptr<T>& o2)
	{
		T* p1 = o1.release();
		T* p2 = o2.release();
		o1.reset(p2);
		o2.reset(p1);
	}
};


RegexFilterDescriptor::RegexFilterDescriptor(
	FilterTag const& tag, FilterGroupTag const& group_tag,
	std::string const& name)
:	m_tag(tag),
	m_groupTag(group_tag),
	m_name(name),
	m_order(0),
	m_matchCountLimit(-1),
	m_replacementType(TEXT),
	m_isEnabled(false)
{
}

RegexFilterDescriptor::RegexFilterDescriptor(RegexFilterDescriptor const& other)
:	m_tag(other.m_tag),
	m_groupTag(other.m_groupTag),
	m_name(other.m_name),
	m_order(other.m_order),
	m_matchCountLimit(other.m_matchCountLimit),
	m_ptrUrlPattern(other.m_ptrUrlPattern),
	m_ptrContentTypePattern(other.m_ptrContentTypePattern),
	m_ptrSearchPattern(other.m_ptrSearchPattern),
	m_replacementType(other.m_replacementType),
	m_ifFlag(other.m_ifFlag),
	m_setFlag(other.m_setFlag),
	m_clearFlag(other.m_clearFlag),
	m_isEnabled(other.m_isEnabled)
{
	if (other.m_ptrReplacement.get()) {
		m_ptrReplacement.reset(new string(*other.m_ptrReplacement));
	}
}

RegexFilterDescriptor::~RegexFilterDescriptor()
{
}

void
RegexFilterDescriptor::swap(RegexFilterDescriptor& other)
{
	::swap(m_tag, other.m_tag);
	::swap(m_groupTag, other.m_groupTag);
	std::swap(m_name, other.m_name);
	std::swap(m_order, other.m_order);
	std::swap(m_matchCountLimit, other.m_matchCountLimit);
	::swap(m_ptrUrlPattern, other.m_ptrUrlPattern);
	::swap(m_ptrContentTypePattern, other.m_ptrContentTypePattern);
	::swap(m_ptrSearchPattern, other.m_ptrSearchPattern);
	ApSwap::swap(m_ptrReplacement, other.m_ptrReplacement);
	std::swap(m_replacementType, other.m_replacementType);
	std::swap(m_ifFlag, other.m_ifFlag);
	std::swap(m_setFlag, other.m_setFlag);
	std::swap(m_clearFlag, other.m_clearFlag);
	std::swap(m_isEnabled, other.m_isEnabled);
}

bool
RegexFilterDescriptor::urlMatches(BString const& url) const
{
	if (!m_ptrUrlPattern.get()) {
		return true;
	}
	
	try {
		if (boost::regex_match(url.begin(), url.end(),
		                       m_ptrUrlPattern->regex())) {
			return true;
		}
	} catch (std::runtime_error& e) {}
	
	return false;
}

bool
RegexFilterDescriptor::contentTypeMatches(BString const& ctype) const
{
	if (!m_ptrContentTypePattern.get()) {
		return true;
	}
	
	try {
		if (boost::regex_match(ctype.begin(), ctype.end(),
		                       m_ptrContentTypePattern->regex())) {
			return true;
		}
	} catch (std::runtime_error& e) {}
	
	return false;
}

bool
RegexFilterDescriptor::isValid() const
{
	if (m_ptrUrlPattern.get() && !m_ptrUrlPattern->isValid()) {
		return false;
	}
	if (m_ptrContentTypePattern.get() && !m_ptrContentTypePattern->isValid()) {
		return false;
	}
	if (!(m_ptrSearchPattern.get() && m_ptrSearchPattern->isValid())) {
		return false;
	}
	if (!m_ptrReplacement.get()) {
		return false;
	}
	return true;
}
